/*****************************************************************/
/*  xdr_socket.c                                                 */
/*  by Rick Phillips                                             */
/*****************************************************************/

/*
 * xdr_socket.c, XDR implementation on a socket.
 *
 * This set of routines implements XDR on a socket stream.
 * XDR_ENCODE serializes onto the stream, XDR_DECODE de-serializes
 * from the stream.
 */

/* Include files */
#include <stdio.h>
#include "general.h"
#include "types.h"
#include "xdr.h"

/* Function prototypes */
static bool_t	xdrsocket_getlong();
static bool_t	xdrsocket_putlong();
static bool_t	xdrsocket_getbytes();
static bool_t	xdrsocket_putbytes();
static u_int	xdrsocket_getpos();
static bool_t	xdrsocket_setpos();
static long *	xdrsocket_inline();
static void		xdrsocket_destroy();

/* Ops vector for socket type XDR */
static struct xdr_ops	xdrsocket_ops = {
	xdrsocket_getlong,	/* deseraialize a long int */
	xdrsocket_putlong,	/* seraialize a long int */
	xdrsocket_getbytes,	/* deserialize counted bytes */
	xdrsocket_putbytes,	/* serialize counted bytes */
	xdrsocket_getpos,	/* get offset in the stream */
	xdrsocket_setpos,	/* set offset in the stream */
	xdrsocket_inline,	/* prime stream for inline macros */
	xdrsocket_destroy	/* destroy stream */
};



/****************************************************************************************
	xdrsocket_create

	Purpose:
		Initialize a socket XDR stream.
		
	Argument List (I=Input, O=Output, I/O=Input/Output):
    	I	xdrs		pointer to the XDR structure to be initialized.
    	I	theSocket	an open socket.
    	I	op			XDR_ENCODE to encode, or XDR_DECODE to decode.
						
	Returns:
		Void.
****************************************************************************************/
void xdrsocket_create(register XDR *xdrs, SOCKET theSocket, enum xdr_op op)
{
	xdrs->x_op = op;
	xdrs->x_ops = &xdrsocket_ops;
	xdrs->x_private = (caddr_t)theSocket;
	xdrs->x_handy = 0;
	xdrs->x_base = 0;
}



/****************************************************************************************
	xdrsocket_destroy

	Purpose:
		Destroy a socket XDR stream.
		
	Argument List (I=Input, O=Output, I/O=Input/Output):
    	I	xdrs		pointer to an XDR structure.
						
	Returns:
		Void.
****************************************************************************************/
static void xdrsocket_destroy(register XDR *xdrs)
{
	xdrs->x_op = 0;
	xdrs->x_ops = 0;
	xdrs->x_private = 0;
	xdrs->x_handy = 0;
	xdrs->x_base = 0;
}



/****************************************************************************************
	xdrsocket_getlong

	Purpose:
		Read a long int from a socket XDR stream.
		
	Argument List (I=Input, O=Output, I/O=Input/Output):
    	I	xdrs		pointer to an XDR structure.
    	O	lp			pointer to a "long" to receive data.
						
	Returns:
		TRUE on success, FALSE on failure.
****************************************************************************************/
static bool_t xdrsocket_getlong(XDR *xdrs, long *lp)
{
	int numBytes;
	
	numBytes = ReceiveData((SOCKET)xdrs->x_private, lp, sizeof(long));
	if ((numBytes == SOCKET_ERROR) || (numBytes != sizeof(long))) {
		fprintf(STDERR"xdrsocket_getlong failed (ReceiveData)\n");
		return(FALSE);
	}
	*lp = ntohl(*lp);
	return(TRUE);
}



/****************************************************************************************
	xdrsocket_putlong

	Purpose:
		Write a long int to a socket XDR stream.
		
	Argument List (I=Input, O=Output, I/O=Input/Output):
    	I	xdrs		pointer to an XDR structure.
    	I	lp			pointer to a "long" that contains the data to write.
						
	Returns:
		TRUE on success, FALSE on failure.
****************************************************************************************/
static bool_t xdrsocket_putlong(XDR *xdrs, long *lp)
{
	long myCopy;
	int numBytes;
	
	myCopy = htonl(*lp);
	numBytes = SendData((SOCKET)xdrs->x_private, &myCopy, sizeof(long));
	if ((numBytes == SOCKET_ERROR) || (numBytes != sizeof(long))) {
		fprintf(STDERR"xdrsocket_putlong failed (SendData)\n");
		return(FALSE);
	}
	return(TRUE);
}



/****************************************************************************************
	xdrsocket_getbytes

	Purpose:
		Read bytes from a socket XDR stream.
		
	Argument List (I=Input, O=Output, I/O=Input/Output):
    	I	xdrs		pointer to an XDR structure.
    	O	addr		pointer to the buffer to read into.
    	I	len			length of the buffer.
						
	Returns:
		TRUE on success, FALSE on failure.
****************************************************************************************/
static bool_t xdrsocket_getbytes(XDR *xdrs, caddr_t addr, u_int len)
{
	int numBytes;
	
	numBytes = ReceiveData((SOCKET)xdrs->x_private, addr, len);
	if ((numBytes == SOCKET_ERROR) || (numBytes != (int)len)) {
		fprintf(STDERR"xdrsocket_getbytes failed (ReceiveData)\n");
		return(FALSE);
	}
	return(TRUE);
}



/****************************************************************************************
	xdrsocket_putbytes

	Purpose:
		Write a long int to a socket XDR stream.
		
	Argument List (I=Input, O=Output, I/O=Input/Output):
    	I	xdrs		pointer to an XDR structure.
    	O	addr		pointer to the buffer to write from.
    	I	len			length of the buffer.
						
	Returns:
		TRUE on success, FALSE on failure.
****************************************************************************************/
static bool_t xdrsocket_putbytes(XDR *xdrs, caddr_t addr, u_int len)
{
	int numBytes;
	
	numBytes = SendData((SOCKET)xdrs->x_private, addr, len);
	if ((numBytes == SOCKET_ERROR) || (numBytes != (int)len)) {
		fprintf(STDERR"xdrsocket_putbytes failed (SendData)\n");
		return(FALSE);
	}
	return(TRUE);
}



/****************************************************************************************
	xdrsocket_getpos

	Purpose:
		Get the current postion in the XDR stream buffer.  Since we are streaming
		directly to/from a socket, this has no meaning here.
		
	Argument List (I=Input, O=Output, I/O=Input/Output):
    	I	xdrs		pointer to an XDR structure.
 						
	Returns:
		0 to indicate that this function is meaningless.
****************************************************************************************/
static u_int xdrsocket_getpos(XDR *xdrs)
{
	fprintf(STDERR"xdrsocket_getpos failed (getpos is not defined for sockets)\n");
	return(0);
}



/****************************************************************************************
	xdrsocket_setpos

	Purpose:
		Set a new postion in the XDR stream buffer.  Since we are streaming
		directly to/from a socket, this has no meaning here.
		
	Argument List (I=Input, O=Output, I/O=Input/Output):
    	I	xdrs		pointer to an XDR structure.
    	I	pos			new postion.
 						
	Returns:
		FALSE to indicate that this function is meaningless.
****************************************************************************************/
static bool_t xdrsocket_setpos(XDR *xdrs, u_int pos) 
{
	fprintf(STDERR"xdrsocket_setpos failed (setpos is not defined for sockets)\n");
	return(FALSE);
}



/****************************************************************************************
	xdrsocket_inline

	Purpose:
		Return a pointer to a piece of the XDR stream's internal buffer.  Since we are
		streaming directly to/from a socket, this has no meaning here.
		
	Argument List (I=Input, O=Output, I/O=Input/Output):
    	I	xdrs		pointer to an XDR structure.
    	I	len			length of the "piece" to return.
 						
	Returns:
		NULL to indicate that this function is meaningless.
****************************************************************************************/
static long *xdrsocket_inline(XDR *xdrs, u_int len)
{
	fprintf(STDERR"xdrsocket_inline failed (inline is not defined for sockets)\n");
	return(NULL);
}
