/*****************************************************************/
/*      netsolveproblemdesc.c                                    */
/*      Henri Casanova, Dorian Arnold                                    */
/*****************************************************************/

#include "core.h"
#include "mnemonics.h"
#include "client.h"
#include "proxyjobdesc.h"
#include "proxy.h"


void displayObject(NS_Object*);
void printCFortranCallingSequence(NS_ProblemDesc*);
void printMnemonicSemantic(NS_ProblemDesc*,char*);

/*
 * Displays Information about a problem
 */
int main(int argc, char **argv)
{
  NS_ProblemDesc *pd;
  int i;
  char *nickname;
  int proxy_pid;

  if (argc != 3)
  {
    fprintf(stderr,"Usage: %s <agent hostname> <problem nickname>\n",
            argv[0]);
    exit(-1);
  }
   
  nickname = argv[2];

  /* Start the proxy */
  if (startProxy("NetSolve",argv[1],&proxy_pid,&proxy_port) == -1)
  {
    fprintf(stderr,"Error while starting the proxy\n");
    exit(-1);
  }

  if (netsolveInfo(nickname,&pd)==-1)
  {
    fprintf(stderr,"%s\n",netsolveErrorMessage(ns_errno));
    exit(-1);
  }


  /* Display the problem */
  fprintf(stderr,"-- %s --",pd->nickname);
  fprintf(stderr,pd->description);
  fprintf(stderr,"* %d objects in INPUT\n",pd->nb_input_objects);
  for (i=0;i<pd->nb_input_objects;i++)
  {
    fprintf(stderr," - input %d: ",i);
    displayObject(pd->input_objects[i]);
  }
  fprintf(stderr,"* %d objects in OUTPUT\n",pd->nb_output_objects);
  for (i=0;i<pd->nb_output_objects;i++)
  {
    fprintf(stderr," - output %d: ",i);
    displayObject(pd->output_objects[i]);
  }
  if (pd->nb_matlab_merge != 0)
  {
    fprintf(stderr,"- - - - - - - - - - - - -\n");
    for (i=0;i<pd->nb_matlab_merge;i++)
    {
      fprintf(stderr,"Output Objects %d and %d can be merged.\n",
                     pd->matlab_merge[i].index1,pd->matlab_merge[i].index2);
    }
    fprintf(stderr,"- - - - - - - - - - - - -\n");
  }
  printCFortranCallingSequence(pd);
  freeProblemDesc(pd);
  exit(0);
}

/*
 * printCFortranCallingSequence()
 */
void printCFortranCallingSequence(NS_ProblemDesc *pd)
{
  int i;
  NS_CallingSequence *c = pd->calling_sequence;

  fprintf(stderr,"* Calling sequence from C or Fortran\n");
  fprintf(stderr,"%d arguments\n",c->nb_args);
  for (i=0;i<c->nb_args;i++)
  {
    char *m = strdup(c->arg_to_mnemonic[i]);
    char *buf;

    fprintf(stderr," - Argument #%d:\n",i);
    buf = strtok(m,",\t \0");
    while (buf != NULL)
    {
      printMnemonicSemantic(pd,buf);
      buf = strtok(NULL,"\t ,");
    }
  }
}

/*
 * printMnemonicSemantic()
 */
void printMnemonicSemantic(NS_ProblemDesc *pd,char *mnemonic)
{
  int io = getIO(mnemonic);
  int index = getIndex(mnemonic);

  switch(mnemonic[0])
  {
    case '?':
      fprintf(stderr,"   - ignored\n");
      break;
    case 'I':
      fprintf(stderr,"   - pointer to input object #%d (%s)\n",index,
                 pd->input_objects[index]->name);
      break;
    case 'O':
      fprintf(stderr,"   - pointer to output object #%d (%s)\n",index,
                 pd->output_objects[index]->name);
      break;
    case 'm':
      if (io == INPUT)

        fprintf(stderr,"   - number of rows of input object #%d (%s)\n",
                    index,pd->input_objects[index]->name);
      else
        fprintf(stderr,"   - number of columns of output object #%d (%s)\n",
                    index,pd->output_objects[index]->name);
      break;
    case 'n':
      if (io == INPUT)
        fprintf(stderr,"   - number of columns of input object #%d (%s)\n",
                    index,pd->input_objects[index]->name);
      else
        fprintf(stderr,"   - number of columns of output object #%d (%s)\n",
                    index,pd->output_objects[index]->name);
      break;
    case 'l':
      if (io == INPUT)
        fprintf(stderr,"   - leading dimension of input object #%d (%s)\n",
                    index,pd->input_objects[index]->name);
      else
        fprintf(stderr,"   - leading dimension of output object #%d (%s)\n",
                    index,pd->output_objects[index]->name);
      break;
    case 'f':
      if (io == INPUT)
        fprintf(stderr,"   - number of non-zero values of input object #%d (%s)\n", index,pd->input_objects[index]->name);
      else
        fprintf(stderr,"   - number of non-zero values of output object #%d (%s)\n", index,pd->output_objects[index]->name);
      break;
    case 'i':
      if (io == INPUT)
        fprintf(stderr,"   - column indices of non-zeros of input object #%d (%s)\n", index,pd->input_objects[index]->name);
      else
        fprintf(stderr,"   - column indices of non-zeros of output object #%d (%s)\n", index,pd->output_objects[index]->name);
      break;
   case 'p':
       fprintf(stderr,"   - row pointers of the sparse matrix #%d (%s)\n", index,pd->input_objects[index]->name); 
       break;
    default:
      break;
  }
}

/*
 * displayObject()
 */
void displayObject(NS_Object *obj)
{
  switch(obj->object_type)
  {
    case NETSOLVE_MATRIX:
      fprintf(stderr,"Matrix ");
      break;
    case NETSOLVE_SPARSEMATRIX:
      fprintf(stderr,"Sparse Matrix ");
      break;
    case NETSOLVE_VECTOR:
      fprintf(stderr,"Vector ");
      break;
    case NETSOLVE_SCALAR:
      fprintf(stderr,"Scalar ");
      break;
    case NETSOLVE_UPF:
      fprintf(stderr,"Fortran/C Function in a file.");
      break;
    case NETSOLVE_FILE:
      fprintf(stderr,"File.");
      break;
    case NETSOLVE_STRINGLIST:
      fprintf(stderr,"String List.");
      break;
    case NETSOLVE_PACKEDFILES:
      fprintf(stderr,"Packed files.");
      break;
    case NETSOLVE_STRING:
      fprintf(stderr,"String.");
      break;
    default:
      fprintf(stderr,"???");
  }
  switch(obj->data_type)
  {
    case NETSOLVE_I:
      fprintf(stderr,"Integer.\n");
      break;
    case NETSOLVE_S:
      fprintf(stderr,"Single Precision Real.\n");
      break;
    case NETSOLVE_D:
      fprintf(stderr,"Double Precision Real.\n");
      break;
    case NETSOLVE_C:
      fprintf(stderr,"Single Precision Complex.\n");
      break;
    case NETSOLVE_Z:
      fprintf(stderr,"Double Precision Complex.\n");
      break;
    case NETSOLVE_B:
      fprintf(stderr,"Byte.\n");
      break;
    case NETSOLVE_CHAR:
      if (obj->object_type == NETSOLVE_PACKEDFILES)
        break;
      if (obj->object_type == NETSOLVE_FILE)
        break;
      if (obj->object_type == NETSOLVE_STRING)
        break;
      if (obj->object_type == NETSOLVE_UPF)
        break;
      fprintf(stderr,"Character.\n");
      break;
    default:
      fprintf(stderr,"???");
  }
  fprintf(stderr,obj->description);
}
