/*
 * Test dgemm sequencing
 * 2000 - Jakob Ostergaard
 *
 * $Id: test_dgemmseq.c,v 1.2 2000/12/14 16:11:42 darnold Exp $
 *
 * The idea of this test is to test the sequencing API of NetSolve, by
 * submitting a number of independent sequences of dependent
 * computations.
 *
 */

#include "regression.h"
#include "netsolve.h"
#include <stdio.h>

#define SEQUENCED 

#define NSEQUENCES 3


/*
 * Matrix sizes are 
 *  l = BASE_SIZE * (n * SEQMULT + 1)
 * where l is side length and n is 1 2 3 
 * (look at test_dgemm.c for something that makes sense)
 */
#define BASE_SIZE 100
#define SEQMULT   1
#define MATLEN(n) ((int)(BASE_SIZE * ((n) * SEQMULT + 1)))

double *A[NSEQUENCES], *B[NSEQUENCES], *Q[NSEQUENCES], *C[NSEQUENCES], *C2[NSEQUENCES];
double alpha1 = 42, alpha2 = (double)-42/86, beta1 = 86, beta2 = (double)1/86;

const char* test_name(int i);

int main(int argc, char** argv)
{
  int seq;
#ifdef SEQUENCED
  int failed = 0;
#endif

  test_init("dgemmseq");

  netslmajor("col");

  for(seq = 0; seq != NSEQUENCES; seq++) {
    
    /*
     * A :  300 x 200 random
     * B :  200 x 300 random
     * Q :  300 x 300 random
     * C :  copy of Q
     */
    A[seq] = tgen_matrix_random(MATLEN(3), MATLEN(2));
    B[seq] = tgen_matrix_random(MATLEN(2), MATLEN(3));
    Q[seq] = tgen_matrix_random(MATLEN(3), MATLEN(3));
    C[seq] = tgen_matrix_copy(Q[seq], MATLEN(3), MATLEN(3));
    C2[seq] = tgen_matrix_random(MATLEN(3), MATLEN(3));
  }
 
#ifdef SEQUENCED
  subtest_submit(test_name(0));
  netsl_sequence_begin();
#endif

  for(seq = 0; seq != NSEQUENCES; seq++) {
#ifndef SEQUENCED
  subtest_submit(test_name(seq));
#endif
    /*
     * C  <-  42 * A * B + 86 * Q
     *
     * dgemm() "N"  "N"  m_A/m_C  n_B/n_C  n_A/m_B  alpha  A  l_A  B l_B  beta C l_C
     */
    netsl("dgemm()", "N", "N",
	  MATLEN(3), MATLEN(3), MATLEN(2),
	  &alpha1, A[seq], MATLEN(3), B[seq], MATLEN(2), &beta1, C[seq], MATLEN(3));


    /*
     * C2 <- C
     */
    netsl("dlacpy()", " ", C2[seq], C[seq], MATLEN(3), MATLEN(3));   

    /*
     * C2 <- -42 / 86 * A * B + 1 / 86 * C2
     *     = Q
     */
    netsl("dgemm()", "N", "N",
	  MATLEN(3), MATLEN(3), MATLEN(2),
	  &alpha2, A[seq], MATLEN(3), B[seq], MATLEN(2), &beta2, C2[seq], MATLEN(3));

#ifndef SEQUENCED
    subtest_complete();

    if(tgen_matrix_compare(C2[seq], Q[seq], MATLEN(3), MATLEN(3))) {
      subtest_fail("C2 != Q - Nonsequenced test failure");
    } else {
      subtest_success();
    }

#endif
  }

#ifdef SEQUENCED
  netsl_sequence_end(NS_NULL); /* XXX: Must take A[0...seq],B[0...seq],C[0...seq] */
  subtest_complete();

  for(seq = 0; seq != NSEQUENCES; seq++) {
    /*
     * Compare Q and C2 - now we can actually check whether the
     * previous calculation and the one just completed actually
     * both worked, or if one or both failed (but we will never
     * know whether only one or both of them failed).
     */
    if(tgen_matrix_compare(C2[seq], Q[seq], MATLEN(3), MATLEN(3))) {
      failed = 1;
    }
  }

  if(failed)
    subtest_fail("C2 != Q");
  else
    subtest_success();
#endif
  
  test_exit();
  return 0;
}

const char* test_name(int i)
{
  static char buf[1024];
  sprintf(buf, "dgemm( %i )", i);
  return buf;
}

