/*
 * Commonly used regression test utility routines
 * 2000 - Jakob Ostergaard
 *
 * $Id: test_common.c,v 1.2 2001/03/01 19:04:05 susan Exp $
 */

#include "regression.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/time.h>
#include <unistd.h>
#include <math.h>

static FILE * timing_file = 0;
static FILE * result_file = 0;
static const char * test_name = 0;

static struct timeval init_time, submit_time, complete_time;
static const char * subtest_name = 0;

static void test_ifail(const char* e)
{
  fprintf(stderr, "\nTEST (%s) FAILURE: %s\n", test_name ? test_name : "unknown", e);
 /* fprintf(stderr, "\nTEST (%s) FAILURE: %s\n", test_name ? : "unknown", e); */
  exit(1);
}

static double sec_diff(const struct timeval* t0,
		       const struct timeval* t1)
{
  double ret = t1->tv_sec - t0->tv_sec;
  double uret = t1->tv_usec - t0->tv_usec;
  return ret + uret / 1E6;
}

void test_init(const char* testname)
{
  char buf[1024];
  test_name = testname;

  sprintf(buf, "test_%s.time", testname);
  if(!(timing_file = fopen(buf, "w")))
    test_ifail("Cannot create .time file");

  sprintf(buf, "test_%s.result", testname);
  if(!(result_file = fopen(buf, "w")))
    test_ifail("Cannot create .result file");

  gettimeofday(&init_time, 0);
}

void subtest_submit(const char* tname)
{
  gettimeofday(&submit_time, 0);
  subtest_name = tname;
}

void subtest_complete(void)
{
  gettimeofday(&complete_time, 0);

  /*
   * Good, write down the times
   */
  if(subtest_name) {
    fprintf(timing_file, "%16s: %6.3f\n", 
	    subtest_name, sec_diff(&submit_time, &complete_time));
  } else {
    fprintf(timing_file, "NO TEST NAME SET!\n");
  }
}

void subtest_fail(const char* text)
{
  if(subtest_name) {
    fprintf(result_file, "%16s: TEST FAILED", subtest_name);
    if(text) fprintf(result_file, " (%s)\n", text);
    else fprintf(result_file, "\n");
    subtest_name = 0;
  } else {
    fprintf(result_file, "NO TEST NAME SET BUT ROUTINE FAILED ANYWAY\n");
  }
  fflush(result_file);
}

void subtest_success(void)
{
  if(subtest_name) {
    fprintf(result_file, "%16s: TEST SUCCEEDED\n", subtest_name);
    subtest_name = 0;
  } else {
    fprintf(result_file, "NO TEST NAME SET BUT ROUTINE SUCCEEDED\n");
  }
  fflush(result_file);
}

void subtest_unchecked(void)
{
  if(subtest_name) {
    fprintf(result_file, "%16s: TEST COMPLETED - RESULT UNCHECKED\n", subtest_name);
    subtest_name = 0;
  } else {
    fprintf(result_file, "NO TEST NAME SET BUT ROUTINE RETURNED (RESULT UNCHECKED)\n");
  }
  fflush(result_file);
}

void test_exit(void)
{
  fclose(timing_file);
  fclose(result_file);
  exit(0);
}

double * tgen_matrix_random(unsigned h, unsigned w)
{
  unsigned i,j,p;
  double * ret = (double*)malloc(h * w * sizeof(double));

  if(!ret) 
    test_ifail("Out of memory in tgen_matrix_random()");
  
  p = 0;
  for(i = 0; i != h; i++)
    for(j = 0; j != w; j++)
      ret[p++] = 1.0 - 2.0 * rand() / RAND_MAX;

  return ret;
}

double * tgen_matrix_copy(const double* A, unsigned i, unsigned j)
{
  double * ret = (double*)malloc(i * j * sizeof(double));

  if(!ret)
    test_ifail("Out of memory in tgen_matrix_copy()");
  
  memcpy(ret, A, i*j*sizeof(double));
  return ret;
}

int tgen_matrix_compare(const double* A, const double* B, unsigned h, unsigned w)
{ 
  unsigned p = h * w;

  if(!p) 
    test_ifail("Don't give zero dimension matrix to matrix_compare");

  while(--p)
    if(fabs(A[p] - B[p]) > 1E-6)
      return 1;
  return 0;
}
