#include <stdio.h>
#include "netsolveclient.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>

#define SYNC   0
#define ASYNC  1

void intTest(int);
void stringTest(int);
void isortTest(int);
void dsortTest(int);
void totalTest(int);
void fileTest(int);
void packedfileTest(int);
void stringlistTest(int);
void upfTest(int);

int main(int argc,char **argv)
{
  char dummy;

  fprintf(stderr,"NetSolve C Interface Test Suite\n");
  fprintf(stderr,"-------------------------------\n");
  fprintf(stderr,"This test suite performs 2x8 tests, in\n");
  fprintf(stderr,"synchronous and asynchronous modes.\n");
  fprintf(stderr,"In the asynchronous mode, a dot ('.')\n");
  fprintf(stderr,"is printed every second to illustrate\n");
  fprintf(stderr,"the fact that this program is not blocked\n");
  fprintf(stderr,"while waiting for the NetSolve request to complete.\n");
  fprintf(stderr,"This might not be observable as tests are short\n");
  fprintf(stderr,"as some tests (e.g. upf test) induce some client overhead.\n");
  fprintf(stderr,"Press <enter> to start the testing...\n");
  scanf("%c",&dummy);

  intTest(SYNC);
  fflush(stdout);
  intTest(ASYNC);
  fflush(stdout);
  stringTest(SYNC);
  stringTest(ASYNC);
  isortTest(SYNC);
  isortTest(ASYNC);
  dsortTest(SYNC);
  dsortTest(ASYNC);
  totalTest(SYNC);
  totalTest(ASYNC);
  fileTest(SYNC);
  fileTest(ASYNC);
  packedfileTest(SYNC);
  packedfileTest(ASYNC);
  stringlistTest(SYNC);
  stringlistTest(ASYNC);
  upfTest(SYNC);
  upfTest(ASYNC);

  return 0;
}

/*
 * intTest()
 */
void intTest(int mode)
{
  int i=3;
  int status;

  fprintf(stderr,"** integer test (");
  if (mode == SYNC)
    fprintf(stderr,"synchronous) **\n");
  else
    fprintf(stderr,"asynchronous) **\n");

  if (mode == SYNC)
    status = netsl("inttest()",&i);
  else
    status = netslnb("inttest()",&i);

  if (status < 0)
  {
    netslerr(status);
    fprintf(stderr, "\n%75s", "Result: Failure\n");
    return;
  }


  if (mode == ASYNC)
  {
    int request = status;
    if (request >= 0)
    {
      status = netslpr(request);
      fprintf(stderr,".");
      while(status == NetSolveNotReady)
      {
        sleep(1);
        fprintf(stderr,".");
        fflush(stderr);
        status = netslpr(request);
      }
      fprintf(stderr,"\n");
      if (status == NetSolveOK)
      {
        status = netslwt(request);
      }
    }
  }

  fflush(stdout);
  if (status != NetSolveOK){
    netslerr(status);
    fprintf(stderr, "\n%75s", "Result: Failure\n");
  }
  else{ 
    if (i != 4){
      fprintf(stderr,"Returned '%d' instead of '4'\n",i);
      fprintf(stderr, "\n%75s", "Result: Failure\n");
    }
    else{
      fprintf(stderr, "\n%75s", "Result: Success\n");
    }
  }
  return;
}

/*
 * stringTest()
 */
void stringTest(int mode)
{
  char *string;
  int status;
  
  string = strdup("aa");

  fprintf(stderr,"** string test (");
  if (mode == SYNC)
    fprintf(stderr,"synchronous) **\n");
  else
    fprintf(stderr,"asynchronous) **\n");

  if (mode == SYNC)
    status = netsl("stringtest()",string);
  else
    status = netslnb("stringtest()",string);

  if (status < 0)
  {
    netslerr(status);
    fprintf(stderr, "\n%75s", "Result: Failure\n");
    return;
  }


  if (mode == ASYNC)
  {
    int request = status;
    if (request >= 0) 
    {
      status = netslpr(request);
      fprintf(stderr,".");
      while(status == NetSolveNotReady)
      {
        sleep(1);
        fprintf(stderr,".");
        status = netslpr(request);
      }
      fprintf(stderr,"\n");
      if (status == NetSolveOK)
        status = netslwt(request);
    }
  }


  if (status != NetSolveOK){
    netslerr(status);
    fprintf(stderr, "\n%75s", "Result: Failure\n");
  }
  else
  { 
    if (strcmp(string,"Xa")){
      fprintf(stderr,"Returned '%s' instead of 'Xa'\n",string);
      fprintf(stderr, "\n%75s", "Result: Failure\n");
    }
    else
      fprintf(stderr, "\n%75s", "Result: Success\n");
  }
  free(string);
  return;
}

/*
 * isortTest()
 */
void isortTest(int mode)
{
  int tosort[4];
  int sorted[4];
  int status;

  tosort[0]=4;
  tosort[1]=3;
  tosort[2]=2;
  tosort[3]=1;

  fprintf(stderr,"** integer sort (");
  if (mode == SYNC)
    fprintf(stderr,"synchronous) **\n");
  else
    fprintf(stderr,"asynchronous) **\n");

  if (mode == SYNC)
    status = netsl("iqsort()",4,tosort,sorted);
  else
    status = netslnb("iqsort()",4,tosort,sorted);

  if (status < 0)
  {
    netslerr(status);
    fprintf(stderr, "\n%75s", "Result: Failure\n");
    return;
  }

  if (mode == ASYNC)
  {
    int request = status;
    fprintf(stderr, "Now probing for results\n");
    if (request >= 0) 
    {
      status = netslpr(request);
      fprintf(stderr,".");
      while(status == NetSolveNotReady)
      {
        sleep(1);
        fprintf(stderr,".");
        status = netslpr(request);
      }
      fprintf(stderr,"\n");
      if (status == NetSolveOK)
        status = netslwt(request);
    }
  }

  if (status != NetSolveOK){
    fprintf(stderr, "\n%75s", "Result: Failure\n");
    netslerr(status);
  }
  else  
  {
    if ((sorted[0] != 1) ||
        (sorted[1] != 2) ||
        (sorted[2] != 3) ||
        (sorted[3] != 4))
    {
      fprintf(stderr,"Array not correctly sorted : %d,%d,%d,%d\n",
                    sorted[0],sorted[1],sorted[2],sorted[3]);
      fprintf(stderr, "\n%75s", "Result: Failure\n");
    }
    else
      fprintf(stderr, "\n%75s", "Result: Success\n");
  }
  return;
}

/*
 * dsortTest()
 */
void dsortTest(int mode)
{
  double tosort[4];
  double sorted[4];
  int status;

  tosort[0]=0.4;
  tosort[1]=0.3;
  tosort[2]=0.2;
  tosort[3]=0.1;

  fprintf(stderr,"** double sort (");
  if (mode == SYNC)
    fprintf(stderr,"synchronous) **\n");
  else
    fprintf(stderr,"asynchronous) **\n");

  if (mode == SYNC)
    status = netsl("dqsort()",4,tosort,sorted);
  else
    status = netslnb("dqsort()",4,tosort,sorted);

  if (status < 0)
  {
    netslerr(status);
    fprintf(stderr, "\n%75s", "Result: Failure\n");
    return;
  }


  if (mode == ASYNC)
  {
    int request = status;
    if (request >= 0) 
    {
      status = netslpr(request);
      fprintf(stderr,".");
      while(status == NetSolveNotReady)
      {
        sleep(1);
        fprintf(stderr,".");
        status = netslpr(request);
      }
      fprintf(stderr,"\n");
      if (status == NetSolveOK)
        status = netslwt(request);
    }
  }


  if (status != NetSolveOK){
    fprintf(stderr, "\n%75s", "Result: Failure\n");
    netslerr(status);
  }
  else 
  {
    if ((sorted[0] != 0.1) ||
        (sorted[1] != 0.2) ||
        (sorted[2] != 0.3) ||
        (sorted[3] != 0.4)) 
    {
      fprintf(stderr,"Array not correctly sorted : %f,%f,%f,%f\n",
                    sorted[0],sorted[1],sorted[2],sorted[3]);
      fprintf(stderr, "\n%75s", "Result: Failure\n");
    }
    else
      fprintf(stderr, "\n%75s", "Result: Success\n");
  }
  return;
}

/*
 * totalTest()
 */
void totalTest(int mode)
{
  int ok = 1;
  int status;

  int i,j;

  int intscalar;
  char charscalar;
  float floatscalar;
  double doublescalar;
  scomplex scomplexscalar;
  dcomplex dcomplexscalar;
  char bytescalar;

  int *intvector;
  int intvectorm=2;
  char *charvector;
  int charvectorm=2;
  float *floatvector;
  int floatvectorm=2;
  double *doublevector;
  int doublevectorm=2;
  scomplex *scomplexvector;
  int scomplexvectorm=2;
  dcomplex *dcomplexvector;
  int dcomplexvectorm=2;
  char *bytevector;
  int bytevectorm=2;

  int *intmatrix;
  int intmatrixm=3;
  int intmatrixn=2;
  int intmatrixl=3;
  char *charmatrix;
  int charmatrixm=3;
  int charmatrixn=2;
  int charmatrixl=4;
  float *floatmatrix;
  int floatmatrixm=3;
  int floatmatrixn=2;
  int floatmatrixl=4;
  double *doublematrix;
  int doublematrixm=3;
  int doublematrixn=2;
  int doublematrixl=4;
  dcomplex *dcomplexmatrix;
  int dcomplexmatrixm=3;
  int dcomplexmatrixn=2;
  int dcomplexmatrixl=4;
  scomplex *scomplexmatrix;
  int scomplexmatrixm=3;
  int scomplexmatrixn=2;
  int scomplexmatrixl=4;
  char *bytematrix;
  int bytematrixm=3;
  int bytematrixn=2;
  int bytematrixl=4;

  char *string;

  fprintf(stderr,"** total test (");
  if (mode == SYNC)
    fprintf(stderr,"synchronous) **\n");
  else
    fprintf(stderr,"asynchronous) **\n");

  /* SCALARS */
  intscalar = 2;
  charscalar = 'a';
  floatscalar = 2.0;
  doublescalar = 2.0;
  scomplexscalar.r = 2.0;
  scomplexscalar.i = 2.0;
  dcomplexscalar.r = 2.0;
  dcomplexscalar.i = 2.0;
  bytescalar = 'a';

  /* VECTORS */
  intvector = (int *)calloc(intvectorm,sizeof(int));
  for (i=0;i<intvectorm;i++)
    intvector[i] = 2;
  charvector = (char *)calloc(charvectorm,sizeof(char));
  for (i=0;i<charvectorm;i++)
    charvector[i] = 'a';
  floatvector = (float *)calloc(floatvectorm,sizeof(float));
  for (i=0;i<floatvectorm;i++)
    floatvector[i] = 2.0;
  doublevector = (double *)calloc(doublevectorm,sizeof(double));
  for (i=0;i<doublevectorm;i++)
    doublevector[i] = 2.0;
  scomplexvector = (scomplex *)calloc(scomplexvectorm,sizeof(scomplex));
  for (i=0;i<scomplexvectorm;i++)
  {
    scomplexvector[i].r = 2.0;
    scomplexvector[i].i = 2.0;
  }
  dcomplexvector = (dcomplex *)calloc(dcomplexvectorm,sizeof(dcomplex));
  for (i=0;i<dcomplexvectorm;i++)
  {
    dcomplexvector[i].r = 2.0;
    dcomplexvector[i].i = 2.0;
  }
  bytevector = (char *)calloc(bytevectorm,sizeof(char));
  for (i=0;i<bytevectorm;i++)
    bytevector[i] = 'a';
  /* MATRICES */
  intmatrix =
    (int *)calloc(intmatrixn*intmatrixl,sizeof(int));
  for (i=0;i<intmatrixn;i++)
    for (j=0;j<intmatrixm;j++)
      intmatrix[i*intmatrixl+j] = 2;
  charmatrix =
    (char *)calloc(charmatrixn*charmatrixl,sizeof(char));
  for (i=0;i<charmatrixn;i++)
    for (j=0;j<charmatrixm;j++)
      charmatrix[i*charmatrixl+j] = 'a';
  floatmatrix =
    (float *)calloc(floatmatrixn*floatmatrixl,sizeof(float));
  for (i=0;i<floatmatrixn;i++)
    for (j=0;j<floatmatrixm;j++)
      floatmatrix[i*floatmatrixl+j] = 2.0;
  doublematrix =
    (double *)calloc(doublematrixn*doublematrixl,sizeof(double));
  for (i=0;i<doublematrixn;i++)
    for (j=0;j<doublematrixm;j++)
      doublematrix[i*doublematrixl+j] = 2.0;
  scomplexmatrix =
    (scomplex *)calloc(scomplexmatrixn*scomplexmatrixl,sizeof(scomplex));
  for (i=0;i<scomplexmatrixn;i++)
    for (j=0;j<scomplexmatrixm;j++)
    {
      scomplexmatrix[i*scomplexmatrixl+j].r = 2.0;
      scomplexmatrix[i*scomplexmatrixl+j].i = 2.0;
    }
  dcomplexmatrix =
    (dcomplex *)calloc(dcomplexmatrixn*dcomplexmatrixl,sizeof(dcomplex));
  for (i=0;i<dcomplexmatrixn;i++)
    for (j=0;j<dcomplexmatrixm;j++)
    {
      dcomplexmatrix[i*dcomplexmatrixl+j].r = 2.0;
      dcomplexmatrix[i*dcomplexmatrixl+j].i = 2.0;
    }
  bytematrix =
    (char *)calloc(bytematrixn*bytematrixl,sizeof(char));
  for (i=0;i<bytematrixn;i++)
    for (j=0;j<bytematrixm;j++)
      bytematrix[i*bytematrixl+j] = 'a';

  /* String */
  string = strdup("ABC"); 

  if (mode == SYNC)
    status = netsl("totaltest()",
                    &intscalar, intvector, intvectorm, intmatrix,
                    intmatrixm, intmatrixn, intmatrixl,
                    &charscalar, charvector, charvectorm, charmatrix,
                    charmatrixm, charmatrixn, charmatrixl,
                    &floatscalar, floatvector, floatvectorm, floatmatrix,
                    floatmatrixm, floatmatrixn, floatmatrixl,
                    &doublescalar, doublevector, doublevectorm, doublematrix,
                    doublematrixm, doublematrixn, doublematrixl,
                    &scomplexscalar, scomplexvector, scomplexvectorm, scomplexmatrix,
                    scomplexmatrixm, scomplexmatrixn, scomplexmatrixl,
                    &dcomplexscalar, dcomplexvector, dcomplexvectorm, dcomplexmatrix,
                    dcomplexmatrixm, dcomplexmatrixn, dcomplexmatrixl,
                    &bytescalar, bytevector, bytevectorm, bytematrix,
                    bytematrixm, bytematrixn, bytematrixl,
                    string);
  else
    status = netslnb("totaltest()",
                    &intscalar, intvector, intvectorm, intmatrix,
                    intmatrixm, intmatrixn, intmatrixl,
                    &charscalar, charvector, charvectorm, charmatrix,
                    charmatrixm, charmatrixn, charmatrixl,
                    &floatscalar, floatvector, floatvectorm, floatmatrix,
                    floatmatrixm, floatmatrixn, floatmatrixl,
                    &doublescalar, doublevector, doublevectorm, doublematrix,
                    doublematrixm, doublematrixn, doublematrixl,
                    &scomplexscalar, scomplexvector, scomplexvectorm, scomplexmatrix,
                    scomplexmatrixm, scomplexmatrixn, scomplexmatrixl,
                    &dcomplexscalar, dcomplexvector, dcomplexvectorm, dcomplexmatrix,
                    dcomplexmatrixm, dcomplexmatrixn, dcomplexmatrixl,
                    &bytescalar, bytevector, bytevectorm, bytematrix,
                    bytematrixm, bytematrixn, bytematrixl,
                    string);

  if (status < 0)
  {
    netslerr(status);
    fprintf(stderr, "\n%75s", "Result: Failure\n");
    return;
  }


  if (mode == ASYNC)
  {
    int request = status;
    if (request >= 0) 
    {
      status = netslpr(request);
      fprintf(stderr,".");
      while(status == NetSolveNotReady)
      {
        sleep(1);
        fprintf(stderr,".");
        status = netslpr(request);
      }
      fprintf(stderr,"\n");
      if (status == NetSolveOK)
        status = netslwt(request);
    }
  }


  if (status != NetSolveOK)
  {
    netslerr(status);
    fprintf(stderr, "\n%75s", "Result: Failure\n");
  }
  else
  {
    /* Checking the validity of the data */
  
    /* SCALARS */
  
    if (intscalar != 3)
    {
      fprintf(stderr,"Error on the 'integer scalar' value\n");
      fprintf(stderr,"            (should be 3 and is %d)\n",intscalar);
      ok = 0;
    }
    if (bytescalar != 'b')
    {
      fprintf(stderr,"Error on the 'byte scalar' value\n");
      fprintf(stderr,"            (should be 'b' and is '%c')\n",bytescalar);
      ok = 0;
    }
    if (charscalar != 'b')
    {
      fprintf(stderr,"Error on the 'character scalar' value\n");
      fprintf(stderr,"            (should be 'b' and is '%c')\n",charscalar);
      ok = 0;
    }
    if (floatscalar != 3.0)
    {
      fprintf(stderr,"Error on the 'float scalar' value\n");
      fprintf(stderr,"            (should be 3.0 and is %f)\n",floatscalar);
      ok = 0;
    }
    if (doublescalar != 3.0)
    {
      fprintf(stderr,"Error on the 'double scalar' value\n");
      fprintf(stderr,"            (should be 3.0 and is %f)\n",doublescalar);
      ok = 0;
    }
    if ((scomplexscalar.r != 3.0)||(scomplexscalar.i != 3.0))
    {
      fprintf(stderr,
        "Error on the 'single complex scalar' value\n");
      fprintf(stderr,"            (should be 3.0+i3.0 and is %f+i%f)\n",
                                  scomplexscalar.r,scomplexscalar.i);
      ok = 0;
    }
    if ((dcomplexscalar.r != 3.0)||(dcomplexscalar.i != 3.0))
    {
      fprintf(stderr,
        "Error on the 'double complex scalar' value\n");
      fprintf(stderr,"            (should be 3.0+i3.0 and is %f+i%f)\n",
                                  dcomplexscalar.r,dcomplexscalar.i);
      ok = 0;
    }
  
  
    /* VECTORS */
  
    for (i=0;i<intvectorm;i++)
      if (intvector[i] != 3)
      {
        fprintf(stderr,"Error on the 'integer vector' value\n");
        fprintf(stderr,"            (should be 3 and is %d)\n",intvector[i]);
        ok = 0;
      }
  
    for (i=0;i<bytevectorm;i++)
      if (bytevector[i] != 'b')
      {
        fprintf(stderr,"Error on the 'byetacter vector' value\n");
        fprintf(stderr,
          "            (should be 'b' and is '%c')\n",bytevector[i]);
        ok = 0;
      }
  
    for (i=0;i<charvectorm;i++)
      if (charvector[i] != 'b')
      {
        fprintf(stderr,"Error on the 'character vector' value\n");
        fprintf(stderr,
          "            (should be 'b' and is '%c')\n",charvector[i]);
        ok = 0;
      }
  
    for (i=0;i<floatvectorm;i++)
      if (floatvector[i] != 3.0)
      {
        fprintf(stderr,"Error on the 'float vector' value\n");
        fprintf(stderr,
          "            (should be 3.0 and is %f)\n",floatvector[i]);
        ok = 0;
      }
  
    for (i=0;i<doublevectorm;i++)
      if (doublevector[i] != 3.0)
      {
        fprintf(stderr,"Error on the 'double vector' value\n");
        fprintf(stderr,
          "            (should be 3.0 and is %f)\n",doublevector[i]);
        ok = 0;
      }
  
    for (i=0;i<scomplexvectorm;i++)
      if ((scomplexvector[i].r != 3.0)||(scomplexvector[i].i != 3.0))
      {
        fprintf(stderr,
                 "Error on the 'single complex vector' value\n");
        fprintf(stderr,
                 "            (should be 3.0+i3.0 and is %f+i%f)\n",
                  scomplexvector[i].r,scomplexvector[i].i);
        ok = 0;
      }
  
    for (i=0;i<dcomplexvectorm;i++)
      if ((dcomplexvector[i].r != 3.0)||(dcomplexvector[i].i != 3.0))
      {
        fprintf(stderr,
                 "Error on the 'double complex vector' value\n");
        fprintf(stderr,
                 "            (should be 3.0+i3.0 and is %f+i%f)\n",
                 dcomplexvector[i].r,dcomplexvector[i].i);
        ok = 0;
      }
  
    /* No TEST for Bytes */
  
  /* MATRICES */
  
    for (i=0;i<intmatrixn;i++)
      for (j=0;j<intmatrixm;j++)
        if (intmatrix[i*intmatrixl+j] != 3)
        {
          fprintf(stderr,
                  "Error on the 'integer matrix' value\n");
          fprintf(stderr,
                  "            (should be 3 and is %d)\n",intmatrix[i*intmatrixl+j]);
          ok = 0;
        }
  
    for (i=0;i<bytematrixn;i++)
      for (j=0;j<bytematrixm;j++)
        if (bytematrix[i*bytematrixl+j] != 'b')
        {
          fprintf(stderr,
                "Error on the 'byteacter matrix' value\n");
          fprintf(stderr,
                "            (should be 'b' and is '%c')\n",bytematrix[i*bytematrixl+j]);
          ok = 0;
        }
  
    for (i=0;i<charmatrixn;i++)
      for (j=0;j<charmatrixm;j++)
        if (charmatrix[i*charmatrixl+j] != 'b')
        {
          fprintf(stderr,
                "Error on the 'character matrix' value\n");
          fprintf(stderr,
                "            (should be 'b' and is '%c')\n",charmatrix[i*charmatrixl+j]);
          ok = 0;
        }
  
    for (i=0;i<floatmatrixn;i++)
      for (j=0;j<floatmatrixm;j++)
        if (floatmatrix[i*floatmatrixl+j] != 3.0)
        {
          fprintf(stderr,
               "Error on the 'float matrix' value\n");
          fprintf(stderr,
               "            (should be 3.0 and is %f)\n",floatmatrix[i*floatmatrixl+j]);
          ok = 0;
        }
  
    for (i=0;i<doublematrixn;i++)
      for (j=0;j<doublematrixm;j++)
        if (doublematrix[i*doublematrixl+j] != 3.0)
        {
          fprintf(stderr,
               "Error on the 'double matrix' value\n");
          fprintf(stderr,
               "            (should be 3.0 and is %f)\n",doublematrix[i*doublematrixl+j]);
          ok = 0;
        }
  
    for (i=0;i<scomplexmatrixn;i++)
      for (j=0;j<scomplexmatrixm;j++)
        if ((scomplexmatrix[i*scomplexmatrixl+j].r != 3.0) ||
            (scomplexmatrix[i*scomplexmatrixl+j].i != 3.0))
        {
          fprintf(stderr,"Error on the 'single complex matrix'\n");
          fprintf(stderr,"            (should be 3.0+i3.0 and is %f+i%f)\n",
                       scomplexmatrix[i*scomplexmatrixl+j].r,scomplexmatrix[i*scomplexmatrixl+j].i);
          ok = 0;
        }
  
    for (i=0;i<dcomplexmatrixn;i++)
      for (j=0;j<dcomplexmatrixm;j++)
        if ((dcomplexmatrix[i*dcomplexmatrixl+j].r != 3.0) ||
            (dcomplexmatrix[i*dcomplexmatrixl+j].i != 3.0))
        {
          fprintf(stderr,
                "Error on the 'double complex matrix' value\n");
          fprintf(stderr,
                "            (should be 3.0+i3.0 and is %f+i%f)\n",
                dcomplexmatrix[i*dcomplexmatrixl+j].r,dcomplexmatrix[i*dcomplexmatrixl+j].i);
          ok = 0;
        }
  
    /* String */
    if (strcmp(string,"BBC"))
    {
      fprintf(stderr,"Error on the string : %s instead of BBC",string);
      ok = 0;
    }
     /* No test for bytes */
  
    if (ok)
      fprintf(stderr, "\n%75s", "Result: Success\n");
    else
      fprintf(stderr, "\n%75s", "Result: Failure\n");
  }
  free(intvector);free(charvector);free(floatvector); free(doublevector);
  free(scomplexvector);free(dcomplexvector);free(bytevector);
  free(intmatrix);free(charmatrix);free(floatmatrix); free(doublematrix);
  free(scomplexmatrix);free(dcomplexmatrix);free(bytematrix);
  free(string);
}

/*
 *  fileTest
 */
void fileTest(int mode)
{
  char *inputfilename  = "./toto";
  char *outputfilename = "./tata";
  FILE *f;
  int status;
  char line[256];

  fprintf(stderr,"** file test (");
  if (mode == SYNC)
    fprintf(stderr,"synchronous) **\n");
  else
    fprintf(stderr,"asynchronous) **\n");

  /* Remove the files */
  remove(inputfilename);
  remove(outputfilename);

  /* Creating a bogus input file */
  f = fopen(inputfilename,"w");
  if (f == NULL)
  {
    fprintf(stderr,"Impossible to create file '%s'\n",inputfilename);
    return;
  } 
  fprintf(f,"INPUT\n");
  fclose(f);
 
  if (mode == SYNC)
    status = netsl("filetest()",inputfilename,outputfilename);
  else
    status = netslnb("filetest()",inputfilename,outputfilename);

  if (status < 0)
  {
    netslerr(status);
    fprintf(stderr, "\n%75s", "Result: Failure\n");
    return;
  }


  if (mode == ASYNC)
  {
    int request = status;
    if (request >= 0)
    {
      status = netslpr(request);
      fprintf(stderr,".");
      while(status == NetSolveNotReady)
      {
        sleep(1);
        fprintf(stderr,".");
        status = netslpr(request);
      }
      fprintf(stderr,"\n");
      if (status == NetSolveOK)
        status = netslwt(request);
    }
  }

  if (status != NetSolveOK){
    netslerr(status);
    fprintf(stderr, "\n%75s", "Result: Failure\n");
  }
  else
  {
    f = fopen(outputfilename,"r");
    if (f == NULL){
      fprintf(stderr,"Output file not created\n");
      fprintf(stderr, "\n%75s", "Result: Failure\n");
    }
    else
    {
      if (fgets(line,256,f) == NULL){
        fprintf(stderr,"Error while reading the output file\n");
        fprintf(stderr, "\n%75s", "Result: Failure\n");
      }
      else
      {
        if (strcmp(line,"INPUT\n")){
          fprintf(stderr,"Corrupted output file\n");
          fprintf(stderr, "\n%75s", "Result: Failure\n");
	}
        else
        {
          if (fgets(line,256,f) == NULL){
            fprintf(stderr,"Error while reading the output file\n");
            fprintf(stderr, "\n%75s", "Result: Failure\n");
	  }
          else
          {
            if (strcmp(line,"DONE\n")){
              fprintf(stderr,"Corrupted output file\n");
              fprintf(stderr, "\n%75s", "Result: Failure\n");
	    }
            else
            {
              if (fgets(line,256,f) != NULL){
                fprintf(stderr,"Corrupted output file\n");
                fprintf(stderr, "\n%75s", "Result: Failure\n");
              }
              else
                fprintf(stderr, "\n%75s", "Result: Success\n");
            } 
          }
        }
      }
      fclose(f);
    }
  }
  
  /* Remove the files */
  remove(inputfilename);
  remove(outputfilename);

  return;
}


/*
 * packedfileTest()
 */
void packedfileTest(int mode)
{
  char *inputfilename[3];
  char *outputfilenamepattern;
  FILE *f;
  int status;
  char buf[256];
  char line[256];
  int i;
 
  fprintf(stderr,"** packed files (");
  if (mode == SYNC) 
    fprintf(stderr,"synchronous) **\n");
  else
    fprintf(stderr,"asynchronous) **\n");

  inputfilename[0] = strdup("toto1");
  inputfilename[1] = strdup("toto2");
  inputfilename[2] = strdup("toto3");

  outputfilenamepattern = strdup("tata");
  

  /* Remove the files */ 
  remove(inputfilename[0]); 
  remove(inputfilename[1]); 
  remove(inputfilename[2]); 
  sprintf(buf,"%s-0",outputfilenamepattern); 
  remove(buf); 
  sprintf(buf,"%s-1",outputfilenamepattern);
  remove(buf);
  sprintf(buf,"%s-2",outputfilenamepattern);
  remove(buf);

  /* Creating bogus input files */
  f = fopen(inputfilename[0],"w");
  if (f == NULL)
  {
    fprintf(stderr,"Impossible to create file '%s'\n",inputfilename[0]);
    fprintf(stderr, "\n%75s", "Result: Failure\n");
    return;
  }
  fprintf(f,"INPUT\n");
  fclose(f);

  f = fopen(inputfilename[1],"w");
  if (f == NULL)
  {
    fprintf(stderr,"Impossible to create file '%s'\n",inputfilename[1]);
    fprintf(stderr, "\n%75s", "Result: Failure\n");
    return;
  }
  fprintf(f,"INPUT\n");
  fclose(f);

  f = fopen(inputfilename[2],"w");
  if (f == NULL)
  {
    fprintf(stderr,"Impossible to create file '%s'\n",inputfilename[2]);
    fprintf(stderr, "\n%75s", "Result: Failure\n");
    return;
  }
  fprintf(f,"INPUT\n");
  fclose(f);

  if (mode == SYNC)
    status = netsl("packedfiletest()",3,inputfilename,outputfilenamepattern);
  else
    status = netslnb("packedfiletest()",3,inputfilename,outputfilenamepattern);

  if (status < 0)
  {
    netslerr(status);
    fprintf(stderr, "\n%75s", "Result: Failure\n");
    return;
  }


  if (mode == ASYNC)
  {
    int request = status;
    if (request >= 0)
    {
      status = netslpr(request);
      fprintf(stderr,".");
      while(status == NetSolveNotReady)
      {
        sleep(1);
        fprintf(stderr,".");
        status = netslpr(request);
      }
      fprintf(stderr,"\n");
      if (status == NetSolveOK)
        status = netslwt(request);
    }
  }
  if (status != NetSolveOK){
    netslerr(status);
    fprintf(stderr, "\n%75s", "Result: Failure\n");
  }
  else
  {
    int ok=1;

    for (i=0;i<3;i++)
    {
      sprintf(buf,"%s-%d",outputfilenamepattern,i);
      f = fopen(buf,"r");
      if (f == NULL)
      {
        fprintf(stderr,"Output file #%d not created\n",i);
        ok = 0;
      }
      else
      {
        if (fgets(line,256,f) == NULL)
        {
          fprintf(stderr,"Error while reading output file #%d\n",i);
          ok = 0;
        }
        else
        {
          if (strcmp(line,"INPUT\n"))
          {
            fprintf(stderr,"Corrupted output file #%d\n",i);
            ok = 0;
          }
          else
          {
            if (fgets(line,256,f) == NULL)
            {
              fprintf(stderr,"Error while reading output file #%d\n",i);
              ok = 0;
            }
            else
            {
              if (strcmp(line,"DONE\n"))
              {
                fprintf(stderr,"Corrupted output file #%d\n",i);
                ok = 0;
              }
              else
              {
                if (fgets(line,256,f) != NULL)
                {
                  fprintf(stderr,"Corrupted output file #%d\n",i);
                  ok = 0;
                }
              }
            }
          }
        }
        fclose(f);
      }
    }
    if (ok)
      fprintf(stderr, "\n%75s", "Result: Success\n");
    else
      fprintf(stderr, "\n%75s", "Result: Failure\n");
  }

  /* Remove the files */
  remove(inputfilename[0]);
  remove(inputfilename[1]);
  remove(inputfilename[2]);
  sprintf(buf,"%s-0",outputfilenamepattern);
  remove(buf);
  sprintf(buf,"%s-1",outputfilenamepattern);
  remove(buf);
  sprintf(buf,"%s-2",outputfilenamepattern);
  remove(buf);
  free(inputfilename[0]);
  free(inputfilename[1]);
  free(inputfilename[2]);
  free(outputfilenamepattern);
  return;
}

/*
 * upfTest()
 */
void upfTest(int mode)
{
  int input_int, output_int;
  int status;

  fprintf(stderr,"** upf test (");
  if (mode == SYNC)
    fprintf(stderr,"synchronous) **\n");
  else
    fprintf(stderr,"asynchronous) **\n");
  input_int = 3;

  if (mode == SYNC)
    status = netsl("upftest()","upf.f","toto.f",&input_int,&output_int);
  else
    status = netslnb("upftest()","upf.f","toto.f",&input_int,&output_int);

  if (status < 0)
  {
    netslerr(status);
    fprintf(stderr, "\n%75s", "Result: Failure\n");
    return;
  }


  if (mode == ASYNC)
  {
    int request = status;
    if (request >= 0)
    {
      status = netslpr(request);
      fprintf(stderr,".");
      while(status == NetSolveNotReady)
      {
        sleep(1);
        fprintf(stderr,".");
        status = netslpr(request);
      }
      fprintf(stderr,"\n");
      if (status == NetSolveOK)
        status = netslwt(request);
    }
  }

  if (status != NetSolveOK){
    netslerr(status);
    fprintf(stderr, "\n%75s", "Result: Failure\n");
  }
  else
  {
    if (output_int != 4*input_int){
      fprintf(stderr,"Returned '%d' instead of '%d'\n",output_int,4*input_int);
      fprintf(stderr, "\n%75s", "Result: Failure\n");
    }
    else
      fprintf(stderr, "\n%75s", "Result: Success\n");
  }
  return; 
}

/*
 * stringlistTest()
 */
void stringlistTest(int mode)
{
  char **input;
  int wc;
  int status;

  fprintf(stderr,"** string list test (");
  if (mode == SYNC)
    fprintf(stderr,"synchronous) **\n");
  else
    fprintf(stderr,"asynchronous) **\n");
  
  input = (char **)calloc(2,sizeof(char*));
  input[0] = strdup("AAA");
  input[1] = strdup("BBBC");

  if (mode == SYNC)
    status = netsl("stringlisttest()",input,2,&wc);
  else
    status = netslnb("stringlisttest()",input,2,&wc);

  if (status < 0)
  {
    netslerr(status);
    fprintf(stderr, "\n%75s", "Result: Failure\n");
    return;
  }
  
  if (mode == ASYNC)
  {
    int request = status;
    if (request >= 0)
    {
      status = netslpr(request);
      fprintf(stderr,".");
      while(status == NetSolveNotReady)
      {
        sleep(1);
        fprintf(stderr,".");
        status = netslpr(request);
      }
      fprintf(stderr,"\n");
      if (status == NetSolveOK)
        status = netslwt(request);
    }
  }

  if (status != NetSolveOK)
    netslerr(status);
  else
  {
    if (wc != 7){
      fprintf(stderr,"Returned '%d' instead of '%d'\n",wc,7);
      fprintf(stderr, "\n%75s", "Result: Failure\n");
    }
    else
      fprintf(stderr, "\n%75s", "Result: Success\n");
  }
  return;
}

