/*****************************************************************/
/*      server_init.c                                            */
/*      Henri Casanova                                           */
/*---------------------------------------------------------------*/
/*  serverInit()                                                 */
/*****************************************************************/

#include <netdb.h>
#include "core.h"
#include "serverglobal.h"
#include "netsolveutil.h"
#include "serverinit.h"
#include "workloadmanager.h"
#ifdef NWS
# include "nwsutils.h"
#endif
#include "netsolvesignals.h"
#include <unistd.h>
#include <fcntl.h>
#include <sys/stat.h>



/*
 * initMySelf()
 *
 * Performs the initialization of the host descriptor
 * the IP port and the listening socket
 */
int initMySelf()
{
  struct hostent *hp;
  char buffer[256];
  
  /* Getting my host name and IPaddr */
  if (gethostname(buffer,256) == -1)
  {
    perror("gethostname()");
    ns_errno = NetSolveSystemError;
    return -1;
  }

  if ((hp = gethostbyname(buffer)) == NULL)
  {
    perror("gethostbyname()");
    ns_errno = NetSolveSystemError;
    return -1;
  }
  BCOPY(hp->h_addr_list[0],
        (void*)&(global.my_self->host_desc->IPaddr),sizeof(NS_IPaddr_type));
  global.my_self->host_desc->hostname = strdup(hp->h_name);

  /* set up the port */
  global.sock = bindToFirstAvailablePort(PORT_INIT,
                                         &(global.my_self->port));
  if (global.sock == -1)
  {
    fprintf(stderr,"   Cannot bind to port !!\n");
    ns_errno = NetSolveSystemError;
    return -1;
  }

  fprintf(stdout,"   Bound on port %d...\n",global.my_self->port);

  global.my_self->host_desc->host_status = HOST_UP;
  global.my_self->host_desc->data_format = getArch();
  if (global.my_self->host_desc->data_format == -1)
  {
    fprintf(stderr,"Impossible to determine my architecture\n");
    return -1;
  }

  fprintf(stdout,"   Running the LINPACK benchmark...");
  fflush(stdout);
  global.my_self->host_desc->speed = kflops();

  fprintf(stdout,"%d Kflops\n",global.my_self->host_desc->speed);

  /* History */

  global.my_self->network_history->latency   = -1;
  global.my_self->network_history->bandwidth = -1;
  global.my_self->workload_history->w        = -1;

  return 0;
}

/*
 * setCondor()
 *
 * Sets the global variables in case of a condor server
 */
int setCondor(char *filename)
{
  FILE *f;
  char line[256];
  int linecount = 0;
  char *token1;
  char *token2,*token3;

  /* Opening the file */
  f = fopen(filename,"r");

  if (f == NULL)
  {
    fprintf(stderr,"Impossible to open file '%s'\n",filename);
    return -1;
  }

  while(readLine(f,line,&linecount))
  {
    if (strncmp(line,"@CONDOR:",8))
      continue;
    fclose(f);
    token1 = (char *)strtok(line,":\t\n");
    token2 = (char *)strtok(NULL," \t");
    token3 = (char *)strtok(NULL," \t\n");
    if ((token2 == NULL)||(token3 == NULL))
    {
      fprintf(stderr,"%s,%d: Syntax error\n",filename,linecount);
      fclose(f);
      return -1;
    }
    global.Condor_path = (char *)strdup(token2);
    global.Condor_nb = atoi(token3);
    global.Condor_path[strlen(global.Condor_path)] = '\0';
    fclose(f);
    return 0;
  }
  fclose(f);
  return 0;
}

/*
 * setDisks()
 */
int setDisks(char *filename)
{
  FILE *f;
  char line[256];
  int linecount = 0;
  char *token1;
  char *token2,*token3;

  /* Opening the file */
  f = fopen(filename,"r");
  if (f == NULL)
  {
    fprintf(stderr,"Impossible to open file '%s'\n",filename);
    return -1;
  }

  while(readLine(f,line,&linecount))
  {
    if (strncmp(line,"@DISK",5))
      continue;
    token1 = (char *)strtok(line," \t\n");
    token2 = (char *)strtok(NULL," \t\n");
    token3 = (char *)strtok(NULL," \t\n");
    if ((token2 == NULL)||(token3 == NULL))
    {
      fprintf(stderr,"%s,%d: Syntax error\n",filename,linecount);
      fclose(f);
      return -1;
    }
    (global.my_self->nb_disks)++; 
    global.my_self->disks = (char **)myRealloc(global.my_self->disks,
                        (global.my_self->nb_disks)*sizeof(char*));
    global.my_self->disks[global.my_self->nb_disks -1] = strdup(token2);
    global.my_self->diskspaces = (int *)myRealloc(global.my_self->diskspaces,
                        (global.my_self->nb_disks)*sizeof(int));
    global.my_self->diskspaces[global.my_self->nb_disks -1] = atoi(token3);
  }
  fclose(f);
  return 0;
}

/*
 * setScratchSpace()
 *
 * Sets the path to the scratch area
 */
int setScratchSpace(char *filename)
{
  FILE *f;
  char line[256];
  int linecount = 0;
  char *token1;
  char *token2;

  /* Opening the file */
  f = fopen(filename,"r");

  if (f == NULL)
  {
    fprintf(stderr,"Impossible to open file '%s'\n",filename);
    return -1;
  }

  while(readLine(f,line,&linecount))
  {
    if (strncmp(line,"@SCRATCH:",9))
      continue;
    fclose(f);
    token1 = (char *)strtok(line,":\t\n");
    token2 = (char *)strtok(NULL," \t\n");
    if (token2 == NULL)
    {
      fprintf(stderr,"%s,%d: Syntax Error\n",filename,linecount);
      return -1;
    }
    free(global.scratch_path);
    global.scratch_path = (char *)strdup(token2);
    global.scratch_path[strlen(global.scratch_path)] = '\0';
    return 0;
  }
  return 0;
}

/*
 * setMPI()
 *
 * Sets the global variables in case of an MPI server
 *
 */
int setMPI(char *filename)
{
  FILE *f;
  char line[256];
  int linecount = 0;
  char *token1;
  char *token2,*token3;

  global.MPI_path = strdup(MPI_DIR);
  fprintf(stderr, "global.MPI_path = %s\n",global.MPI_path);

  /* Opening the file */
  f = fopen(filename,"r");

  if (f == NULL)
  {
    fprintf(stderr,"Impossible to open file '%s'\n",filename);
    return -1;
  }

  global.nb_MPInodes=0;
  global.MPInodefile=NULL;
  while(readLine(f,line,&linecount))
  {
    if (!strncmp(line,"@MPIHOSTS",7))
    {
      token1 = (char *)strtok(line," \t\n");
      token2 = (char *)strtok(NULL," \t\n");
      token3 = (char *)strtok(NULL," \t\n");
      if ((token2 == NULL)||(token3 == NULL))
      {
        fprintf(stderr,"%s,%d: Syntax Error\n",filename,linecount);
        fclose(f);
        return -1;
      }
      global.nb_MPInodes = atoi(token3);
      if (global.nb_MPInodes <= 0)
      {
        fprintf(stderr,"%s,%d: Invalid number of nodes\n",
                       filename,linecount);
        fclose(f);
        return -1;
      }
      global.MPInodefile = strdup(token2);
    }
  }
  fclose(f);

  if (((global.nb_MPInodes == 0)&&(global.MPI_path!= NULL))||
      ((global.nb_MPInodes != 0)&&(global.MPI_path== NULL)))
  {
    fprintf(stderr,"%s: MPI might be missing information (but not on some MPPs)\n",filename);
    return -1;
  }

  return 0;
}

/*
 * setNumberProc()
 *
 * Sets the number of processors available
 */
int setNumberProc(char *filename)
{
  FILE *f;
  char line[256];
  int linecount = 0;

  /* Opening the file */
  f = fopen(filename,"r");

  if (f == NULL)
  {
    fprintf(stderr,"Impossible to open file '%s'\n",filename);
    return -1;
  }

  while(readLine(f,line,&linecount))
  {
    if (strncmp(line,"@PROC:",6) == 0)
    {
      global.my_self->host_desc->number_proc = atoi(&(line[6]));
      if (global.my_self->host_desc->number_proc <= 0)
      {
        fprintf(stderr,"%s, %d: Invalid number of processors\n",
                filename,linecount);
        fclose(f);
        return -1;
      }
      fclose(f);
      return 0;
    }
  }
  fclose(f);
  return 0;
}

/*
 * setWorkloadThreshhold()
 *
 * Sets the workload threshhold
 */
int setWorkloadThreshhold(char *filename)
{
  FILE *f;
  char line[256];
  int linecount = 0;

  /* Opening the file */
  f = fopen(filename,"r");

  if (f == NULL)
  {
    fprintf(stderr,"Impossible to open file '%s'\n",filename);
    return -1;
  }

  while(readLine(f,line,&linecount))
  {
    if (strncmp(line,"@WORKLOADMAX:",13) == 0)
    {

      global.my_self->workload_threshhold = atoi(&(line[13]));
      if (global.my_self->workload_threshhold < -1)
      {
        fprintf(stderr,"%s, %d: Invalid workload threshhold\n",
                filename,linecount);
        fclose(f);
        return -1;
      }
      fclose(f);
      return 0;
    }
  }
  fclose(f);
  return 0;
}

/*
 * setProblems()
 *
 * calls the routine generated by the code generator
 */
int setProblems(char *filename)
{
  int cc,i;
  char **filelist=NULL;
  int listlength=0;
  int linecount;
  int count=0;
  char line[256];
  FILE *f;

  /* Count the number of problem files */
  f = fopen(filename,"r");
  if (f == NULL)
  {
    fprintf(stderr,"Impossible to open file '%s'\n",filename);
    return -1;
  }

  linecount = 0;
  while(readLine(f,line,&linecount))
  {
    if (strncmp(line,"@PROBLEMS:",10))
      continue;
    while(readLine(f,line,&linecount))
    {
      if (line[0]=='@')
        break;
      else
        listlength++;
    }
  }
  fclose(f);

  /* Allocate the space for the list */
  filelist = (char **)calloc(listlength,sizeof(char*));

  /* fill the list with the file names */
  f = fopen(filename,"r");
  if (f == NULL)
  {
    fprintf(stderr,"Impossible to open file '%s'\n",filename);
    return -1;
  }
  linecount = 0;
  while(readLine(f,line,&linecount))
  {
    if (strncmp(line,"@PROBLEMS:",10))
      continue;
    while(readLine(f,line,&linecount))
    {
      if (line[0]=='@')
        break;
      line[strlen(line)-1] = '\0';
      filelist[count++] = strdup(1+strrchr(line,'/'));
    }
  }
  fclose(f);

  /* Call the generated routine */
  cc = problemInit(filelist,listlength);

  for (i=0;i<listlength;i++)
    free(filelist[i]);
  free(filelist);
  if (cc == -1)
    return -1;
  global.nb_problems = NScountList(global.problems);
  return 0;
}

/*
 * setMaster()
 *
 * Build the agent descriptor
 */
int setMaster(char *filename)
{
  FILE *f;
  char line[256];
  struct hostent *hp;
  char *token1;
  char *token2,*token3;
  int linecount = 0;
  char *mastername=NULL;
  NS_AgentDesc *agentdesc;

  /* Opening the file */
  f = fopen(filename,"r");

  if (f == NULL)
  {
    fprintf(stderr,"Impossible to open file '%s'\n",filename);
    return -1;
  }

  while(readLine(f,line,&linecount))
  {
    if (strncmp(line,"@AGENT:",7) == 0)
    {
      token1 = (char *)strtok(line,":\t\n");
      token2 = (char *)strtok(NULL," \t\n");
      token3 = (char *)strtok(NULL," \t\n");
      if (token2 == NULL)
      {
        fprintf(stderr,"%s,%d: Syntax error\n",filename,linecount);
        return -1;
      }
      mastername = (char *)strdup(token2);
      if (token3 != NULL)
      {
        if (!strcmp(token3,"*"))
          global.my_self->broadcast = 1;
        else
          global.my_self->broadcast = 0;
      }
      else
      {
        global.my_self->broadcast = 0;
      }
    }
  }
  fclose(f);
  if (mastername == NULL)
  {
    fprintf(stderr,"%s: No agent specified\n",filename);
    return -1;
  }

  /* Get the full name and IPaddr of the agent */
  hp = gethostbyname(mastername);
  if (hp == NULL)
  {
    fprintf(stderr,"Impossible to resolve hostname '%s'\n",mastername);
    return -1;
  } 
  agentdesc = newAgentDesc();
  agentdesc->port = AGENT_PORT;
  agentdesc->host_desc = newHostDesc();
  agentdesc->host_desc->hostname= 
      strdup(hp->h_name);
  BCOPY(hp->h_addr_list[0],
       (void*)&(agentdesc->host_desc->IPaddr),
       sizeof(NS_IPaddr_type));
  free(mastername);

  addAgent(agentdesc);
  global.master = agentdesc;
  return 0;
}

/*
 * registerToNetSolve
 *
 * Apply for computation
 */
int registerToNetSolve(NS_AgentDesc *agent,int report)
{
  int sock;
  NS_Communicator *comm;
  NS_AgentDesc *ad;
  int nb_agents,i;
  int tag;

  sock = connectToSocket(agent->host_desc->hostname,
                           agent->host_desc->IPaddr,
                           agent->port);
  if (sock == -1)
  {
    netsolvePerror("connectToSocket()");
    fprintf(stderr,"Impossible to contact agent on '%s'\n",
                   agent->host_desc->hostname);
    return -1;
  }

  comm=initTransaction(sock,DATA_XDR);
  if (comm == NULL)
  {
    netsolvePerror("initTransaction()");
    endTransaction(comm);
    return -1;
  }
  if (sendInt(comm,NS_PROT_SV_REGISTER) == -1)
  {
    netsolvePerror("sendInt()");
    endTransaction(comm);
    return -1;
  }
  if (sendServerDesc(comm,global.my_self) == -1)
  {
    netsolvePerror("sendServerDesc()");
    endTransaction(comm);
    return -1;
  }
  if (sendAllProblems(comm) == -1)
  {
    netsolvePerror("sendAllProblems()");
    endTransaction(comm);
    return -1;
  }

  if (recvInt(comm,&tag) == -1)
  {
    netsolvePerror("recvInt()");
    endTransaction(comm);
    return -1;
  }
  if (tag == NS_PROT_REGISTRATION_REFUSED)
  {
    fflush(stdout);
    fprintf(stderr,"Problem description differs from registered problem with same name\n");
    ns_errno = NetSolveNotAllowed;
    endTransaction(comm);
    return -1;
  }
  if (tag == NS_PROT_NOT_ALLOWED)
  {
    fflush(stdout);
    fprintf(stderr,"A server is already running on this machine\n");
    ns_errno = NetSolveNotAllowed;
    endTransaction(comm);
    return -1;
  }
  if (tag != NS_PROT_REGISTRATION_ACCEPTED)
  {
    fflush(stdout);
    ns_errno = NetSolveProtocolError;
    netsolvePerror("Registration not accepted");
    endTransaction(comm);
    return -1;
  }
  fprintf(stdout,"Registration accepted\n");
  if (recvInt(comm,&nb_agents) == -1)
  {
    netsolvePerror("recvInt()");
    endTransaction(comm);
    return -1;
  }
  for (i=0;i<nb_agents;i++)
  {
    ad = recvAgentDesc(comm);
    if (ad == NULL)
    {
      netsolvePerror("recvAgentDesc()");
      endTransaction(comm);
      return -1;
    }
    if ((*lookupAgent(&(ad->host_desc->IPaddr))) == NULL)
      addAgent(ad);
    if (report)
    {
      if (registerToNetSolve(ad,0) == -1)
      {
        fprintf(stderr,"WARNING:Possible incoherence in the system\n");
      }
    }
  }
  endTransaction(comm);
  return 0;
}

/*
 * sendAllProblems()
 */
int sendAllProblems(NS_Communicator *comm)
{
  NS_LinkItem *link;

  if (sendInt(comm,global.nb_problems) == -1)
    return -1;
  link = global.problems->start;
  while(link != NULL)
  {
    if (sendProblemDesc(comm,(NS_ProblemDesc *)(link->content)) == -1)
    {
      netsolvePerror("sendProblemDesc()");
      return -1;
    }
    link = link->next;
  }
  return 0;
}

#ifdef HBM
/*
 * runHBMLocalMonitor()
 *
 * Runs the Heart Beat Local Monitor
 */
int runHBMLocalMonitor()
{
  char commandline[256];
  char path[256];
  int pid;
  char *arch;

  /* clean up the HBM files if any */
  sprintf(commandline,"rm -rf /tmp/hbmlm*");
  system(commandline);

  global.hbmlm_pid = fork();
  if (global.hbmlm_pid == -1)
  {
    ns_errno = NetSolveSystemError;
    netsolvePerror("");
    return -1;
  }

  if (global.hbmlm_pid != 0)
    return 0;
 
  arch = getNetSolveArch();
  if (execl(HBMLM,"hbdlm",(char *)0) == -1)
  {
    fprintf(stderr,"Warning: Unable to start the HBM Local Monitor.\n");
    perror("execl");
    exit(-1);
  }
  exit(0);
}
#endif

/*
 * setRestrictions()
 *
 * Set the user access restrictions
 */
int setRestrictions(char *filename)
{
  NS_RestrictionDesc *r;
  FILE *f;
  char line[256];
  int linecount = 0;
  int count = 0;
  char *token1,*token2;

  /* Count the number of problem files */
  f = fopen(filename,"r");
  if (f == NULL)
  {
    fprintf(stderr,"Impossible to open file '%s'\n",filename);
    ns_errno = NetSolveFileError;
    return -1;
  }

  linecount = 0;
  while(readLine(f,line,&linecount))
  {
    if (strncmp(line,"@RESTRICTIONS:",14))
      continue;
    while(readLine(f,line,&linecount))
    {
      if (line[0]=='@')
        break;
      else
        count++;
    }
  }
  fclose(f);

  if (count == 0)
  {
    fprintf(stderr,"%s,%d: At least one restriction must be specified.\n",
               filename, linecount);
    ns_errno = NetSolveInternalError;
    return -1;
  }

  /* Allocate the space for the list */
  global.my_self->nb_restrictions = count;
  global.my_self->restrictions = 
          (NS_RestrictionDesc **)calloc(count,sizeof(NS_RestrictionDesc*));

  /* fill the list with the file names */
  f = fopen(filename,"r");
  if (f == NULL)
  {
    fprintf(stderr,"Impossible to open file '%s'\n",filename);
    ns_errno = NetSolveFileError;
    return -1;
  }
  linecount = 0;
  count = 0;
  while(readLine(f,line,&linecount))
  {
    if (strncmp(line,"@RESTRICTIONS:",14))
      continue;
    while(readLine(f,line,&linecount))
    {
      if (line[0]=='@')
        break;
      token1 = strtok(line," \t");
      token2 = strtok(NULL," \n");
      if ((token1 == NULL)||(token2 == NULL))
      {
        fprintf(stderr,"%s,%d: Syntax Error\n",filename,linecount);
        fclose(f);
        return -1;
      }
      if (atoi(token2) <= 0)
      {
        fprintf(stderr,"%s,%d: Invalid number of requests\n",filename,linecount);
        fclose(f);
        ns_errno = NetSolveInternalError;
        return -1;
      }
      r = newRestrictionDesc();
      r->domain = strdup(token1); 
      netsolveCaps(r->domain);
      r->max_pending = atoi(token2);
      r->pending = 0;
      (global.my_self->restrictions)[count++] = r;
    }
  }
  fclose(f);
  return 0;
}

/*
 * areCustomizeConflicts()
 */
int areCustomizeConflicts()
{
  int condor;
  NS_ProblemDesc *pd;
  NS_LinkItem *item = global.problems->start;
  char *customized;

  condor =  (global.Condor_path != NULL);

  if (!condor)
    return 0;

  while(item != NULL)
  {
    pd = (NS_ProblemDesc *)(item->content);
    customized = pd->customized;
    if (strcmp(customized,"CONDOR")&&strcmp(customized,"NONE"))
    {
      fprintf(stderr,"Problem '%s': a Condor server cannot handle '%s' customization.\n", 
                       pd->nickname,customized);
      ns_errno = NetSolveInternalError;
      return -1;
    }
    free(pd->customized);
    pd->customized = strdup("CONDOR");
    item = item->next;
  }
  return 0;
}

int set_defaults(){
  char buffer[256];

  global.my_self = newServerDesc();
  global.my_self->host_desc = newHostDesc();
  global.my_self->workload_history = newWorkloadHistory();
  global.my_self->network_history = newNetworkHistory();

  /*** set logfile to $NETSOLVE_ROOT/SERVER_LOGFILE **/
  global.log_file=(char *)calloc(strlen(global.netsolve_root_path) +
                                   strlen(SERVER_LOGFILE) + 2, sizeof(char));
  strcpy(global.log_file, global.netsolve_root_path);
  strcat(global.log_file, "/");
  strcat(global.log_file, SERVER_LOGFILE);

  global.master = NULL;

  sprintf(buffer,"%s/server_config",global.netsolve_root_path);
  global.config_file = strdup(buffer);
  
  global.Condor_path = NULL;
  global.Condor_nb = 0;

  global.scratch_path = (char *)strdup("/tmp/");

  global.my_self->host_desc->number_proc = 1;

  global.my_self->workload_threshhold = -1;
#ifdef KERBEROS5
  global.require_auth = 0;
#endif
  return 0;
}

int check_args(int argc, char **argv){
  int i;

  for(i=1; i<argc; i++){
    if(strcmp(argv[i], "-f") == 0){
      if(i == argc-1){   /* last argument? */
        print_usage(argv[0]);
        return -1;
      }
      else{
        free(global.config_file);
        global.config_file = strdup(argv[i+1]);
        i++;
      }
    }

    else if(strcmp(argv[i], "-l") == 0){
      if(i == argc-1){   /* last argument? */
        print_usage(argv[0]);
        return -1;
      }
      else{
        free(global.log_file);
        global.log_file = (char *)strdup(argv[i+1]);
        i++;
      }
    }
    else if(strcmp(argv[i], "-k") == 0){
#ifdef KERBEROS5
      global.require_auth = 1;
#else
      fprintf(stderr, "-k option only useful when installed with KERBEROS libraries\n");
      return -1;
#endif
    }

    else{
      print_usage(argv[0]);
      return -1;
    }
  }

  return 0;
}

void print_usage(char *command){
  fprintf(stderr,"Usage: %s [ -f config_file ] [ -k ] [ -l logfile ]\n"
                  , command);
  exit(-1);
}

int daemon_init(){
  pid_t pid;
  int fd;

  pid = fork();

  if(pid < 0){
    perror("fork()");
    return -1;
  }

  if(pid != 0) /* parent exits */
    exit(0);

  pid = setsid();   /* become session leader */

  pid = fork();

  if(pid < 0){
    perror("fork()");
    return -1;
  }

  if(pid != 0) /* 1st child exits */
    exit(0);

  chdir(global.netsolve_root_path);
  umask(0);

  fd = open("/dev/null", O_RDWR|O_CREAT);
  if(fd < 0){
    perror("open()");
    return -1;
  }
  close(0);
  if(dup2(fd, 0) < 0){
    perror("dup2()");
    return -1;
  }
  close(fd);

  fprintf(stdout,
    "\n"
    "==============================================================\n"
    "|       -- Server giving up terminal control! --\n|\n"
    "| All further terminal output will be sent to the log file:\n"
    "|\t\"%s\"\n"
    "=============================================================="
    "\n\n", global.log_file);

  remove(global.log_file);
  fd = open(global.log_file, O_RDWR|O_CREAT, 0644);
  if(fd < 0){
    perror("open()");
    return -1;
  }
  close(1);
  if(dup2(fd, 1) < 0){
    perror("dup2()");
    return -1;
  }
  close(2);
  if(dup2(fd, 2) < 0){
    perror("dup2()");
    return -1;
  }
  close(fd);

  return 0;
}

#ifdef NWS
int setNWSSensor()
{
int pid;
pid = startNWSSensor(NWS_DEFAULT_SENSOR_PORT,global.my_self->host_desc->hostname, global.master->host_desc->hostname,NWS_DEFAULT_NAMESERVER_PORT,global.master->host_desc->hostname,NWS_DEFAULT_MEMORY_PORT);
return pid;
}
#endif

#ifdef KERBEROS5
int check_krb5_envvars(){

  if ( getenv("NETSOLVE_KEYTAB") == NULL)
  {
    fprintf (stderr, "Environment variable NETSOLVE_KEYTAB not defined\n");
    return -1;
  }

  if ( getenv("NETSOLVE_USERS") == NULL)
  {
    fprintf (stderr, "Environment variable NETSOLVE_USERS not defined\n");
    return -1;
  }

  return 0;
}
#endif

/*
 *  server_init()
 *
 *  Performs all the initializations required by the server
 */
int server_init(int argc,char **argv)
{

  /* initialize the NETSOLVE_ROOT path */
  fprintf(stdout, "Getting NetSolve Root Path ... ");
  global.netsolve_root_path=getNetSolveRoot();
  if (global.netsolve_root_path == NULL)
  {
    ns_printinfo();
    fprintf(stderr,"Unable to get Netsolve Root Path\n");
    ns_errno = NetSolveSystemError;
    return -1;
  }  
  fprintf(stdout, "%s\n", global.netsolve_root_path);

  set_defaults();

  /* Initialize the NSLinked Lists */
  global.problems = newNSLinkedList();
  global.agents   = newNSLinkedList();

  check_args(argc, argv);

  fprintf(stdout,"Initializing network connection and running CPU benchmark:\n");

  /* Initialize my_self */
  if (initMySelf() == -1)
    return -1;

/*#ifdef 0 Disabling checks for Condor: not currently supported
  fprintf(stdout,"Condor Server?...\n");
  Condor ?
  if (setCondor(global.config_file) == -1)
    return -1;
#endif */

  fprintf(stdout,"Scratch Space ... ");
  if (setScratchSpace(global.config_file) == -1)
    return -1;
  fprintf(stdout, "%s\n", global.scratch_path);

  fprintf(stdout,"MPI ... ");
  /* MPI ? */
  if (setMPI(global.config_file) == -1)
    return -1;
  fprintf(stdout, "machinefile = %s, nodecount = %d\n",
                  global.MPInodefile, global.nb_MPInodes);

  fprintf(stdout,"Number of processors ... ");
  /* Number of processors ? */
  if (setNumberProc(global.config_file) == -1)
    return -1;
  fprintf(stdout, "%d\n", global.my_self->host_desc->number_proc);

  /* workload threshhold ? */
  fprintf(stdout,"Workload Threshhold ... ");
  if (setWorkloadThreshhold(global.config_file) == -1)
    return -1;
  fprintf(stdout, "%d\n", global.my_self->workload_threshhold);

  /* Attached storage spaces */
  fprintf(stdout,"Storage devices ...\n");
  if (setDisks(global.config_file) == -1)
    return -1;

  /* restrictions */
  fprintf(stdout,"User Access Control ...\n");
  if (setRestrictions(global.config_file) == -1)
    return -1;

/* Signals */
  if (setSignals() == -1)
    return -1;

  fprintf(stdout,"Problems ...\n");
  /* Problems */
  if (setProblems(global.config_file) == -1)
    return -1;
  /* Conflicts ? */
  if (areCustomizeConflicts() == -1)
    return -1;

  fprintf(stdout,"Locating Agent ... ");
  /* Agent */
  if (setMaster(global.config_file) == -1)
    return -1;
  fprintf(stdout, "%s\n", global.master->host_desc->hostname);

  fprintf(stdout,"Registering to NetSolve ... ");
  /* Register to the NetSolve system */
  if (registerToNetSolve(global.master,1) == -1)
    return -1;

#ifdef NWS
  fprintf(stdout,"Starting NWS Sensor ... ");
  if (setNWSSensor()==-1){
    fprintf(stdout,"Impossible to start NWS Sensor\n");
    return -1;
  }
  fprintf(stdout, "success\n");
#endif

  fprintf(stdout,"Starting Workload Manager ... ");
  /* Workload Manager */
  global.workload_manager_pid = startWorkloadManager();
  if (global.workload_manager_pid == -1)
  {
    fprintf(stdout,"Impossible to start workload manager\n");
    return -1;
  }
  fprintf(stdout, "success\n");

#ifdef KERBEROS5
  if (global.require_auth)
  {
    fprintf(stdout, "Checking for Kerberos environment variables... ");
    if(check_krb5_envvars() != 0){
      return -1;
    }
    fprintf(stdout, "success\n");
  }
#endif

  if(daemon_init() < 0 ){
    return -1;
  }

  fflush(stderr);
  fflush(stdout);
  return 0;
}
