/*****************************************************************/
/*      generateservice_sequence.c                               */
/*      Henri Casanova                                           */
/*      Dieter Bachmann                                          */
/*---------------------------------------------------------------*/
/*****************************************************************/

#include "core.h"
#include "serverglobal.h"
#include "generateservice.h"
#include "standardservice.h"
#include "condorservice.h"
#include "petscservice.h"
#include "scalapackservice.h"
#include "netsolveupf.h"
#include "serviceutil.h"
#include "lifelink.h"
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <fcntl.h>
#include <unistd.h>
#include <signal.h>
#include <errno.h>
#include <strings.h>

/*
 * Uncomment the following definition for DAG debugging
 * #define SEQUENCING_DEBUG
 */

/*
 * generateServiceProcess_sequence
 *
 */
void generateServiceProcess_sequence(NS_Communicator *comm, int restriction_index,
				     NS_Node *top_node,char *agent_name,
				     NS_IPaddr_type proxy_IPaddr, int proxy_port)
{

  NS_Node *tmp_node;
  int ID;
  int fd;
  int upf,customized,ftp;
  char tmpdir[256];
  int tag;
  char problemdescfile[256];
  int count=0;
  NS_Socket_type listening_socket;
  NS_Socket_type sock;
  int my_port;
  NS_Communicator *comm_to_client;
  int client_major;
  int my_pid;
  int lifelink_pid;
  int wait_pid;
  int wait_status;

  time_t date1,date2;
  int elapsed;


   if (fork())
       return;  
 
  close(global.sock);


  
  /* Open a new socket on a port */
  listening_socket = bindToFirstAvailablePort(3456,&my_port);
  if (listening_socket == -1)
  {
#ifdef VIEW
    fprintf(stderr,"Impossible to bind to a port\n");
#endif
    if (sendInt(comm,-1) == -1)
      netsolvePerror("sendInt()");
    endTransaction(comm);
    notifyServiceFinished(global.my_self->host_desc->hostname,
                global.my_self->port,restriction_index);
    exit(0);
  }

  /* listening on the socket */
  listen(listening_socket,MAX_CONNECTIONS);

  /*********************************************/
  /* Place myself in a new temporary directory */
  /*********************************************/

  while(1)
  {
    sprintf(tmpdir,"%s/netsolve-tmp-%d",global.scratch_path,(int)getpid()+count);
    if (!access(tmpdir,F_OK))
      count++;
    else
      break;
  }
  if (mkdir(tmpdir,0777))
  {
#ifdef VIEW
    fprintf(stderr,"Impossible to create directory %s\n",tmpdir);
#endif
    if (sendInt(comm,-1) == -1)
      netsolvePerror("sendInt()");
    endTransaction(comm);
    notifyServiceFinished(global.my_self->host_desc->hostname,
                global.my_self->port,restriction_index);
    exit(0);    
  } 

  if (chdir(tmpdir) == -1)
  {
#ifdef VIEW
    fprintf(stderr,"Impossible to chdir to %s\n",tmpdir);
#endif
    if (sendInt(comm,-1) == -1)
      netsolvePerror("sendInt()");
    endTransaction(comm);
    notifyServiceFinished(global.my_self->host_desc->hostname,
                          global.my_self->port,restriction_index);
    cleanUp(tmpdir);
    exit(0); 
  }


  /*****************************************/
  /* Save the problem descriptor to a file */
  /*****************************************/

  netsl_assign_file_names(top_node);  /* assigning file names for dependencies */


  tmp_node = top_node;
  while (tmp_node != NULL) { /* for all nodes */
      sprintf(problemdescfile,"./pb_desc_%d",tmp_node->id); /* one file for each node */
      fd = open(problemdescfile,O_WRONLY|O_CREAT,0666);
      if (fd < 0)
	  {
#ifdef VIEW
	      fprintf(stderr,"Impossible to create file '%s'\n",problemdescfile);
#endif
	      if (sendInt(comm,-1) == -1)
		  netsolvePerror("sendInt()");
	      endTransaction(comm);
	      notifyServiceFinished(global.my_self->host_desc->hostname,
				    global.my_self->port,restriction_index);
	      cleanUp(tmpdir);
	      exit(0);
	  }
      
      if (writeProblemDescToFile(fd,tmp_node->pd) == -1)
	  {
#ifdef VIEW
	      fprintf(stderr,"Impossible to save problem descriptor to file %s\n",
		      problemdescfile);
#endif
	      if (sendInt(comm,-1) == -1)
		  netsolvePerror("sendInt()");
	      endTransaction(comm);
	      notifyServiceFinished(global.my_self->host_desc->hostname,
				    global.my_self->port,restriction_index);
	      cleanUp(tmpdir);
	      exit(0);
	  }
      close(fd);
      tmp_node = tmp_node->next;
  } /* for all nodes */

  /*******************************/
  /* Determine problem specifics */
  /*******************************/
  tmp_node = top_node;
  upf = 0;
  customized = 0;
  ftp = 0;

  while (tmp_node != NULL){

      upf = upf || containsUPF(tmp_node->pd);
      customized = customized ||  strcmp(tmp_node->pd->customized,"NONE");
      ftp = ftp || tmp_node->pd->ftp;

      tmp_node = tmp_node -> next;
  }



  
  /**************************************************/
  /* Send back my new port & pid                    */
  /**************************************************/

  my_pid = getpid();
  if ((sendInt(comm,my_port) == -1) ||
      (sendInt(comm,my_pid) == -1))
  {
    endTransaction(comm);
    notifyServiceFinished(global.my_self->host_desc->hostname,
                          global.my_self->port,restriction_index);
    cleanUp(tmpdir);
    exit(0);
  } 

  /* cut the connection to the client proxy */
  endTransaction(comm);

  /*********************************/
  /* Read all the objects to files */
  /*********************************/

  tmp_node = top_node;
  while (tmp_node != NULL){

  /* Wait for a connection from the client */
  sock = acceptConnection(listening_socket);
  if (sock == -1)
  {
#ifdef VIEW
    fprintf(stderr,"Error while accepting client connection\n");
#endif
    notifyServiceFinished(global.my_self->host_desc->hostname,
                          global.my_self->port,restriction_index);
    cleanUp(tmpdir);
    exit(0);
  }


  comm_to_client = acceptTransaction(sock);
  if (comm_to_client == NULL)
  {
#ifdef VIEW
    fprintf(stderr,"Error while accepting client transaction comm\n");
#endif
    notifyServiceFinished(global.my_self->host_desc->hostname,
                          global.my_self->port,restriction_index);
    cleanUp(tmpdir);
    exit(0);
  }

#ifdef KERBEROS5
  if (global.require_auth) {
    /* tell client we need credentials */
    tag = NS_PROT_KRB5_AUTH_REQUIRED;
    if (sendInt (comm_to_client, tag) == -1)
    {
      netsolvePerror ("sendInt()");
      notifyServiceFinished(global.my_self->host_desc->hostname,
                          global.my_self->port,restriction_index);
      notifyProxy(proxy_IPaddr,proxy_port,ns_errno,ID);
      exit(-1);
    }

    /* get credentials from client */
    if (RecvKerberos5Credentials (comm_to_client) < 0) {
      /*
       * if we didn't like the credentials; tell client that the
       * authentication failed, and this will be the error code
       * associated with the problem request.  otherwise, fall
       * through to other error checking below
       */
      tag = NS_PROT_AUTH_FAILED;
      fprintf(stderr, "failed to receive credentials\n");
      if (sendInt (comm_to_client, tag) == -1)
      {
        netsolvePerror ("sendInt()");
        notifyServiceFinished(global.my_self->host_desc->hostname,
                          global.my_self->port,restriction_index);
        notifyProxy(proxy_IPaddr,proxy_port,ns_errno,ID);
        exit(-1);
      }
      fprintf(stderr, "notifying that service failed\n");
      notifyServiceFinished(global.my_self->host_desc->hostname,
                          global.my_self->port,restriction_index);
      notifyProxy(proxy_IPaddr,proxy_port,ns_errno,ID);
      exit(-1);
    }
    fprintf(stderr, "succeeded receiving credentials\n");
    tag = NS_PROT_ACCEPTED;
    if (sendInt (comm_to_client, tag) == -1)
    {
      netsolvePerror ("sendInt()");
      notifyServiceFinished(global.my_self->host_desc->hostname,
                          global.my_self->port,restriction_index);
      notifyProxy(proxy_IPaddr,proxy_port,ns_errno,ID);
      exit(-1);
    }
  }
  else{
    tag = NS_PROT_ACCEPTED;
    if (sendInt (comm_to_client, tag) == -1)
    {
      netsolvePerror ("sendInt()");
      notifyServiceFinished(global.my_self->host_desc->hostname,
                          global.my_self->port,restriction_index);
      notifyProxy(proxy_IPaddr,proxy_port,ns_errno,ID);
      exit(-1);
    }
  }
#else
  tag = NS_PROT_ACCEPTED;
  if (sendInt (comm_to_client, tag) == -1)
  {
    netsolvePerror ("sendInt()");
    notifyServiceFinished(global.my_self->host_desc->hostname,
                          global.my_self->port,restriction_index);
    notifyProxy(proxy_IPaddr,proxy_port,ns_errno,ID);
    exit(-1);
  }
#endif

  /* receiving the ID and the major */
  if ((recvInt(comm_to_client,&ID) == -1) ||
      (recvInt(comm_to_client,&client_major) == -1))
      {
#ifdef VIEW
	  fprintf(stderr,"Error while accepting client transaction\n");
#endif
	  /* Trying to send something back anyway !! */
	  if (sendInt(comm_to_client,NS_PROT_INTERNAL_FAILURE) == -1)
	      netsolvePerror("sendInt()");
	  endTransaction(comm_to_client);
	  notifyServiceFinished(global.my_self->host_desc->hostname,
				global.my_self->port,restriction_index);
	  /*	      cleanUp(tmpdir); */
	  exit(0);
      }
      
  /* Receiving the input objects */
  if (recvInputObjectsToFiles_sequence(comm_to_client,tmp_node->pd,tmp_node->id) == -1)
	  {
#ifdef VIEW
	      fprintf(stderr,"Error while receiving input objects to files\n");
#endif
	      /* Trying to send something back anyway !! */
	      if (sendInt(comm_to_client,NS_PROT_INTERNAL_FAILURE) == -1)
		  netsolvePerror("sendInt()");
	      endTransaction(comm_to_client);
	      notifyServiceFinished(global.my_self->host_desc->hostname,
				    global.my_self->port,restriction_index);
	      /* cleanUp(tmpdir); */
	      exit(0);
	  }
      tmp_node = tmp_node -> next;
      endTransaction(comm_to_client);

  }
  /* Cutting the connection to the client */




  /* set all nodes to waiting */
  tmp_node = top_node;
  while (tmp_node !=NULL){
      tmp_node->exec_status = NS_DEPENDEND;
      tmp_node = tmp_node -> next;
  }

  time(&date1);


      if ((lifelink_pid = startLifeLink(proxy_IPaddr,proxy_port,NS_PROT_JOB_REPORT,
 				    ID,NetSolveServerError)) == -1)
      {
        ns_printinfo();
	  fprintf(stderr,"Warning: cannot start life link....\n");
      } 
 


  /***************************************************/
  /* catching the signals again                      */
  /***************************************************/

  (void)signal(SIGCHLD,SIG_DFL);



  /***************************************************/
  /* Scheduler                                       */
  /***************************************************/

  while (1) {
    /*
     * Find out if we have any nodes that can run
     * (also sets the dependencies of all waiting processes correct)
     */
    if (netsl_may_run(top_node) == -1) {
/*        fprintf(stderr, */
/*  	      "netsl_may_run() on %s returned -1\n", */
/*  	      top_node ? top_node->pd->nickname : "null"); */
      break;
    }

    /*
     * Now fork all runnable nodes
     */
    if (netsl_fork_exec_all(top_node, listening_socket, ID, 
			    client_major, agent_name, restriction_index,
			    tmpdir, proxy_IPaddr, proxy_port) == -1) {
      fprintf(stderr, "netsl_fork_exec_all() returned -1 for top_node=\"%s\"\n",
	      top_node ? top_node->pd->nickname : "null");
      break;
    }

    /*
     * Wait until a child process exits
     */
    while(-1 == (wait_pid = wait(&wait_status)))
      if(errno != EINTR) {
	fprintf(stderr, "wait() failed! (%s)\n", strerror(errno));
	break;
      }

    /*
     * (jakob) XXX: Catch WIFSIGNALED(wait_status) here, and set NS_errno
     * accordingly - pass back error to client program
     */

    /*
     * Now mark all dependencies of results from wait_pid as satisfied
     */
    if (netsl_resolve_dependencies(top_node, wait_pid) == -1) {
      fprintf(stderr, 
	      "netsl_resolve_dependencies() returned -1 for top_node \"%s\"\n",
	      top_node->pd->nickname);
      break;
    }
    
  }

  time(&date2);
  elapsed = (int)(date2-date1);

  netsl_exit_sequence_scheduler(top_node, lifelink_pid, tmpdir, proxy_IPaddr, 
				proxy_port, elapsed, NS_PROT_SOLVED, client_major, 
				restriction_index, ID, listening_socket );

  netsl_delete_Nodes(top_node);

  /* (jakob) XXX:
   * According to POSIX 3.3.1.3 setting the handler for SIGCHLD to SIG_IGN
   * is a very bad thing to do...  SYSV and BSD differ on behaviour in
   * this case - and by the way - why would we want to ignore the signal
   * at all ???
    (void)signal(SIGCHLD,SIG_IGN);
   */

  exit (0);
}







/* send all needed information back to client and proxy */

int netsl_exit_sequence_scheduler(NS_Node *top_node, int lifelink_pid, char *wd, NS_IPaddr_type proxy_IPaddr, 
				  int proxy_port, int elapsed, int solved, int client_major, 
				  int restriction_index, int ID, int listening_socket ){
   NS_Socket_type   sock;
   NS_Communicator  *comm=NULL;
   NS_Node       *tmp_node;
   char stdoutfile[256];


 /* Notifying the proxy of completion */
  kill(lifelink_pid,SIGKILL);
  /* Is the client proxy still around ? */
  if (notifyProxy(proxy_IPaddr,proxy_port,NetSolveOK,ID) == -1)
  {
    notifyServiceFinished(global.my_self->host_desc->hostname, 
			  global.my_self->port,restriction_index);
    cleanUp(wd);
    exit(0);
  }

  /* Listening on the socket */
  if ((sock = acceptConnection(listening_socket)) == -1)
  {
#ifdef VIEW
    fprintf(stderr,"Error while accepting the connection.\n");
#endif
    notifyServiceFinished( global.my_self->host_desc->hostname, 
			   global.my_self->port, restriction_index);
    cleanUp(wd);
    exit(0);
  }
  comm = acceptTransaction(sock);
  if (comm == NULL)
  {
#ifdef VIEW
    fprintf(stderr,"Error while accepting the transaction.\n");
#endif
    notifyServiceFinished(global.my_self->host_desc->hostname, 
			   global.my_self->port, restriction_index);
    cleanUp(wd);
    exit(0);
  }

  /* Sending back the Computational time elapsed */
  if (sendInt(comm,elapsed) == -1)
  {
#ifdef VIEW
    fprintf(stderr,"Error while sending the elapsed time.\n");
#endif
    notifyServiceFinished(global.my_self->host_desc->hostname, 
			   global.my_self->port,restriction_index);
    endTransaction(comm);
    cleanUp(wd);
    exit(0);
  }

  /* Sending back the stdout */
  sprintf(stdoutfile,"%s/netsolve-stdout",wd);
  if (sendFileAsString(comm,stdoutfile) == -1)
  {
#ifdef VIEW
    fprintf(stderr,"Error while sending the stdout.\n");
#endif
    notifyServiceFinished(global.my_self->host_desc->hostname, 
			   global.my_self->port, restriction_index);
    endTransaction(comm);
    cleanUp(wd);
    exit(0);
  }

  /* Error */
  if (solved != NS_PROT_SOLVED)
  {
    sendInt(comm,solved);
    endTransaction(comm);
    notifyServiceFinished(global.my_self->host_desc->hostname, 
			   global.my_self->port ,restriction_index);
    cleanUp(wd);
    exit(0);
  }

  sendInt(comm,solved);

  /* Send back the output */
  /* this takes care of the transpose */
  tmp_node = top_node;
  while(tmp_node != NULL){
      if (sendOutputObjectsFromFiles(comm,tmp_node->pd) == -1)
	  {
#ifdef VIEW
	      fprintf(stderr,"Error while sending the output objects.\n");
	      
#endif
	      fprintf(stderr,"Sleeping 1000 seconds\n");sleep(1000);
	      notifyServiceFinished(global.my_self->host_desc->hostname, 
			   global.my_self->port, restriction_index);
	  
	      endTransaction(comm);
	      cleanUp(wd);
	      exit(0);
	  }
      tmp_node = tmp_node -> next;
  } /* for all nodes */
  /* Terminating the transaction */
  endTransaction(comm);
  notifyServiceFinished(global.my_self->host_desc->hostname, 
			global.my_self->port, restriction_index);

#ifdef VIEW
  fprintf(stderr,"Completed\n");
#endif

  /* Cleaning up */
  close(listening_socket);
  cleanUp(wd);
  return 0;



} /*netsl_exit_agent() */



/* set the execution status of ode with wait_pid  to terminate */
/* update all other dependencies */

int netsl_resolve_dependencies(NS_Node *top_node, int  wait_pid){

    NS_Node *tmp_node = top_node;
    int term_id = -1;
    int i;
    
    /* determine the node number of the terminated process */
    while (tmp_node != NULL){
	if (tmp_node->process_id == wait_pid) {
	    term_id = tmp_node->id;
	    tmp_node->exec_status = NS_TERMINATE;
#ifdef SEQUENCING_DEBUG
	    fprintf(stderr, "Marked termination of %s\n", tmp_node->pd->nickname);
#endif
	}
	tmp_node = tmp_node -> next;
    }
    if (term_id == -1) {
	fprintf(stderr,"Process not found:%d\n",wait_pid);
	return -1;
    }


    /* remove all dependencies from other nodes */
    tmp_node = top_node;
    while (tmp_node != NULL){ 
	for (i = 0; i < tmp_node->pd->nb_input_objects; i++){
	    if (tmp_node->in_dep->dest_node[i] == term_id){ /* input was dependent on terminated node */
		tmp_node->in_dep->dest_node[i] = -1;
		tmp_node->in_dep->dep_type[i] = NETSOLVE_NO_DEPENDENCY;
#ifdef SEQUENCING_DEBUG
		fprintf(stderr, "Resolved input dependency from %s input %i\n",
			tmp_node->pd->nickname, i);
#endif
	    }
	} /* for all input objects*/

	for (i = 0; i < tmp_node->pd->nb_output_objects; i++){
	    if (tmp_node->out_dep->dest_node[i] == term_id){ /* input was dependent on terminated node */
		tmp_node->out_dep->dest_node[i] = -1;
		tmp_node->out_dep->dep_type[i] = NETSOLVE_NO_DEPENDENCY;
#ifdef SEQUENCING_DEBUG
		fprintf(stderr, "Resolved output dependency from %s output %i\n",
			tmp_node->pd->nickname, i);
#endif
	    }
	} /* for all output objects */

#ifdef SEQUENCING_DEBUG
	fprintf(stderr, "---start total dependencies for %s----\n",
		tmp_node->pd->nickname);
	for (i = 0; i < tmp_node->pd->nb_input_objects; i++)
	  fprintf(stderr, " in %i - dep %i  with peer %i\n", 
		  i, tmp_node->in_dep->dep_type[i],
		  tmp_node->in_dep->dest_node[i]);
	for (i = 0; i < tmp_node->pd->nb_output_objects; i++)
	  fprintf(stderr, "out %i - dep %i  with peer %i\n",
		  i, tmp_node->out_dep->dep_type[i],
		  tmp_node->out_dep->dest_node[i]);	
	fprintf(stderr, "---end total dependencies for----\n");	
#endif

	tmp_node=tmp_node->next;
    } /* for all nodes */
    
    return 0;

}/* netsl_resolve_dependencies */

   /* check for processes that may run now */
int netsl_may_run (NS_Node *top_node){

    int i;
    NS_Node *tmp_node;
    int dep;
    tmp_node = top_node;
    
    while (tmp_node != NULL){
	dep = FALSE;
	if (tmp_node-> exec_status == NS_DEPENDEND){
	  /*
	   * Check dependencies on input objects
	   */
	    for (i = 0; i < tmp_node->pd->nb_input_objects; i++){
		if ((tmp_node->in_dep->dep_type[i] == NETSOLVE_INPUT_DEPENDENCY) ||
		    (tmp_node->in_dep->dep_type[i] == NETSOLVE_OUTPUT_DEPENDENCY)){
		    dep = TRUE;
#ifdef SEQUENCING_DEBUG
		    fprintf(stderr, 
			    "netsl_may_run() got early dep=TRUE (%d) for %s i=%d "
			    "and top node %s\n",
			    tmp_node->in_dep->dep_type[i],
			    tmp_node->pd->nickname, 
			    i, top_node->pd->nickname);
#endif
		}
	    }
	    if (dep == FALSE){
	      /*
	       * Check dependencies on output objects
	       */
		for (i = 0; i < tmp_node->pd->nb_output_objects; i++){
		  if (((tmp_node->out_dep->dep_type[i] == NETSOLVE_INPUT_DEPENDENCY) && 
		       (tmp_node->out_dep->dest_node[i] != tmp_node->id ))
/*  		       (tmp_node->out_dep->dest_node[i] != i )) */
		      ||     /* no local dependency */
		      (tmp_node->out_dep->dep_type[i] == NETSOLVE_OUTPUT_DEPENDENCY)){
		    dep = TRUE;
#ifdef SEQUENCING_DEBUG
		    fprintf(stderr,
			    "netsl_may_run() got late dep=TRUE (dep_type=%d) "
			    "for %s out i=%d "
			    "and top node %s\n",
			    tmp_node->out_dep->dep_type[i],
			    tmp_node->pd->nickname, 
			    i, top_node->pd->nickname);
#endif
		  }
		}
		/*
		 * If there are no dependencies stopping the node from
		 * executing, allow it to execute
		 */
		if (dep == FALSE){	    
		    tmp_node->exec_status = NS_WAITING;
#ifdef SEQUENCING_DEBUG
		    fprintf(stderr, "netsl_may_run() set NS_WAITING for %s\n",
			    tmp_node->pd->nickname);
#endif
		}
	    }  /*if */
	}  /*if DEPENDEND */
	tmp_node=tmp_node->next;
    } /* for all nodes */

    /*
     * We must return -1 if no nodes are runnable
     */

    tmp_node = top_node;
    while(tmp_node != NULL){
      if (tmp_node->exec_status != NS_TERMINATE)
	return 0;
/*        if(tmp_node->exec_status == NS_WAITING) */
/*  	return 0; */
      tmp_node = tmp_node->next;
    }


    return -1;
} /* netsl_may run */


/* fork off a new process for each waiting job */
/* store the pid to determine, process termination */

int netsl_fork_exec_all (NS_Node *top_node, int listening_socket, int ID, 
			 int client_major, char* agent_name, int restriction_index,
			 char* tmpdir, int proxy_IPaddr, int proxy_port){

    NS_Node *tmp_node;
    char *xname;
    char *xpath;
    char buffer[256];
    int lifelink_pid = 0;
    int  child_pid;
    

    tmp_node = top_node;
    while (tmp_node != NULL){
	/* may this node run? */
	if (tmp_node->exec_status == NS_WAITING) { /* this node whises to be executed */
	    tmp_node->exec_status = NS_RUNNING; /* set status to running */

	    /***************************************************/
	    /* Set the default path and name of the executable */
	    /***************************************************/
	    sprintf(buffer,"%s/bin/%s/",
		    global.netsolve_root_path,NETSOLVE_ARCH);
	    xpath = strdup(buffer);
	    sprintf(buffer,"service-%s",tmp_node->pd->file);
	    xname = strdup(buffer);

#ifdef SEQUENCING_DEBUG
	    fprintf(stderr, "Ok, we're right about to fork off node id %i\n",
		    tmp_node->id);
#endif

	    child_pid =  fork();

            if (child_pid == 0){


		/*************************************************************/
		/* Start the life link to the proxy                          */
		/*************************************************************/
			forkStandardFileService_sequence(listening_socket,ID,client_major,
							 lifelink_pid,agent_name,tmpdir, 
							 global.my_self->host_desc->hostname,
							 global.my_self->port,restriction_index, 
							 xpath,xname,tmp_node->pd,proxy_IPaddr,proxy_port,tmp_node->id); 
		printf("exiting:%d\n",getpid());	       
		exit (0);
	    }
	    else{
		tmp_node->process_id  = child_pid;

	    }

	}

    tmp_node = tmp_node-> next;
    }

  return 0;
}

/* recursive resolve links */
/* follow all links rekursive and assign the filename of hte first occurence */
char *netsl_resolve_name(NS_Node *act_node, int dep_type, int dest_node_id, int dest_arg){

char *buffer;
char * r_buffer;

buffer = malloc(100 * sizeof (char));

    while((act_node->id != dest_node_id) && (act_node !=0)){
	act_node = act_node->last; /* find the dest_node */
    }

    switch (dep_type) {
    case NETSOLVE_INPUT_DEPENDENCY:
	if (act_node->in_dep->dep_type[dest_arg] == NETSOLVE_NO_DEPENDENCY){
		sprintf(buffer,"input%d_%d", act_node->id, dest_arg);
		r_buffer = strdup(buffer);
		free (buffer);
		return r_buffer;
	}
	else
	    free (buffer);
	    return netsl_resolve_name(act_node, act_node->in_dep->dep_type[dest_arg],  
				      act_node->in_dep->dest_node[dest_arg],  
				      act_node->in_dep->dest_arg[dest_arg]);
	    
	    
	    
	break;
    case NETSOLVE_OUTPUT_DEPENDENCY:
	if (act_node->out_dep->dep_type[dest_arg] == NETSOLVE_NO_DEPENDENCY){
		sprintf(buffer,"output%d_%d", act_node->id, dest_arg);
		r_buffer = strdup(buffer);
		free (buffer);
		return r_buffer;
	}
	else
	    free (buffer);
	    return netsl_resolve_name(act_node, act_node->out_dep->dep_type[dest_arg],  
				      act_node->out_dep->dest_node[dest_arg],  
				      act_node->out_dep->dest_arg[dest_arg]);
	
	break;
    }


    return 0;
} /* char *netsl_resolve_name(NS_Node *act_node, int dep_type, int dest_node_id, int dest_arg) */



/* assigns a filename to each input and output */
int netsl_assign_file_names (NS_Node *top_node){

    int i;
    NS_Node *tmp_node = top_node;
    char *buffer;

    buffer = malloc(100 * sizeof (char));

    while(tmp_node != NULL){
      if (tmp_node->pd->input_names == NULL) 
	  tmp_node->pd->input_names = malloc ( tmp_node->pd->nb_input_objects * sizeof (char *));
	for (i = 0; i< tmp_node->pd->nb_input_objects; i++){
	    if (tmp_node->in_dep->dep_type[i] == NETSOLVE_NO_DEPENDENCY){
		sprintf(buffer,"input%d_%d", tmp_node->id,i); /* no link */
		tmp_node->pd->input_names[i] = strdup(buffer);
	    }
	    else /* recursive link trace */{
		tmp_node->pd->input_names[i] = netsl_resolve_name(tmp_node, tmp_node->in_dep->dep_type[i],  
								  tmp_node->in_dep->dest_node[i],  
								  tmp_node->in_dep->dest_arg[i]);
	    }


	}
      if (tmp_node->pd->output_names == NULL) 
	  tmp_node->pd->output_names = malloc ( tmp_node->pd->nb_output_objects * sizeof (char *));

	for (i = 0; i< tmp_node->pd->nb_output_objects; i++){
	    if (tmp_node->out_dep->dep_type[i] == NETSOLVE_NO_DEPENDENCY){
		sprintf(buffer,"output%d_%d", tmp_node->id,i); /* no link */
		tmp_node->pd->output_names[i]=strdup(buffer);
	    }
	    else{ /* recursive link trace */
		tmp_node->pd->output_names[i] = netsl_resolve_name(tmp_node, tmp_node->out_dep->dep_type[i],  
								  tmp_node->out_dep->dest_node[i],  
								  tmp_node->out_dep->dest_arg[i]);
	    }
	}
	
	tmp_node = tmp_node-> next;
    }
    
    free(buffer);
    return 0;

} /* netsl_assign_file_names */
