/*****************************************************************/
/*      generateservice.c                                        */
/*      Henri Casanova, Dorian Arnold                            */
/*---------------------------------------------------------------*/
/*****************************************************************/

#include "core.h"
#include "serverglobal.h"
#include "generateservice.h"
#include "standardservice.h"
#include "condorservice.h"
#include "petscservice.h"
#include "scalapackservice.h"
#include "netsolveupf.h"
#include "netsolveftp.h"
#include "serviceutil.h"
#include "lifelink.h"
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <fcntl.h>
#include <unistd.h>
#include <signal.h>

/*
 * generateServiceProcess
 *
 */
void generateServiceProcess(NS_Communicator *comm, int restriction_index,
                            NS_ProblemDesc *pd,char *agent_name,
                            NS_IPaddr_type proxy_IPaddr, int proxy_port)
{
  int ID;
  int fd;
  int upf,customized,ftp;
  char tmpdir[256];
  int tag;
  char *xname;
  char *xpath;
  char problemdescfile[256];
  char buffer[256];
  int status;
  int count=0;
  NS_Socket_type listening_socket;
  NS_Socket_type sock;
  int my_port;
  NS_Communicator *comm_to_client;
  int client_major;
  int my_pid;
  int lifelink_pid;

  if (fork())
    return; 

  close(global.sock);
  
  /* Open a new socket on a port */
  listening_socket = bindToFirstAvailablePort(3456,&my_port);
  if (listening_socket == -1)
  {
#ifdef VIEW
    fprintf(stderr,"Impossible to bind to a port\n");
#endif
    if (sendInt(comm,-1) == -1)
      netsolvePerror("sendInt()");
    endTransaction(comm);
    notifyServiceFinished(global.my_self->host_desc->hostname,
                global.my_self->port,restriction_index);
    exit(0);
  }

  /* listening on the socket */
  listen(listening_socket,MAX_CONNECTIONS);

  /*********************************************/
  /* Place myself in a new temporary directory */
  /*********************************************/
  while(1)
  {
    if (!strcmp(pd->customized,"ITER_SOLVE") || 
			!strcmp(pd->customized,"SCALAPACK")){
      sprintf(tmpdir,"%s/src/Server/netsolve-tmp-%d",
                global.netsolve_root_path, (int)getpid()+count);
    }
    else{
      sprintf(tmpdir,"%s/netsolve-tmp-%d",global.scratch_path,
                        (int)getpid()+count);
    }
    if (!access(tmpdir,F_OK))
      count++;
    else
      break;
  }
  if (mkdir(tmpdir,0777))
  {
#ifdef VIEW
    fprintf(stderr,"Impossible to create directory %s\n",tmpdir);
#endif
    if (sendInt(comm,-1) == -1)
      netsolvePerror("sendInt()");
    endTransaction(comm);
    notifyServiceFinished(global.my_self->host_desc->hostname,
                global.my_self->port,restriction_index);
    exit(0);    
  } 

  if (chdir(tmpdir) == -1)
  {
#ifdef VIEW
    fprintf(stderr,"Impossible to chdir to %s\n",tmpdir);
#endif
    if (sendInt(comm,-1) == -1)
      netsolvePerror("sendInt()");
    endTransaction(comm);
    notifyServiceFinished(global.my_self->host_desc->hostname,
                          global.my_self->port,restriction_index);
    cleanUp(tmpdir);
    exit(0); 
  }


  /*****************************************/
  /* Save the problem descriptor to a file */
  /*****************************************/
  sprintf(problemdescfile,"./pb_desc");
  fd = open(problemdescfile,O_WRONLY|O_CREAT,0666);
  if (fd < 0)
  {
#ifdef VIEW
    fprintf(stderr,"Impossible to create file '%s'\n",problemdescfile);
#endif
    if (sendInt(comm,-1) == -1)
      netsolvePerror("sendInt()");
    endTransaction(comm);
    notifyServiceFinished(global.my_self->host_desc->hostname,
                          global.my_self->port,restriction_index);
    cleanUp(tmpdir);
    exit(0);
  }

  if (writeProblemDescToFile(fd,pd) == -1)
  {
#ifdef VIEW
    fprintf(stderr,"Impossible to save problem descriptor to file %s\n",
           problemdescfile);
#endif
    if (sendInt(comm,-1) == -1)
      netsolvePerror("sendInt()");
    endTransaction(comm);
    notifyServiceFinished(global.my_self->host_desc->hostname,
                          global.my_self->port,restriction_index);
    cleanUp(tmpdir);
    exit(0);
  }
  close(fd);

  /*******************************/
  /* Determine problem specifics */
  /*******************************/
  upf = containsUPF(pd);
  customized = strcmp(pd->customized,"NONE");
  ftp = pd->ftp;

  /***************************************************/
  /* Set the default path and name of the executable */
  /***************************************************/
  sprintf(buffer,"%s/bin/%s/",
            global.netsolve_root_path, NETSOLVE_ARCH);
  xpath = strdup(buffer);
  sprintf(buffer,"service-%s",pd->file);
  xname = strdup(buffer);
  
  /**************************************************/
  /* Send back my new port & pid                    */
  /**************************************************/
  my_pid = getpid();
  if ((sendInt(comm,my_port) == -1) ||
      (sendInt(comm,my_pid) == -1))
  {
    endTransaction(comm);
    notifyServiceFinished(global.my_self->host_desc->hostname,
                          global.my_self->port,restriction_index);
    cleanUp(tmpdir);
    exit(0);
  } 

  /* cut the connection to the client proxy */
  endTransaction(comm);

  /*************************************************************/
  /* Use the standard service                                  */
  /* that will read the objects in input and send back the ack */
  /*************************************************************/

  if ((!ftp) && (!upf) && (!customized))
  {
    forkStandardNoFileService(listening_socket,
                              agent_name,
                              tmpdir,
                              global.my_self->host_desc->hostname,
                              global.my_self->port,
                              restriction_index,
                              xpath,
                              xname,
                              pd,
                              proxy_IPaddr,
                              proxy_port);
    cleanUp(tmpdir);
    exit(0);
  }

  /*********************************/
  /* Read all the objects to files */
  /*********************************/

  /* Wait for a connection from the client */
  sock = acceptConnection(listening_socket);
  if (sock == -1)
  {
#ifdef VIEW
    fprintf(stderr,"Error while accepting client connection\n");
#endif
    notifyServiceFinished(global.my_self->host_desc->hostname,
                          global.my_self->port,restriction_index);
    cleanUp(tmpdir);
    exit(0);
  }
#ifdef DEBUG
  fprintf(stderr,"Got connection for input objects\n");
#endif

  comm_to_client = acceptTransaction(sock);
  if (comm_to_client == NULL)
  {
#ifdef VIEW
    fprintf(stderr,"Error while accepting client transaction\n");
#endif
    notifyServiceFinished(global.my_self->host_desc->hostname,
                          global.my_self->port,restriction_index);
    cleanUp(tmpdir);
    exit(0);
  }

#ifdef KERBEROS5
  if (global.require_auth) {
    /* tell client we need credentials */
    tag = NS_PROT_KRB5_AUTH_REQUIRED;
    if (sendInt (comm_to_client, tag) == -1)
    {
      netsolvePerror ("sendInt()");
      notifyServiceFinished(global.my_self->host_desc->hostname,
                          global.my_self->port,restriction_index);
      notifyProxy(proxy_IPaddr,proxy_port,ns_errno,ID);
      freeProblemDesc (pd);
      exit(-1);
    }

    /* get credentials from client */
    if (RecvKerberos5Credentials (comm_to_client) < 0) {
      /*
       * if we didn't like the credentials; tell client that the
       * authentication failed, and this will be the error code
       * associated with the problem request.  otherwise, fall
       * through to other error checking below
       */
      tag = NS_PROT_AUTH_FAILED;
      fprintf(stderr, "failed to receive credentials\n");
      if (sendInt (comm_to_client, tag) == -1)
      {
        netsolvePerror ("sendInt()");
        notifyServiceFinished(global.my_self->host_desc->hostname,
                          global.my_self->port,restriction_index);
        notifyProxy(proxy_IPaddr,proxy_port,ns_errno,ID);
        freeProblemDesc (pd);
        exit(-1);
      }
      fprintf(stderr, "notifying that service failed\n");
      notifyServiceFinished(global.my_self->host_desc->hostname,
                          global.my_self->port,restriction_index);
      notifyProxy(proxy_IPaddr,proxy_port,ns_errno,ID);
      exit(-1);
    }
    fprintf(stderr, "succeeded receiving credentials\n");
    tag = NS_PROT_ACCEPTED;
    if (sendInt (comm_to_client, tag) == -1)
    {
      netsolvePerror ("sendInt()");
      notifyServiceFinished(global.my_self->host_desc->hostname,
                          global.my_self->port,restriction_index);
      notifyProxy(proxy_IPaddr,proxy_port,ns_errno,ID);
      freeProblemDesc (pd);
      exit(-1);
    }
  }
  else{
    tag = NS_PROT_ACCEPTED;
    if (sendInt (comm_to_client, tag) == -1)
    {
      netsolvePerror ("sendInt()");
      notifyServiceFinished(global.my_self->host_desc->hostname,
                          global.my_self->port,restriction_index);
      notifyProxy(proxy_IPaddr,proxy_port,ns_errno,ID);
      freeProblemDesc (pd);
      exit(-1);
    }
  }
#else
  tag = NS_PROT_ACCEPTED;
  if (sendInt (comm_to_client, tag) == -1)
  {
    netsolvePerror ("sendInt()");
    notifyServiceFinished(global.my_self->host_desc->hostname,
                          global.my_self->port,restriction_index);
    notifyProxy(proxy_IPaddr,proxy_port,ns_errno,ID);
    freeProblemDesc (pd);
    exit(-1);
  }
#endif

  /* receiving the ID and the major */
  if ((recvInt(comm_to_client,&ID) == -1) ||
      (recvInt(comm_to_client,&client_major) == -1))
  {
#ifdef VIEW
    fprintf(stderr,"Error while accepting client transaction\n");
#endif
    /* Trying to send something back anyway !! */
    if (sendInt(comm_to_client,NS_PROT_INTERNAL_FAILURE) == -1)
      netsolvePerror("sendInt()");
    endTransaction(comm_to_client);
    notifyServiceFinished(global.my_self->host_desc->hostname,
                          global.my_self->port,restriction_index);
    cleanUp(tmpdir);
    exit(0);
  }

  /* Receiving the input objects */
  if (recvInputObjectsToFiles(comm_to_client,pd) == -1)
  {
#ifdef VIEW
    fprintf(stderr,"Error while receiving input objects to files\n");
#endif
    /* Trying to send something back anyway !! */
    if (sendInt(comm_to_client,NS_PROT_INTERNAL_FAILURE) == -1)
      netsolvePerror("sendInt()");
    endTransaction(comm_to_client);
    notifyServiceFinished(global.my_self->host_desc->hostname,
                          global.my_self->port,restriction_index);
    cleanUp(tmpdir);
    exit(0);
  }

  /* Cutting the connection to the client */
  endTransaction(comm_to_client);

  /*************************************************************/
  /* Start the life link to the proxy                          */
  /*************************************************************/
  if ((lifelink_pid = startLifeLink(proxy_IPaddr,proxy_port,NS_PROT_JOB_REPORT,
            ID,NetSolveServerError)) == -1)
  {
    ns_printinfo();
    fprintf(stderr,"Warning: cannot start life link....\n");
  }

  /* Process possible UPFs */
  if (upf)
  {
    status = processUPF(pd,tmpdir,&xname,&xpath);
    if (status  == -1)
    {
      kill(lifelink_pid,SIGKILL);
      notifyProxy(proxy_IPaddr,proxy_port,ns_errno,ID);
      notifyServiceFinished(global.my_self->host_desc->hostname,
                            global.my_self->port,restriction_index);
      cleanUp(tmpdir);
      exit(0);
    }
  }

  if (ftp)
  {
    status = processFTP(pd,tmpdir,&xname,&xpath);
    if (status == -1)
    {
      kill(lifelink_pid,SIGKILL);
      notifyProxy(proxy_IPaddr,proxy_port,ns_errno,ID);
      notifyServiceFinished(global.my_self->host_desc->hostname,
                            global.my_self->port,restriction_index);
      cleanUp(tmpdir);
      exit(0);

    }
  }
 
  if (!customized)
  {
    /* that will send back the ack */
    forkStandardFileService(listening_socket,ID,client_major,
                            lifelink_pid,agent_name,tmpdir,
                            global.my_self->host_desc->hostname,
                            global.my_self->port,restriction_index,
                            xpath,xname,pd,proxy_IPaddr,proxy_port);
  }

  if (!strcmp(pd->customized,"SCALAPACK"))
  {
    forkScaLAPACKService(listening_socket,ID,client_major,
                         lifelink_pid,agent_name,tmpdir,
                         global.my_self->host_desc->hostname,
                         global.my_self->port,restriction_index,
                         xpath,xname,pd,proxy_IPaddr,proxy_port);
  }
  else if (!strcmp(pd->customized,"CONDOR"))
  {
    forkCondorService(listening_socket,ID,client_major,
                      lifelink_pid,agent_name,tmpdir,
                      global.my_self->host_desc->hostname,
                      global.my_self->port,restriction_index,
                      xpath,xname,pd,proxy_IPaddr,proxy_port);
  }
  else if (!strcmp(pd->customized,"ITER_SOLVE"))
  {
    forkiterativeSolverService(listening_socket,ID,client_major,
                     lifelink_pid,agent_name,tmpdir,
                     global.my_self->host_desc->hostname,
                     global.my_self->port,restriction_index,
                     xpath,xname,pd,proxy_IPaddr,proxy_port);
  }
  else
  {
#ifdef VIEW
    fprintf(stderr,"Unknown Customization\n");
#endif
    kill(lifelink_pid,SIGKILL);
    notifyProxy(proxy_IPaddr,proxy_port,NetSolveInternalError,ID);
    notifyServiceFinished(global.my_self->host_desc->hostname,
                          global.my_self->port,restriction_index);
    cleanUp(tmpdir);
    exit(0);

  }

  exit(0);
}
