/*****************************************************************/
/*      standardservive.c                                        */
/*      Henri Casanova                                           */
/*---------------------------------------------------------------*/
/*****************************************************************/

#include "core.h"
#include "serverglobal.h"
#include "standardservice.h"
#include "serviceutil.h"
#include "lifelink.h"
#include <fcntl.h>
#include <unistd.h>
#include "hbm_wrappers.h"
#include <time.h>
#include <signal.h>

#ifdef sunos
int __main() { }
int MAIN_() { }
#endif

/*
 * standardService()
 */
int main(int argc, char **argv)
{
  time_t date1,date2;
  int elapsed;
  NS_Communicator *comm=NULL;
  char *serverhost;
  int serverport;
  int restriction_index;
  NS_ProblemDesc *pd;
  int solved;
  char *wd=NULL;
  int tag;
  int fd;
  int filemode;
  int client_major;
  int stdoutfd;
  int i;
  char buffer[256];
  /* char *agent_name; */
  char stdoutfile[256];
  NS_IPaddr_type proxy_IPaddr;
  int proxy_port;
  NS_Socket_type listening_socket;
  NS_Socket_type sock;
  int ID;
  int lifelink_pid;
  int node_nr;

  /* Parse the comand line */
  if (argc != 14)
  {
#if (defined(VIEW) || defined(DEBUG))
    ns_printinfo();
    fprintf(stderr,"Wrong number of command line args.\n");
#endif
    cleanUp(wd);
    exit(-1);
  }

  listening_socket = atoi(argv[1]);
  serverhost = argv[2];
  serverport = atoi(argv[3]);
  restriction_index = atoi(argv[4]);
  wd = argv[5];
  filemode = atoi(argv[6]);
  client_major = atoi(argv[7]);
  /* agent_name = argv[8]; */
  proxy_IPaddr = getIPaddr(argv[9]);
  proxy_port = atoi(argv[10]);
  ID = atoi(argv[11]);
  lifelink_pid = atoi(argv[12]);
  node_nr = atoi(argv[13]);

#ifdef KERBEROS5
  if(node_nr == -2)
    global.require_auth = 1;
#endif

  chdir(wd);


  /* Get the Problem Descriptor */
  if (node_nr == -1 || node_nr == -2)
    sprintf(buffer,"./pb_desc"); /* if sequences uses different pb-files */
  else
    sprintf(buffer,"./pb_desc_%d",node_nr);

  fd = open(buffer,O_RDONLY);
  if (fd < 0)
  {
#if (defined(VIEW) || defined(DEBUG))
    ns_printinfo();
    fprintf(stderr,"Impossible to open the Problem Descriptor file.\n");
    perror("open()");
    fprintf(stderr, "Current Dir:");
    system("pwd");
    fprintf(stderr, "Dir Contents:");
    system("ls");
#endif
    notifyServiceFinished(serverhost,serverport,restriction_index);
    if (filemode == FILES)
      kill(lifelink_pid,SIGKILL);
    notifyProxy(proxy_IPaddr,proxy_port,NetSolveInternalError,ID);
    cleanUp(wd);
    exit(-1);
  }
  pd = readProblemDescFromFile(fd);
  close(fd);
  
  if (pd == NULL)
  {
#if (defined(VIEW) || defined(DEBUG))
    fprintf(stderr,"Error while reading the Problem Descriptor file.\n");
#endif
    notifyServiceFinished(serverhost,serverport,restriction_index);
    if (filemode == FILES)
      kill(lifelink_pid,SIGKILL);
    notifyProxy(proxy_IPaddr,proxy_port,NetSolveInternalError,ID);
    cleanUp(wd);
    exit(-1);
  }

#if (defined(VIEW) || defined(DEBUG))
  fprintf(stderr,"Solving '%s' - pid is %i\n",pd->nickname,getpid());
#endif

  /* Receive the input objects */
  if (filemode == NOFILES)
  {
#ifdef DEBUG
    fprintf(stderr, "Receiving Input Objects\n");
#endif
    if ((sock = acceptConnection(listening_socket)) == -1)
    {
#if (defined(VIEW) || defined(DEBUG))
      fprintf(stderr,"Error while accepting the connection.\n");
#endif
      notifyServiceFinished(serverhost,serverport,restriction_index);
      notifyProxy(proxy_IPaddr,proxy_port,ns_errno,ID);
      cleanUp(wd);
      exit(-1);
    } 

    comm = acceptTransaction(sock);
    if (comm == NULL)
    {
#if (defined(VIEW) || defined(DEBUG))
      fprintf(stderr,"Error while accepting the transaction.\n");
#endif
      notifyServiceFinished(serverhost,serverport,restriction_index);
      notifyProxy(proxy_IPaddr,proxy_port,ns_errno,ID);
      cleanUp(wd);
      exit(-1);
    }

#ifdef KERBEROS5
  if (global.require_auth) {
    /* tell client we need credentials */
    tag = NS_PROT_KRB5_AUTH_REQUIRED;
    if (sendInt (comm, tag) == -1)
    {
      netsolvePerror ("sendInt()");
      notifyServiceFinished(serverhost,serverport,restriction_index);
      notifyProxy(proxy_IPaddr,proxy_port,ns_errno,ID);
      freeProblemDesc (pd);
      exit(-1);
    }

    /* get credentials from client */
    if (RecvKerberos5Credentials (comm) < 0) {
      /*
       * if we didn't like the credentials; tell client that the
       * authentication failed, and this will be the error code
       * associated with the problem request.  otherwise, fall
       * through to other error checking below
       */
      tag = NS_PROT_AUTH_FAILED;
      fprintf(stderr, "failed to receive credentials\n");
      if (sendInt (comm, tag) == -1)
      {
        netsolvePerror ("sendInt()");
        notifyServiceFinished(serverhost,serverport,restriction_index);
        notifyProxy(proxy_IPaddr,proxy_port,ns_errno,ID);
        freeProblemDesc (pd);
        exit(-1);
      }
      fprintf(stderr, "notifying that service failed\n");
      notifyServiceFinished(serverhost,serverport,restriction_index);
      notifyProxy(proxy_IPaddr,proxy_port,ns_errno,ID);
      exit(-1);
    }
    fprintf(stderr, "succeeded receiving credentials\n");
    tag = NS_PROT_ACCEPTED;
    if (sendInt (comm, tag) == -1)
    {
      netsolvePerror ("sendInt()");
      notifyServiceFinished(serverhost,serverport,restriction_index);
      notifyProxy(proxy_IPaddr,proxy_port,ns_errno,ID);
      freeProblemDesc (pd);
      exit(-1);
    }
  }
  else{
    tag = NS_PROT_ACCEPTED;
    if (sendInt (comm, tag) == -1)
    {
      netsolvePerror ("sendInt()");
      notifyServiceFinished(serverhost,serverport,restriction_index);
      notifyProxy(proxy_IPaddr,proxy_port,ns_errno,ID);
      freeProblemDesc (pd);
      exit(-1);
    }
  }
#else
  tag = NS_PROT_ACCEPTED;
  if (sendInt (comm, tag) == -1)
  {
    netsolvePerror ("sendInt()");
    notifyServiceFinished(serverhost,serverport,restriction_index);
    notifyProxy(proxy_IPaddr,proxy_port,ns_errno,ID);
    freeProblemDesc (pd);
    exit(-1);
  }
#endif

    /* Receive the ID and the client_major */
    if ((recvInt(comm,&ID) == -1) ||
        (recvInt(comm,&client_major) == -1))
    {
#if (defined(VIEW) || defined(DEBUG))
      fprintf(stderr,"Error while receiving the ID/major.\n");
#endif
      notifyServiceFinished(serverhost,serverport,restriction_index);
      notifyProxy(proxy_IPaddr,proxy_port,ns_errno,ID);
      cleanUp(wd);
      exit(-1);

    }

    if (recvInputObjects(comm,pd) == -1)
    {
#if (defined(VIEW) || defined(DEBUG))
      fprintf(stderr,"Error while receiving the input objects.\n");
#endif
      notifyServiceFinished(serverhost,serverport,restriction_index);
      notifyProxy(proxy_IPaddr,proxy_port,ns_errno,ID);
      cleanUp(wd);
      exit(-1);
    }

    /* Cutting the transaction to the client */
    endTransaction(comm);

    /*************************************************************/
    /* Start the life link to the proxy                          */
    /*************************************************************/
    if ((lifelink_pid = 
           startLifeLink(proxy_IPaddr,proxy_port,NS_PROT_JOB_REPORT,
              ID,NetSolveServerError)) == -1)
    {
      ns_printinfo();
      fprintf(stderr,"Warning: cannot start life link....\n");
    }
  }
  else   /* files */
  {
    if (readInputObjectsFromFiles(pd) == -1)
    {
#if (defined(VIEW) || defined(DEBUG))
      ns_printinfo();
      fprintf(stderr,"Error while reading the input objects from file.\n");
#endif
      notifyServiceFinished(serverhost,serverport,restriction_index);
      kill(lifelink_pid,SIGKILL);
      notifyProxy(proxy_IPaddr,proxy_port,NetSolveFileError,ID);
      cleanUp(wd);
      exit(-1);
    }
  }

#if (defined(VIEW) || defined(DEBUG))
    ns_printinfo();
    fprintf(stderr,"Transposing input Matrices ...\n");
#endif
  /* Transpose */
  for (i=0;i<pd->nb_input_objects;i++)
  {
    if (pd->input_objects[i]->object_type == NETSOLVE_MATRIX){
      if (pd->input_objects[i]->attributes.matrix_attributes.major != pd->major)
        transposeMatrix(pd->input_objects[i]);
    }
    else if (pd->input_objects[i]->object_type == NETSOLVE_SPARSEMATRIX){
      if (pd->input_objects[i]->attributes.sparsematrix_attributes.major != pd->major)
        transposeSparseMatrix(pd->input_objects[i]);
    }
    else
      continue;
  }
 
  /* Initializes the output objects */
  if (initOutputObjects(pd) == -1)
  {
#if (defined(VIEW) || defined(DEBUG))
    ns_printinfo();
    fprintf(stderr,"Error while initializing the output objects.\n");
#endif
    notifyServiceFinished(serverhost,serverport,restriction_index);
    kill(lifelink_pid,SIGKILL);
    notifyProxy(proxy_IPaddr,proxy_port,NetSolveInternalError,ID);
    cleanUp(wd);
    exit(-1);
  }

  /* Redirecting the stdout */
  sprintf(stdoutfile,"%s/netsolve-stdout",wd);
  stdoutfd = redirectStdout(stdoutfile);
  if (stdoutfd == -1)
  {
#if (defined(VIEW) || defined(DEBUG))
    ns_printinfo();
    fprintf(stderr,"Error while redirecting the stdout.\n");
#endif
    notifyServiceFinished(serverhost,serverport,restriction_index);
    kill(lifelink_pid,SIGKILL);
    notifyProxy(proxy_IPaddr,proxy_port,NetSolveSystemError,ID);
    cleanUp(wd);
    exit(-1);
  }
  
#if (defined(VIEW) || defined(DEBUG))
    ns_printinfo();
    fprintf(stderr,"Calling the \"solve()\" routine.\n");
#endif
  /* Solve the problem */
  time(&date1);
  solved = solve(pd,pd->input_objects,pd->output_objects);
  time(&date2);
  elapsed = (int)(date2-date1);


  if (node_nr != -1 && node_nr != -2) { /*sequencing mode */
      /* Error */
      if (solved != NS_PROT_SOLVED){
#if (defined(VIEW) || defined(DEBUG))
	  fprintf(stderr,"Error while solving the problem.\n");
#endif
	  cleanUp(wd);
	  exit(solved);
      }

      /* Send back the output */
      /* this takes care of the transpose */
      
      if (writeOutputObjectsToFiles(pd) == -1){
#if (defined(VIEW) || defined(DEBUG))
	  fprintf(stderr,"Error while writing the output objects.\n");
#endif
	  cleanUp(wd);
	  exit(-1);
      }


#if (defined(VIEW) || defined(DEBUG))
      fprintf(stderr,"Completed (pid=%i)\n", getpid());
#endif

      /* Cleaning up */
      fflush(stdout);
      close(stdoutfd);
      exit(-1);
  } /* sequencing mode */


  /* Notifying the proxy of completion */
  kill(lifelink_pid,SIGKILL);
  /* Is the client proxy still around ? */
  if (notifyProxy(proxy_IPaddr,proxy_port,NetSolveOK,ID) == -1)
  {
    notifyServiceFinished(serverhost,serverport,restriction_index);
    cleanUp(wd);
    exit(-1);
  }

  /* Listening on the socket */
  if ((sock = acceptConnection(listening_socket)) == -1)
  {
#if (defined(VIEW) || defined(DEBUG))
    fprintf(stderr,"Error while accepting the connection.\n");
#endif
    notifyServiceFinished(serverhost,serverport,restriction_index);
    cleanUp(wd);
    exit(-1);
  }
  comm = acceptTransaction(sock);
  if (comm == NULL)
  {
#if (defined(VIEW) || defined(DEBUG))
    fprintf(stderr,"Error while accepting the transaction.\n");
#endif
    notifyServiceFinished(serverhost,serverport,restriction_index);
    cleanUp(wd);
    exit(-1);
  }

  /* Sending back the Computational time elapsed */
  if (sendInt(comm,elapsed) == -1)
  {
#if (defined(VIEW) || defined(DEBUG))
    fprintf(stderr,"Error while sending the elapsed time.\n");
#endif
    notifyServiceFinished(serverhost,serverport,restriction_index);
    endTransaction(comm);
    cleanUp(wd);
    exit(-1);
  }

  /* Sending back the stdout */
  fflush(stdout);
  close(stdoutfd);
  if (sendFileAsString(comm,stdoutfile) == -1)
  {
#if (defined(VIEW) || defined(DEBUG))
    fprintf(stderr,"Error while sending the stdout.\n");
#endif
    notifyServiceFinished(serverhost,serverport,restriction_index);
    endTransaction(comm);
    cleanUp(wd);
    exit(-1);
  }

  /* Error */
  if (solved != NS_PROT_SOLVED)
  {
    sendInt(comm,solved);
    endTransaction(comm);
    notifyServiceFinished(serverhost,serverport,restriction_index);
    cleanUp(wd);
    exit(-1);
  }

  sendInt(comm,solved);

  /* Send back the output */
  /* this takes care of the transpose */
  if (sendOutputObjects(comm,pd,client_major) == -1)
  {
#if (defined(VIEW) || defined(DEBUG))
    ns_printinfo();
    fprintf(stderr,"Error while sending the output objects.\n");
#endif
    notifyServiceFinished(serverhost,serverport,restriction_index);
    endTransaction(comm);
    cleanUp(wd);
    exit(-1);
  }

  /* Terminating the transaction */
  endTransaction(comm);
  notifyServiceFinished(serverhost,serverport,restriction_index);

#if (defined(VIEW) || defined(DEBUG))
  fprintf(stderr,"Completed (pid=%i)\n", getpid());
#endif

  /* Cleaning up */
  close(listening_socket);
  cleanUp(wd);
  exit(0);
}

/*
 * sendOutputObjects()
 */
int sendOutputObjects(NS_Communicator *comm,NS_ProblemDesc *pd,int client_major)
{
  NS_Object **list = pd->output_objects;
  int nb = pd->nb_output_objects;
  int i;

  for (i=0;i<nb;i++)
  {
    switch(list[i]->object_type)
    {
      case NETSOLVE_MATRIX:  /* transpose ? */
        if (pd->major != client_major)
        {
          if (transposeMatrix(list[i]) == -1)
          {
#if (defined(VIEW) || defined(DEBUG))
            fprintf(stderr,"Error while transposing matrix\n");
#endif
            return -1;
          }
        }
        break;
      case NETSOLVE_SPARSEMATRIX:  /* transpose ? */
        if (pd->major != client_major)
        {
          if (transposeSparseMatrix(list[i]) == -1)
          {
#if (defined(VIEW) || defined(DEBUG))
            fprintf(stderr,"Error while transposing sparse matrix\n");
#endif
            return -1;
          }
        }
        break;
      default:  /* Nothing */
        break;
    }
    if (sendObject(comm,list[i]) == -1)
    {
      netsolvePerror("sendObject()");
      return -1;
    }
  } 

  return 1;
}

/*
 * initOutputObjects()
 */
int initOutputObjects(NS_ProblemDesc *pd)
{
  NS_Object **list = pd->output_objects;
  int nb = pd->nb_output_objects;
  char buffer[256];
  int i;
    
  for (i=0;i<nb;i++)
  {
    /* do special initializations */
    switch(list[i]->object_type)
    {
      case NETSOLVE_FILE: /* Create a file name */
        sprintf(buffer,"fileoutput%d",i);
        list[i]->attributes.file_attributes.filename = strdup(buffer);
        break;
      case NETSOLVE_MATRIX: /* Set the major */
        list[i]->attributes.matrix_attributes.major = pd->major;
        list[i]->attributes.matrix_attributes.l = -1;
        break;
      case NETSOLVE_SPARSEMATRIX: /* Set the major */
        list[i]->attributes.sparsematrix_attributes.major = pd->major;
        break;
      default: /* Do nothing */
        break;
    }
  }
  return 1;
}

/*
 * recvInputObjects()
 */
int recvInputObjects(NS_Communicator *comm,NS_ProblemDesc *pd)
{
  NS_Object **list = pd->input_objects;
  int nb = pd->nb_input_objects;
  char buffer[256];
  int i;

  for (i=0;i<nb;i++)
  {
    /* do initializations */
    switch(list[i]->object_type)
    {
      case NETSOLVE_MATRIX: 
        list[i]->attributes.matrix_attributes.m = -1;
        list[i]->attributes.matrix_attributes.n = -1;
        list[i]->attributes.matrix_attributes.l = -1;
        list[i]->attributes.matrix_attributes.major = -1;
        list[i]->attributes.matrix_attributes.ptr = NULL;
        list[i]->attributes.matrix_attributes.d = NULL;
        break;
      case NETSOLVE_SPARSEMATRIX: 
        list[i]->attributes.sparsematrix_attributes.m = -1;
        list[i]->attributes.sparsematrix_attributes.n = -1;
        list[i]->attributes.sparsematrix_attributes.major = -1;
        list[i]->attributes.sparsematrix_attributes.f = -1;
        list[i]->attributes.sparsematrix_attributes.rc_ptr = NULL;
        list[i]->attributes.sparsematrix_attributes.rc_index = NULL;
        list[i]->attributes.sparsematrix_attributes.ptr = NULL;
        list[i]->attributes.sparsematrix_attributes.d = NULL;
        break;
      case NETSOLVE_VECTOR:
        list[i]->attributes.vector_attributes.m = -1;
        list[i]->attributes.vector_attributes.ptr = NULL;
        break;
      case NETSOLVE_SCALAR:
        list[i]->attributes.scalar_attributes.ptr = NULL;
        break;
      case NETSOLVE_STRING:
        list[i]->attributes.string_attributes.ptr = NULL;
        break;
      case NETSOLVE_STRINGLIST:
        list[i]->attributes.stringlist_attributes.strings = NULL;
        list[i]->attributes.stringlist_attributes.m = -1;
        break;
      case NETSOLVE_FILE: /* Create a file name */
        sprintf(buffer,"fileinput%d",i);
        list[i]->attributes.file_attributes.filename = strdup(buffer);
        break;
      case NETSOLVE_PACKEDFILES: /* Create a file name */
        sprintf(buffer,"packedfileinput%d",i);
        list[i]->attributes.packedfiles_attributes.defaultprefix = strdup(buffer);
        list[i]->attributes.packedfiles_attributes.filenames = NULL;
        list[i]->attributes.packedfiles_attributes.m = -1;
        break;
      case NETSOLVE_UPF: /* Create a file name */
#if (defined(VIEW) || defined(DEBUG))
        fprintf(stderr,"There should not be any UPF here !!!\n");
#endif
        return -1;
        break;
      default: /* Nothing to so */
        break;
    }

    /* receive the object from the net */
    if (recvObject(comm,list[i]) == -1)
    {
      netsolvePerror("recvObject()");
      return -1;
    }

    /* Perform special post-processing !! */
    switch(list[i]->object_type)
    {
      case NETSOLVE_MATRIX: /* transpose */
        if (list[i]->attributes.matrix_attributes.major == pd->major)
          break;
        if (transposeMatrix(list[i]) == -1)
        {
#if (defined(VIEW) || defined(DEBUG))
          fprintf(stderr,"Error while transposing matrix\n");
#endif
          return -1;
        }
        break;
      case NETSOLVE_SPARSEMATRIX: /* transpose */
        if (list[i]->attributes.sparsematrix_attributes.major == pd->major)
          break;
        if (transposeSparseMatrix(list[i]) == -1)
        {
#if (defined(VIEW) || defined(DEBUG))
          fprintf(stderr,"Error while transposing sparse matrix\n");
#endif
          return -1;
        }
        break;
      default: /* Nothing to do */
        break;
    }
  }
  return 1;
}

