/*****************************************************************/
/*      forkcondorservice.c                                      */
/*      Henri Casanova                                           */
/*---------------------------------------------------------------*/
/*****************************************************************/

#include "core.h"
#include "serverglobal.h"
#include "condorservice.h"
#include "serviceutil.h"
#include <unistd.h>
#include <fcntl.h>
#include <time.h>
#include <signal.h>

/*
 * forkCondorService()
 */
void forkCondorService(NS_Socket_type listening_socket, int ID, int client_major,
                         int lifelink_pid,
                         char *agent_name,
                         char *wd, char *serverhost,
                         int serverport, int restriction_index, 
                         char *xpath,char *xname,
                         NS_ProblemDesc *pd,NS_IPaddr_type proxy_IPaddr, 
                         int proxy_port)
{
  time_t date1,date2;
  int elapsed;
  char command[256];
  char stdoutfile[256];
  char cmdfile[256];
  FILE *f;
  char verdict[20];
  NS_Socket_type sock;
  NS_Communicator *comm;


  /* Condor */
  if (createCondorCmdFile(cmdfile,pd,client_major,wd) == -1)
  {
#ifdef VIEW
    fprintf(stderr,"Impossible to create the Condor cmd file\n");
#endif
    notifyServiceFinished(serverhost,serverport,restriction_index);
    kill(lifelink_pid,SIGKILL);
    notifyProxy(proxy_IPaddr,proxy_port,NetSolveCondorError,ID);
    cleanUp(wd);
    exit(0);
  }

#ifdef VIEW
  fprintf(stderr,"Submitting to condor\n");
#endif
  sprintf(command,"%s/bin/condor_submit %s",global.Condor_path,cmdfile);

  time(&date1);
  system(command);

 
  /* waiting for the done file */
  while((f = fopen("./done","r")) == NULL)
    sleep(3);
  time(&date2);

  /* Getting the verdict */
  fscanf(f,"%s",verdict);
  fclose(f);

  /* Notifying the proxy of completion */
  kill(lifelink_pid,SIGKILL);
  if (notifyProxy(proxy_IPaddr,proxy_port,NetSolveOK,ID) == -1)
  {
    cleanUp(wd);
    notifyServiceFinished(serverhost,serverport,restriction_index);
    exit(0); 
  }

  /* From now on, the life link is dead */

  /* Listening on the socket */
  if ((sock = acceptConnection(listening_socket)) == -1)
  {
#ifdef VIEW
    fprintf(stderr,"Error while accepting the connection.\n");
#endif
    notifyServiceFinished(serverhost,serverport,restriction_index);
    cleanUp(wd);
    exit(0);
  }
  comm = acceptTransaction(sock);
  if (comm == NULL)
  {
#ifdef VIEW
    fprintf(stderr,"Error while accepting the transaction.\n");
#endif
    notifyServiceFinished(serverhost,serverport,restriction_index);
    cleanUp(wd);
    exit(0);
  }


  /* Sending back the elapsed time */
  elapsed = (int)(date2-date1);
  if (sendInt(comm,elapsed) == -1)
  {
#ifdef VIEW
    fprintf(stderr,"Error while sending the elapsed time.\n");
#endif
    netsolvePerror("sendInt()");
    notifyServiceFinished(serverhost,serverport,restriction_index);
    endTransaction(comm);
    cleanUp(wd);
    exit(0);
  }

  /* sending back whatever stdout has been generated */
  sprintf(stdoutfile,"%s/netsolve-stdout",wd);
  if (sendFileAsString(comm,stdoutfile) == -1)
  {
#ifdef VIEW
    fprintf(stderr,"Error while sending the stdout.\n");
#endif
    netsolvePerror("sendFileAsString()");
    notifyServiceFinished(serverhost,serverport,restriction_index);
    endTransaction(comm);
    cleanUp(wd);
    exit(0);
  }

  /* Error ? */
  if (!strcmp(verdict,"ERROR"))
  {
#ifdef VIEW
    fprintf(stderr,"Condor: ERROR\n");
#endif
    notifyServiceFinished(serverhost,serverport,restriction_index);
    sendInt(comm,NS_PROT_INTERNAL_FAILURE);
    endTransaction(comm);
    cleanUp(wd);
    exit(0);
  }

  /* SUCCESS !! */
  sendInt(comm,NS_PROT_SOLVED);

  /* Sending back the output */
  if (sendOutputObjectsFromFiles(comm,pd) == -1)
  {
#ifdef VIEW
    fprintf(stderr,"Error while sending the output objects from files.\n");
#endif
    netsolvePerror("sendOutputObjectsFromFiles()");
    notifyServiceFinished(serverhost,serverport,restriction_index);
    endTransaction(comm);
    cleanUp(wd);
    exit(0);
  }

  /* Terminating the transaction */
  endTransaction(comm);
  notifyServiceFinished(serverhost,serverport,restriction_index);

#ifdef VIEW
  fprintf(stderr,"Completed\n");
#endif

  /* Cleaning up */
  cleanUp(wd);
  exit(0);
}

/*
 * createCondorCmdFile()
 */
int createCondorCmdFile(char *filename,NS_ProblemDesc *pd,int client_major,char *wd)
{
  FILE *f;

  sprintf(filename,"./do.cmd");
  f = fopen(filename,"w");
  if (f == NULL)
    return -1;

  fprintf(f,"notify = NEVER\n");
  fprintf(f,"executable = %s/bin/%s/service-%s\n",global.netsolve_root_path,
                       getenv("NETSOLVE_ARCH"),pd->file);
  fprintf(f,"output = ./condor.stdout\n");
  fprintf(f,"error = ./condor.stderr\n");
  fprintf(f,"arguments =");
  fprintf(f," %d ",client_major);
  fprintf(f," %s ",wd);
  fprintf(f,"\n");
  fprintf(f,"queue\n");

  fclose(f);
  return 1;
}
