#include <stdlib.h>
#ifdef petsc29
#include "petscmat.h"
#include "petscvec.h"
#else
#include "mat.h"
#include "vec.h"
#endif
#include "error_returns.h"
#include "mec.h"

int read_utah_mat
(MPI_Comm comm,char *mat_name,char *rhs_name,int band,
 int *r_first,int *r_last,int *r_nnz,int *r_total,
 int **r_ptr,int **r_idx,double **r_val,double **r_rhs)
{
  FILE* file;
  int mytid,ntids,first,last,m,n,nnz,my_nnz,ierr;

  MPI_Comm_size(comm,&ntids); MPI_Comm_rank(comm,&mytid);

  /* get matrix elements */
  file = fopen(mat_name,"r");
  fscanf(file,"%d %d %d",&m,&n,&nnz);
  if (m!=n) ERR_REPORTii("Matrix not square",m,n);
  *r_total = m;
  { /* find block structure 
     * this should be done only on one processor */
    int *left,*above,nb,*blocks,first_block,last_block;

    /* find skyline structure */
    ALLOCATE(left,m,int,"leftmost"); ALLOCATE(above,m,int,"topmost");
    for (n=0; n<m; n++) {left[n] = 0; above[n] = 0;}
    for (n=0; n<nnz; n++) {
      int i,j; double v;
      if (feof(file)) ERR_REPORTi("Premature end of matrix file at",n);
      fscanf(file,"%d %d %lf",&i,&j,&v);
      if (j<i && j>i-band && i-j>left[i] && v!=0.) left[i] = i-j;
      if (j>i && j<i+band && i-j>above[j] && v!=0.) above[j] = i-j;
    }
    /* overestimate number of blocks. this can be improved */
    nb = 1;
    for (n=0; n<m; n++) if (left[n]==0 && above[n]==0) nb++;
    ALLOCATE(blocks,nb+1,int,"block structure");
    nb = 0;
    for (n=0; n<m; n++)
      if (left[n]==0 && above[n]==0) {
	int k,found=1;
	for (k=1; k<band && n+k<m; k++)
	  if (left[n+k]>k || above[n+k]>k) {found=0; break;}
	if (found) blocks[nb++] = n;
      }
    blocks[nb] = m;
    PetscPrintf(comm,"Blocks found %d:",nb);
    for (n=0; n<=nb; n++) PetscPrintf(comm," %d",blocks[n]);
    PetscPrintf(comm,"\n");
    /* up to here it should be done centrally, then broadcast;
     * now figure out the local part */
    first_block = (nb*mytid)/ntids; last_block = (nb*(mytid+1))/ntids;
    first = blocks[first_block]; last = blocks[last_block];
    *r_first = first; *r_last = last;
    printf("[%d] I get blocks %d--%d, elements %d--%d\n",
	   mytid,first_block,last_block,first,last);
    free(left); free(above); free(blocks);
  }
  rewind(file);
  /* count how many nonzeros are ours */
  {
    my_nnz = 0;
    /* lose first line: */ {int m,n,nnz; fscanf(file,"%d %d %d",&m,&n,&nnz);}
    for (n=0; n<nnz; n++) {
      int i,j; double v,z=0.;
      if (feof(file)) ERR_REPORTi("Premature end of matrix file at",n);
      fscanf(file,"%d %d %lf\n",&i,&j,&v);
      if (i>=first && i<last) my_nnz++;
    }
    *r_nnz = my_nnz;
    if (!feof(file)) ERR_REPORT("Elements left at end of matrix file");
  }
  rewind(file);
  /* now actually construct the arrays */
  {
    int *ico,*ptr,*idx,inz,local_size = last-first; double *val,*rhs;
    ALLOCATE(ico,my_nnz,int,"pointers");
    ALLOCATE(idx,my_nnz,int,"indices");
    ALLOCATE(val,my_nnz,double,"values");
    /* lose first line: */ {int m,n,nnz; fscanf(file,"%d %d %d",&m,&n,&nnz);}
    inz = 0;
    for (n=0; n<nnz; n++) {
      int i,j; double v,z=0.;
      if (feof(file)) ERR_REPORTi("Premature end of matrix file at",n);
      fscanf(file,"%d %d %lf\n",&i,&j,&v);
      if (i>=first && i<last) {
	ico[inz] = i; idx[inz] = j; val[inz] = v; inz++;
      }
    }
    if (!feof(file)) ERR_REPORT("Elements left at end of matrix file");
    fclose(file);
    ierr = coo_to_crs
      (ico,&ptr,idx,val,first,last-first,my_nnz); ERR_RETURN(ierr);
    free(ico); *r_ptr = ptr; *r_idx = idx; *r_val = val;
  }
  /* get right hand side */
  {
    double *rhs; int i,nrhs,size,ielt;
    file = fopen(rhs_name,"r");
    fscanf(file,"%d %d",&nrhs,&size);
    if (nrhs!=1) ERR_REPORT("Can only handle one rhs");
    if (size!=m) ERR_REPORTii("Rhs not same size as matrix",size,m);
    ALLOCATE(rhs,last-first,double,"rhs");
    ielt = 0;
    for (i=0; i<size; i++) {
      double v;
      if (feof(file)) ERR_REPORTi("Premature end of rhs file at",i);
      fscanf(file,"%lf",&v);
      if (i>=first && i<last) {
	rhs[i-first] = v;}
    }
    *r_rhs = rhs;
    if (!feof(file)) ERR_REPORT("Elements left at end of rhs file");
    fclose(file);
  }
  return 0;
}
