/*****************************************************************/
/*      nsprox_process_messages.c                                */
/*      Henri Casanova                                           */
/*---------------------------------------------------------------*/
/*****************************************************************/

#include "core.h"
#include "proxyjobdesc.h"
#include "proxy.h"
#include "netsolvesignals.h"
#include "agentprocessmessage.h"

#include <sys/types.h>
#include <netinet/in.h>
#include <unistd.h>

/*
 * proxyProcessRequest()
 *
 * This function implements all the system-independent proxy 
 * functionalities and calls the system-specific functions. Those
 * functions are implemented in sub-directories (e.g. NetSolve, Globus)
 */
void proxyProcessRequest(NS_Communicator *comm,int listeningsocket)
{
  int rc;
  int tag;

  if (recvInt(comm,&tag) == -1)
  {
    netsolvePerror("recvInt()");
    return;
  }
  
  /* No fork: some of the proxy functions opperate on the 
   * state of the proxy and must therefore no be performed
   * by another process
   */
  if (tag == NS_PROT_DIE)
  {
    close(listeningsocket);
    endTransaction(comm);
    exit(0);
  } 
  if (tag == NS_PROT_NEW_JOB_DESC)
  {
    processNewJobDesc(comm);
    endTransaction(comm);
    return;
  }
  if (tag == NS_PROT_RM_JOB_DESC)
  {
    processRmJobDesc(comm);
    endTransaction(comm);
    return;
  }
  if (tag == NS_PROT_JOB_REPORT)
  {
    processJobReport(comm);
    endTransaction(comm);
    return;
  }
  if (tag == NS_PROT_JUST_CHECKING) {
    endTransaction(comm);
    return;
  }
  if (tag == NS_PROT_AWAIT_COMPLETION) {
    processAwaitCompletion(comm);
    /* important: do *NOT* end the transaction for this particular call */
    return;
  }

  /* fork for everything else */
  if (-1 == (rc = fork())) {
    /* fork failed */
    /* XXX: We should not give a protocol error -
     * something else should be introduced for resource
     * related errors.
     */
    if (sendInt(comm,NetSolveSystemError) == -1)
      netsolvePerror("sendInt()");
    endTransaction(comm);
    return;
  }
  /* parent returns imediately */
  if(rc) {
    endTransaction(comm);
    return;
  }
 
  close(listeningsocket);

  switch(tag)
  {
    case NS_PROT_NB_SERVERS:
      processNumberOfServers(comm);
      break;
    case NS_PROT_AGENT_LIST:
      processAgentList(comm);
      break;
    case NS_PROT_SERVER_LIST:
      processServerList(comm);
      break;
    case NS_PROT_PROBLEM_LIST:
      processProblemList(comm);
      break;
    case NS_PROT_PROBLEM_INFO:
      processProblemInfo(comm);
      break;
    case NS_PROT_STORE_HANDLE:
      processStoreHandle(comm);
      break;
    case NS_PROT_GET_HANDLE:
      processGetHandle(comm);
      break;
    case NS_PROT_SV_FAILURE:
      processServerFailure(comm);
      break;
    case NS_PROT_KILL_JOB:
      processKillJob(comm);
      break;
    case NS_PROT_SEND_ASSIGNMENT:
      processSendAssignment(comm);
      break;
    case NS_PROT_SEND_REQUEST:
      processSendRequest(comm);
      break;
    case NS_PROT_SEND_SEQUENCE_REQUEST:
      processSend_sequence_Request(comm);
      break;
    case NS_PROT_PROBE_REQUEST:
      processProbeRequest(comm);
      break;
    case NS_PROT_JOB_COMPLETED:
      processJobCompleted(comm);
      break;
    default: /* ERROR */
      if (sendInt(comm,NS_PROT_PROTOCOL_ERROR) == -1)
        netsolvePerror("sendInt()");
      break;
  }
  /* 
   * XXX: Some of the calls above (all of them ??) will call
   * endTransaction(), and we can't detect if it has happened
   * or not. This will cause a double free() of the comm area.
   * Good design would probably be to either pass &comm, so that
   * endTransaction() could set *comm=0 after the free(*comm),
   * or eventually move the free() entirely out of endTransaction()
   */
  /*  So far this is disabled... endTransaction(comm); */
  exit(0);
}

/*
 * processJobReport()
 *
 * One of my children has been contacted by a server that reported
 * its status. This child then reports it to me so that it is possible
 * to update the global state.
 */
void processJobReport(NS_Communicator *comm)
{
  int ID;
  int error_code;
  NS_NotificationWaiter * iter, * piter = 0;

  if ((recvInt(comm,&ID) == -1) ||
      (recvInt(comm,&error_code) == -1))
  { 
    netsolvePerror("recvInt()");
    return;
  } 
  if ((ID<0)||(ID>NB_MAX_JOBS))
    return;
  if (jobs[ID] == NULL)
    return;

  jobs[ID]->error_code = error_code;

  /*
   * If this was a job completion report, look tru
   * the waiter list, and send back a reply to whoever
   * cares about this
   */
  if(error_code == NetSolveNotReady)
    return;

  for(iter = notification_waiters; iter; iter = iter->next) {
    if(iter->id == ID) {
      /* We found it ! */
      if ((sendInt(iter->comm, NetSolveOK) == -1) ||
	  (sendInt(iter->comm, error_code) == -1)) {
	netsolvePerror("sendInt()");
	return;
      }
      /* kill the comms channel */
      endTransaction(iter->comm);
      /* Now unchain and free */
      if(!piter)
	notification_waiters = iter->next;
      else
	piter->next = iter->next;
      free(iter);
      break;
    }
    piter = iter;
  }
}

/*
 *  processNewJobDesc()
 *
 *  One of my childs has successfully launched a job and 
 *  asks me for a new job descriptor ID.
 */
void processNewJobDesc(NS_Communicator *comm)
{
  NS_JobDesc *jd;
  int ID;

  if ((jd = recvJobDesc(comm)) == NULL)
  {
    netsolvePerror("recvJobDesc()");
    return;
  }
  
  ID = insertJobDesc(jd);

  if (sendInt(comm,ID) == -1)
  {
    netsolvePerror("sendInt()");
    return;
  }
  return; 
}

/*
 * processRmJobDesc()
 *
 * one of my child has been asked to remove a job descriptor.
 */
void processRmJobDesc(NS_Communicator *comm)
{
  int ID;

  if (recvInt(comm,&ID) == -1)
  {
    netsolvePerror("recvInt()");
    return;
  }
  if ((ID<0)||(ID>NB_MAX_JOBS))
    return;
  if (jobs[ID] == NULL)
    return;
  freeJobDesc(jobs[ID]);
  jobs[ID] = NULL;
  return;
}
