/*****************************************************************/
/*      proxyutil.c                                              */
/*      Henri Casanova                                           */
/*---------------------------------------------------------------*/
/*****************************************************************/

#include "core.h"
#include "client.h"

#ifndef WIN32
#include <unistd.h>
#include <sys/socket.h>
#endif /* UNIX */

#include <signal.h>
#include <time.h>

#include "netsolvejobdesc.h"
#include "netsolveproxy.h"

/*
 * SubmitProblemToAgent()
 */
int submitProblemToAgent(char *nickname,int input_size, int output_size,
                         int problem_size, int *nb_servers, char ***hostlist,
                         NS_IPaddr_type **IPaddrlist, int **portlist,
                         int **dataformatlist, int **predictedtimelist,
                         char *agent)
{
  NS_Communicator *comm;
  char *username;
  char *hostname;
  int tag;
  int i;

  username = getUserName();
  if(username == NULL){
    username = strdup("NONE");
  }

  hostname = getHostName();
  if(hostname == NULL){
    hostname = strdup("NONE");
  }

  comm = contactHost(agent,0,AGENT_PORT,DATA_XDR);
  if (comm == NULL)
    return -1;

  /* Sending */
  if ((sendInt(comm,NS_PROT_PROBLEM_SUBMIT) == -1) ||
      (sendString(comm,nickname) == -1) ||
      (sendInt(comm,input_size) == -1) ||
      (sendInt(comm,output_size) == -1) ||
      (sendInt(comm,problem_size) == -1) ||
      (sendString(comm,username) == -1) ||
      (sendString(comm,hostname) == -1))
  {
    endTransaction(comm);
    return -1;
  }

  /* receiving */
  if (recvInt(comm,&tag) == -1)
  {
    endTransaction(comm);
    return -1;
  }
  if (tag == NS_PROT_PROBLEM_NOT_FOUND)
  {
    endTransaction(comm);
    ns_errno = NetSolveUnknownProblem;
    return -1;
  }
  if (tag != NS_PROT_OK)
  {
    endTransaction(comm);
    ns_errno = NetSolveProtocolError;
    return -1;
  }

  if (recvInt(comm,nb_servers) == -1)
  {
    endTransaction(comm);
    return -1;
  }
  if (*nb_servers == 0)
  {
    endTransaction(comm);
    ns_errno = NetSolveNoServer;
    return -1;
  }
 
  *hostlist          = (char **)calloc(*nb_servers,sizeof(char*));
  *IPaddrlist        = (NS_IPaddr_type *)calloc(*nb_servers,sizeof(NS_IPaddr_type));
  *portlist          = (int *)calloc(*nb_servers,sizeof(int));
  *dataformatlist    = (int *)calloc(*nb_servers,sizeof(int));
  *predictedtimelist = (int *)calloc(*nb_servers,sizeof(int));

  for (i=0;i<*nb_servers;i++)
  {
    if ((recvString(comm,&((*hostlist)[i])) == -1) ||
       (recvIPaddr(comm,&((*IPaddrlist)[i])) == -1) ||
       (recvInt(comm,&((*portlist)[i])) == -1) ||
       (recvInt(comm,&((*dataformatlist)[i])) == -1) ||
       (recvInt(comm,&((*predictedtimelist)[i])) == -1))
    {
      endTransaction(comm);
      return -1;
    }
  }
  endTransaction(comm);
  ns_errno = NetSolveOK;
  return 1;
}

/*
 * submitProblemToServer()
 */
int submitProblemToServer(NS_ProblemDesc *pd, NS_IPaddr_type IPaddr,
                          int port, int *start_time, int *newport, int *pid)
{
  NS_Communicator *comm;
  int tag;
  time_t now; 
  
  comm = contactHost(NULL,IPaddr,port,DATA_XDR);
  if (comm == NULL)
    return -1;

  /* Sending */
  if (sendInt(comm,NS_PROT_PROBLEM_SOLVE) == -1){
    endTransaction(comm);
    return -1;
  }

  if (recvInt(comm, &tag) == -1)
  {
    endTransaction(comm);
    return -1;
  }

  if(tag != NS_PROT_OK){
    endTransaction (comm);
    return -1;
  }

  if((sendProblemDesc(comm,pd) == -1) ||
      (sendString(comm,agent) == -1) ||
      (sendIPaddr(comm,&myIPaddr) == -1) ||
      (sendInt(comm,myport) == -1))
    
  {
    endTransaction(comm);
    return -1;
  }

  /* Receiving */
  if (recvInt(comm,&tag) == -1)
  {
    endTransaction(comm);
    return -1;
  }
  if (tag != NS_PROT_ACCEPTED)
  {
    endTransaction(comm);
    switch(tag)
    {
      case NS_PROT_PROBLEM_NOT_FOUND:
        ns_errno = NetSolveUnknownProblem;
        break;
      case NS_PROT_BAD_SPECIFICATION:
        ns_errno = NetSolveBadProblemSpecification;
        break;
      case NS_PROT_NOT_ALLOWED:
        ns_errno = NetSolveNotAllowed;
        break;
      default:
        ns_errno = NetSolveProtocolError;
        break;
    }
    return -1;
  }

  if (recvInt(comm,newport) == -1)
  {
    endTransaction(comm);
    ns_errno = NetSolveProtocolError;
    return -1;
  }
  if (*newport == -1)
  {
    endTransaction(comm);
    ns_errno = NetSolveServerError;
    return -1;
  }

  if (recvInt(comm,pid) == -1)
  {
    endTransaction(comm);
    ns_errno = NetSolveProtocolError;
    return -1;
  }
  endTransaction(comm);

  time(&now);
  *start_time = (int)now;

  return 1;
}

/*
 * forwardFailure()
 */
int forwardFailure(NS_IPaddr_type IPaddr, char *nickname, int error_type)
{
  NS_Communicator *comm_to_agent;

    /* Contact the agent */
  comm_to_agent = contactHost(agent,0,AGENT_PORT,DATA_XDR);
  if (comm_to_agent == NULL)
  {
    netsolvePerror("contactHost()");
    return -1;
  }

  if ((sendInt(comm_to_agent,NS_PROT_SV_FAILURE) == -1) ||
     (sendIPaddr(comm_to_agent,&IPaddr) == -1) ||
     (sendString(comm_to_agent,nickname) == -1) ||
     (sendInt(comm_to_agent,error_type) == -1))
  {
    netsolvePerror("");
    endTransaction(comm_to_agent);
    return -1;
  }
  endTransaction(comm_to_agent);
  return 1;
}

/*
 * rmJobDesc()
 */
void rmJobDesc(int ID)
{
  NS_Communicator *comm;

  comm = contactHost(NULL,getMyIPaddr(),myport,DATA_RAW);
  if (comm == NULL)
  {
    netsolvePerror("contactHost()");
    return;
  } 
  
  if ((sendInt(comm,NS_PROT_RM_JOB_DESC) == -1)||
      (sendInt(comm,ID) == -1))
  {
    endTransaction(comm);
    netsolvePerror("sendInt()");
    return;
  }
  endTransaction(comm);
  return; 
}

/*
 * requestInsertJobDesc()
 *
 * returns an ID
 */
int requestInsertJobDesc(NS_JobDesc *jd)
{
  NS_Communicator *comm;
  int ID;

  comm = contactHost(NULL,getMyIPaddr(),myport,DATA_RAW);
  if (comm == NULL)
  {
    netsolvePerror("contactHost()");
    return -1;
  } 
  
  if ((sendInt(comm,NS_PROT_NEW_JOB_DESC) == -1)||
      (sendJobDesc(comm,jd) == -1))
  {
    endTransaction(comm);
    netsolvePerror("");
    return -1;
  }
  if (recvInt(comm,&ID) == -1)
  {
    endTransaction(comm);
    netsolvePerror("recvInt()");
    return -1;
  }
  endTransaction(comm);
  return ID;
}

/*
 * getAServer()
 */
NS_ServerDesc *getAServer(NS_IPaddr_type IPaddr, char *agent_name)
{
  int status;
  NS_Communicator *comm;
  NS_ServerDesc *sd;

  comm = contactHost(agent_name,0,AGENT_PORT,DATA_XDR);
  if (comm == NULL)
  {
    return NULL;
  }

  if ((sendInt(comm,NS_PROT_SV_INFO) == -1) ||
      (sendIPaddr(comm,&IPaddr) == -1))
  {
    endTransaction(comm);
    return NULL;
  }
  
  if (recvInt(comm,&status) == -1)
  {
    endTransaction(comm);
    return NULL;
  }
  if (status == NS_PROT_SV_NOT_FOUND)
  {
    ns_errno = NetSolveUnknownServer;
    return NULL;
  }
  sd = recvServerDesc(comm);
  if (sd == NULL)
    return NULL;

  ns_errno = NetSolveOK;
  return sd;
}

/*
 * insertJobDesc()
 */
int insertJobDesc(NS_JobDesc *jd)
{
  int i;

  for (i=0;i<NB_MAX_JOBS;i++)
  {
    if (jobs[i] == NULL)
    {
      jobs[i] = jd;
      return i;
    }
  }
  return -1;
}
