/*****************************************************************/
/*      netsolveproxybasics.c                                    */
/*      Henri Casanova                                           */
/*---------------------------------------------------------------*/
/*****************************************************************/

#include "core.h"
#include <sys/types.h>
#include <netinet/in.h>
#include "netsolvejobdesc.h"
#include "netsolveproxy.h"
#include "netsolvesignals.h"

NS_NotificationWaiter* notification_waiters = 0;

/*
 * processNumberOfServers()
 */
void processNumberOfServers(NS_Communicator *comm)
{
  int nb_servers;
  NS_Communicator *comm_to_agent;

  /* contact the agent */
  comm_to_agent = contactHost(agent,0,AGENT_PORT,DATA_XDR);
  if (comm_to_agent == NULL)
  {
    sendInt(comm,NetSolveCannotContactAgent);
    netsolvePerror("");
    return;
  }
   
  /* Sending */
  if (sendInt(comm_to_agent,NS_PROT_NB_SERVERS) == -1)
  {
    sendInt(comm,NetSolveNetworkError);
    netsolvePerror("");
    endTransaction(comm_to_agent);
    return;
  }

  if (recvInt(comm_to_agent,&nb_servers) == -1)
  {
    sendInt(comm,NetSolveNetworkError);
    netsolvePerror("");
    endTransaction(comm_to_agent);
    return;
  }

  endTransaction(comm_to_agent);

  /* Forward this to the client */
  if (sendInt(comm,NetSolveOK) == -1)
  {
    netsolvePerror("");
    return;
  }
  if (sendInt(comm,nb_servers) == -1)
  {
    netsolvePerror("");
    return;
  }
  return; 
}

/*
 * processAgentList()
 */
void processAgentList(NS_Communicator *comm)
{
  NS_AgentDesc **list;
  int nb_agents,i;
  NS_Communicator *comm_to_agent;

  /* contact the agent */
  comm_to_agent = contactHost(agent,0,AGENT_PORT,DATA_XDR);
  if (comm_to_agent == NULL)
  {
    sendInt(comm,NetSolveCannotContactAgent);
    netsolvePerror("");
    return;
  }
   
  /* Sending */
  if (sendInt(comm_to_agent,NS_PROT_AGENT_LIST) == -1)
  {
    sendInt(comm,NetSolveNetworkError);
    netsolvePerror("");
    endTransaction(comm_to_agent);
    return;
  }

  if (recvInt(comm_to_agent,&nb_agents) == -1)
  {
    sendInt(comm,NetSolveNetworkError);
    netsolvePerror("");
    endTransaction(comm_to_agent);
    return;
  }

  list = (NS_AgentDesc **)calloc(nb_agents,sizeof(NS_AgentDesc*));
  for (i=0;i<nb_agents;i++)
  {
    list[i] = recvAgentDesc(comm_to_agent);
    if (list[i] == NULL)
    {
    sendInt(comm,NetSolveNetworkError);
      netsolvePerror("");
      endTransaction(comm_to_agent);
      return;
    }
  }
  endTransaction(comm_to_agent);

  /* Forward this to the proxy */
  if (sendInt(comm,NetSolveOK) == -1)
  {
    netsolvePerror("");
    return;
  }
  if (sendInt(comm,nb_agents) == -1)
  {
    netsolvePerror("");
    return;
  }
  for (i=0;i<nb_agents;i++)
  {
    if (sendAgentDesc(comm,list[i]) == -1)
    {
      netsolvePerror("");
      return;
    }
  }
 
  return; 
}

/*
 * processServerList()
 */
void processServerList(NS_Communicator *comm)
{
  NS_ServerDesc **list;
  int nb_servers,i;
  NS_Communicator *comm_to_agent;

  /* contact the agent */
  comm_to_agent = contactHost(agent,0,AGENT_PORT,DATA_XDR);
  if (comm_to_agent == NULL)
  {
    sendInt(comm,NetSolveCannotContactAgent);
    netsolvePerror("");
    return;
  }
   
  /* Sending */
  if (sendInt(comm_to_agent,NS_PROT_SERVER_LIST) == -1)
  {
    sendInt(comm,NetSolveNetworkError);
    netsolvePerror("");
    endTransaction(comm_to_agent);
    return;
  }

  if (recvInt(comm_to_agent,&nb_servers) == -1)
  {
    sendInt(comm,NetSolveNetworkError);
    netsolvePerror("");
    endTransaction(comm_to_agent);
    return;
  }

  list = (NS_ServerDesc **)calloc(nb_servers,sizeof(NS_ServerDesc*));
  for (i=0;i<nb_servers;i++)
  {
    list[i] = recvServerDesc(comm_to_agent);
    if (list[i] == NULL)
    {
      sendInt(comm,NetSolveNetworkError);
      netsolvePerror("");
      endTransaction(comm_to_agent);
      return;
    }
  }
  endTransaction(comm_to_agent);

  /* Forward this to the proxy */
  
  if (sendInt(comm,NetSolveOK) == -1)
  {
    netsolvePerror("");
    return;
  }
  if (sendInt(comm,nb_servers) == -1)
  {
    netsolvePerror("");
    return;
  }
  for (i=0;i<nb_servers;i++)
  {
    if (sendServerDesc(comm,list[i]) == -1)
    {
      netsolvePerror("");
      return;
    }
  }
 
  return; 
}

/*
 * processProblemList()
 */
void processProblemList(NS_Communicator *comm)
{
  NS_ProblemDesc **list;
  int nb_problems,i;
  NS_Communicator *comm_to_agent;

  /* contact the agent */
  comm_to_agent = contactHost(agent,0,AGENT_PORT,DATA_XDR);
  if (comm_to_agent == NULL)
  {
    sendInt(comm,NetSolveCannotContactAgent);
    netsolvePerror("");
    return;
  }
   
  /* Sending */
  if (sendInt(comm_to_agent,NS_PROT_PROBLEM_LIST) == -1)
  {
    sendInt(comm,NetSolveNetworkError);
    netsolvePerror("");
    endTransaction(comm_to_agent);
    return;
  }

  if (recvInt(comm_to_agent,&nb_problems) == -1)
  {
    sendInt(comm,NetSolveNetworkError);
    netsolvePerror("");
    endTransaction(comm_to_agent);
    return;
  }

  list = (NS_ProblemDesc **)calloc(nb_problems,sizeof(NS_ProblemDesc*));
  for (i=0;i<nb_problems;i++)
  {
    list[i] = recvProblemDesc(comm_to_agent);
    if (list[i] == NULL)
    {
      sendInt(comm,NetSolveNetworkError);
      netsolvePerror("");
      endTransaction(comm_to_agent);
      return;
    }
  }
  endTransaction(comm_to_agent);

  /* Forward this to the proxy */
  
  if (sendInt(comm,NetSolveOK) == -1)
  {
    netsolvePerror("");
    return;
  }
  if (sendInt(comm,nb_problems) == -1)
  {
    netsolvePerror("");
    return;
  }
  for (i=0;i<nb_problems;i++)
  {
    if (sendProblemDesc(comm,list[i]) == -1)
    {
      netsolvePerror("");
      return;
    }
  }
 
  return; 
}

/*
 * processKillJob()
 */
void processKillJob(NS_Communicator *comm)
{
  int ID;
  NS_JobDesc *jd;
  NS_Communicator *comm_to_server;
  int NS_errno;

  if (recvInt(comm,&ID) == -1)
  {
    netsolvePerror("");
    return;
  }
  if ((ID < 0)||(ID>NB_MAX_JOBS))
  {
    sendInt(comm,NetSolveInvalidRequestID);
    return;
  }

  if (jobs[ID] == NULL)
  {
    sendInt(comm,NetSolveInvalidRequestID);
    return;
  }

  jd = jobs[ID];

  /* Contact the server */
  comm_to_server = contactHost(NULL,jd->IPaddr,jd->port,DATA_XDR);
  if (comm_to_server == NULL)
  {
    sendInt(comm,NetSolveNetworkError);
    netsolvePerror("");
    return;
  }
  if ((sendInt(comm_to_server,NS_PROT_KILL_JOB) == -1) ||
      (sendInt(comm_to_server,jd->pid) == -1))
  {
    sendInt(comm,NetSolveNetworkError);
    netsolvePerror("");
    return;
  }
  if (recvInt(comm_to_server,&NS_errno) == -1)
  {
    sendInt(comm,NetSolveNetworkError);
    netsolvePerror("");
    return;
  } 
  endTransaction(comm_to_server);
  if (sendInt(comm,NS_errno) == -1)
  {
    netsolvePerror("");
    return;
  }
  rmJobDesc(ID);
  return;
}

/*
 * processProblemInfo()
 */
void processProblemInfo(NS_Communicator *comm)
{
  char *nickname;
  NS_Communicator *comm_to_agent;
  int tag;
  NS_ProblemDesc *pd;

  if (recvString(comm,&nickname) == -1)
  {
    netsolvePerror("");
    return;
  }
  
  /* contact the agent */
  comm_to_agent = contactHost(agent,0,AGENT_PORT,DATA_XDR);
  if (comm_to_agent == NULL)
  {
    sendInt(comm,NetSolveCannotContactAgent);
    netsolvePerror("");
    return;
  }
   
  /* Sending */
  if (sendInt(comm_to_agent,NS_PROT_PROBLEM_INFO) == -1)
  {
    sendInt(comm,NetSolveNetworkError);
    netsolvePerror("");
    endTransaction(comm_to_agent);
    return;
  }
  if (sendString(comm_to_agent,nickname) == -1)
  {
    sendInt(comm,NetSolveNetworkError);
    netsolvePerror("");
    endTransaction(comm_to_agent);
    return;
  }

  /* Receiving */
  if (recvInt(comm_to_agent,&tag) == -1)
  {
    sendInt(comm,NetSolveNetworkError);
    netsolvePerror("");
    endTransaction(comm_to_agent);
    return;
  }

  /* Forward the tag */
  if (tag == NS_PROT_PROBLEM_DESC)
  {
    pd = recvProblemDesc(comm_to_agent);
    if (pd == NULL)
    {
      netsolvePerror("");
      sendInt(comm,NetSolveNetworkError);
      endTransaction(comm_to_agent);
      return;
    } 
  } else {
    /* (jakob) XXX:  What should we do if tag != NS_PROT_PROBLEM_DESC ?? */
    /* For now, we set pd = 0, and check for it when we call sendProblemDesc() 
     * further down... */
    pd = 0;
  }
  endTransaction(comm_to_agent);
  if (tag == NS_PROT_PROBLEM_NOT_FOUND)
  {
    if (sendInt(comm,NetSolveUnknownProblem) == -1)
    {
      netsolvePerror("");
      return;
    }
    return;
  }
  if (sendInt(comm,NetSolveOK) == -1)
  {
    netsolvePerror("");
    return; 
  }

  /* Forward the actual descriptor */
  if (pd && sendProblemDesc(comm,pd) == -1)
  {
    netsolvePerror("");
    return;
  }
  return;
}
 
/*
 * processSendRequest()
 */
void processSendRequest(NS_Communicator *comm)
{
  NS_JobDesc *jd;
  NS_ProblemDesc *pd;
  int input_size,output_size,problem_size;
  int ID;
  int status;
  int nb_servers;
  char **hostlist;
  NS_IPaddr_type *IPaddrlist;
  int *portlist;
  int *dataformatlist;
  int *predictedtimelist;
  int i;
  time_t now;
  int tmp;
  int predicted_time = 0;
  int start_time;
  int newport,pid;


  /* receive the info */
  if ((pd = recvProblemDesc(comm)) == NULL)
  {
    netsolvePerror("");
    return;
  }
  
  if ((recvInt(comm,&input_size) == -1) ||
      (recvInt(comm,&output_size) == -1) ||
      (recvInt(comm,&problem_size) == -1))
  {
    netsolvePerror("");
    return;
  }
        
  /* Do the actual submission !!! */
  status = submitProblemToAgent(pd->nickname,input_size,
                 output_size,problem_size,
                 &nb_servers, &hostlist, &IPaddrlist,
                 &portlist, &dataformatlist,
                 &predictedtimelist,agent);

  if (status == -1)
  {
    if (sendInt(comm,ns_errno) == -1)
    {
      netsolvePerror("");
      return;
    }
    return;
  }

  /********************************************/
  /* contact the servers successively         */
  /********************************************/

  jd = NULL;
  for (i=0;i<nb_servers;i++)
  {
    if (jd != NULL)
      freeJobDesc(jd);
    jd = newJobDesc();
    predicted_time = predictedtimelist[i];
    time(&now);
    jd->IPaddr = IPaddrlist[i];
    jd->port = portlist[i];

    status = submitProblemToServer(pd,IPaddrlist[i],
                                   portlist[i],&start_time,
                                   &newport,&pid);
    if (status == -1)
    {
# if defined(VIEW)
    netsolvePerror("ProcessSendRequest()");
#endif
      switch(ns_errno)
      {
        case NetSolveNoServer:
        case NetSolveConnectionRefused:
          forwardFailure(IPaddrlist[i],pd->nickname,HOST_ERROR);
          continue;
        case NetSolveNetworkError:
        case NetSolveUnknownProblem:
        case NetSolveServerError:
        case NetSolveProtocolError:
          forwardFailure(IPaddrlist[i],pd->nickname,SERVER_ERROR);
          continue;
        case NetSolveNotAllowed:
          continue;
        case NetSolveUPFError:
        case NetSolveUPFUnsafe:
        case NetSolveBadProblemSpecification:
        case NetSolveFileError:
        case NetSolveInternalError:
        case NetSolveBadValues:
        case NetSolveDimensionMismatch:
        case NetSolveNoSolution:
        case NetSolveSystemError:
          tmp = ns_errno;
          freeJobDesc(jd);
          freeCharStarList(hostlist,nb_servers);
          free(IPaddrlist);
          free(portlist);
          free(dataformatlist);
          free(predictedtimelist);
          ns_errno = tmp;
          if (sendInt(comm,ns_errno) == -1)
          {
            netsolvePerror("");
            return;
          }
          return;
        case NetSolveOK:
          break;
        default:
          continue;
      }
    }
    break;
  }

  if (i == nb_servers || status == -1)
  {
    freeCharStarList(hostlist,nb_servers);
    free(IPaddrlist);
    free(portlist);
    free(dataformatlist);
    free(predictedtimelist);
    freeJobDesc(jd);
    ns_errno = NetSolveNoServer;
    if (sendInt(comm,ns_errno) == -1)
    {
      netsolvePerror("");
      return;
    }
    return;
  }

  /* Build the Job Descriptor */
  jd = newJobDesc();
  jd->error_code = NetSolveNotReady;
  jd->IPaddr = IPaddrlist[i];
  jd->port = portlist[i];
  jd->pid = pid;

  /* Request the ID number from the daemon */
  ID = requestInsertJobDesc(jd);
  if (ID == -1)
  {
    status = -1;
    ns_errno = NetSolveTooManyPendingRequests;
    if (sendInt(comm,ns_errno) == -1) 
    {
      netsolvePerror("");
      return;
    }
    return;
  }


  if ((sendInt(comm,NetSolveOK) == -1)                ||
     (sendInt(comm,ID) == -1)       ||
     (sendInt(comm,predicted_time) == -1)       ||
     (sendInt(comm,start_time) == -1)           ||
     (sendIPaddr(comm,&(IPaddrlist[i])) == -1)     ||
     (sendString(comm,hostlist[i]) == -1)     ||
     (sendInt(comm,newport) == -1)          ||
     (sendInt(comm,dataformatlist[i]) == -1))
  {
    netsolvePerror("");
  }
  freeCharStarList(hostlist,nb_servers);
  free(IPaddrlist);
  free(portlist);
  free(dataformatlist);
  free(predictedtimelist);

  endTransaction(comm);
  return; 
}

/*
 * processServerFailure()
 */
void processServerFailure(NS_Communicator *comm)
{
  NS_IPaddr_type IPaddr;
  char *nickname;
  int error_type;
  int ID;
 
  if ((recvIPaddr(comm,&IPaddr) == -1) ||
      (recvString(comm,&nickname) == -1) ||
      (recvInt(comm,&ID) == -1) ||
      (recvInt(comm,&error_type) == -1))
  {
    netsolvePerror("");
    return;
  }     

  /* contact the agent */
  forwardFailure(IPaddr,nickname,error_type);

  /* contact the daemon */
  rmJobDesc(ID);

  return;
}

/*
 * processProbeRequest()
 */
void processProbeRequest(NS_Communicator *comm)
{
  int ID;

  if (recvInt(comm,&ID) == -1)
  {   
    netsolvePerror("");
    return;
  } 
  
  if ((ID<0)||(ID>NB_MAX_JOBS))
  {
    sendInt(comm,NetSolveInvalidRequestID);
    return;
  }
  if (jobs[ID] == NULL)
  {
    sendInt(comm,NetSolveInvalidRequestID);
    return;  
  }
  if ((sendInt(comm,NetSolveOK) == -1) ||
      (sendInt(comm,jobs[ID]->error_code) == -1)){
    netsolvePerror("sendInt");
  }
  return;
}

/*
 * processAwaitCompletion()
 */
void processAwaitCompletion(NS_Communicator *comm)
{
  int ID;
  NS_NotificationWaiter * nnw;

  if (recvInt(comm,&ID) == -1)
  {   
    netsolvePerror("");
    return;
  } 
  
  if ((ID<0)||(ID>NB_MAX_JOBS))
  {
    sendInt(comm,NetSolveInvalidRequestID);
    return;
  }
  if (jobs[ID] == NULL)
  {
    sendInt(comm,NetSolveInvalidRequestID);
    return;  
  }

  /* First, see if job is complete already */
  if(jobs[ID]->error_code != NetSolveNotReady) {
    if ((sendInt(comm,NetSolveOK) == -1) ||
	(sendInt(comm,jobs[ID]->error_code) == -1)){
      netsolvePerror("");
    }
/*      fprintf(stderr, "In awaitCompletion(): Job already complete...\n"); */
    return;
  }

  /* Ok, register this client in the waiter list
   * for immediate notification once the request
   * completes
   */
  nnw = (NS_NotificationWaiter*)malloc(sizeof(NS_NotificationWaiter));
  if(!nnw) {
    /* malloc() failed - let someone know */
    if ((sendInt(comm,NetSolveSystemError) == -1) ||
	(sendInt(comm,jobs[ID]->error_code) == -1))
      netsolvePerror("");
    return;
  }
  nnw->next = notification_waiters;
  notification_waiters = nnw;
  nnw->comm = comm;
  nnw->id = ID;
}


/*
 * processJobCompleted()
 */
void processJobCompleted(NS_Communicator *comm)
{
  NS_IPaddr_type IPaddr;
  int ID;
  NS_Communicator *comm_to_agent;

  if ((recvIPaddr(comm,&IPaddr) == -1)||
      (recvInt(comm,&ID) == -1))
  {
    netsolvePerror("");
    return;
  }

  /* contact the agent */
  comm_to_agent = contactHost(agent,0,AGENT_PORT,DATA_XDR);
  if (comm_to_agent == NULL)
  {
    netsolvePerror("");
    return;
  }

  /* Sending */
  if (sendInt(comm_to_agent,NS_PROT_JOB_COMPLETED) == -1)
  {
    netsolvePerror("");
    endTransaction(comm_to_agent);
    return;
  }
  if (sendIPaddr(comm_to_agent,&IPaddr) == -1)
  {
    netsolvePerror("");
    endTransaction(comm_to_agent);
    return;
  }
  endTransaction(comm_to_agent);

  /* contact the daemon */
  rmJobDesc(ID);

  return;
}


/*
 * processSendAssignment()
 */
void processSendAssignment(NS_Communicator *comm)
{
  NS_JobDesc *jd;
  NS_ServerDesc *sd;
  NS_ProblemDesc *pd;
  int ID;
  int status;
  NS_IPaddr_type IPaddr;
  time_t now;
  int tmp;
  int start_time;
  int newport,pid;


  /* receive the info */
  if ((pd = recvProblemDesc(comm)) == NULL)
  {
    netsolvePerror("");
    return;
  }
  
  if (recvIPaddr(comm,&IPaddr) == -1)
  {
    netsolvePerror("");
    return;
  }
        
  /* Receive the server info from the agent */
  if ((sd = getAServer(IPaddr,agent)) == NULL)
  {
    fprintf(stderr,"Server not found !!!\n");
    if (sendInt(comm,ns_errno) == -1)
    {
      netsolvePerror("");
      return;
    }
    return;
  }

  /********************************************/
  /* contact the server                       */
  /********************************************/
 
  jd = NULL;
  jd = newJobDesc();
  time(&now);
  status = submitProblemToServer(pd,IPaddr,sd->port,&start_time,&newport,&pid);
  if (status == -1)
  {
    switch(ns_errno)
    {
      case NetSolveNoServer:
      case NetSolveConnectionRefused:
        forwardFailure(IPaddr,pd->nickname,HOST_ERROR);
        break;
      case NetSolveNetworkError:
      case NetSolveUnknownProblem:
      case NetSolveServerError:
      case NetSolveProtocolError:
        forwardFailure(IPaddr,pd->nickname,SERVER_ERROR);
        break;
      case NetSolveNotAllowed:
        break;
      case NetSolveUPFError:
      case NetSolveUPFUnsafe:
      case NetSolveBadProblemSpecification:
      case NetSolveFileError:
      case NetSolveInternalError:
      case NetSolveBadValues:
      case NetSolveDimensionMismatch:
      case NetSolveNoSolution:
      case NetSolveSystemError:
        tmp = ns_errno;
        freeJobDesc(jd);
        ns_errno = tmp;
        status = -1;
        if (sendInt(comm,ns_errno) == -1) 
        {
          netsolvePerror("");
          return;
        }
        break;
      case NetSolveOK:
        break;
      default:
        break;
    }
    if (sendInt(comm,ns_errno) == -1)
    {
      netsolvePerror("");
      return;
    }
    return;
  }

  /* Build the Job Descriptor */
  jd = newJobDesc();
  jd->error_code = NetSolveNotReady;
  jd->IPaddr = IPaddr;
  jd->port = sd->port;
  jd->pid = pid;
  
  /* Request an ID number */
  ID = requestInsertJobDesc(jd);
  if (ID == -1)
  {
    ns_errno = NetSolveTooManyPendingRequests;
    if (sendInt(comm,ns_errno) == -1)
    {
      netsolvePerror("");
      return;
    }
    return;
  }

  if ((sendInt(comm,NetSolveOK) == -1)                ||
     (sendInt(comm,ID) == -1)       ||
     (sendInt(comm,start_time) == -1)           ||
     (sendInt(comm,newport) == -1)          ||
     (sendInt(comm,sd->host_desc->data_format) == -1))
  {
    netsolvePerror("");
  }
  endTransaction(comm);
  return; 
}

/*
 * processStoreHandle()
 */
void processStoreHandle(NS_Communicator *comm)
{
  DSI_OBJECT* dsi_obj;
  NS_Communicator *comm_to_agent;

  dsi_obj = (DSI_OBJECT *)malloc(sizeof(DSI_OBJECT));
  dsi_obj->dsi_file = (DSI_FILE *)malloc(sizeof(DSI_FILE));
  if(recvDsiObj(comm, dsi_obj) == -1)
  {
    netsolvePerror("");
    return;
  }

  /* contact the agent */
  comm_to_agent = contactHost(agent,0,AGENT_PORT,DATA_XDR);
  if (comm_to_agent == NULL)
  {
    sendInt(comm,NetSolveCannotContactAgent);
    netsolvePerror("");
    return;
  }

  /* Sending */
  if (sendInt(comm_to_agent,NS_PROT_STORE_HANDLE) == -1)
  {
    sendInt(comm,NetSolveNetworkError);
    netsolvePerror("");
    endTransaction(comm_to_agent);
    return;
  }
  if(sendDsiObj(comm_to_agent, dsi_obj)	 == -1)
  {
    sendInt(comm,NetSolveNetworkError);
    netsolvePerror("");
    endTransaction(comm_to_agent);
    return;
  }

  endTransaction(comm_to_agent);

  return;
}

/*
 * processGetHandle()
 */
void processGetHandle(NS_Communicator *comm)
{
char* name;
int tag;
NS_Communicator *comm_to_agent;
DSI_OBJECT* dsi_obj = NULL;

  if(recvString(comm, &name) == -1)
  {
    netsolvePerror("");
    return;
  }

  /* contact the agent */
  comm_to_agent = contactHost(agent,0,AGENT_PORT,DATA_XDR);
  if (comm_to_agent == NULL)
  {
    sendInt(comm,NetSolveCannotContactAgent);
    netsolvePerror("");
    return;
  }

  /* Sending */
  if (sendInt(comm_to_agent,NS_PROT_GET_HANDLE) == -1)
  {
    sendInt(comm,NetSolveNetworkError);
    netsolvePerror("");
    endTransaction(comm_to_agent);
    return;
  }

  if(sendString(comm_to_agent, name) == -1)
  {
    sendInt(comm,NetSolveNetworkError);
    netsolvePerror("");
    endTransaction(comm_to_agent);
    return;
  }

  /* Receiving */
  if (recvInt(comm_to_agent,&tag) == -1)
  {
    sendInt(comm,NetSolveNetworkError);
    netsolvePerror("");
    endTransaction(comm_to_agent);
    return;
  }

  /* Forward the tag */
  if (tag == NS_PROT_HANDLE)
  {
    dsi_obj = (DSI_OBJECT *)malloc(sizeof(DSI_OBJECT));
    dsi_obj->dsi_file = (DSI_FILE *)malloc(sizeof(DSI_FILE));
    if(recvDsiObj(comm_to_agent, dsi_obj) == -1)
    {
      sendInt(comm,NetSolveNetworkError);
	netsolvePerror("");
	endTransaction(comm_to_agent);
	return;
    }
  }

  endTransaction(comm_to_agent);

  if (tag == NS_PROT_HANDLE_NOT_FOUND)
  {
    if (sendInt(comm,NetSolveUnknownHandle) == -1)
    {
      netsolvePerror("");
	return;
    }
    return;
  }
  if (sendInt(comm,NetSolveOK) == -1)
  {
    netsolvePerror("");
    return;
  }

  /* send the dsi object back to the client */
  if(sendDsiObj(comm, dsi_obj)	== -1)
  {
    netsolvePerror("");
    return;
  }

  return;
}
