import java.util.Enumeration;
import java.util.StringTokenizer;
import java.util.NoSuchElementException;
import java.io.*;

import org.w3c.dom.*;
import org.xml.sax.*;
import javax.xml.parsers.SAXParserFactory;
import javax.xml.parsers.SAXParser;


public class PDFParser implements DocumentHandler
{
  private ProblemDescription curProb;
  private java.util.Vector probVec;
  private String curBuf;
  private Objects curObject;
  private IOSequence callSeq;
  private int seqCount;

  private int status;

  private static final int PROCESSING_INPUTS = 0;
  private static final int PROCESSING_OUTPUTS = 1;
  private static final int PROCESSING_OTHER = 2;

  public PDFParser(){
    probVec = new java.util.Vector();
  }
  public ProblemDescription [] parseFile (String filename) throws IOException, PDFParsingException
  {
    ProblemDescription [] newprobs = null;

    try {
      String uri = "file:" + new java.io.File(filename).getAbsolutePath();
      Parser parser;

      SAXParserFactory spf = SAXParserFactory.newInstance();

      spf.setValidating(true);

      SAXParser sp = spf.newSAXParser();
      parser = sp.getParser();

      // parser.setDocumentHandler (new PDFParser());
      parser.setDocumentHandler(this);
      parser.setErrorHandler(new MyErrorHandler());
      parser.parse(uri);

    } catch (SAXParseException err) {
      throw new PDFParsingException( "** Parsing error" + 
        ", line " + err.getLineNumber() + "\n" +
        err.getMessage()+ "\n");
    } catch (javax.xml.parsers.ParserConfigurationException err2) {
      throw new PDFParsingException( "** Parser configuration error.");
    } catch (SAXException e) {
      throw new PDFParsingException("** Parsing error: " + e.getMessage());
    }

    newprobs = new ProblemDescription[probVec.size()];

    int i=0;
    for (Enumeration e = probVec.elements(); e.hasMoreElements(); i++)
      newprobs[i] = (ProblemDescription) e.nextElement();

    return newprobs;
  }

  static class MyErrorHandler extends HandlerBase
  {
    // treat validation errors as fatal
    public void error (SAXParseException e) throws SAXParseException
    {
      throw e;
    }

    // dump warnings too
    public void warning (SAXParseException err) throws SAXParseException
    {
      System.out.println ("** Warning" 
        + ", line " + err.getLineNumber ()
        + ", uri " + err.getSystemId ());
      System.out.println("   " + err.getMessage ());
    }
  }

  class IntPair {
    int num1,num2;
   
    IntPair(AttributeList a) {
      if(a.getName(0).equals("num1")) {
        num1 = Integer.parseInt(a.getValue(0));
        num2 = Integer.parseInt(a.getValue(1));
      }
      else { 
        num2 = Integer.parseInt(a.getValue(0));
        num1 = Integer.parseInt(a.getValue(1));
      }
    }
  }

  private Writer	out;

  // here are all the SAX DocumentHandler methods

  public void setDocumentLocator (Locator l)
  {
    // we'd record this if we needed to resolve relative URIs
    // in content or attributes, or wanted to give diagnostics.
  }

  public void startDocument ()
  throws SAXException
  {
    try {
        out = new OutputStreamWriter (System.out, "UTF8");
    } catch (IOException e) {
        throw new SAXException ("I/O error", e);
    }
  }

  public void endDocument ()
  throws SAXException
  {
    try {
        out.write ('\n');
        out.flush ();
        out = null;
    } catch (IOException e) {
        throw new SAXException ("I/O error", e);
    }
  }

  public void startElement (String tag, AttributeList attrs)
  throws SAXException
  {
    // check all attributes to make sure they are non-empty
    if(attrs != null) {
      for(int i = 0; i < attrs.getLength(); i++) {
        if(firstWord(attrs.getValue(i)).length() == 0) {
          throw new SAXException("Empty '" + attrs.getName(i) + 
              "' in '" + tag + "' not allowed.");
        }
      }
    }

    if( tag.equals("NetSolvePDF") ) NetSolvePDF(attrs); 
    else if( tag.equals("Problem") ) Problem(attrs); 
    else if( tag.equals("Function") ) Function(attrs); 
    else if( tag.equals("Include") ) Include(attrs); 
    else if( tag.equals("DashI") ) DashI(attrs); 
    else if( tag.equals("Lib") ) Lib(attrs); 
    else if( tag.equals("Language") ) Language(attrs); 
    else if( tag.equals("Major") ) Major(attrs); 
    else if( tag.equals("Path") ) Path(attrs); 
    else if( tag.equals("Description") ) Description(attrs); 
    else if( tag.equals("Input") ) Input(attrs); 
    else if( tag.equals("Output") ) Output(attrs); 
    else if( tag.equals("MatlabMerge") ) MatlabMerge(attrs); 
    else if( tag.equals("Customized") ) Customized(attrs); 
    else if( tag.equals("Complexity") ) Complexity(attrs); 
    else if( tag.equals("CallingSequence") ) CallingSequence(attrs); 
    else if( tag.equals("Arg") ) Arg(attrs); 
    else if( tag.equals("Code") ) Code(attrs); 
    else if( tag.equals("Scalar_B") || tag.equals("Scalar_I") || 
             tag.equals("Scalar_CHAR") || tag.equals("Scalar_S") || 
             tag.equals("Scalar_D") || tag.equals("Scalar_C") || 
             tag.equals("Scalar_Z") || tag.equals("Vector_B") || 
             tag.equals("Vector_I") || tag.equals("Vector_CHAR") || 
             tag.equals("Vector_S") || tag.equals("Vector_D") || 
             tag.equals("Vector_C") || tag.equals("Vector_Z") || 
             tag.equals("Matrix_B") || tag.equals("Matrix_I") || 
             tag.equals("Matrix_CHAR") || tag.equals("Matrix_S") || 
             tag.equals("Matrix_D") || tag.equals("Matrix_C") || 
             tag.equals("Matrix_Z") || tag.equals("SparseMatrix_B") || 
             tag.equals("SparseMatrix_I") || tag.equals("SparseMatrix_CHAR") || 
             tag.equals("SparseMatrix_S") || tag.equals("SparseMatrix_D") || 
             tag.equals("SparseMatrix_C") || tag.equals("SparseMatrix_Z") || 
             tag.equals("NetSolveFile") || tag.equals("NetSolvePackedFiles") || 
             tag.equals("NetSolveUPF") || tag.equals("NetSolveString") || 
             tag.equals("NetSolveStringList") )
      curObject = new Objects(firstWord(attrs.getValue(0)));
    else
      System.err.println("WARNING: Unknown tag: " + tag); 
  }

  private String firstWord(String s)
  {
    StringTokenizer st = new StringTokenizer(s," ");
    String newStr;

    try {
      newStr = st.nextToken();
    } catch(NoSuchElementException e) {
      newStr = "";
    }

    return newStr;
  }

  private void NetSolvePDF(AttributeList attrs)
  {
    // this begins the PDF file.  for now, nothing really to do here.
  }

  private void Problem(AttributeList attrs)
  {
    // initialize class variables
    curProb = new ProblemDescription();
    status = PROCESSING_OTHER;
    seqCount = 0;

    curProb.setName(firstWord(attrs.getValue(0)));
  }
  private void Function(AttributeList attrs)
  {
    curProb.addFunction(firstWord(attrs.getValue(0)));
  }
  private void Include(AttributeList attrs)
  {
    curProb.addInclude(firstWord(attrs.getValue(0)));
  }
  private void DashI(AttributeList attrs)
  {
    curProb.addDashI(firstWord(attrs.getValue(0)));
  }
  private void Lib(AttributeList attrs)
  {
    String l = firstWord(attrs.getValue(0));

    // look for -L which signifies libpath.  otherwise, add to libs.
    if(l.regionMatches(0,"-L",0,2))
      curProb.addLibdir(l);
    else
      curProb.addLib(l);
  }
  private void Language(AttributeList attrs)
  {
    curProb.setLanguage(firstWord(attrs.getValue(0)));
  }
  private void Major(AttributeList attrs)
  {
    curProb.setMajor(firstWord(attrs.getValue(0)));
  }
  private void Path(AttributeList attrs)
  {
    curProb.setPath(firstWord(attrs.getValue(0)));
  }
  private void Description(AttributeList attrs)
  {
    // begin description.  do nothing now, but set description 
    // when we see the ending Description tag.
  }
  private void Input(AttributeList attrs)
  {
    // begin INput block.
    status = PROCESSING_INPUTS;
  }
  private void Output(AttributeList attrs)
  {
    // begin Output block
    status = PROCESSING_OUTPUTS;
  }
  private void MatlabMerge(AttributeList attrs) throws SAXException
  {
    IntPair p = null;

    try {
      p = new IntPair(attrs);    
    } catch(NumberFormatException e) {
      throw new SAXException("Bad Specification for Matlab Merge: " + e.getMessage());
    }

    if((p.num1 >= curProb.numOutputObjects()) || (p.num2 >= curProb.numOutputObjects()))
      throw new SAXException("Matlab Merge: not that many outputs.");
 
    if(p.num1 != p.num2-1)
      throw new SAXException("Matlab Merge numbers must be consecutive.");

    curProb.setMatlabMerge(new MatlabMerge(p.num1, p.num2));
  }
  private void Customized(AttributeList attrs)
  {
    curProb.setCustomName(firstWord(attrs.getValue(0)));
  }
  private void Complexity(AttributeList attrs) throws SAXException
  {
    IntPair p = null;

    try {
      p = new IntPair(attrs);    
    } catch(NumberFormatException e) {
      throw new SAXException("Bad Specification for Complexity: " + e.getMessage());
    }

    curProb.setComplexity(new Complexity(p.num1, p.num2));
  }
  private void CallingSequence(AttributeList attrs)
  {
    // begin calling sequence.  nothing to do here.
  }
  private void Arg(AttributeList attrs) throws SAXException
  {
    String mnemonic = firstWord(attrs.getValue(0));
    StringTokenizer st = new StringTokenizer(mnemonic, ",");
    int idx = -1;
     
    try{
      idx = curProb.findMnemonic(st.nextToken());
    }catch(NoSuchElementException e) {
      throw new SAXException("Error parsing mnemonic: " + mnemonic);
    }

    if(idx == -1)
      throw new SAXException("Cannot find mnemonic: " + mnemonic);

    callSeq.setElementAt(idx,seqCount);

    seqCount++;
  }
  private void Code(AttributeList attrs)
  {
    // begin code.  nothing to do here.  catch text at end of tag.
  }

  public void endElement (String name)
  throws SAXException
  {
    if( name.equals("Problem") ) {
      probVec.addElement(curProb);
      curProb = new ProblemDescription();
    }
    else if( name.equals("Description") ) {
      curProb.setDescription(curBuf);
    }
    else if( name.equals("Code") ) {
      curProb.setCode(curBuf);
    }
    else if( name.equals("Input") ) {
      status = PROCESSING_OTHER;
      curProb.setInputObjectSequence(new IOSequence(curProb.numInputObjects()));
    }
    else if( name.equals("Output") ) {
      status = PROCESSING_OTHER;
      curProb.setArguments(initializeArgs());
      callSeq = new IOSequence(curProb.getArgCount(), -1);
      curProb.setOutputObjectSequence(new IOSequence(curProb.numOutputObjects()));
    }
    else if( name.equals("CallingSequence") ) {
      curProb.setCallSequence(callSeq);
    }
    else if( name.equals("Scalar_B") || 
             name.equals("Scalar_I") || 
             name.equals("Scalar_CHAR") || 
             name.equals("Scalar_S") || 
             name.equals("Scalar_D") || 
             name.equals("Scalar_C") || 
             name.equals("Scalar_Z") || 
             name.equals("Vector_B") || 
             name.equals("Vector_I") || 
             name.equals("Vector_CHAR") || 
             name.equals("Vector_S") || 
             name.equals("Vector_D") || 
             name.equals("Vector_C") || 
             name.equals("Vector_Z") || 
             name.equals("Matrix_B") || 
             name.equals("Matrix_I") || 
             name.equals("Matrix_CHAR") || 
             name.equals("Matrix_S") || 
             name.equals("Matrix_D") || 
             name.equals("Matrix_C") || 
             name.equals("Matrix_Z") || 
             name.equals("SparseMatrix_B") || 
             name.equals("SparseMatrix_I") || 
             name.equals("SparseMatrix_CHAR") || 
             name.equals("SparseMatrix_S") || 
             name.equals("SparseMatrix_D") || 
             name.equals("SparseMatrix_C") || 
             name.equals("SparseMatrix_Z") || 
             name.equals("NetSolveFile") || 
             name.equals("NetSolvePackedFiles") || 
             name.equals("NetSolveUPF") || 
             name.equals("NetSolveString") || 
             name.equals("NetSolveStringList") )
    {
      StringTokenizer st = new StringTokenizer(name,"_");
      String obj = null, data = null;

      try {
        obj = st.nextToken();

        if(obj.equals("Matrix"))
          curObject.setObjectType(Objects.NETSOLVE_MATRIX);
        else if(obj.equals("Vector"))
          curObject.setObjectType(Objects.NETSOLVE_VECTOR);
        else if(obj.equals("Scalar"))
          curObject.setObjectType(Objects.NETSOLVE_SCALAR);
        else if(obj.equals("NetSolveUPF"))
          curObject.setObjectType(Objects.NETSOLVE_UPF);
        else if(obj.equals("NetSolveFile"))
          curObject.setObjectType(Objects.NETSOLVE_FILE);
        else if(obj.equals("NetSolveString"))
          curObject.setObjectType(Objects.NETSOLVE_STRING);
        else if(obj.equals("NetSolvePackedFiles"))
          curObject.setObjectType(Objects.NETSOLVE_PACKEDFILES);
        else if(obj.equals("NetSolveStringList"))
          curObject.setObjectType(Objects.NETSOLVE_STRINGLIST);
        else if(obj.equals("SparseMatrix"))
          curObject.setObjectType(Objects.NETSOLVE_SPARSEMATRIX);
        else
          curObject.setObjectType(Objects.NETSOLVE_NOTYPE);

        data = st.nextToken();

        if(data.equals("B"))
          curObject.setDataType(Objects.NETSOLVE_B);
        else if(data.equals("I"))
          curObject.setDataType(Objects.NETSOLVE_I);
        else if(data.equals("CHAR"))
          curObject.setDataType(Objects.NETSOLVE_CHAR);
        else if(data.equals("S"))
          curObject.setDataType(Objects.NETSOLVE_S);
        else if(data.equals("D"))
          curObject.setDataType(Objects.NETSOLVE_D);
        else if(data.equals("C"))
          curObject.setDataType(Objects.NETSOLVE_C);
        else if(data.equals("Z"))
          curObject.setDataType(Objects.NETSOLVE_Z);
        else
          curObject.setDataType(Objects.NETSOLVE_NOTYPE);

      }
      catch(NoSuchElementException e) {
        // not expecting an exception on the first call to nextToken(),
        // but for untyped objects (e.g. NetSolveFile), the second call
        // will throw an exception, so we just set the type empty here.
        curObject.setDataType(Objects.NETSOLVE_NOTYPE);
      }

      curObject.setDesc(curBuf);

      try {
        curObject.setCallObj((ObjectType)Class.forName(obj).newInstance());
      } catch(Exception e) {
        throw new SAXException("Internal error creating data object.");
      }

      if(status == PROCESSING_INPUTS)
        curProb.addInputObject(curObject);
      else if(status == PROCESSING_OUTPUTS)
        curProb.addOutputObject(curObject);
      else
        throw new SAXException("Internal error (this case should not be reached).");
    }
  }

  private Arguments initializeArgs()
  {
    Arguments newArgs;
    int count = 0;

    // finished with inputs and outputs, so initialize the
    // arguments now.

    newArgs = new Arguments( curProb.getArgCount() );

    count = processArgs(curProb.getInputs(), newArgs, count, "I");
    count = processArgs(curProb.getOutputs(), newArgs, count, "O");

    return newArgs;
  }

  private int processArgs(Enumeration obj, Arguments a, int cur_count, String io)
  {
    ObjectType thisCallObj;
    Objects thisObject;
    int i=0;

    while(obj.hasMoreElements()) {
      thisObject = (Objects) obj.nextElement();
      thisCallObj = thisObject.getCallObj();

      for(int j=0;j< thisCallObj.getArgCount(); j++) {
        a.setUserArg(thisObject.getName() + "." + thisCallObj.getArg(j), cur_count);
        a.setNetsolveArgs(thisCallObj.getMnemonic(j) + io + Integer.toString(i),cur_count);
        a.setArgTypes(thisCallObj.getType(j),cur_count);
        cur_count++;
      }

      i++;
    }

    return cur_count;
  }

  public void characters (char buf [], int offset, int len)
  throws SAXException
  {
    curBuf = new String(buf, offset, len);
  }

  public void ignorableWhitespace (char buf [], int offset, int len)
  throws SAXException
  {
    // ignore
  }

  public void processingInstruction (String target, String data)
  throws SAXException
  {
    // ignore
  }
}
