/*****************************************************************/
/*      mxnetsolve.c                                             */
/*      Henri Casanova          				 */
/*****************************************************************/

#include "mex.h"
#include "core.h"
#include "client.h"
#include "matlabclient.h"

/*
 * Blocking call to NetSolve from Matlab
 */
void mexFunction(int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[])
{
  char buffer[256];
  NS_ProblemDesc *pd;
  int request_id;
  int status;
  int *input_allocated=NULL;
  int i;
  int elapsed;

  my_major = COL_MAJOR;

#ifdef WIN32
  if (!WinSockInit())
  {
    mxSetNSErrno(NetSolveNetworkError);
    for (i=0;i<nlhs;i++)
      plhs[i] = mxCreateDoubleMatrix(0,0,mxREAL);
    return;
  }
#endif

  /* find a slot for the request */
  request_id = 0;
  while(requests[request_id] != NULL)
  {
    request_id++;
    if (request_id == NB_MAX_REQUESTS)
    {
      ns_errno = NetSolveTooManyPendingRequests;
      return;
    }
  }

  proxy_port = mxGetProxyPort();
  if (proxy_port == -1) { 
    char *agent_name;

    agent_name = getNetSolveAgent();
    if (agent_name == NULL) {
      ns_errno = NetSolveSetNetSolveAgent;
      mexPrintf("%s\n",netsolveErrorMessage(ns_errno));
      mxSetNSErrno(ns_errno);
      for (i=0;i<nlhs;i++)
        plhs[i] = mxCreateDoubleMatrix(0,0,mxREAL);
      return;
    }

    /* Start the proxy !! */
    if (startProxy("NetSolve",agent_name,&proxy_pid,&proxy_port) == -1)      
    {
      mexPrintf("%s\n",netsolveErrorMessage(ns_errno));
      mxSetNSErrno(ns_errno);
      for (i=0;i<nlhs;i++)
        plhs[i] = mxCreateDoubleMatrix(0,0,mxREAL);
      return;
    }

    mxSetProxyPort(proxy_port);
  }

  /* List of Problems */
  if (nrhs == 0)
  {
    mxDisplayListOfProblems();
    for (i=0;i<nlhs;i++)
      plhs[i] = mxCreateDoubleMatrix(0,0,mxREAL);
    mxSetNSErrno(ns_errno);
    return;
  }

  /* Error */
  if (!mxIsChar(prhs[0]))
  {
    mexPrintf("Invalid first parameter : should be a string\n");
    for (i=0;i<nlhs;i++)
      plhs[i] = mxCreateDoubleMatrix(0,0,mxREAL);
    mxSetNSErrno(NetSolveBadValues);
    return;
  }

  mxGetString(prhs[0],buffer,256);

  /* List of servers */
  if (!strcmp(buffer,"?"))
  {
    mxDisplayListOfServersAgents();
    for (i=0;i<nlhs;i++)
      plhs[i] = mxCreateDoubleMatrix(0,0,mxREAL);
    mxSetNSErrno(ns_errno);
    return;
  }

  /* Information about a particular problem */
  if ((nlhs==0)&&(nrhs==1))
  {
    mxDisplayProblemDesc(buffer);
    for (i=0;i<nlhs;i++)
      plhs[i] = mxCreateDoubleMatrix(0,0,mxREAL);
    mxSetNSErrno(ns_errno);
    return;
  }

  /* Get the problem descriptor */
  status = netsolveInfo(buffer,&pd);  
  if (status < 0)
  {
    mexPrintf("%s\n",netsolveErrorMessage(ns_errno));
    for (i=0;i<nlhs;i++)
      plhs[i] = mxCreateDoubleMatrix(0,0,mxREAL);
    mxSetNSErrno(ns_errno);
    return;
  }

  /* Checking for SparseMatrix of char or bytes (not implemented in output) */
  for(i=0; i<pd->nb_output_objects; i++){
    if(pd->output_objects[i]->object_type == NETSOLVE_SPARSEMATRIX){
      if(pd->output_objects[i]->data_type == NETSOLVE_CHAR ||
         pd->output_objects[i]->data_type == NETSOLVE_B){
        mexPrintf("NetSolve: Output Object %d -- Sparse matrices of 'characters'"
                  " or 'bytes' not supported in output from Matlab interface\n", i);

        ns_errno = NetSolveNotAllowed;
        for (i=0;i<nlhs;i++)
          plhs[i] = mxCreateDoubleMatrix(0,0,mxREAL);

        mxSetNSErrno(ns_errno);
        return;
      }
    }
  }

  /* Check the input */
  status = mxInitializeInputObjects(pd,nrhs-1,
               (mxArray**)(&(prhs[1])),&input_allocated);
  if (status == -1)
  {
    mexPrintf("%s\n",netsolveErrorMessage(ns_errno));
    for (i=0;i<nlhs;i++)
      plhs[i] = mxCreateDoubleMatrix(0,0,mxREAL);
    if (input_allocated != NULL)
    {
      for (i=0;i<pd->nb_input_objects;i++)
        if (input_allocated[i])
          freeObjectData(pd->input_objects[i]);
    }
    freeProblemDesc(pd);
    mxSetNSErrno(ns_errno);
    return;
  }

  /* check the output */
  if ((pd->nb_output_objects < nlhs)||
      (nlhs < pd->nb_output_objects - pd->nb_matlab_merge))
  {
    mexPrintf("'%s' requires %d objects in output and %d of those can be merged\n",
              pd->nickname,pd->nb_output_objects,pd->nb_matlab_merge);
    for (i=0;i<nlhs;i++)
      plhs[i] = mxCreateDoubleMatrix(0,0,mxREAL);
    ns_errno = NetSolveBadProblemSpecification;
    mxSetNSErrno(ns_errno);
    return;
  }

  /* Initialize the output */
  status = mxInitializeOutputObjects(pd);
  if (status == -1)
  {
    mexPrintf("%s\n",netsolveErrorMessage(ns_errno));
    for (i=0;i<nlhs;i++)
      plhs[i] = mxCreateDoubleMatrix(0,0,mxREAL);
    for (i=0;i<pd->nb_input_objects;i++)
      if (input_allocated[i])
        freeObjectData(pd->input_objects[i]);
    freeProblemDesc(pd);
    mxSetNSErrno(ns_errno);
    return;
  }

  /* solve */
  status = submit_problem(NS_BLOCK, NS_NOASSIGNMENT, NULL,
            pd,pd->input_objects,pd->output_objects,&elapsed, request_id);

  if (status == -1)
  {
    mexPrintf("%s\n",netsolveErrorMessage(ns_errno));
    for (i=0;i<nlhs;i++)
      plhs[i] = mxCreateDoubleMatrix(0,0,mxREAL);
    for (i=0;i<pd->nb_input_objects;i++)
      if (input_allocated[i])
        freeObjectData(pd->input_objects[i]);
    free(input_allocated);
    freeProblemDesc(pd);
    mxSetNSErrno(ns_errno);
    return;
  }

  /* Take care of the output */
  status = mxFillInOutputObjects(pd,nlhs,plhs);
  if (status == -1)
  {
    mexPrintf("%s\n",netsolveErrorMessage(ns_errno));
    for (i=0;i<nlhs;i++)
      plhs[i] = mxCreateDoubleMatrix(0,0,mxREAL);
    for (i=0;i<pd->nb_input_objects;i++)
      if (input_allocated[i])
        freeObjectData(pd->input_objects[i]);
    free(input_allocated);
    freeProblemDesc(pd);
    mxSetNSErrno(ns_errno);
    return;
  }
  for (i=0;i<pd->nb_input_objects;i++)
  if (input_allocated[i])
    freeObjectData(pd->input_objects[i]);
  free(input_allocated);
  freeProblemDesc(pd);
  ns_errno = NetSolveOK;
  mxSetNSErrno(ns_errno);
  return;
}
