/*  nsmath.c
 * 
 *  Mathematica interface to NetSolve
 *  $Id: nsmath.c,v 1.3 2001/07/23 19:56:05 shi Exp $
 *
 */

#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <stdlib.h>
#include <signal.h>

#include "mathlink.h"

#include "core.h"
#include "netsolveerror.h"
#include "client.h"

#include "nsmath.h"
#include "nsmath_gateway.h"
#include "ns_utils.h"


char netsolve_agent[64];

#define NSMATH_MAX_REQUESTS 256

NS_RequestDesc *NB_requests[NSMATH_MAX_REQUESTS];

extern int flag;

void nsmath_usage()
{
   log(1, "usage()");
   log(2, "  MLError: %s", MLErrorMessage(stdlink));
   nsmath_msg(NSM_USAGE);
   nsmath_return_null();
}

void nsmath_list_problems()
{
   char **list, *err;
   int	i, n;

   log(1, "nsmath_list_problems()");
   if ((err = ns_list_problems(&n, &list)) != NULL) {
      log(2, "\tError: %s", err);
      nsmath_printf_msg("NetSolve: %s", err);
      nsmath_return_null();
      return;
   }
   log(2, "\tSuccessfull: %d problems returned", n);
   for(i=0; i<n; i++) 
      nsmath_msg(list[i]);
   nsmath_msg("--------------------");
   nsmath_printf_msg("Handle %d problem(s).",n); 
   nsmath_msg("--------------------");
   if (n==0) 
     nsmath_printf_msg("None.");
   free_ptr_array(n, (void **) list);
   nsmath_return_null();
}

void nsmath_list_servers()
{
   char *err;
   char **list;
   int	i, n;

   log(1, "nsmath_list_servers()");
   if ((err = ns_list_servers(&n, &list)) != NULL) {
      log(2, "\tError: %s", err);
      nsmath_msg(err);
      nsmath_return_null();
      return;
   }
   log(2, "\tSuccessfull...");
   for(i=0; i<n; i++) 
      nsmath_msg(list[i]);
   
   if (n==0) 
     nsmath_printf_msg("None.");
   free_ptr_array(n, (void **) list);
   nsmath_return_null();
}

char *data_type_str(int data_type)
{
   switch (data_type) {
    case NETSOLVE_I: 	return "Integer";
    case NETSOLVE_B: 	return "Byte";
    case NETSOLVE_CHAR: return "Character";
    case NETSOLVE_S: 	return "Single precision real";
    case NETSOLVE_D: 	return "Double precision real";
    case NETSOLVE_C: 	return "Single precision complex";
    case NETSOLVE_Z: 	return "Double precision complex";
    default: 		return "???";
   }
}

char *object_type_str(int object_type)
{
   switch (object_type) {
    case NETSOLVE_FILE: 	return "File";
    case NETSOLVE_PACKEDFILES:	return "Packed File";
    case NETSOLVE_STRINGLIST:	return "String List";
    case NETSOLVE_MATRIX: 	return "Matrix";
    case NETSOLVE_VECTOR: 	return "Vector";
    case NETSOLVE_SCALAR: 	return "Scalar";
    case NETSOLVE_UPF: 		return "User supplied function";
    case NETSOLVE_STRING:	return "String";
    default: 			return "???";
   }
}
     
void nsmath_msg_pb_args(char *arg_type, int n, NS_Object **objs)
{
   int	i;
   
   nsmath_msg("");
   nsmath_printf_msg(arg_type);
   for(i=0; i<n; i++) {
      nsmath_printf_msg("\t#%2d : %s %s", i, data_type_str(objs[i]->data_type), 
			 object_type_str(objs[i]->object_type));
      nsmath_printf_msg("\t%s", objs[i]->description);
   }
}


char obj_label(NS_Object *obj)
{
   char c;
   
   switch (obj->data_type) {
    case NETSOLVE_I:
      c = 'i';
      break;
    case NETSOLVE_S:
      c = 's';
      break;
    case NETSOLVE_D:
      c = 'd';
      break;
    case NETSOLVE_C:
      c = 'c';
      break;
    case NETSOLVE_Z:
      c = 'z';
      break;
    case NETSOLVE_B:
    case NETSOLVE_CHAR:
      c = 'b';
      break;
/*    case NETSOLVE_ASCII:
      c = 'f';
      break;
    case NETSOLVE_EXTERNAL:
      c = 'u';
      break;    
 */
    default:
      c = 'x';
   }
   if (obj->object_type == NETSOLVE_MATRIX ||
       obj->object_type == NETSOLVE_VECTOR) {
      c &= 0xdf;
   }
   return c;
}
       
char *nsmath_sprintf_args(int sz, char *buf, char *prefix, int n, NS_Object **objs)
{
   int i, len;

   *buf = '\0';
   for(i=0; i<n && (len = strlen(buf)) < sz; i++) {
      if (i<n-1) sprintf(&buf[len], "%s%c%d,", prefix, obj_label(objs[i]), i);
      else sprintf(&buf[len], "%s%c%d", prefix, obj_label(objs[i]), i);
   }
   
   return buf;
}


void nsmath_problem_info(char *problem)
{
   char 	*name, junk[128], junk2[128];
   NS_ProblemDesc  *pb_desc;
   char *agent_name;

/* pedantic -- but better than to find odd errors instead :) */   
   if (!problem || strlen(problem)<=2) return;
   
   log(1, "nsmath_problem_info(\"%s\")", problem);
   name = (char *) malloc(strlen(problem)+1);

/* remove ending '[]' */
   strcpy(name, problem);
   name[strlen(problem)-2] = '\0';

   agent_name = getNetSolveAgent();
   if (agent_name == NULL)
   {
     ns_errno == NetSolveSetNetSolveAgent; 
     log(2, "\tError: %s", netsolve_error(ns_errno));
     nsmath_printf_msg("NetSolve: %s", netsolve_error(ns_errno));
     nsmath_return_null();
   }
   if ((netsolveInfo(name, &pb_desc)) == -1) {
      log(2, "\tError: %s", netsolve_error(ns_errno));
      nsmath_printf_msg("NetSolve: %s", netsolve_error(ns_errno));
   }
   else {
      log(2, "\tSuccess");
      nsmath_printf_msg("%s: %s", pb_desc->nickname, pb_desc->description);
      nsmath_msg_pb_args("Input:", pb_desc->nb_input_objects, pb_desc->input_objects);
      nsmath_msg_pb_args("Output:", pb_desc->nb_output_objects, pb_desc->output_objects);
      
/*    { O1, O2, ..., On } = netsolve[ %name[I1, I1, ..., In ] ] */
      nsmath_printf_msg("\nMathematica example:\n%c%s%c = NetSolve[%s[%s]]",
			(pb_desc->nb_output_objects > 1) ? '{' : ' ',
			 nsmath_sprintf_args(sizeof(junk), junk, "r",
					     pb_desc->nb_output_objects, pb_desc->output_objects),
			(pb_desc->nb_output_objects > 1) ? '}' : ' ',
			 pb_desc->nickname,
			 nsmath_sprintf_args(sizeof(junk2), junk2, "",
					     pb_desc->nb_input_objects, pb_desc->input_objects));
      nsmath_printf_msg("\nexamples for types:\n");
      nsmath_printf_msg("        Char     Byte/Integer  Single/Double  Complex");
      nsmath_printf_msg("Scalar: \\\"c\\\"           42          66.32       4 - 7 I ");
      nsmath_printf_msg("Vector: \\\"vector\\\"    {1,2,3}     {3,4.5,7}    {3, -5+3I, 8}");
      nsmath_printf_msg("Matrix: {\\\"line 1\\\", {{1,2,3},   {{6.4,2,1},    {{1+2I, 3+4I},");
      nsmath_printf_msg("         \\\"line 2\\\"}  {4,5,6}}    {-7,1.2,4}}    {5-6I, 7}}");
   }
   free(agent_name);
   freeProblemDesc(pb_desc);
   free(name);
   nsmath_return_null();
}

void nsmath_info(char *keyword)
{
   if (!keyword) {
      nsmath_return_null();
      return;
   }
   log(1, "nsmath_info(\"%s\")", keyword);
   if (strcmp(keyword, "?problems")==0) nsmath_list_problems();
   else if (strcmp(keyword, "?servers")==0) nsmath_list_servers();
   else if (keyword[0] == '?' && 
	    strstr(keyword, "[]") == keyword+strlen(keyword)-2) 
           nsmath_problem_info(&keyword[1]);
   else nsmath_usage();
}
   
void nsmath_probe(int rq)
{
   int elapsed;
   NS_RequestDesc *rd = NB_requests[rq]; 
   log(1, "nsmath_probe()");
   if (rq<0 || rq>=NSMATH_MAX_REQUESTS || !NB_requests[rq])  {
      ns_errno = NetSolveInvalidRequestID;
      log_and_printf_msg(2, netsolveErrorMessage(ns_errno));
      nsmath_return_null();
      return;
   }
     
   MLPutInteger(stdlink, netsolveProbeRequest(rd->assignment,&(NB_requests[rq])));
}

void nsmath_wait(int rq)
{
   int elapsed;
   NS_RequestDesc *rd = NB_requests[rq];

   
   log(1, "nsmath_wait()");
   if (rq<0 || rq>=NSMATH_MAX_REQUESTS || !NB_requests[rq])  {
      ns_errno = NetSolveInvalidRequestID;
      log_and_printf_msg(2, netsolveErrorMessage(ns_errno));
      nsmath_return_null();
      return;
   }
   
   netsolveWaitRequest(rd->assignment,&NB_requests[rq],&elapsed);

   if (ns_errno == NetSolveOK)
     gateway_to_ML(NB_requests[rq]->pd);
   else {
      log_and_printf_msg(2, netsolve_error(ns_errno));
      nsmath_return_null();
   }
   freeProblemDesc(NB_requests[rq]->pd);
   freeRequestDesc(NB_requests[rq]);
   NB_requests[rq] = NULL;
}


void nsmath_get_agent()
{
   char *s;
   
   ns_errno = NetSolveOK;
   s = getenv("NETSOLVE_AGENT");
   log(2, "nsmath_get_agent = %s", s);
   MLPutString(stdlink, (s) ? s : "");
}

void nsmath_set_agent(char *agent)
{
   char buf[sizeof(netsolve_agent)];
   
/* just to make sure there is no buffer overrun */
   
   ns_errno = NetSolveOK;   
 strncpy(buf, agent, sizeof(netsolve_agent) - strlen(NETSOLVE_AGENT) - 2);
   buf[sizeof(buf)-1] = '\0';

/* sprintf(netsolve_agent, "%s=%s", NETSOLVE_AGENT, buf);*/ 
  sprintf(netsolve_agent, "NETSOLVE_AGENT=%s",buf);
   if(putenv(netsolve_agent)!=0)
    MLPutString(stdlink,"cannot change environment variable NETSOLVE_AGENT");
   log(1, "nsmath_set_agent(%s)", buf);
   flag=1;
   nsmath_return_null();
}

void nsmath_error_msg(int code)
{
   MLPutString(stdlink, netsolve_error(code));
}

int nsmath_error()
{
   return ns_errno;
}

void nsmath_solve_problem()
{
   NS_ProblemDesc *pd;
   int elapsed;

   log(1, "nsmath_solve_problem()");
   
   ns_errno = NetSolveOK;

   setMajorDefault("Row");
   if ((pd = ML_to_gateway()) != NULL)
    {
     submit_problem(NS_BLOCK,NS_NOASSIGNMENT,NULL,pd, pd->input_objects, pd->output_objects,&elapsed,0);
     
    } 
   if (ns_errno == NetSolveOK && pd != NULL) 
      gateway_to_ML(pd);
   else {
      log_and_printf_msg(2, netsolve_error(ns_errno));
      nsmath_return_null();
   }
   freeProblemDesc(pd);
}

void nsmath_solve_problem_nb()
{
   NS_ProblemDesc  *pd;
   NS_RequestDesc  *rd;
   char 	*s;
   int		i, rq;

   log(1, "nsmath_solve_problem_nb()");
   
   ns_errno = NetSolveOK;

   setMajorDefault("Row");
   
   if ((pd = ML_to_gateway()) != NULL)
     rd = netsolveSendRequest(NS_NOASSIGNMENT,NULL,pd, pd->input_objects, pd->output_objects);
   
   if (ns_errno == NetSolveOK && pd && rd) {
      rq = -1;
      for(i=0; i<NSMATH_MAX_REQUESTS; i++)
	if (!NB_requests[i]) {
	   rq = i;
	   break;
	}
      if (rq == -1) {
	 ns_errno = NetSolveTooManyPendingRequests;
	 log_and_printf_msg(2, netsolve_error(ns_errno));
	 nsmath_return_null();
      }
      NB_requests[rq] = rd;
      if (!MLPutInteger(stdlink, rq)) {
	 s = strdup(MLErrorMessage(stdlink));
	 MLClearError(stdlink);
	 log_and_printf_msg(2, "MLError: %s", s);
	 free(s);
	 nsmath_return_null();
      }
   }
   else {
      log_and_printf_msg(2, netsolve_error(ns_errno));
      nsmath_return_null();
   }
}

#ifdef SYSVSIGNAL
void nsmath_sigpipe_handler(int s)
{
   log(1, "Computational server failed");
   signal(SIGPIPE, nsmath_sigpipe_handler);
}
#endif

int main(int argc, char **argv)
{
#ifdef DBGLOG
   int i;
   nsmath_initlog();

   for(i=0; i<argc; i++)
     log(8, "argc[%d] = %s", i, argv[i]);
#endif
#ifdef SYSVSIGNAL
   signal(SIGPIPE, nsmath_sigpipe_handler);
#endif
   /* agent_name here ? */
   for(i=0; i<NSMATH_MAX_REQUESTS; i++)
     NB_requests[i] = NULL;
   
   return MLMain(argc, argv);
}
