/*****************************************************************/
/*      iserv_spawn.c                                            */
/*      Jakob Oestergaard                                        */
/*---------------------------------------------------------------*/
/*  spawn_infoserver.c                                           */
/*****************************************************************/

#include "infoserver.h"
#include "infoserver_int.h"
#include "socketutil.h"

#include <errno.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <sys/un.h>
#include <string.h>
#include <stdio.h>
#include <signal.h>

/*
 * For use by the infoserver itself
 */
int infoserver_input_fd = -1;
int infoserver_client_fd = -1;

/*
 * Spawn local infoserver
 */
int spawn_infoserver(void)
{
  return spawn_infoserver_int(NS_IS_PRIVATE);
}



/*
 * Main spawn routine
 */
int spawn_infoserver_int(iserv_access_t accesst)
{
  int cpid;
  struct sockaddr_in iaddr;
  int rc;

  /*
   * Bind to client port
   */
  if(0 > (infoserver_client_fd = socket(PF_INET, SOCK_STREAM, 0))) {
    fprintf(stderr, "InfoServer could not create client socket (%s)\n",
	    strerror(errno));
    return 1;
  }
  rc = 1;
  if(0 >  setsockopt(infoserver_client_fd, SOL_SOCKET, SO_REUSEADDR, &rc, sizeof(rc)))
    fprintf(stderr, "Couldn't set socket option for client socket (%s)\n",
	    strerror(errno));

  iaddr.sin_family = AF_INET;
  iaddr.sin_port = htons(NS_IS_CLIENT_PORT);
  iaddr.sin_addr.s_addr = INADDR_ANY;
  if(0 > (rc = bind(infoserver_client_fd, (struct sockaddr*)&iaddr, sizeof(iaddr)))) {
    fprintf(stderr, "InfoServer could not bind to client socket port %i (%s)\n",
	    NS_IS_CLIENT_PORT, strerror(errno));
    return 1;
  }

  /*
   * Depending on InfoServer type (public/private)
   * we either bind to a TCP socket for accepting client
   * connections, or we bind to a UN*X socket.
   */
  if(accesst == NS_IS_PUBLIC) {
    /* 
     * Go public
     */
    if(0 > (infoserver_input_fd = socket(PF_INET, SOCK_STREAM, 0))) {
      fprintf(stderr, "InfoServer could not create public input socket (%s)\n",
	      strerror(errno));
      return 1;
    }
    rc = 1;
    if(0 >  setsockopt(infoserver_input_fd, SOL_SOCKET, SO_REUSEADDR, &rc, sizeof(rc)))
      fprintf(stderr, "Couldn't set socket option for input socket (%s)\n",
	      strerror(errno));
    iaddr.sin_port = htons(NS_IS_INPUT_PORT);
    if(0 > (rc = bind(infoserver_input_fd, (struct sockaddr*)&iaddr, sizeof(iaddr)))) {
      fprintf(stderr, "InfoServer could not bind to public input port %i (%s)\n",
	      NS_IS_INPUT_PORT, strerror(errno));
      return 1;
    }
  } else {
    struct sockaddr_un addr;
    /*
     * Create UN*X socket
     */
    if(0 > (infoserver_input_fd = socket(PF_UNIX, SOCK_STREAM, 0))) {
      fprintf(stderr, "InfoServer could not create public input socket (%s)\n",
	      strerror(errno));
      return 1;
    }
    addr.sun_family = AF_UNIX;
    sprintf(addr.sun_path, "/tmp/infoserver-%i", getpid());
    unlink(addr.sun_path);
    if(0 > bind(infoserver_input_fd, (struct sockaddr*)&addr, sizeof(addr))) {
      fprintf(stderr, "Inforserver could not bind to local socket (%s)\n",
	      strerror(errno));
      return 1;
    }    
    if(!(infoserver_input_path = strdup(addr.sun_path))) {
      fprintf(stderr, "Not enough memory to remember path\n");
      return 1;
    }
  }

  /*
   * Fork and detach from old session
   */

  if(0 > (cpid = fork())) {
    fprintf(stderr, "spawn_infoserver() cannot fork! (%s)\n",
	    strerror(errno));
    return 1;
  }

  if(cpid) {
    close(infoserver_input_fd);
    fprintf(stderr, "InfoServer spawned (pid=%i)\n", cpid);
    return 0;
  }

  setsid();

  signal(SIGPIPE, SIG_IGN);

  /*
   * Initialize DBMS
   */
  if(0 > dbms_initialize()) {
    fprintf(stderr, "Couldn't initialize database engine\n");
    return 0;
  }

  /*
   * Load on-disk database
   */
  iserv_readdb();

  /*
   * Run main loop
   */
  iserv_main_loop();

  return 0;
}
