%{
#include "infoserver_int.h"
#include <ctype.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>

/*
 * We need this global variable for knowing what fd the
 * client we're parsing for is hanging on.
 */
int parse_client = -1;
/*
 * And the parser can be in two modes, one for parsing
 * ``interactive'' client stuff, and one for parsing the
 * log file.
 */
NS_IS_PMode_t parser_mode = NS_IS_PMode_Interactive;


/*
 * Stuff for statistical purposes
 */
static time_t   t_lastput = 0;
static time_t   t_lastget = 0;
static unsigned n_puts = 0;
static unsigned n_gets = 0;


time_t req_stat_lastput(void)
{
  return t_lastput;
}

time_t req_stat_lastget(void)
{
  return t_lastget;
}

unsigned req_stat_puts(void)
{
  return n_puts;
}

unsigned req_stat_gets(void)
{
  return n_gets;
}


int yyerror(char * err)
{
  fprintf(stderr, "yyerror(): %s\n", err);
  if(parse_client != -1) {
    char * tmp = (char*)malloc(strlen(err) + 200);
    if(!tmp) {
      push_to_client(parse_client,
		     "ERROR (parse error - no clarification due to memory shortage)\n");
      return 0;
    }
    sprintf(tmp, "ERROR (%s)\n", err);
    push_to_client(parse_client, tmp);
    free(tmp);
  }
  return 0;
}

/*
 * Real code
 */

static NS_IS_Expression * new_expression(void) 
{
  NS_IS_Expression * ret = (NS_IS_Expression*)malloc(sizeof(NS_IS_Expression));
  if(ret) memset(ret, 0, sizeof(NS_IS_Expression));
  return ret;
}

static NS_IS_Insertion * new_insertion(void) 
{
  NS_IS_Insertion * ret = (NS_IS_Insertion*)malloc(sizeof(NS_IS_Insertion));
  if(ret) memset(ret, 0, sizeof(NS_IS_Insertion));
  return ret;
}

static void delete_expression(NS_IS_Expression* e)
{
  if(!e)
    return;

  switch(e->etype) {
  case NS_IS_Exp_KeyValue:
    if(e->value) {
      free(e->value);
      e->value = 0;
    }
    /* fallthru */
  case NS_IS_Exp_KeyReq:
    if(e->key) {
      free(e->key);
      e->key = 0;
    }
    break;
  case NS_IS_Exp_Combination:
    delete_expression(e->lexp);
    e->lexp = 0;
    delete_expression(e->rexp);
    e->rexp = 0;
    break;
  default:
    fprintf(stderr, "Unknown expression type encountered\n");
  }
}

static void delete_insertion(NS_IS_Insertion* e)
{
  if(!e)
    return;
  if(e->key)
    free(e->key);
  if(e->value)
    free(e->value);
  if(e->next)
    delete_insertion(e->next);
  free(e);
}

%}

%union {
  NS_IS_Expression * expr;
  NS_IS_Insertion * pexpr;
  char * keyname;
  char * value;
  NS_IS_Exp_RelOp_t rel_op;
  char * text;
}

%token TOK_AND TOK_OR TOK_EQUALITY TOK_SUCCEEDS 
       TOK_GET TOK_ENDQUERY TOK_PUT TOK_STAT
       TOK_LPAR TOK_RPAR TOK_QUITREQ

%token <text> TOK_ALNUMS
%token <text> TOK_QALNUMS
%token <text> TOK_INVALID

%left TOK_OR
%left TOK_AND
%left TOK_EQUALITY TOK_SUCCEEDS

%nonassoc LOWER_THAN_KEYVAL
%nonassoc USING_KEYVAL

%type <rel_op> relation
%type <expr> query
%type <expr> qexp
%type <keyname> keyname
%type <value> value
%type <rel_op> relation
%type <pexpr> insertion
%type <pexpr> pexp

%%

session : request | session request;

request : query {
  /*
   * Process the query and kill the expression
   */
  if(parser_mode != NS_IS_PMode_Interactive) {
    fprintf(stderr, "Got interactive query in non-interactive mode\n");
  } else {    
    dbms_process_query(parse_client, $1);
  }
  delete_expression($1);
  /* Account */
  t_lastget = time(0);
  n_gets ++;
} | insertion {
  /*
   * Process the insertion and kill the expression
   */
  if(parser_mode != NS_IS_PMode_Interactive) {
    /*
     * Process a non-interactive insertion 
     */
    dbms_process_insertion(-1, $1);
  } else {
    /*
     * Process an interactive insertion 
     */
    dbms_process_insertion(parse_client, $1);
  } 
  delete_insertion($1);
  /* Account */
  t_lastput = time(0);
  n_puts ++;
} | TOK_QUITREQ {
  /*
   * Quit connection
   */
  treat_quit(parse_client);
} | TOK_STAT {
  /*
   * Treat stats request
   */
  treat_stats(parse_client);
} | TOK_INVALID {
  char obuf[1024]; /* fixed length content  - stack is ok */
  sprintf(obuf, "Bad character: 0x%X \"%c\"\n", $1[0], isgraph($1[0]) ? ($1[0]) : '.');
  fprintf(stderr, "%s", obuf);
  if(parse_client != -1)
    push_to_client(parse_client, obuf);
};


query : TOK_GET qexp TOK_ENDQUERY
{
  $$ = $2;
};

insertion : TOK_PUT pexp TOK_ENDQUERY
{
  $$ = $2;
}

relation: TOK_EQUALITY {
  $$ = NS_IS_Equality;
} | TOK_SUCCEEDS {
  $$ = NS_IS_Succeeds;
};


qexp : keyname %prec LOWER_THAN_KEYVAL {
  if(!($$ = new_expression())) {
    if($1) free($1);
    fprintf(stderr, "Allocation error in qexp keyname parser\n");
  } else {
    $$->etype = NS_IS_Exp_KeyReq;
    $$->key = $1;
    $$->key_ndx = find_key_index($1);
  }  
} | keyname relation value %prec USING_KEYVAL  {
  if(!($$ = new_expression())) {
    if($1) free($1);
    if($3) free($3);
    fprintf(stderr, "Allocation error in qexp keyval parser\n");
  } else {
    $$->etype = NS_IS_Exp_KeyValue;
    $$->key = $1;
    $$->key_ndx = find_key_index($1);
    $$->rel_op = $2;
    $$->value = $3;
  }  
} | TOK_LPAR qexp TOK_RPAR {
  $$ = $2;
} | qexp TOK_AND qexp {
  if(!($$ = new_expression())) {
    delete_expression($1);
    delete_expression($3);
    fprintf(stderr, "Allocation error in qexp combination parser\n");
  } else {
    $$->etype = NS_IS_Exp_Combination;
    $$->lexp = $1;
    $$->comb_op = NS_IS_And;
    $$->rexp = $3;
  }  
} | qexp TOK_OR qexp {
  if(!($$ = new_expression())) {
    delete_expression($1);
    delete_expression($3);
    fprintf(stderr, "Allocation error in qexp combination parser\n");
  } else {
    $$->etype = NS_IS_Exp_Combination;
    $$->lexp = $1;
    $$->comb_op = NS_IS_Or;
    $$->rexp = $3;
  }  
};

pexp : keyname TOK_EQUALITY value
{
  if(!($$ = new_insertion())) {
    if($1) free($1);
    if($3) free($3);
    fprintf(stderr, "Allocation error in pexp head parser\n");
  } else {
    $$->key = $1;
    $$->value = $3;
    $$->next = 0;
  }
} | keyname TOK_EQUALITY value pexp
{
  if(!($$ = new_insertion())) {
    if($1) free($1);
    if($3) free($3);
    if($4) delete_insertion($4);
    fprintf(stderr, "Allocation error in pexp head parser\n");
  } else {
    $$->key = $1;
    $$->value = $3;
    $$->next = $4;
  }
};

keyname : TOK_ALNUMS {
  $$ = strdup($1);
};

value : TOK_QALNUMS {
  $$ = strdup($1);
};


