/*****************************************************************/
/*      iserv_dbms.c                                             */
/*      Jakob Oestergaard                                        */
/*---------------------------------------------------------------*/
/*  DBMS routines for the InfoServer information store           */
/*****************************************************************/

#include "infoserver.h"
#include "infoserver_int.h"

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>

/*
 * A record consists of:
 *   an array of values (either 0 or a valid value pointer for each index)
 *
 * The database is a chronologically ordered
 * list of such records
 *
 * We also keep a list of key names that maps into array indices, so
 * that we now at which positiion in the keyname/value array we should
 * look for that particular key.
 *
 *******************************************************
 * So, a database with two records could hold:
 *
 * key_list:     "HOSTNAME", "WORKLOAD", "TYPE"
 *
 * record_list:
 *  1:            "torc0", 0, "SERVER_REGISTER"
 *  2:            "torc1", "1.2", 0
 *
 * Meaning:
 *  1:            HOSTNAME="torc0" and TYPE="SERVER_REGISTER"
 *  2:            HOSTNAME="torc1" and WORKLOAD="1.2"
 *******************************************************
 *
 * We simply step thru all records when searching, returning the ones
 * that match completely.
 *
 * Thus, for a database with n records and a search including
 * p directives, with a total of s distinct keys in the database,
 * the complexity of a search will be  O(n*p + p*s).
 *
 * However, s and p will be "small" numbers, usually: p < s << n
 * (p is 1-5 maybe, s could be 10-50, n could well be around 10^5 -
 *  10^7)
 *
 * The database should be "vaccumed" once in a while, making sure
 * that limits on the number of records kept are held, for the
 * various types of records.
 *
 */

/*
 * Mandatory keys are:  TIME and URID
 */
#define NS_IS_MANDATORY_KEYS 2

NS_IS_Record * record_head;
NS_IS_Record * record_tail;

NS_IS_KeyNameIdx * keyname_index_head;
NS_IS_KeyNameIdx * keyname_index_tail;

/*
 * Statistics routines
 */
static unsigned n_records = 0;
static unsigned n_reckeys = 0;
static unsigned n_ukeys = 0;

unsigned dbms_stat_records(void)
{
  return n_records;
}

unsigned dbms_stat_ukeys(void)
{
  return n_ukeys;
}

double dbms_stat_keyrec(void)
{
  if(!n_records) return 0;
  return (double)n_reckeys/(double)n_records;
}

/*
 * Returns the index where a key named as given
 * should be inserted
 */
int find_key_index(const char * key)
{
  int index = 0;
  NS_IS_KeyNameIdx * iter;
  for(iter = keyname_index_head; iter; iter = iter->next)
    if(!strcmp(key, iter->keyname)) break;
    else index++;
  if(iter) 
    return index;
  return -1;
}

/*
 * Returns the key at the given index
 * XXX: This should be O(1) !!
 */
const char * find_index_key(unsigned index)
{
  NS_IS_KeyNameIdx * iter;
  for(iter = keyname_index_head; iter; iter = iter->next)
    if(iter->index == index)
      break;
  if(iter)
    return iter->keyname;
  else 
    return 0;
}

/*
 * Allocates a new key index
 */
int new_key_index(const char * key)
{
  NS_IS_KeyNameIdx * nidx;
  
  if(!(nidx = (NS_IS_KeyNameIdx*)malloc(sizeof(NS_IS_KeyNameIdx)))) {
    fprintf(stderr, "Out of memory when allocating new key index for \"%s\"\n",
	    key);
    return -1;
  }
  if(!(nidx->keyname = strdup(key))) {
    fprintf(stderr, "Out of memory when duplicating key name for key \"%s\"\n",
	    key);
    free(nidx);
    return -1;
  }
  nidx->index = n_ukeys++;
  /*
   * Hook to the end of the list, as new keys might be less likely to
   * be used often...  maybe...
   * This can be optimized by reordering keys in the key index list,
   * and in all records - I don't know if that would pay off. We could
   * gather statistics on the queries to make an informed decision to
   * re-order all of this.
   * It would be a lot of fun with pointers though ;)
   */
  nidx->next = 0;
  if(!keyname_index_head) {
    keyname_index_head = keyname_index_tail = nidx;
  } else {
    keyname_index_tail->next = nidx;
    keyname_index_tail = nidx;
  }
  return nidx->index;
}

/*
 * new_record():  Allocate new record
 */
static NS_IS_Record * new_record(const NS_IS_Insertion * keyvals)
{
  const NS_IS_Insertion * iter;
  NS_IS_Record * ret;
  unsigned nkeys, i;

  ret = (NS_IS_Record*)malloc(sizeof(NS_IS_Record));
  if(!ret) {
    fprintf(stderr, "Out of memory allocating new record\n");
    return 0;
  }
  /*
   * Allocate room for the keys  --  allocate extra space to the
   * mandatory events. Also zero the value pointer array.
   * keys to alloc :  mandatory keys + keys in record + keys in dbms
   */
  nkeys = NS_IS_MANDATORY_KEYS;
  for(iter = keyvals; iter; iter = iter->next)
    nkeys++;
  nkeys += n_ukeys;

  if(!(ret->value = (char**)calloc(nkeys,sizeof(char*)))) {
    fprintf(stderr, "Out of memory while allocating value array\n");
    free(ret);
    return 0;
  }
  ret->valuelen = nkeys;
  /*
   * Fill in values
   */
  for(i = 0, iter = keyvals; iter; iter = iter->next, i++) {
    int idx;
    if(0 > (idx = find_key_index(iter->key))) {
      /*
       * Key doesn't exist - create it
       */
      if(0 > (idx = new_key_index(iter->key))) {
	/*
	 * Key creation failed - let's ignore this key
	 *  - same reasoning as below with strdup()
	 */
	fprintf(stderr, "Couldn't allocate key for \"%s\" = \"%s\"\n",
		iter->key, iter->value);
	continue;
      }
    }
    assert(idx < ret->valuelen);
    /*
     * If string duplication fails, we print it out
     * but don't do more about it. This ok as it will
     * just look like that key never existed for this
     * record.  Queries requiring it will not return
     * this record then.
     */    
    if(!(ret->value[idx] = strdup(iter->value)))
      fprintf(stderr, "Couldn't allocate value string for \"%s\" = \"%s\"\n",
	      iter->key, iter->value);
  }
  /*
   * Return
   */
  return ret;
}

/*
 * Record deletion
 */
void delete_record(NS_IS_Record* rec)
{
  if(rec->value && rec->valuelen)
    free(rec->value);
  rec->value = 0;
  rec->valuelen = 0;
  rec->prev = 0;
  rec->next = 0;
  free(rec);
}

/*
 * insert_record():  This one chains a record into the
 * main record list  -   it will update the accounting
 * (statistics) information, and some day it should also
 * perform vacuuming of the list.
 */
void insert_record(NS_IS_Record* rec)
{
  unsigned i;
  /*
   * Chain it in
   */
  rec->prev = record_tail;
  rec->next = 0;
  if(record_tail)
    record_tail->next = rec;
  if(!record_head)
    record_head = rec;
  record_tail = rec;

  /*
   * Account
   */
  n_records ++;  

  for(i = 0; i != rec->valuelen; i++)
    if(rec->value[i])
      n_reckeys++;
}


/*
 * Allocation of mandatory keys
 */
int dbms_initialize(void)
{
  /*
   * Allocate mandatory keys
   */
  if(0 > new_key_index("TIME")
     || 0 > new_key_index("URID"))
    return -1;
  /*
   * Randomize for the URID generator
   */
  srand(time(0));
  /*
   * Ok
   */
  return 0;
}

/*
 * Basic output routines for returning query results
 * to clients
 */
static void return_end_results(int fd)
{
  push_to_client(fd, "END\n");
}

static void return_end_failure(int fd, const char* txt)
{
  static char * sbuf = 0;
  if(!sbuf) {
    sbuf = (char*)malloc(NS_IS_MAX_REQLEN);
    if(!sbuf) {
      fprintf(stderr, "Out of memory when allocating ref string buffer\n");
      push_to_client(fd, "Out of memory in ref string buffer allocation\n");
      return;
    }
  }
  sprintf(sbuf, "ERROR (%s)\n", txt);
  push_to_client(fd, sbuf);
}

int urid_exists(unsigned urid)
{
  NS_IS_Record * rec;
  int urid_key_ndx = find_key_index("URID");
  if(urid_key_ndx < 0) {
    fprintf(stderr, "Sorry, urid_exists() could't resolve URID key\n");
    return 0;
  }
  for(rec = record_head; rec; rec = rec->next)
    if(atol(rec->value[urid_key_ndx]) == urid)
      break;
  return !!rec;
}


/*
 * Insertion engine
 */
void dbms_process_insertion(int fd, const NS_IS_Insertion* e)
{
  NS_IS_Record * rec;
  int ndx;
  char buf[128];  /* this buffer is safe - only used for int's */
  unsigned urid;

  /*
   * Construct the new record
   */
  if(!(rec = new_record(e))) {
    return_end_failure(fd, "Insertion error");
    return;
  }

  /*
   * If we're interactive, fill in mandatory records
   */
  if(fd != -1) {
    ndx = find_key_index("TIME");
    assert(ndx >= 0);  /* if dbms is initialized, TIME will exist */
    sprintf(buf, "%i", (int)time(0));
    rec->value[ndx] = strdup(buf);

    ndx = find_key_index("URID");
    assert(ndx >= 0);  /* if dbms is initialized, URID will exist */
    /*
     * Find a unique id for this record. The amortized complexity
     * for n records should be  O(n * (1 + n/(2^32))),  so for
     * n << 2^32 we have roughly O(n)  (but going towards O(n^2))
     */
    while(urid_exists(urid = rand()));
    sprintf(buf, "%u", urid);
    rec->value[ndx] = strdup(buf);  
  }

  /*
   * Verify that subject is allowed to import this object into
   * the database 
   */
  if(!refmon_grant(NS_IS_RM_Import, rec, 0)) {
    fprintf(stderr, "Object import refused - ditching record\n");
    delete_record(rec);
    return;
  }

  /*
   * Chain it into the database
   */
  insert_record(rec);

  /*
   * If we're interactive, go write it to persistent store
   */
  if(fd != -1)
    iserv_putrecord(rec);
}


/*
 * The match routine for records
 */
static int record_matches(const NS_IS_Expression * e, const NS_IS_Record * rec, char * evalmap)
{
  switch(e->etype) {
  case NS_IS_Exp_KeyValue: { /* Must match both key and value */
    /*
     * Check key
     */
    if(e->key_ndx < 0)
      return 0;
    if(e->key_ndx >= rec->valuelen)
      return 0;
    if(!rec->value[e->key_ndx])
      return 0;
    /*
     * Ok, now see if value matches (use relop)
     */
    switch(e->rel_op) {
    case NS_IS_Equality:
      /*
       * Equality is a simple strcmp(),  so if a = b, then strcmp(a,b)==0
       */
      return (evalmap[e->key_ndx] |= !strcmp(e->value, rec->value[e->key_ndx]));
    case NS_IS_Succeeds: {
      /*
       * Succeeds is defined as,   r_p > r_q  iff  !strcmp(r_(p-s),r_q) for some s>0
       *
       * First:  Find the record that has value == e->value
       * Then:   see if this record succeeds that one chronologically
       *
       * Obviously, this query makes the most sense on record IDs (URIDs)
       */      
      NS_IS_Record * iter;
      for(iter = record_tail; iter; iter = iter->prev)
	if(iter->value[e->key_ndx] && !strcmp(iter->value[e->key_ndx], e->value))
	  break;
      /*
       * If we can't find equality (eg. the expression doesn't exist in the database)
       * the expression has no successors in the database either.
       */
      if(!iter)
	return 0;
      /*
       * Now let's see if rec succeeds iter
       */
      for(iter = iter->next; iter && iter != rec; iter = iter->next);
      return (evalmap[e->key_ndx] |= !!iter);
    }
    }
    return 0;
  }
  case NS_IS_Exp_KeyReq: { /* Must match key */
    /*
     * Check key
     */
    if(e->key_ndx < 0)
      return 0;
    if(e->key_ndx >= rec->valuelen)
      return 0;
    return (evalmap[e->key_ndx] |= !!rec->value[e->key_ndx]);
  }
  case NS_IS_Exp_Combination: { /* Combinations */
    switch(e->comb_op) {
    case NS_IS_And: {
      if(record_matches(e->lexp, rec, evalmap)) 
	return record_matches(e->rexp, rec, evalmap);
      else return 0;
    }
    case NS_IS_Or: {
      int a;
      char * temap = (char*)malloc(n_ukeys);
      if(!temap) {
	fprintf(stderr, "Out of memory error when allocating temporary evalmap\n");
	return 0;
      }
      memcpy(temap, evalmap, n_ukeys);
      a = record_matches(e->lexp, rec, temap);
      if(a) {
	memcpy(evalmap, temap, n_ukeys);
	free(temap);
	return 1;
      } else {
	free(temap);
	return record_matches(e->rexp, rec, evalmap);
      }
    }
    default:
      fprintf(stderr, "match engine got bad combination (%i)\n",
	      e->comb_op);
      return 0;
    }
  }
  default:
    fprintf(stderr, "match engine got bad exression type (%i)\n", 
	    e->etype);
    return 0;
  }
  return 0;
}

/*
 * Query engine
 */
void dbms_process_query(int fd, const NS_IS_Expression* e)
{
  static char * sbuf = 0;
  static char * rbuf = 0; 
  static unsigned rbuflen = 0;
  NS_IS_Record * rec;
  /*
   * Allocate static reply buffer
   */
  if(!sbuf) {
    sbuf = (char*)malloc(2 * NS_IS_MAX_REQLEN);
    if(!sbuf) {
      fprintf(stderr, "Out of memory when allocating rse string buffer\n");
      return_end_failure(fd, "Out of memory in rse string buffer allocation");
      return;
    }
  }
  /*
   * Allocate static return-buffer
   */
  if(rbuflen < n_ukeys) {
    if(rbuf) free(rbuf);
    rbuf = (char*)malloc( n_ukeys );
    if(!rbuf) {
      rbuflen = 0;
      fprintf(stderr, "Out of memory when allocating record return buffer\n");
      return_end_failure(fd, "Out of memory in record return buffer allocation");
      return;
    }
    rbuflen = n_ukeys;
  }

  /*
   * Seek thru all records, returning those that match
   */
  for(rec = record_head; rec; rec = rec->next) {
    unsigned i;
    int slen;

    /*
     * All requirements must be met.
     */
    memset(rbuf, 0, rbuflen);
    if(!record_matches(e, rec, rbuf))
      continue;

    /*
     * Verify that subject is allowed access to the object via. given
     * channel
     */
    if(!refmon_grant(NS_IS_RM_Export, rec, 0))
      continue;

    /*
     * Print it
     */
    slen = 0;
    for(i = 0; i != rbuflen; i++)
      if(rbuf[i]) {
	if(slen + strlen(rec->value[i]) < NS_IS_MAX_REQLEN)
	  slen += sprintf(sbuf + slen, "%s=\"%s\" ", 
			  find_index_key(i), rec->value[i]);
	else
	  fprintf(stderr, "Sorry, buffer overflow avoided, skipping keys...\n");
      }

    /*
     * End with newline instead of space
     */
    if(slen)
      sbuf[slen-1] = '\n';

    /*
     * And out it goes
     */
    push_to_client(fd, sbuf); 
  }

  return_end_results(fd);
}

