/*****************************************************************/
/*      socketutil.c                                             */
/*      Henri Casanova                                           */
/*---------------------------------------------------------------*/
/*****************************************************************/

#include <stdio.h>
#include <sys/types.h>
#include <errno.h>

#ifndef WIN32
#include <strings.h>
#include <sys/ioctl.h>
#if (defined(solaris))
#include <sys/filio.h>
#endif
#include <sys/socket.h>
#include <sys/un.h>
#include <netinet/in.h>
#include <netdb.h>
#include <unistd.h>
#endif /* UNIX */

#ifdef WIN32
#include "win_util.h"
#endif /* WIN32 */

#include "core.h"

#define HOSTNAME_LENGTH 256

/* Establish Socket is only for UNIX */
#ifndef WIN32

/*
 *  Creation of an INET STREAM socket given :
 *     - the desired port number
 *
 * It is *VERY* important that reuse is true *ONLY*
 * where we know that we at all times will have one
 * and only one process trying to bind on that
 * given port.  
 * For example:  agent and server can have reuse=1
 *               but proxy must *never* ever have reuse=1
 *               as that will cause all sorts of funny
 *               races to occur...
 *
 *  return >0 is success
 */

int establishSocket(int *port, int reuse)
{
  int desc;                       
  struct sockaddr_in addr;        
  int addr_length;               
  char hostname[HOSTNAME_LENGTH];
  struct hostent *hp;
  int option_value = 1;    

  if((desc =socket(AF_INET,SOCK_STREAM,0)) == -1)
  {
    ns_errno = NetSolveSystemError;
    return -1;
  }
  if(reuse) {
    if (setsockopt(desc,SOL_SOCKET,SO_REUSEADDR,
                   (char *)(&option_value),sizeof(int)) == -1)
      {
        ns_errno = NetSolveSystemError;
        return -1;
      }
  }
  BZERO((char *)&addr,sizeof(addr));
  addr.sin_port = htons(*port);
  if (gethostname(hostname,HOSTNAME_LENGTH) == -1)
  {
    ns_errno = NetSolveSystemError;
    close(desc);
    return -1;
  }
  hp = gethostbyname(hostname);
  if (hp == NULL)
  {
    ns_errno = NetSolveSystemError;
    close(desc);
    return -1;
  }
  BCOPY((void*)hp->h_addr_list[0],(void*)&addr.sin_addr,hp->h_length);
  addr.sin_family = AF_INET;
  if(bind(desc,(struct sockaddr *)&addr,sizeof(addr)))
  {
    if (errno == EADDRINUSE)
    {
      ns_errno = NetSolveCannotBind;
      close(desc);
      return -2;
    }
    ns_errno = NetSolveSystemError;
    close(desc);
    return -1;
  }
  addr_length = sizeof(addr);
  if ((getsockname(desc,(struct sockaddr *)&addr,&addr_length)) == -1)
  {
    ns_errno = NetSolveSystemError;
    close(desc);
    return -1;
  }
  *port = ntohs(addr.sin_port);
  return(desc);
}

#endif /* UNIX */

/*
 * connectToSocket()
 */

/* UNIX Version */
#ifndef WIN32
NS_Socket_type connectToSocket(char *hostname,NS_IPaddr_type IPaddr,int port)
{
  NS_Socket_type sock;
  struct sockaddr_in inet_addr;
  struct hostent *hp;
  NS_IPaddr_type tmpIPaddr;
  static NS_IPaddr_type myIPaddr=(NS_IPaddr_type)0;
  char *hn = 0;

  tmpIPaddr = IPaddr;

  if ((sock = socket(AF_INET,SOCK_STREAM,0)) == -1)
  {
    ns_errno = NetSolveSystemError;
    return -1;
  }
  BZERO((void*)&inet_addr,sizeof(inet_addr));

  /* if NULL,0 : this host, otherwise distant host */
  if ((hostname == NULL)&&(IPaddr == 0))
  {
    if (myIPaddr == (NS_IPaddr_type)0)
      myIPaddr = getMyIPaddr();
    tmpIPaddr = myIPaddr;
  }
  else if (IPaddr != 0)
  {
    tmpIPaddr = IPaddr;
  }
  else if (hostname != NULL)
  {
    hn = hostname;
  }

  /* Try to do a gethostbyaddr() first() */
  if (tmpIPaddr != (NS_IPaddr_type)0)
    hp = gethostbyaddr((char *)&tmpIPaddr,sizeof(NS_IPaddr_type),AF_INET); 
  else 
    hp = NULL;

  if (hp == NULL)
  {
    /* Try to do a gethostbyname() */
    if (hn != NULL)
      hp = gethostbyname(hn);
    else
      hp = NULL;
    if (hp == NULL)
    {
      close(sock);
      ns_errno = NetSolveUnknownHost;
      return -1;
    }
  }
  BCOPY((void*)hp->h_addr,(void*)&inet_addr.sin_addr,hp->h_length);
  inet_addr.sin_family = AF_INET;
  inet_addr.sin_port = htons(port);
  
#ifdef DEBUG
  if (hn != NULL)
    fprintf(STDERR"Connecting to %s ...",hn);
#endif
  if (connect(sock,(struct sockaddr *)&inet_addr,sizeof(inet_addr)) == -1)
  {
#ifdef DEBUG
    fprintf(STDERR"failure\n",hn);
#endif
    close(sock);
    ns_errno = NetSolveConnectionRefused;
    return -1;
  }
#ifdef DEBUG
  fprintf(STDERR"success\n",hn);
#endif
  return sock;
}
#endif /* UNIX */

/* Win32 version */
#ifdef WIN32
NS_Socket_type connectToSocket(char *hostname,NS_IPaddr_type IPaddr,int port)
{
  NS_Socket_type sock;
  FAR struct sockaddr_in inet_addr;
  LPHOSTENT hp = NULL;
  NS_IPaddr_type tmpIPaddr;
  char *hn;
  static char *thishost = NULL;

  tmpIPaddr = IPaddr;

  /* Allocate a socket */
  if ((sock = socket(AF_INET,SOCK_STREAM,0)) == INVALID_SOCKET) 
  {
    fprintf(STDERR "socket error: %d\n",WSAGetLastError());
    ns_errno = NetSolveSystemError;
    return SOCKET_ERROR;
  }

  /* If hostname == NULL, this host */
  if (hostname == NULL)
  {
    if (thishost == NULL)
    {
      char buf[256];
      if (gethostname(buf,256) == -1)
      {
        perror("gethostname()");
        ns_errno = NetSolveSystemError;
        return -1;
      }
      else
        thishost = strdup(buf);
    }
    hn = thishost;
  }
  else
    hn = hostname;


  BZERO((void*)&inet_addr,sizeof(inet_addr));

  /* Try to build a IP addr if non is supplied */
  if (tmpIPaddr == (NS_IPaddr_type)0)
    tmpIPaddr = getIPaddr(hostname);

  /* Try to do a gethostbyaddr() */
  if (tmpIPaddr)
    hp = gethostbyaddr((char *)&tmpIPaddr,sizeof(NS_IPaddr_type),AF_INET);
  else
    hp = NULL;
  if (hp == NULL) 
  {
    /* Try to do a gethostbyname() */
    if (hostname != NULL)
      hp = gethostbyname(hostname);
    else
      hp = NULL;
    if (hp == NULL)   
    {
      closesocket(sock);
      ns_errno = NetSolveUnknownHost;
      return SOCKET_ERROR;
    }
  }
  BCOPY((void*)hp->h_addr,(void*)&(inet_addr.sin_addr),hp->h_length);
  inet_addr.sin_family = AF_INET;
  inet_addr.sin_port = htons((short)port);

#ifdef DEBUG
  if (hostname != NULL)
    fprintf(STDERR"Connecting to %s ...\n",hostname);
#endif
  if (connect(sock,(struct sockaddr *)&inet_addr,sizeof(inet_addr)) 
                 == SOCKET_ERROR) 
  {
    closesocket(sock);
    ns_errno = NetSolveConnectionRefused;
    return SOCKET_ERROR;
  }
  else
    return sock;
}

#endif /* WIN32 */

/*
 * isThereSomethingOnTheSocket()
 *
 *  returns :   0 if connection closed by peer
 *              1 if something on the socket
 *             -1 if error and sets the errno
 *
 *  mm - rename function to be more clear about its purpose: isConnectAlive()
 *  This function is only called in startLifeLink() which only tests for 0.
 */
int isThereSomethingOnTheSocket(NS_Socket_type sock)
{
  int off =0;
  int on  =1;
  char buf[1];
  int try;

#ifndef WIN32
  ioctl(sock,FIONBIO,&on);
  try = recv(sock,buf,1,MSG_PEEK);
  ioctl(sock,FIONBIO,&off);
#endif /* UNIX */
#ifdef WIN32
  ioctlsocket(sock,FIONBIO,&on);
  try = recv(sock,buf,1,MSG_PEEK);
  ioctlsocket(sock,FIONBIO,&off);
#endif /* WIN32 */

/* mm notes - found bug.  recv() above is failing with a return of EWOULDBLOCK
 * because the socket is non-blocking, but this recv would block since there
 * is nothing waiting to be read from the socket.  This is a polling interface
 * which will never received data.  It only checks for endpoint exit.
 *
 * recv() returns 0 if no messages are available and the peer has closed the
 * connections.  recv() returns -1 and sets errno if error.  Upon success,
 * recv() returns number of bytes received.
 */

  if (isSocketError(try) && (errno != EWOULDBLOCK) && (errno != EAGAIN))
  {
    ns_errno = NetSolveNetworkError;
    return -1;
  }

  if (try == 0)
  {
    ns_errno = NetSolveOK;
    return 0;
  }
  ns_errno = NetSolveOK;
  return 1;
}

/*
 * getIPaddr()
 */
NS_IPaddr_type getIPaddr(char *hostname)
{
  unsigned int u1,u2,u3,u4;
  NS_IPaddr_type IPaddr;

  if (hostname == NULL)
    return (NS_IPaddr_type)0;

  if (sscanf(hostname,"%u.%u.%u.%u",&u1,&u2,&u3,&u4) != 4)
  {
    return (NS_IPaddr_type)0;
  }
  ((unsigned char *)&IPaddr)[0] = (unsigned char)u1;
  ((unsigned char *)&IPaddr)[1] = (unsigned char)u2;
  ((unsigned char *)&IPaddr)[2] = (unsigned char)u3;
  ((unsigned char *)&IPaddr)[3] = (unsigned char)u4;
  return IPaddr;
}


/*
 * closeSocket()
 */
int closeSocket(NS_Socket_type s)
{
#ifdef WIN32
  return closesocket(s);
#endif
#ifndef WIN32
  return close(s);
#endif
}

/*
 * isSocketError()
 */
int isSocketError(NS_Socket_type s)
{
#ifdef WIN32
  return (s == SOCKET_ERROR);
#endif
#ifndef WIN32
  return (s == -1);
#endif
}


/*
 * contactHost()
 *
 * This is a high level wrapper
 */
NS_Communicator *contactHost(char *hostname, NS_IPaddr_type IPaddr, int port,
            int encoding)
{
  NS_Socket_type sock;
  NS_Communicator *comm;

  sock = connectToSocket(hostname,IPaddr,port);
  if (isSocketError(sock))
    return NULL;

  comm = initTransaction(sock,encoding);
  if (comm == NULL)
  {
    closeSocket(sock);
    return NULL;
  }
  return comm;
}

/*
 * acceptConnection()
 */
NS_Socket_type acceptConnection(NS_Socket_type listening_socket)
{
  struct sockaddr_in addr;      /* INET socket address */
  int addrlen;     /* address length      */
  NS_Socket_type sock;

  addrlen = sizeof(addr);
  sock = accept(listening_socket,
              (struct sockaddr *)&addr,&addrlen);
  while(sock < 0 && errno == EINTR){

    sock = accept(listening_socket,
              (struct sockaddr *)&addr,&addrlen);
  }
  return sock;
}

/*
 * bindToFirstAvailablePort()
 */
int bindToFirstAvailablePort(int initport, int *found_port)
{
  int port;
  int tryport;
  int sock;

  port = initport;
  tryport = port;
  /*
   * It's important that we don't reuse the addresses here,
   * as we will just attempt the next port if the current
   * one fails - no problem...  This call is typically used
   * where the exact port doesn't matter much, but where
   * we have multiple processes on the local host concurrently
   * trying this call with the same starting port number.
   * The proxy does this, and setting reuse=1 in the
   * establishSocket() call will spawn an unholy number of
   * proxy processes.
   */
  while((sock=establishSocket(&tryport,0)) == -2)
  {
    port++;
    tryport = port;
    if (port > initport + 200)
    {
#ifdef VIEW
      fprintf(STDERR "Impossible to bind to a port\n");
#endif
      return -1;
    }
  }
  if (isSocketError(sock))
    return -1;
  *found_port = port;
  return sock;
}
