/*****************************************************************/
/*      restrictiondesc.c                                        */
/*      Henri Casanova                                           */
/*---------------------------------------------------------------*/
/*****************************************************************/

#include <sys/types.h>

#ifndef WIN32
#include <sys/socket.h>
#include <sys/un.h>
#include <netinet/in.h>
#include <netdb.h>
#endif
#include "core.h"

/*
 * freeRestrictionDesc()
 */
void freeRestrictionDesc(NS_RestrictionDesc *d)
{
  if (d == NULL)
    return;
  if (d->domain != NULL)
    free(d->domain);
  free(d);
}

/*
 * newRestrictionDesc()
 */
NS_RestrictionDesc *newRestrictionDesc()
{
  return (NS_RestrictionDesc *)calloc(1,sizeof(NS_RestrictionDesc));
}

/*
 * sendRestrictionDesc()
 *
 * Sends a workload History on the network
 */
int sendRestrictionDesc(NS_Communicator *comm, NS_RestrictionDesc* d)
{
  if (sendString(comm,d->domain) == -1)
    return -1;
  if (sendInt(comm,d->max_pending) == -1)
    return -1;
  if (sendInt(comm,d->pending) == -1)
    return -1;
  return 1;
}

/*
 * recvRestrictionDesc()
 *
 * receives a workload history from a communicator
 */
NS_RestrictionDesc *recvRestrictionDesc(NS_Communicator *comm)
{
  NS_RestrictionDesc *new;
  new = newRestrictionDesc();
  if (recvString(comm,&(new->domain)) == -1)
  {
    free(new);
    return NULL;
  }
  if (recvInt(comm,&(new->max_pending)) == -1)
  {
    free(new);
    return NULL;
  }
  if (recvInt(comm,&(new->pending)) == -1)
  {
    free(new);
    return NULL;
  }
  return new;
}

/*
 * checkRestriction()
 *
 * Not only checks, but also increments.
 */
int checkRestriction(NS_Communicator *comm, NS_RestrictionDesc **restrictions,
                     int nb_restrictions)
{
  char *peername;
  NS_RestrictionDesc *current;
  NS_RestrictionDesc *most_restrictive;
  int index=-1;
  int i;


  peername = getPeerName(comm);
  if (peername == NULL)
  {
    free(peername);
    return -1;
  }

  most_restrictive = NULL;

  for (i=0;i<nb_restrictions;i++)
  {
    current = restrictions[i];
    if (!isInDomain(current->domain,peername))
      continue; 
    if (most_restrictive == NULL)
    {
      most_restrictive = current;
      index = i;
    }
    else if (strlen(most_restrictive->domain) < strlen(current->domain))
    {
      most_restrictive = current;
      index = i;
    }
  }
  free(peername);
  if (most_restrictive == NULL)
  {
    ns_errno = NetSolveOK;
    return -1;
  } 
  if (most_restrictive->pending < most_restrictive->max_pending)
  {
    (most_restrictive->pending)++;
    return (index);
  }
  else
  {
#ifdef VIEW
    fprintf(STDERR "Too many pending requests already (%d)\n",
            most_restrictive->pending);
#endif
    ns_errno = NetSolveNotAllowed;
    return -1;
  }
}

/*
 * isInDomain()
 */
int isInDomain(char *domain, char *hostname)
{
  char *lastdot1;
  char *lastdot2;
  char *domaintmp,*hostnametmp;

  if (strcmp(domain,"*") == 0)
    return 1;

  domaintmp = (char *)strdup(domain);
  hostnametmp = (char *)strdup(hostname);

  while(1)
  {
    lastdot1 = (char *)strrchr(domaintmp,'.');
    lastdot2 = (char *)strrchr(hostnametmp,'.');

    if (lastdot1 == NULL)
    {
      if ((strcmp(domaintmp,"*") == 0)||
          (strcmp(domaintmp,hostnametmp) == 0))
      {
        free(domaintmp); free(hostnametmp);
        return 1;
      }
      else
      {
        free(domaintmp); free(hostnametmp);
        return 0;
      }
    }
    if (lastdot2 == NULL)
    {
      free(domaintmp); free(hostnametmp);
      return 0;
    }

    if((strcmp(lastdot1,".*") == 0)||
       (strcmp(lastdot1,lastdot2) == 0))
    {
      lastdot1[0] = '\0';
      lastdot2[0] = '\0';
    }
    else
    {
      free(domaintmp); free(hostnametmp);
      return 0;
    }
  }
  return 0;
}

/*
 * getPeerName()
 */
char *getPeerName(NS_Communicator *comm)
{
  struct sockaddr_in name;
  int namelen;
  NS_IPaddr_type IPaddr=0;
  struct hostent *hp=NULL;
  char *peername;

  namelen = sizeof(name);
  if (getpeername(comm->sock,(struct sockaddr *)&name,&namelen) != 0)
  {
#ifdef VIEW
    fprintf(STDERR "Unknown peer!!\n");
#endif
    ns_errno = NetSolveSystemError;
    return NULL;
  }
  BCOPY((const void*)(&(name.sin_addr)),(void*)&IPaddr,sizeof(IPaddr));
  hp = gethostbyaddr((char *)&IPaddr,sizeof(IPaddr),AF_INET);
  if (hp == NULL)
  {
#ifdef VIEW
    fprintf(STDERR "Unresolved peer!!\n");
#endif
    ns_errno = NetSolveSystemError;
    return NULL;
  }
  peername = strdup(hp->h_name);
  netsolveCaps(peername);
  return peername;
}
