/*****************************************************************/
/*      problemdesc.c                                            */
/*      Henri Casanova, Dorian Arnold                            */
/*---------------------------------------------------------------*/
/*****************************************************************/

#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#ifndef WIN32
#include <unistd.h>
#endif

#include "core.h"

/*
 * newObject()
 *
 * This function allocate the memory for an object. 
 * It only performs the calloc of the data structure
 * without doing anyother allocation. It's role is really
 * to make the source code shorter and easier to read at 
 * above levels.
 */
NS_Object* newObject()
{
NS_Object *tmp_object;
  tmp_object = (NS_Object *)calloc(1,sizeof(NS_Object));
  tmp_object->pass_back = TRUE;
  return tmp_object;
}

/*
 * newProblemDesc()
 *
 * This function allocate the memory for a problem descriptor.
 * It only performs the calloc of the data structure
 * without doing anyother allocation. It's role is really
 * to make the source code shorter and easier to read at 
 * above levels.
 */
NS_ProblemDesc* newProblemDesc()
{
  NS_ProblemDesc *tmp_pd;
  tmp_pd = (NS_ProblemDesc *)calloc(1,sizeof(NS_ProblemDesc));
  tmp_pd->input_names = NULL;
  tmp_pd->output_names = NULL;
  return tmp_pd;
}

/*
 * freeProblemDesc()
 *
 * This routine frees the memory for a problem descriptor.
 * Unlike newProblemDesc(), it goes inside the structure
 * to free sub-structures, etc....
 */
void freeProblemDesc(void *d)
{
  NS_ProblemDesc *pd=(NS_ProblemDesc *)d;
  int i;

  if (pd == NULL)
    return;
  if (pd->name != NULL)
    free(pd->name);
  if (pd->nickname != NULL)
    free(pd->nickname);
  if (pd->description != NULL)
    free(pd->description);
  if (pd->file != NULL)
    free(pd->file);
  if (pd->customized != NULL)
    free(pd->customized);
  if (pd->matlab_merge != NULL)
    free(pd->matlab_merge);
  if (pd->complexity != NULL)
    free(pd->complexity);
  for (i=0;i<pd->nb_input_objects;i++)
    freeObject(pd->input_objects[i]);
  free(pd->input_objects);
  for (i=0;i<pd->nb_output_objects;i++)
    freeObject(pd->output_objects[i]);
  free(pd->output_objects);

  if (pd->input_names != NULL)
      for (i=0;i<pd->nb_input_objects;i++)
          free (pd->input_names[i]);

  if (pd->output_names != NULL)
      for (i=0;i<pd->nb_output_objects;i++)
          free (pd->output_names[i]);
   
  freeCallingSequence(pd->calling_sequence);
  free(pd);
}

/*
 * freeHandle()
 *
 */
void freeHandle(void *d)
{
#if defined(DSI_IBP)
DSI_OBJECT* dsi_obj = (DSI_OBJECT*)d;
  if(dsi_obj->dsi_file->storage_system == IBP)
    free(dsi_obj->dsi_file->dsi_specifics.ibp.cap);
  free(dsi_obj);
#endif
}

/*
 * compHandleName()
 *
 */
int compHandleName(void *key,void *d)
{
  char *name;

  if (key == NULL)
    return 1;

  name = ((DSI_OBJECT *)d)->name;

  if (!strcmp((char *)key,name))
    return 1;

  return 0;
}

/*
 * compProblemDescKey()
 *
 * This function returns 1 (true) is the problem descriptor
 * at address d has a nickname equal to the string at address key.
 * This routine is needed by our linked list for the search and
 * remove operations. It allows us to have a somewhat "polymorphic"
 * linked list in C.
 */
int compProblemDescKey(void *key,void *d)
{
  char *nickname;
  
  if (key == NULL)
    return 1;

  nickname = ((NS_ProblemDesc *)d)->nickname;

  if (!strcasecmp((char *)key,nickname))
    return 1;

  return 0;
}

/*
 * freeObject()
 *
 * This function frees the memory for an object. It goes inside
 * the structure to free all sub-structures.
 * WARNING: It does not free any actual DATA, but only
 * NetSolve specific structures.
 */
void freeObject(NS_Object *obj)
{
  if (obj == NULL)
    return;

  if (obj->name != NULL)
    free(obj->name);

  if (obj->description != NULL)
    free(obj->description);

  switch(obj->object_type)
  {
  case NETSOLVE_MATRIX:
    if (obj->attributes.matrix_attributes.d != NULL)
      free(obj->attributes.matrix_attributes.d);
    break;
  case NETSOLVE_SPARSEMATRIX:
    if (obj->attributes.sparsematrix_attributes.d != NULL)
      free(obj->attributes.sparsematrix_attributes.d);
    break;
  case NETSOLVE_VECTOR:
    if (obj->attributes.vector_attributes.d != NULL)
      free(obj->attributes.vector_attributes.d);
    break;
  case NETSOLVE_SCALAR:
    break;
  case NETSOLVE_FILE:
    if (obj->attributes.file_attributes.filename != NULL)
      free(obj->attributes.file_attributes.filename);
    break;
  case NETSOLVE_PACKEDFILES:
    if (obj->attributes.packedfiles_attributes.defaultprefix != NULL)
      free(obj->attributes.packedfiles_attributes.defaultprefix);
    else
    {
      if (obj->attributes.packedfiles_attributes.filenames != NULL)
      {
        int i;
        for (i=0;i<obj->attributes.packedfiles_attributes.m;i++)
        {
          if (obj->attributes.packedfiles_attributes.filenames[i] != NULL)
            free(obj->attributes.packedfiles_attributes.filenames[i]);
        }
        free(obj->attributes.packedfiles_attributes.filenames);
      }
    }
    break;
  case NETSOLVE_STRINGLIST: /* Nothing */
    break;
  case NETSOLVE_UPF:
    if (obj->attributes.upf_attributes.filename != NULL)
      free(obj->attributes.upf_attributes.filename);
    if (obj->attributes.upf_attributes.funcname != NULL)
      free(obj->attributes.upf_attributes.funcname);
    break;
  case NETSOLVE_STRING: /* Nothing */
    break;
  }
  free(obj);
}

/*
 * writeProblemDescToFile()
 *
 * This function writes a problem description to a file in
 * a certain format (binary). The problem descriptor can
 * then be read by readProblemDescFromFile(). 
 * It is written on top of sendProblemDesc() which would
 * normally be used for networking. That's why we create
 * a Communicator object with a UNIX file descriptor
 * that is really a file and not a socket descriptor.
 */ 
int writeProblemDescToFile(int fd, NS_ProblemDesc *pd)
{
  NS_Communicator *comm;
  int cc;
  
  comm = newCommunicator(fd,DATA_RAW);
  cc = sendProblemDesc(comm,pd);
  free(comm);
  if (cc == -1)
  {
    ns_errno = NetSolveFileError;
    return -1;
  }
  return 1;
}

/*
 * readProblemDescFromFile()
 *
 * This function reqds a problem descriptor into a new
 * ProblemDesc structure, from a file. That file must have been
 * written by writeProblemDescToFile(). As for writeProblemDescToFile(),
 * readProblemDescFromFile() is written on top of recvProblemDesc()
 * that would normally be used for networking. That's why
 * a Communicator object with a UNIX file descriptor
 * that is really a file and not a socket descriptor.
 */
NS_ProblemDesc *readProblemDescFromFile(int fd)
{
  NS_ProblemDesc *new = NULL;
  NS_Communicator *comm;
 
  comm = newCommunicator(fd,DATA_RAW);
  new = recvProblemDesc(comm);
  free(comm); 
  return new;
}

/*
 * sendProblemDesc()
 *
 * This function sends a problem descriptor over the network. 
 * WARNING: It send the list of Object in input and output,
 * but does not send the objects' attributes, only
 * their specification. sendObject() must be called to send
 * and object along with it's attributes (i.e. data)
 * 
 */
int sendProblemDesc(NS_Communicator *comm, NS_ProblemDesc *pd)
{
  int i;
  int customized_names;

  if (sendString(comm,pd->name) == -1)
    return -1;
  if (sendString(comm,pd->nickname) == -1)
      return -1;
  if (sendString(comm,pd->description) == -1)
    return -1;
  if (sendString(comm,pd->file) == -1)
    return -1;
  
  if (sendInt(comm,pd->major) == -1)
    return -1;

  if (sendString(comm,pd->customized) == -1)
    return -1;
  if (sendInt(comm,pd->nb_matlab_merge) == -1)
    return -1;

  for (i=0;i<pd->nb_matlab_merge;i++)
  {
    if (sendInt(comm,pd->matlab_merge[i].index1) == -1)
      return -1;
    if (sendInt(comm,pd->matlab_merge[i].index2) == -1)
      return -1;
  }
  if (sendString(comm,pd->complexity) == -1)
    return -1;
  
  /* sending the calling sequence */
  if (sendCallingSequence(comm,pd->calling_sequence) == -1)
    return -1;

  if (sendInt(comm,pd->nb_input_objects) == -1)
    return -1;
  for (i=0;i<pd->nb_input_objects;i++)
  {
    if (sendInt(comm,pd->input_objects[i]->object_type) == -1)
      return -1;
    if (sendInt(comm,pd->input_objects[i]->data_type) == -1)
      return -1;
    if (sendInt(comm,pd->input_objects[i]->pass_back) == -1)
      return -1;
    if (sendString(comm,pd->input_objects[i]->description) == -1)
      return -1;
    if (sendString(comm,pd->input_objects[i]->name) == -1)
          return -1;
  }
  if (sendInt(comm,pd->nb_output_objects) == -1)
    return -1;
  for (i=0;i<pd->nb_output_objects;i++)
  {
    if (sendInt(comm,pd->output_objects[i]->object_type) == -1)
      return -1;
    if (sendInt(comm,pd->output_objects[i]->data_type) == -1)
      return -1;
    if (sendInt(comm,pd->output_objects[i]->pass_back) == -1)
      return -1;
    if (sendString(comm,pd->output_objects[i]->description) == -1)
      return -1;
    if (sendString(comm,pd->output_objects[i]->name) == -1)
          return -1;
  }
  /* if there are filenames send them */
 if (pd->input_names == NULL) customized_names = 0;
 else customized_names = 1;
 if (sendInt(comm,customized_names) == -1)
     return -1;

  if (customized_names == 1) {
      for (i=0;i<pd->nb_input_objects;i++){
          if (sendString(comm,pd->input_names[i]) == -1)
              return -1;

      }

      for (i=0;i<pd->nb_output_objects;i++){
          if (sendString(comm,pd->output_names[i]) == -1)
              return -1;

      }
      


  }



  return 1;
}

/*
 * freeCallingSequence()
 * 
 * This function frees the memory for a CallingSequence structure.
 * It goes inside the structures and follows pointers to
 * free sub-structures.
 */
void freeCallingSequence(NS_CallingSequence *c)
{
  int i;
  
  if (c == NULL)
    return;

  for (i=0;i<c->nb_args;i++)
  {
    free(c->arg_to_mnemonic[i]);
  }
  free(c->arg_to_mnemonic);
  if (c->nb_constants>0)
  {
    for (i=0;i<c->nb_constants;i++)
    {
      if (c->constants[i].mnemonic != NULL)
        free(c->constants[i].mnemonic);
      if (c->constants[i].value != NULL)
        free(c->constants[i].value);
    }
    free(c->constants);
  }
  if (c->nb_formulae>0)
  {
    for (i=0;i<c->nb_formulae;i++)
    {
      if (c->formulae[i].mnemonic != NULL)
        free(c->formulae[i].mnemonic);
      if (c->formulae[i].expression != NULL)
        free(c->formulae[i].expression);
    }
    free(c->formulae);
  }
  free(c);
  return; 
}

/*
 * sendCallingSequence()
 *
 * This function sends a CallingSequence structure over the network.
 */
int sendCallingSequence(NS_Communicator *comm,NS_CallingSequence *cs)
{
  int i;

  if (sendInt(comm,cs->nb_args) == -1)
    return -1;
  for (i=0;i<cs->nb_args;i++)
  {
    if (sendString(comm,cs->arg_to_mnemonic[i]) == -1)
      return -1;
  }
  if (sendInt(comm,cs->nb_constants) == -1)
    return -1;
  for (i=0;i<cs->nb_constants;i++)
  {
    if (sendString(comm,cs->constants[i].mnemonic) == -1)
      return -1;
    if (sendString(comm,cs->constants[i].value) == -1)
      return -1;
  }  
  if (sendInt(comm,cs->nb_formulae) == -1)
    return -1;
  for (i=0;i<cs->nb_formulae;i++)
  {
    if (sendString(comm,cs->formulae[i].mnemonic) == -1)
      return -1;
    if (sendString(comm,cs->formulae[i].expression) == -1)
      return -1;
  }
  return 1;
}

/*
 * recvProblemDesc()
 *
 * This funcions receives a problem descriptor of the network.
 * A new ProblemDesc structure is thereby created.
 * WARNING: as said for sendProblemDesc, the objects come only
 * with their specifications, not their attributes.
 */
NS_ProblemDesc *recvProblemDesc(NS_Communicator *comm)
{
  NS_ProblemDesc *pd;
  int i;
  int names;

  pd = newProblemDesc();
  if (pd == NULL)
  {
    ns_errno = NetSolveInternalError;
    return NULL;
  }
  if ((recvString(comm,&(pd->name)) == -1) ||
      (recvString(comm,&(pd->nickname)) == -1) ||
      (recvString(comm,&(pd->description)) == -1) ||
      (recvString(comm,&(pd->file)) == -1) ||
      (recvInt(comm,&(pd->major)) == -1) ||
      (recvString(comm,&(pd->customized)) == -1) ||
      (recvInt(comm,&(pd->nb_matlab_merge)) == -1))
  {
    freeProblemDesc(pd);
    return NULL;
  }
  pd->matlab_merge = (NS_MatlabMerge *)calloc(pd->nb_matlab_merge,
                              sizeof(NS_MatlabMerge));
  for (i=0;i<pd->nb_matlab_merge;i++)
  {
    if (recvInt(comm,&(pd->matlab_merge[i].index1)) == -1)
    {
      freeProblemDesc(pd);
      return NULL;
    }
    if (recvInt(comm,&(pd->matlab_merge[i].index2)) == -1)
    {
      freeProblemDesc(pd);
      return NULL;
    }
  }
  if (recvString(comm,&(pd->complexity)) == -1)
  {
    freeProblemDesc(pd);
    return NULL;
  }
  
  /* receiving the calling sequence */
  pd->calling_sequence = recvCallingSequence(comm);
  if (pd->calling_sequence == NULL)
  {
    freeProblemDesc(pd);
    return NULL;
  }

  if (recvInt(comm,&(pd->nb_input_objects)) == -1)
  {
    freeProblemDesc(pd);
    return NULL;
  }
  pd->input_objects = (NS_Object **)calloc(
               pd->nb_input_objects,sizeof(NS_Object**)); 
  for (i=0;i<pd->nb_input_objects;i++)
  {
    pd->input_objects[i] = newObject();
    if (pd->input_objects[i] == NULL)
    {
      freeProblemDesc(pd);
      ns_errno = NetSolveInternalError;
      return NULL;
    }
    if (recvInt(comm,&(pd->input_objects[i]->object_type)) == -1)
    {
      freeProblemDesc(pd);
      return NULL;
    }
    if (recvInt(comm,&(pd->input_objects[i]->data_type)) == -1)
    {
      freeProblemDesc(pd);
      return NULL;
    }
    if (recvInt(comm,&(pd->input_objects[i]->pass_back)) == -1)
    {
      freeProblemDesc(pd);
      return NULL;
    }
    if (recvString(comm,&(pd->input_objects[i]->description)) == -1)
    {
      freeProblemDesc(pd);
      return NULL;
    }
    if (recvString(comm,&(pd->input_objects[i]->name)) == -1)
    {
      freeProblemDesc(pd);
      return NULL;
    }
  }
  if (recvInt(comm,&(pd->nb_output_objects)) == -1)
  {
    freeProblemDesc(pd);
    return NULL;
  }
  pd->output_objects = (NS_Object **)calloc(
               pd->nb_output_objects,sizeof(NS_Object**));
  for (i=0;i<pd->nb_output_objects;i++)
  {

    pd->output_objects[i] = newObject();
    if (pd->output_objects[i] == NULL)
    {
      freeProblemDesc(pd);
      ns_errno = NetSolveInternalError;
      return NULL;
    }




    if (recvInt(comm,&(pd->output_objects[i]->object_type)) == -1)
    {
      freeProblemDesc(pd);
      return NULL;
    }
    if (recvInt(comm,&(pd->output_objects[i]->data_type)) == -1)
    {
      freeProblemDesc(pd);
      return NULL;
    }

    if (recvInt(comm,&(pd->output_objects[i]->pass_back)) == -1)
    {
      freeProblemDesc(pd);
      return NULL;
    }



    if (recvString(comm,&(pd->output_objects[i]->description)) == -1)
    {
      freeProblemDesc(pd);
      return NULL;
    }
    if (recvString(comm,&(pd->output_objects[i]->name)) == -1)
    {
      freeProblemDesc(pd);
      return NULL;
    }
  }


  /* recv file names if available */
  if (recvInt(comm,&names) == -1) /* 1 if no file_names; 0 if file names */
    {
      freeProblemDesc(pd);
      return NULL;
    }

  
  if (names == 1) {
      pd -> input_names = (char**)
			calloc (pd -> nb_input_objects, sizeof (char *));
      pd -> output_names =  (char**)
			calloc (pd -> nb_output_objects, sizeof (char *));
      
      for (i=0;i<pd->nb_input_objects;i++){
          if (recvString(comm,&(pd->input_names[i])) == -1)
              {
                  freeProblemDesc(pd);
                  return NULL;
              }
        }
      
      for (i=0; i<pd->nb_output_objects; i++){
          if (recvString(comm,&(pd->output_names[i])) == -1)
              {
                  freeProblemDesc(pd);
                  return NULL;
              }
      }
      
  } /* customized names */
   

  return pd;
}

/*
 * recvCallingSequence()
 *
 * This function receives a CallingSequence structure from the network.
 * A new structure is thereby created.
 */
NS_CallingSequence *recvCallingSequence(NS_Communicator *comm)
{
  NS_CallingSequence *cs;
  int i;

  cs = (NS_CallingSequence *)calloc(1,sizeof(NS_CallingSequence));
  if (recvInt(comm,&(cs->nb_args)) == -1)
  {
    return NULL;
  }
  cs->arg_to_mnemonic = (char **)calloc(cs->nb_args,sizeof(char*));
  for (i=0;i<cs->nb_args;i++)
  {
    if (recvString(comm,&(cs->arg_to_mnemonic[i])) == -1)
      return NULL;
  }
  if (recvInt(comm,&(cs->nb_constants)) == -1)
    return NULL;
  if (cs->nb_constants != 0)
    cs->constants = (NS_Constant *)calloc(cs->nb_constants,sizeof(NS_Constant));
  else 
    cs->constants = 0;
  for (i=0;i<cs->nb_constants;i++)
  {
    if (recvString(comm,&(cs->constants[i].mnemonic)) == -1)
      return NULL;
    if (recvString(comm,&(cs->constants[i].value)) == -1)
      return NULL;
  }  
  if (recvInt(comm,&(cs->nb_formulae)) == -1)
    return NULL;
  if (cs->nb_formulae != 0)
    cs->formulae = (NS_Formula *)calloc(cs->nb_formulae,sizeof(NS_Formula));
  else
    cs->nb_formulae = 0;
  for (i=0;i<cs->nb_formulae;i++)
  {
    if (recvString(comm,&(cs->formulae[i].mnemonic)) == -1)
      return NULL;
    if (recvString(comm,&(cs->formulae[i].expression)) == -1)
      return NULL;
  } 
  return cs;
}
 
/*
 * compareProblemDesc()
 *
 * This function compares two problem descriptions and 
 * returns 1 if they are compatible, 0 otherwise. The semantics
 * of this function are bound to change when new rules/paradigms
 * are created for the coherency of problems and servers.
 */
int compareProblemDesc(NS_ProblemDesc *d1,NS_ProblemDesc *d2)
{
  int i;

  if (strcmp(d1->name,d2->name))
    return 0;
  if (strcmp(d1->nickname,d2->nickname))
    return 0;

  /* Ignore description and file */

  if (d1->major != d2->major)
    return 0;

  /* Ignore Customized */
  
  if (d1->nb_matlab_merge != d2->nb_matlab_merge)
    return 0;
  for (i=0;i<d1->nb_matlab_merge;i++)
  {
    if (d1->matlab_merge[i].index1 != d2->matlab_merge[i].index1)
      return 0;
    if (d1->matlab_merge[i].index2 != d2->matlab_merge[i].index2)
      return 0;
  }
  if (strcmp(d1->complexity,d2->complexity))
    return 0;
  if (!compareCallingSequence(d1->calling_sequence,d2->calling_sequence))
    return 0;
  if (d1->nb_input_objects != d2->nb_input_objects)
    return 0;
  for (i=0;i<d1->nb_input_objects;i++)
  {
    if (d1->input_objects[i]->data_type != 
                  d2->input_objects[i]->data_type)
      return 0;
    if (d1->input_objects[i]->object_type != 
                  d2->input_objects[i]->object_type)
      return 0;
    /* Ignore the description */
  } 
  if (d1->nb_output_objects != d2->nb_output_objects) 
    return 0;
  for (i=0;i<d1->nb_output_objects;i++)
  {
    if (d1->output_objects[i]->data_type != 
                  d2->output_objects[i]->data_type)
      return 0;
    if (d1->output_objects[i]->object_type != 
                  d2->output_objects[i]->object_type)
      return 0;
    /* Ignore the description */
  }
  return 1;
}

/*
 * compareCallingSequence()
 *
 * This function compares two calling sequences and returns
 * 1 if they are compatible, 0 otherwise.
 */
int compareCallingSequence(NS_CallingSequence *c1,NS_CallingSequence *c2)
{
  int i;
  if (c1->nb_args != c2->nb_args)
    return 0;
 
  for (i=0;i<c1->nb_args;i++)
    if (strcmp(c1->arg_to_mnemonic[i],c2->arg_to_mnemonic[i]))
      return 0;
  if (c1->nb_constants != c2->nb_constants)
    return 0;
  for (i=0;i<c1->nb_constants;i++)
  {
    if (strcmp(c1->constants[i].mnemonic,c2->constants[i].mnemonic))
      return 0;
    if (strcmp(c1->constants[i].value,c2->constants[i].value))
      return 0;
  }
  if (c1->nb_formulae != c2->nb_formulae)
    return 0;
  for (i=0;i<c1->nb_formulae;i++)
  {
    if (strcmp(c1->formulae[i].mnemonic,c2->formulae[i].mnemonic))
      return 0;
    if (strcmp(c1->formulae[i].expression,c2->formulae[i].expression))
      return 0;
  }
  return 1;
}

/*
 * sendObject()
 *
 * This function sends an object over the network, ALONG WITH ITS ATTRIBUTES.
 */
int sendObject(NS_Communicator *comm,NS_Object *obj)
{
  int k;

  if (sendInt(comm,obj->object_type) == -1)
    return -1;
  if (sendInt(comm,obj->data_type) == -1)
    return -1;
  if (sendInt(comm,obj->distributed) == -1)
    return -1;
  if(obj->distributed == 1){
    if(sendDsiObj(comm, obj->dsi_obj)  == -1)
      return -1;
  }

  if (sendInt(comm,obj->pass_back) == -1)
    return -1;

  /* Ignore the description */
  switch(obj->object_type)
  {
    case NETSOLVE_MATRIX:
    {
      int major,m,n,l;
      void *ptr;
  
      major = obj->attributes.matrix_attributes.major;
      m = obj->attributes.matrix_attributes.m;
      n = obj->attributes.matrix_attributes.n;
      l = obj->attributes.matrix_attributes.l;
      ptr = obj->attributes.matrix_attributes.ptr;
  
      if (sendInt(comm,major) == -1)
        return -1;
      if (sendInt(comm,m) == -1)
        return -1;
      if (sendInt(comm,n) == -1)
        return -1;
      /* Send the data */
      if(obj->distributed == 0){
      if ((l == -1) || (l == m))
      {
        if (sendArray(comm,obj->data_type,ptr, m*n) == -1)
          return -1;
      }
      else
      {
        for (k=0;k<n;k++)
          if (sendArray(comm,obj->data_type,
              (char *)ptr+k*l*netsolve_sizeof(obj->data_type),m) == -1)
            return -1;
      }
      }
      break;
    }
    case NETSOLVE_SPARSEMATRIX:
    {
      int major,m,n,f, *rc_ptr, *rc_index;
    
      void *ptr;
  
      major = obj->attributes.sparsematrix_attributes.major;
      m = obj->attributes.sparsematrix_attributes.m;
      n = obj->attributes.sparsematrix_attributes.n;
      f = obj->attributes.sparsematrix_attributes.f;
      rc_ptr = obj->attributes.sparsematrix_attributes.rc_ptr;
      rc_index = obj->attributes.sparsematrix_attributes.rc_index;
      ptr = obj->attributes.sparsematrix_attributes.ptr;
  
      if (sendInt(comm,major) == -1)
        return -1;
      if (sendInt(comm,m) == -1)
        return -1;
      if (sendInt(comm,n) == -1)
        return -1;
      if (sendInt(comm,f) == -1)
        return -1;

      /*send indices */
      if (sendArray(comm,NETSOLVE_I,rc_ptr,m+1) == -1)
        return -1;
      if (sendArray(comm,NETSOLVE_I,rc_index,f) == -1)
        return -1;

      /* Send the data */
      if(obj->distributed == 0){
      if (sendArray(comm,obj->data_type,ptr, f) == -1)
        return -1;
      }
      break;
    }
    case NETSOLVE_VECTOR:
    {
      int m;
      void *ptr;
  
      m = obj->attributes.vector_attributes.m;
      ptr = obj->attributes.vector_attributes.ptr;
  
      if (sendInt(comm,m) == -1)
        return -1;
      if(obj->distributed == 0){
      if (sendArray(comm,obj->data_type,ptr,m) == -1)
        return -1;
      }
      break;
    }
    case NETSOLVE_SCALAR:
    { 
      void *ptr;
      ptr = obj->attributes.scalar_attributes.ptr;
      if (sendArray(comm,obj->data_type,ptr,1) == -1)
        return -1;
      break;
    } 
    case NETSOLVE_FILE:
    {
      int fd;
      char *filename=obj->attributes.file_attributes.filename;
      int size;
      struct stat st;
  
      if (!isRemoteFile(filename))
      {
        fd = open(filename,O_RDONLY,0666);
        if (fd < 0)
        {
#ifdef VIEW
          fprintf(STDERR "File '%s' not found\n",filename);
#endif 
          ns_errno = NetSolveFileError;
          return -1; 
        }

        if (stat(filename,&st) == -1)
        {
          close(fd);
          ns_errno = NetSolveFileError;
          return -1;
        }
        size = (int)(st.st_size);

        if (sendInt(comm,size) == -1)
        {
          close(fd);
          return -1;
        }
        if (sendArrayFromFile(comm,NETSOLVE_CHAR,fd,size) == -1)
        {
          close(fd);
          return -1;
        }
        close(fd);
      }
      else /* Remote file */
      {
        fprintf(stderr,"REMOTE !!!\n");
        if (sendInt(comm,-1) == -1)
          return -1;
        if (sendString(comm,filename) == -1)
          return -1;
      }
      break;
    }
    case NETSOLVE_PACKEDFILES:
    {
      int fd;
      char buffer[256];
      int nb_files=obj->attributes.packedfiles_attributes.m;
      char *prefix=obj->attributes.packedfiles_attributes.defaultprefix;
      char **filenames=obj->attributes.packedfiles_attributes.filenames;
      int size;
      int i;
      struct stat st;

      if (sendInt(comm,nb_files) == -1)
        return -1;
   
      for (i=0;i<nb_files;i++)
      {
        if (prefix == NULL)
          sprintf(buffer,"%s",filenames[i]);
        else
          sprintf(buffer,"%s-%d",prefix,i);

        if (sendString(comm,buffer) == -1)
          return -1;

        if (!isRemoteFile(buffer))
        {
          fd = open(buffer,O_RDONLY,0666);
          if (fd < 0)
          {
#ifdef VIEW
            fprintf(STDERR "File '%s' not found\n",buffer);
#endif
            ns_errno = NetSolveFileError;
            return -1;
          }
  
          if (stat(buffer,&st) == -1)
          {
            close(fd);
#ifdef VIEW
            fprintf(STDERR "Impossible to access file '%s'\n",buffer);
#endif
            ns_errno = NetSolveFileError;
            return -1;
          }
          size = (int)(st.st_size);
    
          if (sendInt(comm,size) == -1)
          {
            close(fd);
            return -1;
          }
          if (sendArrayFromFile(comm,NETSOLVE_CHAR,fd,size) == -1)
          {
            close(fd);
            return -1;
          }
          close(fd);
        }
      }
      break;
    }
    case NETSOLVE_STRINGLIST:
    {
      int nb_strings=obj->attributes.stringlist_attributes.m;
      char **strings=obj->attributes.stringlist_attributes.strings;
      int i;

      if (sendInt(comm,nb_strings) == -1)
        return -1;

      for (i=0;i<nb_strings;i++)
      {
        if (sendString(comm,strings[i]) == -1)
          return -1;
      }
      break;
    } 
    case NETSOLVE_UPF:
    {
      int fd;
      char *filename=obj->attributes.upf_attributes.filename;
      char *funcname=obj->attributes.upf_attributes.funcname;
      int size;
      struct stat st;
      int language=obj->attributes.upf_attributes.language;

      fd = open(filename,O_RDONLY,0666);
      if (fd < 0)
      {
#ifdef VIEW
        fprintf(STDERR "Impossible to open file '%s'\n",filename);
#endif
        ns_errno = NetSolveFileError;
        return -1;
      }
      if (stat(filename,&st) == -1)
      {
#ifdef VIEW
        fprintf(STDERR "Impossible to open file '%s'\n",filename);
#endif
        close(fd);
        ns_errno = NetSolveFileError;
        return -1;
      }
      size = (int)(st.st_size);
      if (sendInt(comm,language) == -1)
      {
        close(fd);
        return -1;
      }
      if (sendString(comm,funcname) == -1)
      {
        close(fd);
        return -1;
      }
      if (sendInt(comm,size) == -1)
      {
        close(fd);
        return -1;
      }
      if (sendArrayFromFile(comm,NETSOLVE_CHAR,fd,size) == -1)
      {
        close(fd);
        return -1;
      }
      close(fd);
      break;
    }
    case NETSOLVE_STRING:
    {
      char *ptr=obj->attributes.string_attributes.ptr;
      if (sendString(comm,ptr) == -1)
        return -1;
      break;
    }
    default:
#ifdef VIEW
      fprintf(STDERR "Unknown Object type\n");
#endif
      ns_errno = NetSolveInternalError;
      return -1;
  }
  return 1;
}

/*
 * recvObject()
 *
 * This function receives an object over the network. Thi object
 * should have been sent by sendObject(). The object arrives
 * WITH ITS ATTRIBUTES. The 'obj' structure passed to recvObject()
 * can contain partial information about the object expected. If there
 * is a mismatch, then this function returns -1 and sets ns_errno.
 * Usually, integer values of -1 or pointer values of NULL are wildcards
 * and will match the parameters of any incoming objects. Of course,
 * it is advised to use wildcards only when there is no other alternative.
 */
int recvObject(NS_Communicator *comm,NS_Object *obj)
{
  int k;
  int object_type;
  int data_type;
  int return_val;

  if (obj == NULL)
  {
    ns_errno = NetSolveInternalError;
    return -1;
  }

  if (recvInt(comm,&object_type) == -1)
    return -1;
  if (recvInt(comm,&data_type) == -1)
    return -1;
  if (recvInt(comm,&(obj->distributed)) == -1)
    return -1;

  if(obj->distributed == 1){
    obj->dsi_obj = (DSI_OBJECT *)calloc(1, sizeof(DSI_OBJECT));
    obj->dsi_obj->dsi_file = (DSI_FILE *)calloc(1, sizeof(DSI_FILE));
    if(recvDsiObj(comm, obj->dsi_obj) == -1)
      return -1;
  }

  if (recvInt(comm,&obj->pass_back) == -1)
    return -1;


  if (obj->object_type != -1)
  {
    if (obj->object_type != object_type)
    {
/*        fprintf(stderr, "pid%i *OBJTYP mismatch %i (expected) != %i (actual)\n",  */
/*            getpid(), obj->object_type, object_type); */
      ns_errno = NetSolveMismatch;
      return -1;
    }
  }
  if (obj->data_type != -1)
  {
    if (obj->data_type != data_type)
    {
/*        fprintf(stderr, "pid%i *DATATYP mismatch %i != %i\n", getpid(), */
/*            obj->data_type, data_type); */
      ns_errno = NetSolveMismatch;
      return -1;
    }
   }
  

  /* Ignore the description */

  switch(obj->object_type)
  {
    case NETSOLVE_MATRIX:
    {
      int major,m,n;
      void *ptr;
  
      if (recvInt(comm,&major) == -1)
        return -1;
      if (obj->attributes.matrix_attributes.major != -1)
      { 
        if (obj->attributes.matrix_attributes.major != major)
        {
          ns_errno = NetSolveMismatch;
          return -1;
        }
      }
      obj->attributes.matrix_attributes.major = major;
      if (recvInt(comm,&m) == -1)
        return -1;
      if (obj->attributes.matrix_attributes.m != -1)
      {
        if (obj->attributes.matrix_attributes.m != m)
        {
          ns_errno = NetSolveMismatch;
          return -1;
        }
      }
      obj->attributes.matrix_attributes.m = m;
      if (recvInt(comm,&n) == -1)
        return -1;
      if (obj->attributes.matrix_attributes.n != -1)
      {
        if (obj->attributes.matrix_attributes.n != n)
        {
          ns_errno = NetSolveMismatch;
          return -1;
        }
      }

      obj->attributes.matrix_attributes.n = n;

      if (obj->attributes.matrix_attributes.l  == -1)
        obj->attributes.matrix_attributes.l = m;

      /* Receive the data */
      if (obj->attributes.matrix_attributes.ptr == NULL)
      {
        if (obj->attributes.matrix_attributes.l != m)
          obj->attributes.matrix_attributes.l = m;
        /* Malloc the data space */
        obj->attributes.matrix_attributes.ptr =
          (void *)calloc(m*n,netsolve_sizeof(data_type));
      }
      ptr = obj->attributes.matrix_attributes.ptr;

      /* Recv the data */
      if(obj->distributed == 1){
        ns_dsi_read_matrix(obj->dsi_obj, ptr, m, n, data_type);
      }
      else{
      if (obj->attributes.matrix_attributes.l == m)
      {
        if (recvArray(comm,data_type,ptr,m*n) == -1)
          return -1;

      }
      else if (obj->attributes.matrix_attributes.l == -1)
      {
        if (recvArray(comm,data_type,ptr,m*n) == -1)
          return -1;
        obj->attributes.matrix_attributes.l = m;
      }
      else
      {
        for (k=0;k<n;k++)
          if (recvArray(comm,data_type,
              (char *)ptr+k*obj->attributes.matrix_attributes.l*
              netsolve_sizeof(data_type),m) == -1)
            return -1;
      }
      }
      break;
    }
    case NETSOLVE_SPARSEMATRIX:
    {
      int major,m,n,f,*rc_ptr,*rc_index;
      void *ptr;
 
      if (recvInt(comm,&major) == -1)
        return -1;
      if (obj->attributes.sparsematrix_attributes.major != -1)
      {
        if (obj->attributes.sparsematrix_attributes.major != major)
        {
          ns_errno = NetSolveMismatch;
          return -1;
        }
      }
      obj->attributes.sparsematrix_attributes.major = major;

      if (recvInt(comm,&m) == -1)
        return -1;
      if (obj->attributes.sparsematrix_attributes.m != -1)
      {
        if (obj->attributes.sparsematrix_attributes.m != m)
        {
          ns_errno = NetSolveMismatch;
          return -1;
        }
      }
      obj->attributes.sparsematrix_attributes.m = m;

      if (recvInt(comm,&n) == -1)
        return -1;
      if (obj->attributes.sparsematrix_attributes.n != -1)
      {
        if (obj->attributes.sparsematrix_attributes.n != n)
        {
          ns_errno = NetSolveMismatch;
          return -1;
        }
      }
      obj->attributes.sparsematrix_attributes.n = n;
 
      if (recvInt(comm,&f) == -1)
        return -1;
      if (obj->attributes.sparsematrix_attributes.f != -1)
      {
        if (obj->attributes.sparsematrix_attributes.f != f)
        {
          ns_errno = NetSolveMismatch;
          return -1;
        }
      }
      obj->attributes.sparsematrix_attributes.f = f;
 
      /* Malloc space for the indices and data ptrs */
      if (obj->attributes.sparsematrix_attributes.ptr == NULL)
      {
        obj->attributes.sparsematrix_attributes.ptr =
          (void *)calloc(f,netsolve_sizeof(data_type));
      }
      if (obj->attributes.sparsematrix_attributes.rc_ptr == NULL)
      {
        obj->attributes.sparsematrix_attributes.rc_ptr =
          (void *)calloc(m+1,netsolve_sizeof(NETSOLVE_I));
      }
      if (obj->attributes.sparsematrix_attributes.rc_index == NULL)
      {
        obj->attributes.sparsematrix_attributes.rc_index =
          (void *)calloc(f,netsolve_sizeof(NETSOLVE_I));
      }

      rc_ptr = obj->attributes.sparsematrix_attributes.rc_ptr;
      rc_index = obj->attributes.sparsematrix_attributes.rc_index;
      ptr = obj->attributes.sparsematrix_attributes.ptr;

      /* Receive the indices and data */
      if (recvArray(comm,NETSOLVE_I,rc_ptr,m+1) == -1)
        return -1;
      if (recvArray(comm,NETSOLVE_I,rc_index,f) == -1)
        return -1;
      if (recvArray(comm,data_type,ptr,f) == -1)
        return -1;
 
      break;
    }
    case NETSOLVE_VECTOR:
    {
      int m;
      void *ptr;
  
      if (recvInt(comm,&m) == -1)
        return -1;
      if (obj->attributes.vector_attributes.m != -1)
      {
        if (obj->attributes.vector_attributes.m != m)
        {
          ns_errno = NetSolveMismatch; 
          return -1;
        }
      }
      obj->attributes.vector_attributes.m = m;

      if (obj->attributes.vector_attributes.ptr == NULL)
      {
        /* Malloc the data space */
        obj->attributes.vector_attributes.ptr =
          (void *)calloc(m,netsolve_sizeof(data_type));
      }
      ptr = obj->attributes.vector_attributes.ptr;

      if(obj->distributed == 1){
        return_val = ns_dsi_read_vector(obj->dsi_obj, ptr, m, data_type);
        if (return_val < 0)
          return -1;
      }
      else{
      if (recvArray(comm,data_type,ptr,m) == -1)
        return -1;
      }
      break;
    }
    case NETSOLVE_SCALAR:
    { 
      void *ptr;
      if (obj->attributes.scalar_attributes.ptr == NULL)
      {
        /* Malloc the data space */
        obj->attributes.scalar_attributes.ptr =
          (void *)calloc(1,netsolve_sizeof(data_type));
      }
      ptr = obj->attributes.scalar_attributes.ptr;
      if (recvArray(comm,data_type,ptr,1) == -1)
        return -1;
      
      break;
    } 
    case NETSOLVE_FILE:
    {
      int fd;
      char *filename=obj->attributes.file_attributes.filename;
      int size;
  
      if (filename == NULL)
      {
        ns_errno = NetSolveInternalError;
        return -1;
      }

      if (recvInt(comm,&size) == -1)
        return -1;

      if (size == -1) /* Remote file */
      {
        char *localname;

        if (recvString(comm,&localname) == -1)
          return -1;
        free(obj->attributes.file_attributes.filename);
        obj->attributes.file_attributes.filename = strdup(localname);
        break;
      }
    
      /* File on the socket */
      fd = open(filename,O_CREAT|O_WRONLY,0666);
      if (fd < 0)
      {
        ns_errno = NetSolveFileError;
        return -1; 
      }
      if (recvArrayToFile(comm,NETSOLVE_CHAR,fd,size) == -1)
      {
        close(fd);
        return -1;
      }
      close(fd);
      break;
    }
    case NETSOLVE_PACKEDFILES:
    {
      int fd;
      char **filenames=obj->attributes.packedfiles_attributes.filenames;
      char *prefix=obj->attributes.packedfiles_attributes.defaultprefix;
      int nb_files;
      int size,i;
      char buffer[256];
      char *original_filename;

      if (recvInt(comm,&nb_files) == -1)
        return -1;
 
      if (obj->attributes.packedfiles_attributes.m != -1)
      {
        if (obj->attributes.packedfiles_attributes.m != nb_files)
        {
          ns_errno = NetSolveMismatch;
          return -1;
        }
      }
      obj->attributes.packedfiles_attributes.m = nb_files;
      
      for (i=0;i<nb_files;i++)
      {
        if (recvString(comm,&original_filename) == -1)
          return -1;


        if (prefix != NULL)
          sprintf(buffer,"%s-%d", prefix,i);
        else
        {
          if (filenames != NULL)
            sprintf(buffer,"%s",filenames[i]);
          else
            sprintf(buffer,"%s",original_filename);
        }

        if (isRemoteFile(original_filename))
        {
          char command[256];
          sprintf(command,"ln -s %s %s",getRemoteFileName(original_filename),
                                        buffer);
          free(original_filename);
          continue;
        }

        free(original_filename);

        fd = open(buffer,O_CREAT|O_WRONLY,0666);
       
        if (fd < 0)
        {
#ifdef VIEW
          fprintf(STDERR "Impossible to create file '%s'\n",buffer);
#endif
          ns_errno = NetSolveFileError;
          return -1;
        }
        if (recvInt(comm,&size) == -1)
        {
          close(fd);
          return -1;
        }
        if (recvArrayToFile(comm,NETSOLVE_CHAR,fd,size) == -1)
        {
          close(fd);
          return -1;
        }
        close(fd);
      }
      /* convert to prefix = NULL */
      if (prefix != NULL)
      {
        obj->attributes.packedfiles_attributes.filenames =
             (char **)calloc(nb_files,sizeof(char*));
        for (i=0;i<nb_files;i++)
        {
          sprintf(buffer,"%s-%d",prefix,i);
          obj->attributes.packedfiles_attributes.filenames[i] = strdup(buffer);
        }

        obj->attributes.packedfiles_attributes.defaultprefix = NULL;
      }
      break;
    }
    case NETSOLVE_STRINGLIST:
    {
      char **strings=obj->attributes.stringlist_attributes.strings;
      int nb_strings;
      int i;

      if (recvInt(comm,&nb_strings) == -1)
        return -1;

      if (obj->attributes.stringlist_attributes.m != -1)
      {
        if (obj->attributes.stringlist_attributes.m != nb_strings)
        {
          ns_errno = NetSolveMismatch;
          return -1;
        }
      }
      obj->attributes.stringlist_attributes.m = nb_strings;

      if (strings == NULL)
      {
        obj->attributes.stringlist_attributes.strings =
           (char **)calloc(nb_strings,sizeof(char*));
        strings = obj->attributes.stringlist_attributes.strings;
      }

      for (i=0;i<nb_strings;i++)
      {
        if (recvString(comm,&(strings[i])) == -1)
          return -1;
      }
      break;
    }
    case NETSOLVE_UPF:
    {
      int fd;
      char *filename=obj->attributes.upf_attributes.filename;
      int size;
      int language;
      char *funcname;

      if (filename == NULL)
      {
        ns_errno = NetSolveInternalError;
        return -1;
      }

      fd = open(filename,O_CREAT|O_WRONLY,0666);
      if (fd < 0)
      {
#ifdef VIEW
        fprintf(STDERR "Cannot open file '%s'\n",filename);
#endif
        ns_errno = NetSolveFileError;
        return -1;
      }      
      if (recvInt(comm,&language) == -1)
      {
        close(fd);
        return -1;
      }
      if (obj->attributes.upf_attributes.language != -1)
      {
        if (obj->attributes.upf_attributes.language != language) 
        {
          ns_errno = NetSolveMismatch;
          return -1;
        }
      }
      obj->attributes.upf_attributes.language = language;

      if (recvString(comm,&funcname) == -1)
      {
        close(fd);
        return -1;
      }
      if (obj->attributes.upf_attributes.funcname != NULL)
      {
        if (strcmp(obj->attributes.upf_attributes.funcname,funcname))
        {
          ns_errno = NetSolveMismatch;
          return -1;
        }
      }
      else
        obj->attributes.upf_attributes.funcname = funcname;
      if (recvInt(comm,&size) == -1)
      {
        close(fd);
        return -1;
      }
      if (recvArrayToFile(comm,NETSOLVE_CHAR,fd,size) == -1)
      {
        close(fd);
        return -1;
      }
      close(fd);
      break;
    }
    case NETSOLVE_STRING:
    {
      if (recvString(comm,&(obj->attributes.string_attributes.ptr)) == -1)
        return -1;
      break;
    }
    default:
#ifdef VIEW
      fprintf(STDERR "Unknown object type\n");
#endif
      ns_errno = NetSolveInternalError;
      return -1;
  }
  return 1;
}

/*
 * createMatrixObject()
 *
 * This function creates Matrix Object with its attributes. 
 * it takes in input the matrix's data type, number of rows, columns, 
 * leading dimension and pointer to the matrix's first element.
 * The ScaLAPACK descriptor in the Matrix Object data structure is used
 * only inside the ScaLAPACK customized server and is of no concern
 * at this level.
 */
NS_Object *createMatrixObject(int data_type,void*ptr,int major,
            int m,int n,int l)
{
  NS_Object *obj;

  obj = newObject();
  obj->object_type = NETSOLVE_MATRIX;
  obj->data_type = data_type;
  obj->attributes.matrix_attributes.ptr = ptr;
  obj->attributes.matrix_attributes.major = major;
  obj->attributes.matrix_attributes.m = m;
  obj->attributes.matrix_attributes.n = n;
  obj->attributes.matrix_attributes.l = l;
  return obj;
}

NS_Object *createSparseMatrixObject(int data_type,void*ptr,int major,
            int m,int n,int f, int *rc_ptr, int *rc_index)
{
  NS_Object *obj;
 
  obj = newObject();
  obj->object_type = NETSOLVE_SPARSEMATRIX;
  obj->data_type = data_type;
  obj->attributes.sparsematrix_attributes.ptr = ptr;
  obj->attributes.sparsematrix_attributes.major = major;
  obj->attributes.sparsematrix_attributes.m = m;
  obj->attributes.sparsematrix_attributes.n = n;
  obj->attributes.sparsematrix_attributes.f = f;
  obj->attributes.sparsematrix_attributes.rc_ptr = rc_ptr;
  obj->attributes.sparsematrix_attributes.rc_index = rc_index;
  return obj;
}

/*
 * createVectorObject()
 *
 * This function creates a vector object with attributes. It takes
 * in input the vector's data type, number of rows and a pointer
 * to the vector's first element.
 */
NS_Object *createVectorObject(int data_type,void *ptr,int m)
{
  NS_Object *obj;

  obj = newObject();
  obj->object_type = NETSOLVE_VECTOR;
  obj->data_type = data_type;
  obj->attributes.vector_attributes.ptr = ptr;
  obj->attributes.vector_attributes.m = m;
  return obj;
}

/*
 * createScalarObject()
 *
 * This function creates a scalar object with its attributes. It takes
 * in input the scalar's data type and a pointer to it's value.
 */
NS_Object *createScalarObject(int data_type,void *ptr)
{
  NS_Object *obj;

  obj = newObject();
  obj->object_type = NETSOLVE_SCALAR;
  obj->data_type = data_type;
  obj->attributes.scalar_attributes.ptr = ptr;
  return obj;
}

/*
 * createFileObject()
 *
 * This function creates a file objects with its attributes. It takes
 * in input the filename.
 */
NS_Object *createFileObject(char *filename)
{
  NS_Object *obj;

  obj = newObject();
  obj->object_type = NETSOLVE_FILE;
  obj->data_type = NETSOLVE_CHAR;
  if (filename != NULL)
    obj->attributes.file_attributes.filename = strdup(filename);
  return obj;
}

/*
 * createPackedFilesObject()
 *
 * This function creates a packed file object. It takes in input
 * a possible default prefix for the files, a list of filenames
 * and the number of files to be packed. Typically, when one wants
 * to send a packedfile object, on calls createPackedFilesObject() as:
 *     obj = createPackedFilesObject(NULL,list,10);
 * and when on want to receive a packed file object of unknown 
 * specification, on calls:
 *     obj = createPackedFilesObject("stuff",NULL,-1);
 * and the files received are numbered: stuff0,stuff1,stuff2,etc...
 */
NS_Object *createPackedFilesObject(char *def,char **filenames,int m)
{
  NS_Object *obj;
  int i;

  obj = newObject();
  obj->object_type = NETSOLVE_PACKEDFILES;
  obj->data_type = NETSOLVE_CHAR;
  obj->attributes.packedfiles_attributes.m = m;

  if (def != NULL)
  {
    obj->attributes.packedfiles_attributes.defaultprefix = strdup(def);
    obj->attributes.packedfiles_attributes.filenames = NULL;
  }
  else if (filenames != NULL)
  {
    obj->attributes.packedfiles_attributes.defaultprefix = NULL;

    obj->attributes.packedfiles_attributes.filenames = 
      (char **)calloc(m,sizeof(char*));
    for (i=0;i<m;i++)
      obj->attributes.packedfiles_attributes.filenames[i] =
            strdup(filenames[i]);
  }
  else
  {
    obj->attributes.packedfiles_attributes.defaultprefix = NULL;
    obj->attributes.packedfiles_attributes.filenames = NULL;
  }

  return obj;
}

/*
 * createStringListObject()
 *
 * This function creates a packed file object. It takes in input
 * a possible default prefix for the files, a list of filenames
 * and the number of files to be packed. Typically, when one wants
 * to send a packedfile object, on calls createPackedFilesObject() as:
 *     obj = createPackedFilesObject(NULL,list,10);
 * and when on want to receive a packed file object of unknown
 * specification, on calls:
 *     obj = createPackedFilesObject("stuff",NULL,-1);
 * and the files received are numbered: stuff0,stuff1,stuff2,etc...
 */
NS_Object *createStringListObject(char **strings,int m)
{
  NS_Object *obj;

  obj = newObject();
  obj->object_type = NETSOLVE_STRINGLIST;
  obj->data_type = NETSOLVE_CHAR;
  obj->attributes.stringlist_attributes.m = m;
  obj->attributes.stringlist_attributes.strings = strings;
  return obj;
}



/*
 * createUPFObject()
 *
 * This function creates a UPF object with its attributes.
 * It takes in input the language of the UPF, (UPF_LANG_FORTRAN, UPF_LANG_C)
 * the filename and the name of the function.
 */
NS_Object *createUPFObject(int language, char *filename,
                        char *funcname)
{
  NS_Object *obj;

  obj = newObject();
  obj->object_type = NETSOLVE_UPF;
  obj->data_type = NETSOLVE_CHAR;
  obj->attributes.upf_attributes.language = language;
  if (filename != NULL)
    obj->attributes.upf_attributes.filename = strdup(filename);
  else
    obj->attributes.upf_attributes.filename = NULL;
  if (funcname == NULL)
    obj->attributes.upf_attributes.funcname = NULL;
  else
    obj->attributes.upf_attributes.funcname = strdup(funcname);
  return obj;
}

/*
 * createStringObject()
 *
 * This function creates a string object. It takes in
 * input a pointer to the first character of the null-termincated string.
 */
NS_Object *createStringObject(char *s)
{
  NS_Object *obj;

  obj = newObject();
  obj->object_type = NETSOLVE_STRING;
  obj->data_type = NETSOLVE_CHAR;
  obj->attributes.string_attributes.ptr = s;
  return obj;
}

/*
 * containsUPF()
 *
 * This function returns 1 if the problem descriptor passed in input
 * as at least one UPF object in input, 0 otherwise.
 */
int containsUPF(NS_ProblemDesc *pd)
{
  int i;

  for (i=0;i<pd->nb_input_objects;i++)
    if ((pd->input_objects)[i]->object_type == NETSOLVE_UPF)
      return 1;
  return 0;
}

/*
 * recvInputObjectsToFiles()
 *
 * receives all the objects (data) to files in the working
 * directory. The files are names input0,input1,...
 * if the input is a file or a upf, then the file
 * inputx countains no real data, just the name 
 * of the file containing the data (fileinputx or upfinputx)
 */
int recvInputObjectsToFiles(NS_Communicator *comm,NS_ProblemDesc *pd)
{
  int i;
  NS_Object *obj;
  int fd;
  char buffer[256];
  void *misc=NULL;

  for (i=0;i<pd->nb_input_objects;i++)
  {
    if (pd->input_objects[i]->pass_back == FALSE) continue; /* do i really have to pass the object */
    if (pd->input_names == NULL) sprintf(buffer,"input%d",i);
    else sprintf(buffer,"%s",pd->input_names[i]);
    fd = open(buffer,O_CREAT|O_WRONLY,0666);
    if (fd < 0)
    {
#ifdef VIEW
      fprintf(STDERR "Impossible to create file '%s'\n",buffer);
#endif
      ns_errno = NetSolveFileError;
      return -1;
    }
    obj = createEmptyObject(pd->input_objects[i]->object_type,
                           pd->input_objects[i]->data_type);

    if (obj->object_type == NETSOLVE_UPF)
      misc = buffer;
    if (obj->object_type == NETSOLVE_FILE)
      misc = buffer;
    if (obj->object_type == NETSOLVE_PACKEDFILES)
      misc = buffer;
    if (recvObjectToFile(comm,fd,obj,misc) == -1)
    {
      close(fd);
      return -1;
    }
    freeObject(obj);
    close(fd);
  }
  return 1;
}
/*
 * recvInputObjectsToFiles_sequence()
 *
 * receives all the objects (data) to files in the working
 * directory. The files are names input0,input1,...
 * if the input is a file or a upf, then the file
 * inputx countains no real data, just the name 
 * of the file containing the data (fileinputx or upfinputx)
 */
int recvInputObjectsToFiles_sequence(NS_Communicator *comm,NS_ProblemDesc *pd, int node_nr)
{
  int i;
  NS_Object *obj;
  int fd;
  char buffer[256];
  void *misc=NULL;

  for (i=0;i<pd->nb_input_objects;i++)
  {
    if (pd->input_objects[i]->pass_back == FALSE) continue; /* do i really have to pass the object */
    if (pd->input_names == NULL) sprintf(buffer,"input%d",i);
    else sprintf(buffer,"%s",pd->input_names[i]);
    fd = open(buffer,O_CREAT|O_WRONLY,0666);
    if (fd < 0)
    {
#ifdef VIEW
      fprintf(STDERR "Impossible to create file '%s'\n",buffer);
#endif
      ns_errno = NetSolveFileError;
      return -1;
    }
    obj = createEmptyObject(pd->input_objects[i]->object_type,
                           pd->input_objects[i]->data_type);

    if (obj->object_type == NETSOLVE_UPF)
      misc = buffer;
    if (obj->object_type == NETSOLVE_FILE)
      misc = buffer;
    if (obj->object_type == NETSOLVE_PACKEDFILES)
      misc = buffer;

    if (recvObjectToFile(comm,fd,obj,misc) == -1)
    {
      close(fd);
      return -1;
    }
    freeObject(obj);
    close(fd);
  }
  return 1;
}

/*
 * createEmptyObject()
 *
 * This function allocates memory for an object of a given
 * type, and sets all the parameters to wildcards (-1 or NULL)
 */
NS_Object *createEmptyObject(int object_type,int data_type)
{
  NS_Object *obj;

  switch(object_type)
  {
    case NETSOLVE_MATRIX:
      obj = createMatrixObject(data_type,NULL,-1,-1,-1,-1); 
      break;
    case NETSOLVE_SPARSEMATRIX:
      obj = createSparseMatrixObject(data_type,NULL,-1,-1,-1,-1,NULL,NULL); 
      break;
    case NETSOLVE_VECTOR:
      obj = createVectorObject(data_type,NULL,-1); 
      break;
    case NETSOLVE_SCALAR:
      obj = createScalarObject(data_type,NULL); 
      break;
    case NETSOLVE_STRING:
      obj = createStringObject(NULL); 
      break;
    case NETSOLVE_FILE:
      obj = createFileObject(NULL); 
      break;
    case NETSOLVE_PACKEDFILES:
      obj = createPackedFilesObject(NULL,NULL,-1); 
      break;
    case NETSOLVE_STRINGLIST:
      obj = createStringListObject(NULL,-1); 
      break;
    case NETSOLVE_UPF:
      obj = createUPFObject(-1,NULL,NULL); 
      break;
    default:
#ifdef VIEW
      fprintf(STDERR "Unknown object type\n");
#endif
      ns_errno = NetSolveInternalError;
      return NULL;
  }
  return obj;
}

/*
 * readObjectFromFile()
 *
 */
int readObjectFromFile(int fd,NS_Object *obj)
{
  int object_type;
  int data_type;
  int pass_back;

  if (read(fd,&object_type,sizeof(int)) != sizeof(int))
  {
    ns_errno = NetSolveFileError;
    return -1;
  }
  if (read(fd,&data_type,sizeof(int)) != sizeof(int))
  {
    ns_errno = NetSolveFileError;
    return -1;
  }

  if (read(fd,&pass_back,sizeof(int)) != sizeof(int))
  {
    ns_errno = NetSolveFileError;
    return -1;
  }

  if (object_type != obj->object_type)
  {
    ns_errno = NetSolveMismatch;
    return -1;
  }
  if (data_type != obj->data_type)
  {
    ns_errno = NetSolveMismatch;
    return -1;
  }

  switch(object_type)
  {
    case NETSOLVE_MATRIX:
    {
      int m,n,major,k;
      void *ptr;

      if (read(fd,&major,sizeof(int)) != sizeof(int))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
      if (read(fd,&m,sizeof(int)) != sizeof(int))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
      if (read(fd,&n,sizeof(int)) != sizeof(int))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
 
      if (obj->attributes.matrix_attributes.major != -1)
      {
        if (obj->attributes.matrix_attributes.major != major)
        {
          ns_errno = NetSolveMismatch;
          return -1;
        }
      }
      obj->attributes.matrix_attributes.major = major;
      if (obj->attributes.matrix_attributes.m != -1)
      {
        if (obj->attributes.matrix_attributes.m != m)
        {
          ns_errno = NetSolveMismatch;
          return -1;
        }
      }
      obj->attributes.matrix_attributes.m = m;
      if (obj->attributes.matrix_attributes.n != -1)
      {
        if (obj->attributes.matrix_attributes.n != n)
        {
          ns_errno = NetSolveMismatch;
          return -1;
        }
      }
      obj->attributes.matrix_attributes.n = n;

      if (obj->attributes.matrix_attributes.ptr == NULL)
      {
        if (obj->attributes.matrix_attributes.l != m)
          obj->attributes.matrix_attributes.l = m;
        /* Malloc the data space */
        obj->attributes.matrix_attributes.ptr =
          (void *)calloc(m*n,netsolve_sizeof(data_type));
      }
      ptr = obj->attributes.matrix_attributes.ptr;

      /* Read the data */
      if (obj->attributes.matrix_attributes.l == m)
      {
        if (read(fd,ptr,netsolve_sizeof(data_type)*m*n) !=
                  netsolve_sizeof(data_type)*m*n)
        {
          ns_errno = NetSolveFileError;
          return -1;
        }
      }
      else if (obj->attributes.matrix_attributes.l == -1)
      {
        if (read(fd,ptr,netsolve_sizeof(data_type)*m*n) !=
                  netsolve_sizeof(data_type)*m*n)
        {
          ns_errno = NetSolveFileError;
          return -1;
        }
        obj->attributes.matrix_attributes.l = m;
      }
      else
      {
        for (k=0;k<n;k++)
          if (read(fd,
              (char *)ptr+k*obj->attributes.matrix_attributes.l*
              netsolve_sizeof(data_type),m*netsolve_sizeof(data_type)) !=
                    m*netsolve_sizeof(data_type))
          {
            ns_errno = NetSolveFileError;
            return -1;
          }
      }
      break;
    }
    case NETSOLVE_SPARSEMATRIX:
    {
      int m,n,major,f,*rc_ptr,*rc_index;
      void *ptr;
 
      if (read(fd,&major,sizeof(int)) != sizeof(int))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
      if (read(fd,&m,sizeof(int)) != sizeof(int))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
      if (read(fd,&n,sizeof(int)) != sizeof(int))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
      if (read(fd,&f,sizeof(int)) != sizeof(int))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
 
      if (obj->attributes.sparsematrix_attributes.major != -1)
      {
        if (obj->attributes.sparsematrix_attributes.major != major)
        {
          ns_errno = NetSolveMismatch;
          return -1;
        }
      }
      obj->attributes.sparsematrix_attributes.major = major;

      if (obj->attributes.sparsematrix_attributes.m != -1)
      {
        if (obj->attributes.sparsematrix_attributes.m != m)
        {
          ns_errno = NetSolveMismatch;
          return -1;
        }
      }
      obj->attributes.sparsematrix_attributes.m = m;

      if (obj->attributes.sparsematrix_attributes.n != -1)
      {
        if (obj->attributes.sparsematrix_attributes.n != n)
        {
          ns_errno = NetSolveMismatch;
          return -1;
        }
      }
      obj->attributes.sparsematrix_attributes.n = n;

      if (obj->attributes.sparsematrix_attributes.f != -1)
      {
        if (obj->attributes.sparsematrix_attributes.f != f)
        {
          ns_errno = NetSolveMismatch;
          return -1;
        }
      }
      obj->attributes.sparsematrix_attributes.f = f;
 
    /*allocate for the indices and data*/
      if (obj->attributes.sparsematrix_attributes.ptr == NULL)
      {
        obj->attributes.sparsematrix_attributes.ptr =
          (void *)calloc(f,netsolve_sizeof(data_type));
      }
      ptr = obj->attributes.sparsematrix_attributes.ptr;
      if (obj->attributes.sparsematrix_attributes.rc_ptr == NULL)
      {
        obj->attributes.sparsematrix_attributes.rc_ptr =
          (void *)calloc(m+1,netsolve_sizeof(NETSOLVE_I));
      }
      rc_ptr = obj->attributes.sparsematrix_attributes.rc_ptr;
      if (obj->attributes.sparsematrix_attributes.rc_index == NULL)
      {
        obj->attributes.sparsematrix_attributes.rc_index =
          (void *)calloc(f,netsolve_sizeof(NETSOLVE_I));
      }
      rc_index = obj->attributes.sparsematrix_attributes.rc_index;

 
      /* Read the indices and data */
      if (read(fd,rc_ptr,netsolve_sizeof(NETSOLVE_I)*(m+1)) !=
                netsolve_sizeof(NETSOLVE_I)*(m+1))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
      if (read(fd,rc_index,netsolve_sizeof(NETSOLVE_I)*f) !=
                netsolve_sizeof(NETSOLVE_I)*f)
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
      if (read(fd,ptr,netsolve_sizeof(data_type)*f) !=
                netsolve_sizeof(data_type)*f)
      {
        ns_errno = NetSolveFileError;
        return -1;
      }

      break;
    }
    case NETSOLVE_VECTOR:
    {
      void *ptr;
      int m;

      if (read(fd,&m,sizeof(int)) != sizeof(int))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
      if (obj->attributes.vector_attributes.m != -1)
      {
        if (obj->attributes.vector_attributes.m != m)
        {
          ns_errno = NetSolveMismatch;
          return -1;
        }
      }
      obj->attributes.vector_attributes.m = m;

      if (obj->attributes.vector_attributes.ptr == NULL)
      {
        /* Malloc the data space */
        obj->attributes.vector_attributes.ptr =
          (void *)calloc(m,netsolve_sizeof(data_type));
      }
      ptr = obj->attributes.vector_attributes.ptr;

      if (read(fd,ptr,m*netsolve_sizeof(data_type)) != 
               m*netsolve_sizeof(data_type))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
      break;
    }
    case NETSOLVE_SCALAR:
    {
      void *ptr;
      if (obj->attributes.scalar_attributes.ptr == NULL)
      {
        /* Malloc the data space */
        obj->attributes.scalar_attributes.ptr =
          (void *)calloc(1,netsolve_sizeof(data_type));
      }
      ptr = obj->attributes.scalar_attributes.ptr;
 
      if (read(fd,ptr,1*netsolve_sizeof(data_type)) != 
                 1*netsolve_sizeof(data_type))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
      break;
    }
    case NETSOLVE_FILE:
    {
      int len;
      char *s;

      if (read(fd,&len,sizeof(int)) != sizeof(int))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
      s = (char *)calloc(len+1,sizeof(char));
      if (read(fd,s,len*sizeof(char)) != len*sizeof(char))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
      if (obj->attributes.file_attributes.filename != NULL)
      {
        if (strcmp(s,obj->attributes.file_attributes.filename))
        {
          free(s);
          ns_errno = NetSolveMismatch;
          return -1;
        }
        free(s);
      }
      else
        obj->attributes.file_attributes.filename = s;
      break;
    }
    case NETSOLVE_PACKEDFILES:
    {
      int nb_files;
      int len,i;
      char *s;

      if (read(fd,&nb_files,sizeof(int)) != sizeof(int))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }

      if (obj->attributes.packedfiles_attributes.m != -1) 
      {
        if (obj->attributes.packedfiles_attributes.m != nb_files)
        {
          ns_errno = NetSolveMismatch;
          return -1;
        }
      }
      obj->attributes.packedfiles_attributes.m = nb_files;

      obj->attributes.packedfiles_attributes.defaultprefix = NULL;

      obj->attributes.packedfiles_attributes.filenames =
            (char **)calloc(nb_files,sizeof(char*));

      for (i=0;i<nb_files;i++)
      {
        if (read(fd,&len,sizeof(int)) != sizeof(int))
        {
          ns_errno = NetSolveFileError;
          return -1;
        }
        s = (char *)calloc(len+1,sizeof(char));
        if (read(fd,s,len*sizeof(char)) != len*sizeof(char))
        {
          ns_errno = NetSolveFileError;
          return -1;
        }
        obj->attributes.packedfiles_attributes.filenames[i] = s;
      }
      break;
    }
    case NETSOLVE_STRINGLIST:
    {
      int nb_strings;
      int len,i;
      char *s;

      if (read(fd,&nb_strings,sizeof(int)) != sizeof(int))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }

      if (obj->attributes.stringlist_attributes.m != -1)
      {
        if (obj->attributes.stringlist_attributes.m != nb_strings)
        {
          ns_errno = NetSolveMismatch;
          return -1;
        }
      }
      obj->attributes.stringlist_attributes.m = nb_strings;

      obj->attributes.stringlist_attributes.strings =
            (char **)calloc(nb_strings,sizeof(char*));

      for (i=0;i<nb_strings;i++)
      {
        if (read(fd,&len,sizeof(int)) != sizeof(int))
        {
          ns_errno = NetSolveFileError;
          return -1;
        }
        s = (char *)calloc(len+1,sizeof(char));
        if (read(fd,s,len*sizeof(char)) != len*sizeof(char))
        {
          ns_errno = NetSolveFileError;
          return -1;
        }
        obj->attributes.stringlist_attributes.strings[i] = s;
      }
      break;
    }
    case NETSOLVE_UPF:
    {
      int language;
      int len; 
      char *s;

      if (read(fd,&len,sizeof(int)) != sizeof(int))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
      s = (char *)calloc(1+len,sizeof(char));
      if (read(fd,s,len*sizeof(char)) != len*sizeof(char))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
      if (obj->attributes.upf_attributes.filename != NULL)
      {
        if (strcmp(s,obj->attributes.upf_attributes.filename))
        {
          free(s);
          ns_errno = NetSolveMismatch;
          return -1;
        }
        free(s);
      }
      else
        obj->attributes.upf_attributes.filename = s;

    
      if (read(fd,&language,sizeof(int)) != sizeof(int))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
      if (obj->attributes.upf_attributes.language != -1)
      {
        if (obj->attributes.upf_attributes.language != language)
        {
          ns_errno = NetSolveMismatch;
          return -1;
        }
      }
      obj->attributes.upf_attributes.language = language;
 
      if (read(fd,&len,sizeof(int)) != sizeof(int))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
      s = (char *)calloc(len+1,sizeof(char));
      if (read(fd,s,len*sizeof(char)) != len*sizeof(char))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
      if (obj->attributes.upf_attributes.funcname != NULL)
      {
        if (strcmp(s,obj->attributes.upf_attributes.funcname))
        {
          free(s);
          ns_errno = NetSolveMismatch;
          return -1;
        }
        free(s);
      }
      else
        obj->attributes.upf_attributes.funcname = s;
      break;
    }
    case NETSOLVE_STRING:
    {
      int len;
      if (read(fd,&len,sizeof(int)) != sizeof(int))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
      obj->attributes.string_attributes.ptr = (char*)calloc(len,sizeof(char)); 
      if (read(fd, obj->attributes.string_attributes.ptr,len*sizeof(char)) != 
              len*sizeof(char))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
      break;

    }
    default:
#ifdef VIEW
      fprintf(STDERR "Unknown object type\n");
#endif
      ns_errno = NetSolveInternalError;
      return -1;
  }
  return 1;
}


/*
 * recvObjectToFile()
 *
 * If the object is a file or a UPF, then the filename should
 * be preset.
 */
int recvObjectToFile(NS_Communicator *comm,int fd,NS_Object *obj, void *misc)
{
  int data_type;
  int object_type;

  if (obj == NULL)
  {
    ns_errno = NetSolveInternalError;
    return -1;
  }

  if (recvInt(comm,&object_type) == -1)
    return -1;
  if (recvInt(comm,&data_type) == -1)
    return -1;
  if (recvInt(comm,&(obj->distributed)) == -1)
    return -1;

  if(obj->distributed == 1){
    obj->dsi_obj = (DSI_OBJECT *)calloc(1, sizeof(DSI_OBJECT));
    obj->dsi_obj->dsi_file = (DSI_FILE *)calloc(1, sizeof(DSI_FILE));
    if(recvDsiObj(comm, obj->dsi_obj) == -1)
      return -1;
  }

  if (recvInt(comm,&obj->pass_back) == -1)
    return -1;

  if (object_type != obj->object_type)
  {
    ns_errno = NetSolveMismatch;
    return -1;
  }
  if (data_type != obj->data_type)
  {
    ns_errno = NetSolveMismatch;
    return -1;
  }

  if (write(fd,&object_type,sizeof(int)) != sizeof(int))
  {
    ns_errno = NetSolveFileError;
    return -1;
  }
  if (write(fd,&data_type,sizeof(int)) != sizeof(int))
  {
    ns_errno = NetSolveFileError;
    return -1;
  }
  if (write(fd,&obj->pass_back,sizeof(int)) != sizeof(int))
  {
    ns_errno = NetSolveFileError;
    return -1;
  }

  switch(object_type)
  {
    case NETSOLVE_MATRIX:
    {
      void* ptr;
      int major,m,n;
      if (recvInt(comm,&major) == -1)
        return -1;
      if (obj->attributes.matrix_attributes.major != -1)
      {
        if (major != obj->attributes.matrix_attributes.major)
        {
          ns_errno = NetSolveMismatch;
          return -1;
        }
      }
      obj->attributes.matrix_attributes.major = major;
      if (recvInt(comm,&m) == -1)
        return -1;
      if (obj->attributes.matrix_attributes.m != -1)
      {
        if (m != obj->attributes.matrix_attributes.m)
        {
          ns_errno = NetSolveMismatch;
          return -1;
        }
      }
      obj->attributes.matrix_attributes.m = m;
      if (recvInt(comm,&n) == -1)
        return -1;
      if (obj->attributes.matrix_attributes.n != -1)
      {
        if (n != obj->attributes.matrix_attributes.n)
        {
          ns_errno = NetSolveMismatch;
          return -1;
        }
      }
      obj->attributes.matrix_attributes.n = n;

      if (write(fd,&major,sizeof(int)) != sizeof(int))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
      if (write(fd,&m,sizeof(int)) != sizeof(int))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
      if (write(fd,&n,sizeof(int)) != sizeof(int))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }

      if(obj->distributed == 1){
        ptr = (void *)calloc(m*n,netsolve_sizeof(data_type));
        ns_dsi_read_matrix(obj->dsi_obj, ptr, m, n, data_type);
        if (write(fd,ptr,netsolve_sizeof(data_type)*m*n) !=
                  netsolve_sizeof(data_type)*m*n)
        {
          ns_errno = NetSolveFileError;
          return -1;
        }
      }
      else{
        if (recvArrayToFile(comm,data_type,fd,m*n) == -1)
          return -1;
      }
      break;
    }
    case NETSOLVE_SPARSEMATRIX:
    {
      int major,m,n,f;

      if (recvInt(comm,&major) == -1)
        return -1;
      if (obj->attributes.sparsematrix_attributes.major != -1)
      {
        if (major != obj->attributes.sparsematrix_attributes.major)
        {
          ns_errno = NetSolveMismatch;
          return -1;
        }
      }
      obj->attributes.sparsematrix_attributes.major = major;

      if (recvInt(comm,&m) == -1)
        return -1;
      if (obj->attributes.sparsematrix_attributes.m != -1)
      {
        if (m != obj->attributes.sparsematrix_attributes.m)
        {
          ns_errno = NetSolveMismatch;
          return -1;
        }
      }
      obj->attributes.sparsematrix_attributes.m = m;

      if (recvInt(comm,&n) == -1)
        return -1;
      if (obj->attributes.sparsematrix_attributes.n != -1)
      {
        if (n != obj->attributes.sparsematrix_attributes.n)
        {
          ns_errno = NetSolveMismatch;
          return -1;
        }
      }
      obj->attributes.sparsematrix_attributes.n = n;

      if (recvInt(comm,&f) == -1)
        return -1;
      if (obj->attributes.sparsematrix_attributes.f != -1)
      {
        if (f != obj->attributes.sparsematrix_attributes.f)
        {
          ns_errno = NetSolveMismatch;
          return -1;
        }
      }
      obj->attributes.sparsematrix_attributes.f = f;

      if (write(fd,&major,sizeof(int)) != sizeof(int))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
      if (write(fd,&m,sizeof(int)) != sizeof(int))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
      if (write(fd,&n,sizeof(int)) != sizeof(int))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
      if (write(fd,&f,sizeof(int)) != sizeof(int))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
      if (recvArrayToFile(comm,NETSOLVE_I,fd,m+1) == -1)
        return -1;
      if (recvArrayToFile(comm,NETSOLVE_I,fd,f) == -1)
        return -1;

      if (recvArrayToFile(comm,data_type,fd,f) == -1)
        return -1;
      break;
    }

    case NETSOLVE_VECTOR:
    {
      void* ptr;
      int m;
      if (recvInt(comm,&m) == -1)
        return -1;
      if (obj->attributes.vector_attributes.m != -1)
      {
        if (m != obj->attributes.vector_attributes.m)
        {
          ns_errno = NetSolveMismatch;
          return -1;
        }
      }
      obj->attributes.vector_attributes.m = m;

      if (write(fd,&m,sizeof(int)) != sizeof(int))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
      if(obj->distributed == 1){
        ptr = (void *)calloc(m,netsolve_sizeof(data_type));
        ns_dsi_read_vector(obj->dsi_obj, ptr, m, data_type);
        if (write(fd,ptr,m*netsolve_sizeof(data_type)) !=
               m*netsolve_sizeof(data_type))
        {
          ns_errno = NetSolveFileError;
          return -1;
        }
      }
      else{
        if (recvArrayToFile(comm,data_type,fd,m) == -1)
          return -1;
      }
      break;
    }
    case NETSOLVE_SCALAR:
    {
      if (recvArrayToFile(comm,data_type,fd,1) == -1)
        return -1;
      break;
    }
    case NETSOLVE_FILE:
    {
      int size;
      int filed;
      char buffer[256];
      int namelen;

      sprintf(buffer,"%s-file",(char*)misc);
      obj->attributes.file_attributes.filename = strdup(buffer);

      if (recvInt(comm,&size) == -1)
        return -1;
      filed = open(obj->attributes.file_attributes.filename,O_CREAT|O_WRONLY,0666);
      if (filed < 0)
      {
#ifdef VIEW
        fprintf(STDERR "Impossible to create file '%s'\n",
                obj->attributes.file_attributes.filename);
#endif
        ns_errno = NetSolveFileError;
        return -1;
      }
      if (recvArrayToFile(comm,NETSOLVE_CHAR,filed,size) == -1)
      {
        close(filed);
        return -1;
      }
      close(filed);
      namelen = strlen(obj->attributes.file_attributes.filename)+1;
      if (write(fd,&namelen,sizeof(int)) != sizeof(int))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
      if (write(fd,obj->attributes.file_attributes.filename,
                namelen*sizeof(char)) != namelen*sizeof(char))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
      break;
    }
    case NETSOLVE_PACKEDFILES:
    {
      int size;
      int filed;
      char buffer[256];
      int namelen,i;
      int nb_files;
      char *original_filename;

      if (recvInt(comm,&nb_files) == -1)
        return -1;

      if (write(fd,&nb_files,sizeof(int)) != sizeof(int))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }

      obj->attributes.packedfiles_attributes.m = nb_files;
      obj->attributes.packedfiles_attributes.defaultprefix = NULL;
      obj->attributes.packedfiles_attributes.filenames = (char **)
            calloc(nb_files,sizeof(char*));
      for (i=0;i<nb_files;i++)
      {
        if (recvString(comm,&original_filename) == -1)
          return -1;
        free(original_filename);

        sprintf(buffer,"%s-file%d",(char*)misc,i);
        obj->attributes.packedfiles_attributes.filenames[i] = strdup(buffer);

        if (recvInt(comm,&size) == -1)
          return -1;
        filed = open(obj->attributes.packedfiles_attributes.filenames[i],
                O_CREAT|O_WRONLY,0666);
        if (filed < 0)
        {
#ifdef VIEW
          fprintf(STDERR "Impossible to create file '%s'\n",
                  obj->attributes.packedfiles_attributes.filenames[i]);
#endif
          ns_errno = NetSolveFileError;
          return -1;
        }
        if (recvArrayToFile(comm,NETSOLVE_CHAR,filed,size) == -1)
        {
          close(filed);
          return -1;
        }
        close(filed);
        namelen = strlen(obj->attributes.packedfiles_attributes.filenames[i])+1;        if (write(fd,&namelen,sizeof(int)) != sizeof(int))
        {
          ns_errno = NetSolveFileError;
          return -1;
        }
        if (write(fd,obj->attributes.packedfiles_attributes.filenames[i],
                  namelen*sizeof(char)) != namelen*sizeof(char))
        {
          ns_errno = NetSolveFileError;
          return -1;
        }
        break;
      }
    }
    case NETSOLVE_STRINGLIST:
    {
      int namelen,i;
      char *tmp;
      int nb_strings;

      if (recvInt(comm,&nb_strings) == -1)
        return -1;

      if (write(fd,&nb_strings,sizeof(int)) != sizeof(int))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }

      if (obj->attributes.stringlist_attributes.m != -1)
      {
        if (obj->attributes.stringlist_attributes.m != nb_strings)
        {
          ns_errno = NetSolveMismatch;
          return -1;
        }
      }
      obj->attributes.stringlist_attributes.m = nb_strings;
      for (i=0;i<nb_strings;i++)
      {
        if (recvString(comm,&tmp) == -1)
          return -1;
        namelen = strlen(tmp)+1;
        if (write(fd,&namelen,sizeof(int)) != sizeof(int))
        {
          ns_errno = NetSolveFileError;
          return -1;
        }
        if (write(fd,tmp,namelen*sizeof(char)) != namelen*sizeof(char))
        {
          ns_errno = NetSolveFileError;
          return -1;
        }
        free(tmp);
      }
      break;
    }
    case NETSOLVE_UPF:
    {
      int size;
      int filed;
      char *fcnname;
      int fcnlen;
      int language;
      int namelen;
      char buffer[256];

      if (recvInt(comm,&language) == -1)
        return -1;
      if (recvString(comm,&fcnname) == -1)
        return -1;

      sprintf(buffer,"%s-upf",(char*)misc);
      obj->attributes.upf_attributes.filename = strdup(buffer);
      if (obj->attributes.upf_attributes.language != -1)
      {
        if (language != obj->attributes.upf_attributes.language)
        {
          ns_errno = NetSolveMismatch;
          return -1;
        }
      }
      obj->attributes.upf_attributes.language = language;
      if (obj->attributes.upf_attributes.funcname != NULL)
      {
        if (strcmp(fcnname,obj->attributes.upf_attributes.funcname))
        {
          ns_errno = NetSolveMismatch;
          return -1;
        }
      }
      obj->attributes.upf_attributes.funcname = strdup(fcnname);

      filed = open(obj->attributes.upf_attributes.filename,O_CREAT|O_WRONLY,0666);
      if (filed < 0)
      {
#ifdef VIEW
        fprintf(STDERR "Impossible to create file '%s'\n",
                obj->attributes.upf_attributes.filename);
#endif
        ns_errno = NetSolveFileError;
        return -1;
      }
      if (recvInt(comm,&size) == -1)
        return -1;
      if (recvArrayToFile(comm,NETSOLVE_CHAR,filed,size) == -1)
      {
        close(filed);
        return -1;
      }
      close(filed);
      namelen = strlen(obj->attributes.upf_attributes.filename)+1;
      if (write(fd,&namelen,sizeof(int)) != sizeof(int))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
      if (write(fd,obj->attributes.upf_attributes.filename,
        namelen*sizeof(char)) != namelen*sizeof(char))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
      if (write(fd,&language,sizeof(int)) != sizeof(int))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
      fcnlen = strlen(fcnname)+1;
      if (write(fd,&fcnlen,sizeof(int)) != sizeof(int))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
      if (write(fd,fcnname,fcnlen*sizeof(char)) != fcnlen*sizeof(char))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
      free(fcnname);
      break;
    }
    case NETSOLVE_STRING:
    {
      int len;
      char *s;
      if (recvString(comm,&s) == -1)
        return -1;
      len = strlen(s)+1;
      if (write(fd,&len,sizeof(int)) != sizeof(int))
      {
        free(s);
        ns_errno = NetSolveFileError;
        return -1;
      }
      if (write(fd,s,len*sizeof(char)) != len*sizeof(char))
      {
        free(s);
        ns_errno = NetSolveFileError;
        return -1;
      }
      free(s);
      break;
    }
    default:
#ifdef VIEW
      fprintf(STDERR "Unknown object type\n");
#endif
      ns_errno = NetSolveInternalError;
      return -1;
  }
  return 1;
}

/*
 * getObjectByteSize()
 *
 * the amount of data to be sent over the net
 */
int getObjectByteSize(NS_Object *obj)
{
  int size;
  struct stat st;
  int i;

  if (obj == NULL)
  {
    ns_errno = NetSolveInternalError;
    return -1;
  }

  size = 0;
  switch(obj->object_type)
  {
    case NETSOLVE_MATRIX:
      size += 5*sizeof(int);
      if (obj->attributes.matrix_attributes.m == -1)
        size += 0;
      else if (obj->attributes.matrix_attributes.n == -1)
        size += 0;
      else
        size += obj->attributes.matrix_attributes.m *
               obj->attributes.matrix_attributes.n *
               netsolve_sizeof(obj->data_type);
      break;
    case NETSOLVE_SPARSEMATRIX:
      size += 5*sizeof(int);
      if (obj->attributes.sparsematrix_attributes.m == -1)
        size += 0;
      else if (obj->attributes.sparsematrix_attributes.n == -1)
        size += 0;
      else
        size += obj->attributes.sparsematrix_attributes.f *
               netsolve_sizeof(obj->data_type);
     size += obj->attributes.sparsematrix_attributes.f *
             netsolve_sizeof(NETSOLVE_I); 
     size += (obj->attributes.sparsematrix_attributes.m + 1) *
             netsolve_sizeof(NETSOLVE_I); 
      break;
    case NETSOLVE_VECTOR:
      size +=3*sizeof(int);
      if (obj->attributes.vector_attributes.m == -1)
        size += 0;
      else
        size += obj->attributes.vector_attributes.m *
               netsolve_sizeof(obj->data_type);
      break;
    case NETSOLVE_SCALAR:
      size +=2*sizeof(int);
      size += netsolve_sizeof(obj->data_type);
      break;
    case NETSOLVE_FILE:
      size += 3*sizeof(int);
      if (stat(obj->attributes.file_attributes.filename,&st))
        size += 0;
      else 
        size += st.st_size*netsolve_sizeof(NETSOLVE_CHAR);
      break;
    case NETSOLVE_PACKEDFILES:
      size += 3*sizeof(int);
      if (obj->attributes.packedfiles_attributes.m == -1)
      {
        size = 1;
        break;
      }
      if (obj->attributes.packedfiles_attributes.defaultprefix != NULL)
      {
        size = 1;
        break;
      }
      for (i=0;i<obj->attributes.packedfiles_attributes.m;i++)
      {
        size += sizeof(int);
        if (stat(obj->attributes.packedfiles_attributes.filenames[i],&st))
          size += 0;
        else
          size += st.st_size*netsolve_sizeof(NETSOLVE_CHAR);
      }
      break;
    case NETSOLVE_STRINGLIST:
      size += 3*sizeof(int);
      for (i=0;i<obj->attributes.stringlist_attributes.m;i++)
      {
        if (obj->attributes.stringlist_attributes.strings[i] != NULL)
          size += strlen(obj->attributes.stringlist_attributes.strings[i]) +1;
      }
      break;
    case NETSOLVE_UPF:
      size += 6*sizeof(int);
      size += strlen(obj->attributes.upf_attributes.funcname) *
              netsolve_sizeof(NETSOLVE_CHAR);
      if (stat(obj->attributes.upf_attributes.filename,&st))
        size += 0;
      else
        size += st.st_size*netsolve_sizeof(NETSOLVE_CHAR);
      break;
    case NETSOLVE_STRING:
      size += 3*sizeof(int);
      if (obj->attributes.string_attributes.ptr != NULL)
        size += strlen(obj->attributes.string_attributes.ptr) *
                netsolve_sizeof(NETSOLVE_CHAR);
      else
        size += 0;
      break;
    default:
#ifdef VIEW
      fprintf(STDERR "Unknown object type\n");
#endif
      ns_errno = NetSolveInternalError;
      size = -1;
      break;
  }
  return size; 
}

/*
 * sendObjectFromFile()
 */
int sendObjectFromFile(NS_Communicator *comm,int fd,NS_Object  *obj)
{
  int data_type;
  int object_type;
  int pass_back;

  if (read(fd,&object_type,sizeof(int)) == -1)
  {
    ns_errno = NetSolveFileError;
    return -1;
  }
  if (read(fd,&data_type,sizeof(int)) == -1)
  {
    ns_errno = NetSolveFileError;
    return -1;
  }

  if (read(fd,&pass_back,sizeof(int)) == -1)
  {
    ns_errno = NetSolveFileError;
    return -1;
  }

  if (object_type != obj->object_type)
  {
    ns_errno = NetSolveMismatch;
    return -1;
  }
  if (data_type != obj->data_type)
  {
    ns_errno = NetSolveMismatch;
    return -1;
  }

  if (sendInt(comm,object_type)  == -1)
    return -1;
  if (sendInt(comm,data_type)  == -1)
    return -1;
  obj->distributed = 0;
  if (sendInt(comm,obj->distributed) == -1)
    return -1;

  if (sendInt(comm,pass_back)  == -1)
    return -1;

  switch(object_type)
  {
    case NETSOLVE_MATRIX:
    {
      int major,m,n;
      if (read(fd,&major,sizeof(int)) == -1)
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
      if (read(fd,&m,sizeof(int)) == -1)
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
      if (read(fd,&n,sizeof(int)) == -1)
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
      if (obj->attributes.matrix_attributes.major != -1)
      {
        if (major != obj->attributes.matrix_attributes.major)
        {
          ns_errno = NetSolveMismatch;
          return -1;
        }
      }
      obj->attributes.matrix_attributes.major = major;
      if (obj->attributes.matrix_attributes.m != -1)
      {
        if (m != obj->attributes.matrix_attributes.m)
        {
          ns_errno = NetSolveMismatch;
          return -1;
        }
      }
      obj->attributes.matrix_attributes.m = m;
      if (obj->attributes.matrix_attributes.n != -1)
      {
        if (n != obj->attributes.matrix_attributes.n)
        {
          ns_errno = NetSolveMismatch;
          return -1;
        }
      }
      obj->attributes.matrix_attributes.n = n;
      if (sendInt(comm,major) == -1)
        return -1;
      if (sendInt(comm,m) == -1)
        return -1;
      if (sendInt(comm,n) == -1)
        return -1;
      if (sendArrayFromFile(comm,data_type,fd,m*n) == -1)
        return -1;
      break;    
    }
    case NETSOLVE_SPARSEMATRIX:
    {
      int major,m,n,f;
      if (read(fd,&major,sizeof(int)) == -1)
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
      if (read(fd,&m,sizeof(int)) == -1)
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
      if (read(fd,&n,sizeof(int)) == -1)
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
      if (read(fd,&f,sizeof(int)) == -1)
      {
        ns_errno = NetSolveFileError;
        return -1;
      }

      if (obj->attributes.sparsematrix_attributes.major != -1)
      {
        if (major != obj->attributes.sparsematrix_attributes.major)
        {
          ns_errno = NetSolveMismatch;
          return -1;
        }
      }
      obj->attributes.sparsematrix_attributes.major = major;

      if (obj->attributes.sparsematrix_attributes.m != -1)
      {
        if (m != obj->attributes.sparsematrix_attributes.m)
        {
          ns_errno = NetSolveMismatch;
          return -1;
        }
      }
      obj->attributes.sparsematrix_attributes.m = m;

      if (obj->attributes.sparsematrix_attributes.n != -1)
      {
        if (n != obj->attributes.sparsematrix_attributes.n)
        {
          ns_errno = NetSolveMismatch;
          return -1;
        }
      }
      obj->attributes.sparsematrix_attributes.n = n;

      if (obj->attributes.sparsematrix_attributes.f != -1)
      {
        if (f != obj->attributes.sparsematrix_attributes.f)
        {
          ns_errno = NetSolveMismatch;
          return -1;
        }
      }
      obj->attributes.sparsematrix_attributes.f = f;

      if (sendInt(comm,major) == -1)
        return -1;
      if (sendInt(comm,m) == -1)
        return -1;
      if (sendInt(comm,n) == -1)
        return -1;
      if (sendInt(comm,f) == -1)
        return -1;

      if (sendArrayFromFile(comm,NETSOLVE_I,fd,m+1) == -1)
        return -1;
      if (sendArrayFromFile(comm,NETSOLVE_I,fd,f) == -1)
        return -1;
      if (sendArrayFromFile(comm,data_type,fd,f) == -1)
        return -1;
      break;    
    }
    
    case NETSOLVE_VECTOR:
    {
      int m;
      if (read(fd,&m,sizeof(int)) != sizeof(int))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
      if (obj->attributes.vector_attributes.m != -1)
      {
        if (m != obj->attributes.vector_attributes.m)
        {
          ns_errno = NetSolveMismatch;
          return -1;
        }
      }
      obj->attributes.vector_attributes.m = m;

      if (sendInt(comm,m) == -1)
        return -1;
      if (sendArrayFromFile(comm,data_type,fd,m) == -1)
        return -1;
      break;
    }
    case NETSOLVE_SCALAR:
    {
      if (sendArrayFromFile(comm,data_type,fd,1) == -1)
        return -1;
      break;
    }
    case NETSOLVE_FILE:
    {
      int size;
      int filed;
      char buffer[256];
      int namelen;
      struct stat st;

      /* Get the filename */
      if (read(fd,&namelen,sizeof(int)) != sizeof(int))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
      if (read(fd,buffer,namelen*sizeof(char)) != namelen*sizeof(char))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
      if (stat(buffer,&st))
      {
#ifdef VIEW
        fprintf(STDERR "Imposible to open file '%s'\n",buffer);
#endif
        ns_errno = NetSolveFileError;
        return -1;
      }
      size = st.st_size;

      if (sendInt(comm,size) == -1)
        return -1;

      filed = open(buffer,O_RDONLY,0666);
      if (filed < 0)
      {
#ifdef VIEW
        fprintf(STDERR "Impossible to open file '%s'\n",buffer);
#endif
        ns_errno = NetSolveFileError;
        return -1;
      } 
      if (sendArrayFromFile(comm,NETSOLVE_CHAR,filed,size) == -1)
      {
        close(filed);
        return -1;
      }
      close(filed);
      break;
    }
    case NETSOLVE_PACKEDFILES:
    {
      int size;
      int filed;
      char buffer[256];
      int namelen;
      struct stat st;
      int nb_files,i;

      if (read(fd,&nb_files,sizeof(int)) != sizeof(int))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
    
      if (sendInt(comm,nb_files) == -1)
        return -1;

      for (i=0;i<nb_files;i++)
      {
        /* Get the filename */
        if (read(fd,&namelen,sizeof(int)) != sizeof(int))
        {
          ns_errno = NetSolveFileError;
          return -1;
        }
        if (read(fd,buffer,namelen*sizeof(char)) != namelen*sizeof(char))
        {
          ns_errno = NetSolveFileError;
          return -1;
        }

        if (sendString(comm,lastInPath(buffer)) == -1)
          return -1;

        if (stat(buffer,&st))
        {
#ifdef VIEW
          fprintf(STDERR "Imposible to open file '%s'\n",buffer);
#endif
          ns_errno = NetSolveFileError;
          return -1;
        }
        size = st.st_size;
  
        if (sendInt(comm,size) == -1)
          return -1;
  
        filed = open(buffer,O_RDONLY,0666);
        if (filed < 0)
        {
#ifdef VIEW
          fprintf(STDERR "Impossible to open file '%s'\n",buffer);
#endif
          ns_errno = NetSolveFileError;
          return -1;
        }
        if (sendArrayFromFile(comm,NETSOLVE_CHAR,filed,size) == -1)
        {
          close(filed);
          return -1;
        }
        close(filed);
      }
      break;
    }
    case NETSOLVE_STRINGLIST:
    {
      char *buffer;
      int namelen;
      int nb_strings,i;

      if (read(fd,&nb_strings,sizeof(int)) != sizeof(int))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
      
      if (sendInt(comm,nb_strings) == -1)
        return -1;

      for (i=0;i<nb_strings;i++)
      {
        /* Get the strings */
        if (read(fd,&namelen,sizeof(int)) != sizeof(int))
        {
          ns_errno = NetSolveFileError;
          return -1;
        }
        buffer = (char *)calloc(namelen,sizeof(char));
        if (read(fd,buffer,namelen*sizeof(char)) != namelen*sizeof(char))
        {
          ns_errno = NetSolveFileError;
          return -1;
        }

        if (sendString(comm,buffer) == -1)
          return -1;

        free(buffer);
      }
      break;
    }
    case NETSOLVE_UPF:
    {
      int size;
      int filed;
      char buffer[256];
      char *fcnname;
      int fcnlen;
      int language;
      int namelen;
      struct stat st;

      /* read the filename */
      if (read(fd,&namelen,sizeof(int)) != sizeof(int))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
      if (read(fd,buffer,namelen*sizeof(char)) != namelen*sizeof(char))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }

      if (read(fd,&language,sizeof(int)) != sizeof(int))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
      if (obj->attributes.upf_attributes.language != -1)
      {
        if (language != obj->attributes.upf_attributes.language)
        {
          ns_errno = NetSolveMismatch;
          return -1;
        }
      }
      obj->attributes.upf_attributes.language = language;

      if (read(fd,&fcnlen,sizeof(int)) != sizeof(int))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
      fcnname = (char *)calloc(fcnlen,sizeof(char));
      if (read(fd,fcnname,fcnlen*sizeof(char)) != fcnlen*sizeof(char))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
      if (obj->attributes.upf_attributes.funcname != NULL)
      {
        if (strcmp(fcnname,obj->attributes.upf_attributes.funcname))
        {
          ns_errno = NetSolveMismatch;
          return -1;
        }
      }
      obj->attributes.upf_attributes.funcname = strdup(fcnname);  


      if (stat(buffer,&st))
      {
#ifdef VIEW
        fprintf(STDERR "Impossible to open file '%s'\n",buffer);
#endif
        free(fcnname);
        ns_errno = NetSolveFileError;
        return -1;
      }
      size = st.st_size;

      if (sendInt(comm,language) == -1)
      {
        free(fcnname);
        return -1;
      }
      if (sendString(comm,fcnname) == -1)
      {
        free(fcnname);
        return -1;
      }
      free(fcnname);

      filed = open(buffer,O_RDONLY,0666);
      if (filed < 0)
      {
#ifdef VIEW
        fprintf(STDERR "Impossible to open file '%s'\n",buffer);
#endif
        ns_errno = NetSolveFileError;
        return -1;
      }
      if (sendInt(comm,size) == -1)
        return -1;
      if (sendArrayFromFile(comm,NETSOLVE_CHAR,filed,size) == -1)
      {
        close(filed);
        return -1;
      }
      close(filed);
      break;
    }
    case NETSOLVE_STRING:
    {
      int len;
      char *s;
      if (read(fd,&len,sizeof(int)) != sizeof(int))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
      s = (char*)calloc(len,sizeof(char));
      if (read(fd,s,len*sizeof(char)) != len*sizeof(char))
      {
        free(s);
        ns_errno = NetSolveFileError;
        return -1;
      }
      if (sendString(comm,s) == -1)
      {
        free(s);
        return -1;
      }
      free(s);
      break;
    }
    default:
#ifdef VIEW
      fprintf(STDERR "Unknown object type\n");
#endif
      ns_errno = NetSolveInternalError;
      return -1;
  }
  return 1;
}

/*
 * sendOutputObjectsFromFiles()
 *
 * receives all the objects (data) to files in the working
 * directory. The files are names input0,input1,...
 * if the input is a file or a upf, then the file
 * inputx countains no real data, just the name 
 * of the file containing the data (fileinputx or upfinputx)
 */
int sendOutputObjectsFromFiles(NS_Communicator *comm,NS_ProblemDesc *pd)
{
  int i;
  NS_Object *obj;
  int fd;
  char buffer[256];

  for (i=0;i<pd->nb_output_objects;i++)
  {
    if (pd->output_objects[i]->pass_back == FALSE) continue; /* do i really have to pass the object */
    if (pd->output_names == NULL) sprintf(buffer,"output%d",i);
    else sprintf(buffer,"%s",pd->output_names[i]);

    fd = open(buffer,O_RDONLY,0666);
    if (fd < 0)
    {
#ifdef VIEW
      fprintf(STDERR "Impossible to open file '%s'\n",buffer);
#endif
      ns_errno = NetSolveFileError;
      return -1;
    }
    obj = createEmptyObject(pd->output_objects[i]->object_type,
                           pd->output_objects[i]->data_type);
    if (sendObjectFromFile(comm,fd,obj) == -1)
    {
#ifdef VIEW
      fprintf(STDERR "Impossible to send object from file '%s'\n",buffer);
#endif
      close(fd);
      return -1;
    }
    freeObject(obj);
    close(fd);
  }
  return 1;
}

/*
 * writeObjectToFile()
 */
int writeObjectToFile(int fd,NS_Object *obj)
{
  if (write(fd,&(obj->object_type),sizeof(int)) != sizeof(int))
  {
    ns_errno = NetSolveFileError;
    return -1;
  }
  if (write(fd,&(obj->data_type),sizeof(int)) != sizeof(int))
  {
    ns_errno = NetSolveFileError;
    return -1;
  }

  if (write(fd,&(obj->pass_back),sizeof(int)) != sizeof(int))
  {
    ns_errno = NetSolveFileError;
    return -1;
  }

  switch(obj->object_type)
  {
    case NETSOLVE_MATRIX:
    {
      void *ptr;
      int m,n,k;

      m = obj->attributes.matrix_attributes.m;
      n = obj->attributes.matrix_attributes.n;

      if (write(fd,&(obj->attributes.matrix_attributes.major),sizeof(int)) != sizeof(int))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
      if (write(fd,&m,sizeof(int)) != sizeof(int))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
      if (write(fd,&n,sizeof(int)) != sizeof(int))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
 
      ptr = obj->attributes.matrix_attributes.ptr;

      if (obj->attributes.matrix_attributes.l == m)
      {
        if (write(fd,obj->attributes.matrix_attributes.ptr,
                 netsolve_sizeof(obj->data_type)*m*n) !=
                 netsolve_sizeof(obj->data_type)*m*n)
        {
          ns_errno = NetSolveFileError;
          return -1;
        }
      }
      else if (obj->attributes.matrix_attributes.l == -1)
      {
        if (write(fd,obj->attributes.matrix_attributes.ptr,
                 netsolve_sizeof(obj->data_type)*m*n) !=
                 netsolve_sizeof(obj->data_type)*m*n)
        {
          ns_errno = NetSolveFileError;
          return -1;
        }
        obj->attributes.matrix_attributes.l = m;
      }
      else
      {
        for (k=0;k<n;k++)
          if (read(fd,
              (char *)ptr+k*obj->attributes.matrix_attributes.l*
              netsolve_sizeof(obj->data_type),m*netsolve_sizeof(obj->data_type)) !=
                    m*netsolve_sizeof(obj->data_type))
          {
            ns_errno = NetSolveFileError;
            return -1;
          }
      }
      break;
    }
    case NETSOLVE_SPARSEMATRIX:
    {
      int m,n,f;

      m = obj->attributes.sparsematrix_attributes.m;
      n = obj->attributes.sparsematrix_attributes.n;
      f = obj->attributes.sparsematrix_attributes.f;

      if (write(fd,&(obj->attributes.sparsematrix_attributes.major),sizeof(int)) != sizeof(int))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
      if (write(fd,&m,sizeof(int)) != sizeof(int))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
      if (write(fd,&n,sizeof(int)) != sizeof(int))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
      if (write(fd,&f,sizeof(int)) != sizeof(int))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
 
      if (write(fd,obj->attributes.sparsematrix_attributes.rc_ptr,
               netsolve_sizeof(NETSOLVE_I)*(m+1)) !=
               netsolve_sizeof(NETSOLVE_I)*(m+1))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
      if (write(fd,obj->attributes.sparsematrix_attributes.rc_index,
               netsolve_sizeof(NETSOLVE_I)*f) !=
               netsolve_sizeof(NETSOLVE_I)*f)
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
      if (write(fd,obj->attributes.sparsematrix_attributes.ptr,
               netsolve_sizeof(obj->data_type)*f) !=
               netsolve_sizeof(obj->data_type)*f)
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
      break;
    }
    
    case NETSOLVE_VECTOR:
    {
      void *ptr;
      int m;

      m = obj->attributes.vector_attributes.m;
      if (write(fd,&m,sizeof(int)) != sizeof(int))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }

      ptr = obj->attributes.vector_attributes.ptr;

      if (write(fd,ptr,m*netsolve_sizeof(obj->data_type)) != 
               m*netsolve_sizeof(obj->data_type))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
      break;
    }
    case NETSOLVE_SCALAR:
    {
      void *ptr;
      ptr = obj->attributes.scalar_attributes.ptr;
 
      if (write(fd,ptr,1*netsolve_sizeof(obj->data_type)) != 
                 1*netsolve_sizeof(obj->data_type))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
      break;
    }
    case NETSOLVE_FILE:
    {
      int len;

      len = strlen(obj->attributes.file_attributes.filename)+1;
      if (write(fd,&len,sizeof(int)) != sizeof(int))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
      if (write(fd,obj->attributes.file_attributes.filename,len*sizeof(char)) !=
                 len*sizeof(char))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
      break;
    }
    case NETSOLVE_PACKEDFILES:
    {
      int len;
      char buffer[256];
      int i,nb_files;

      nb_files = obj->attributes.packedfiles_attributes.m;
      if (write(fd,&nb_files,sizeof(int)) != sizeof(int))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }

      for (i=0;i<nb_files;i++) 
      {
        if (obj->attributes.packedfiles_attributes.defaultprefix != NULL)
        {
          sprintf(buffer,"%s-%d",
                 obj->attributes.packedfiles_attributes.defaultprefix,i);
        }
        else
        {
          sprintf(buffer,"%s",
                 obj->attributes.packedfiles_attributes.filenames[i]);
        }
        len = strlen(buffer)+1;
        if (write(fd,&len,sizeof(int)) != sizeof(int))
        {
          ns_errno = NetSolveFileError;
          return -1;
        }
        if (write(fd,buffer,
                  len*sizeof(char)) != len*sizeof(char))
        {
          ns_errno = NetSolveFileError;
          return -1;
        }
      }
      break;
    }
    case NETSOLVE_STRINGLIST:
    {
      int len;
      int i,nb_strings;
      char **strings = obj->attributes.stringlist_attributes.strings;

      nb_strings = obj->attributes.stringlist_attributes.m;
      if (write(fd,&nb_strings,sizeof(int)) != sizeof(int))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }

      for (i=0;i<nb_strings;i++)
      {
        len = strlen(strings[i])+1;
        if (write(fd,&len,sizeof(int)) != sizeof(int))
        {
          ns_errno = NetSolveFileError;
          return -1;
        }
        if (write(fd,strings[i],
                  len*sizeof(char)) != len*sizeof(char))
        {
          ns_errno = NetSolveFileError;
          return -1;
        }
      }
      break;
    }
    case NETSOLVE_UPF:
    {
      int len; 

      len = strlen(obj->attributes.upf_attributes.filename) +1;
      if (write(fd,&len,sizeof(int)) != sizeof(int))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
      if (write(fd,obj->attributes.upf_attributes.filename,len*sizeof(char)) !=
                 len*sizeof(char))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }

      if (write(fd,&(obj->attributes.upf_attributes.language),sizeof(int)) != 
                 sizeof(int))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
      len = strlen(obj->attributes.upf_attributes.funcname) +1;
      if (write(fd,&len,sizeof(int)) != sizeof(int))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
      if (write(fd,obj->attributes.upf_attributes.funcname,len*sizeof(char)) != 
                 len*sizeof(char))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
      break;
    }
    case NETSOLVE_STRING:
    {
      int len = 1+strlen(obj->attributes.string_attributes.ptr);
      if (write(fd,&len,sizeof(int)) != sizeof(int))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
      if (write(fd,obj->attributes.string_attributes.ptr,len*sizeof(char)) !=
               len*sizeof(char))
      {
        ns_errno = NetSolveFileError;
        return -1;
      }
      break;
    }
    default:
#ifdef VIEW
      fprintf(STDERR "Unknown object type\n");
#endif
      ns_errno = NetSolveInternalError;
      return -1;
  }
  return 1;
}

/*
 * readInputObjectsFromFiles()
 */
int readInputObjectsFromFiles(NS_ProblemDesc *pd)
{
  NS_Object **list = pd->input_objects;
  int nb = pd->nb_input_objects;
  char buffer[256];
  int i,fd;

  for (i=0;i<nb;i++)
  {
     if (pd->input_names == NULL) sprintf(buffer,"input%d",i);
    else sprintf(buffer,"%s",pd->input_names[i]);

    fd = open(buffer,O_RDONLY);
    if (fd < 0)
    {
#ifdef VIEW
      fprintf(STDERR "Impossible to open file '%s'\n",buffer);
#endif
      ns_errno = NetSolveFileError;
      return -1;
    }

    /* do initializations */
    switch(list[i]->object_type)
    {
      case NETSOLVE_MATRIX:
        list[i]->attributes.matrix_attributes.m = -1;
        list[i]->attributes.matrix_attributes.n = -1;
        list[i]->attributes.matrix_attributes.l = -1;
        list[i]->attributes.matrix_attributes.major = -1;
        list[i]->attributes.matrix_attributes.ptr = NULL;
        list[i]->attributes.matrix_attributes.d = NULL;
        break;
      case NETSOLVE_SPARSEMATRIX:
        list[i]->attributes.sparsematrix_attributes.m = -1;
        list[i]->attributes.sparsematrix_attributes.n = -1;
        list[i]->attributes.sparsematrix_attributes.f = -1;
        list[i]->attributes.sparsematrix_attributes.major = -1;
        list[i]->attributes.sparsematrix_attributes.ptr = NULL;
        list[i]->attributes.sparsematrix_attributes.rc_ptr = NULL;
        list[i]->attributes.sparsematrix_attributes.rc_index = NULL;
        list[i]->attributes.sparsematrix_attributes.d = NULL;
        break;
      case NETSOLVE_VECTOR:
        list[i]->attributes.vector_attributes.m = -1;
        list[i]->attributes.vector_attributes.ptr = NULL;
        list[i]->attributes.vector_attributes.d = NULL;
        break;
      case NETSOLVE_SCALAR:
        list[i]->attributes.scalar_attributes.ptr = NULL;
        break;
      case NETSOLVE_STRING:
        list[i]->attributes.matrix_attributes.ptr = NULL;
        break;
      case NETSOLVE_FILE:
        list[i]->attributes.file_attributes.filename = NULL;
        break;
      case NETSOLVE_PACKEDFILES:
        list[i]->attributes.packedfiles_attributes.filenames = NULL;
        list[i]->attributes.packedfiles_attributes.defaultprefix = NULL;
        list[i]->attributes.packedfiles_attributes.m = -1;
        break;
      case NETSOLVE_STRINGLIST:
        list[i]->attributes.stringlist_attributes.strings = NULL;
        list[i]->attributes.stringlist_attributes.m = -1;
        break;
      case NETSOLVE_UPF: /* Create a file name */
        list[i]->attributes.upf_attributes.filename = NULL;
        list[i]->attributes.upf_attributes.funcname = NULL;
        list[i]->attributes.upf_attributes.language = -1;
        break;
      default: /* Nothing to so */
        break;
    }

    /* read an object from file */
    if (readObjectFromFile(fd,list[i]) == -1)
      return -1;

    /* Perform special post-processing !! */
    switch(list[i]->object_type)
    {
      case NETSOLVE_MATRIX: /* transpose */
        if (list[i]->attributes.matrix_attributes.major == pd->major)
          break;
        if (transposeMatrix(list[i]) == -1)
        {
#ifdef VIEW
          fprintf(STDERR "Error while transposing matrix\n");
#endif
          return -1;
        }
        break;
      case NETSOLVE_SPARSEMATRIX: /* transpose */
        if (list[i]->attributes.sparsematrix_attributes.major == pd->major)
          break;
        if (transposeSparseMatrix(list[i]) == -1)
        {
#ifdef VIEW
          fprintf(STDERR "Error while transposing sparse matrix\n");
#endif
          return -1;
        }
        break;
      default: /* Nothing to do */
        break;
    }
  }
  return 1;
}

/*
 * writeOutputObjectsToFiles()
 *
 * write all the objects (data) to files in the working
 * directory. The files are names input0,input1,...
 * if the input is a file or a upf, then the file
 * inputx countains no real data, just the name
 * of the file containing the data (fileinputx or upfinputx)
 */
int writeOutputObjectsToFiles(NS_ProblemDesc *pd)
{
  int i;
  int fd;
  char buffer[256];
  
  for (i=0;i<pd->nb_output_objects;i++)
  { 
    if (pd->output_names == NULL)sprintf(buffer,"./output%d",i); 
    else sprintf(buffer,"%s",pd->output_names[i]);

    fd = open(buffer,O_WRONLY|O_CREAT,0666);
    if (fd < 0)
    {
#ifdef VIEW
      fprintf(STDERR "pid%i: Impossible to create file '%s' (%s)\n", (int)getpid(), buffer, strerror(errno));
#endif
      ns_errno = NetSolveFileError;
      return -1;
    }
    if (writeObjectToFile(fd,pd->output_objects[i]) == -1)  
    {
#ifdef VIEW
      fprintf(STDERR "pid%i: Couldn't write object to file '%s' (%s)\n", (int)getpid(), buffer, strerror(errno));
#endif
      close(fd);
      return -1;
    } 
    close(fd);
  } 
  return 1;
} 

/*
 * problemOrder()
 */
#ifdef sunos
int problemOrder(void *s1, void *s2)
#else
int problemOrder(const void *s1, const void *s2)
#endif
{
  NS_ProblemDesc **pd1 = (NS_ProblemDesc **)s1;
  NS_ProblemDesc **pd2 = (NS_ProblemDesc **)s2;

  if (strcmp((*pd1)->name,(*pd2)->name) > 0)
    return 1;
  if (strcmp((*pd1)->name,(*pd2)->name) < 0)
    return -1;
  return 0;
}

/*
 * freeObjectData()
 */
void freeObjectData(NS_Object *obj)
{
  int i;

  if (obj == NULL)
    return;
  
  switch(obj->object_type)
  {
    case NETSOLVE_MATRIX:
      if (obj->attributes.matrix_attributes.ptr != NULL)
        free(obj->attributes.matrix_attributes.ptr);
      break;
    case NETSOLVE_SPARSEMATRIX:
      if (obj->attributes.sparsematrix_attributes.rc_ptr != NULL)
        free(obj->attributes.sparsematrix_attributes.rc_ptr);
      if (obj->attributes.sparsematrix_attributes.rc_index != NULL)
        free(obj->attributes.sparsematrix_attributes.rc_index);
      if (obj->attributes.sparsematrix_attributes.ptr != NULL)
        free(obj->attributes.sparsematrix_attributes.ptr);
      break;
    case NETSOLVE_VECTOR:
      if (obj->attributes.vector_attributes.ptr != NULL)
        free(obj->attributes.vector_attributes.ptr);
      break;
    case NETSOLVE_SCALAR:
      if (obj->attributes.scalar_attributes.ptr != NULL)
        free(obj->attributes.scalar_attributes.ptr);
      break;
    case NETSOLVE_STRINGLIST:
      if (obj->attributes.stringlist_attributes.strings != NULL)
      {
        for (i=0;i<obj->attributes.stringlist_attributes.m;i++)
          if (obj->attributes.stringlist_attributes.strings[i] != NULL)
            free(obj->attributes.stringlist_attributes.strings[i]);
        free(obj->attributes.stringlist_attributes.strings);
      }
      break;
    case NETSOLVE_STRING:
      if (obj->attributes.string_attributes.ptr != NULL)
        free(obj->attributes.string_attributes.ptr);
      break;
    default:  /* NOTHING */
      break;
  }
  return;
}

/*
 * transposeMatrix()
 */
int transposeMatrix(NS_Object *mat)
{
  int m,n;
  void **ptr;
  int original_major;
  int block_size=0;
  int block_number=0;
  int i,j;
  void *new=NULL;
  void *tmp=NULL;
  int count;

  if (mat->object_type != NETSOLVE_MATRIX)
  {
    ns_errno = NetSolveInternalError;
    return -1;
  }
  m = mat->attributes.matrix_attributes.m;
  n = mat->attributes.matrix_attributes.n;
  ptr = &(mat->attributes.matrix_attributes.ptr);
  original_major = mat->attributes.matrix_attributes.major;

  switch(original_major)
  {
    case COL_MAJOR:
      block_size = m;
      block_number = n;
      break;
    case ROW_MAJOR:
      block_size = n;
      block_number = m;
      break;
  }

  switch(mat->data_type)
  {
    case NETSOLVE_CHAR:
      new = (char *)calloc(m*n,sizeof(char));
      break;

    case NETSOLVE_B:
      new = (char *)calloc(m*n,sizeof(char));
      break;

    case NETSOLVE_I:
      new = (int *)calloc(m*n,sizeof(int));
      break;

    case NETSOLVE_S:
      new = (float *)calloc(m*n,sizeof(float));
      break;

    case NETSOLVE_D:
      new = (double *)calloc(m*n,sizeof(double));
      break;

    case NETSOLVE_C:
      new = (scomplex *)calloc(m*n,sizeof(scomplex));
      break;

    case NETSOLVE_Z:
      new = (dcomplex *)calloc(m*n,sizeof(dcomplex));
      break;

  }

  count = 0;
  for (i=0;i<block_size;i++)
  {
    for (j=0;j<block_number;j++)
    {
      switch(mat->data_type)
      {
        case NETSOLVE_CHAR:
          ((char *)new)[count] = ((char *)*ptr)[j*block_size+i];
          break;

        case NETSOLVE_B:
          ((char *)new)[count] = ((char *)*ptr)[j*block_size+i];
          break;

        case NETSOLVE_I:
          ((int *)new)[count] = ((int *)*ptr)[j*block_size+i];
          break;
        case NETSOLVE_S:
          ((float *)new)[count] = ((float *)*ptr)[j*block_size+i];
          break;

        case NETSOLVE_D:
          ((double *)new)[count] = ((double *)*ptr)[j*block_size+i];
          break;

        case NETSOLVE_C:
          ((scomplex *)new)[count] = ((scomplex *)*ptr)[j*block_size+i];
          break;

        case NETSOLVE_Z:
          ((dcomplex *)new)[count] = ((dcomplex *)*ptr)[j*block_size+i];
          break;
      }
      count++;
    }
  }
  tmp = *ptr;
  *ptr = new;
  free(tmp);

  switch(original_major)
  {
    case ROW_MAJOR:
      mat->attributes.matrix_attributes.major = COL_MAJOR;
      break;
    case COL_MAJOR:
      mat->attributes.matrix_attributes.major = ROW_MAJOR;
      break;
  }

  return 1;
}

/*
 * transposeSparseMatrix()
 */
int transposeSparseMatrix(NS_Object *mat)
{
  int original_major, datatype;
  int * rc_index, *rc_ptr, *newrc_index, *newrc_ptr;
  int m, n, f;
  void *ptr, *newptr;

  if (mat->object_type != NETSOLVE_SPARSEMATRIX)
  {
#ifdef DEBUG
    fprintf(stderr, "NetSolve: trying to do a sparse transpose on an object"
                    "that is not a sparse matrix\n");
#endif
    ns_errno = NetSolveInternalError;
    return -1;
  }

  m = mat->attributes.sparsematrix_attributes.m;
  n = mat->attributes.sparsematrix_attributes.n;
  f = mat->attributes.sparsematrix_attributes.f;
  rc_index = mat->attributes.sparsematrix_attributes.rc_index;
  rc_ptr = mat->attributes.sparsematrix_attributes.rc_ptr;
  ptr = mat->attributes.sparsematrix_attributes.ptr;
  original_major = mat->attributes.sparsematrix_attributes.major;
  datatype = mat->data_type;


  /* Allocate storage for another copy of the matrix. */
  newptr = (void *)calloc(f, netsolve_sizeof(datatype));
  newrc_index = (int *) calloc(f, sizeof(int));
  newrc_ptr = (int *) calloc(n+1, sizeof(int));

  switch(original_major)
  {
    case ROW_MAJOR:
      CompRow_to_CompCol(m, n, f, ptr, rc_index, rc_ptr,
             newptr, newrc_index, newrc_ptr, datatype);
      free(ptr); mat->attributes.sparsematrix_attributes.ptr = newptr;
      free(rc_index); mat->attributes.sparsematrix_attributes.rc_index = newrc_index;
      free(rc_ptr); mat->attributes.sparsematrix_attributes.rc_ptr = newrc_ptr;
      mat->attributes.sparsematrix_attributes.major = COL_MAJOR;
      break;

    case COL_MAJOR:
      CompCol_to_CompRow(m, n, f, ptr, rc_index, rc_ptr,
             newptr, newrc_index, newrc_ptr, datatype);
      free(ptr); mat->attributes.sparsematrix_attributes.ptr = newptr;
      free(rc_index); mat->attributes.sparsematrix_attributes.rc_index = newrc_index;
      free(rc_ptr); mat->attributes.sparsematrix_attributes.rc_ptr = newrc_ptr;
      mat->attributes.sparsematrix_attributes.major = ROW_MAJOR;
      break;
  }
  return 1;
}

/*
 * Convert a row compressed storage into a column compressed storage.
 */

void CompRow_to_CompCol(int m, int n, int nnz,
            void *a, int *colind, int *rowptr,
            void *at, int *rowind, int *colptr, int datatype)
{
  register int i, j, col, relpos;
  int *marker;
 
  marker = (int *) calloc(n, sizeof(int));
 
  /* Get counts of each column of A, and set up column pointers */
  for (i = 0; i < m; ++i)
    for (j = rowptr[i]; j < rowptr[i+1]; ++j) ++marker[colind[j]];

  colptr[0] = 0;

  for (j = 0; j < n; ++j) {
    colptr[j+1] = colptr[j] + marker[j];
    marker[j] = colptr[j];
  }
 
  /* Transfer the matrix into the compressed column storage. */
  for (i = 0; i < m; ++i) {
    for (j = rowptr[i]; j < rowptr[i+1]; ++j) {
      col = colind[j];
      relpos = marker[col];
      rowind[relpos] = i;

      switch(datatype)
      {
        case NETSOLVE_CHAR:
          ((char *)at)[relpos] = ((char *)a)[j];
          break;
 
        case NETSOLVE_B:
          ((char *)at)[relpos] = ((char *)a)[j];
          break;
 
        case NETSOLVE_I:
          ((int *)at)[relpos] = ((int *)a)[j];
          break;
        case NETSOLVE_S:
          ((float *)at)[relpos] = ((float *)a)[j];
          break;
 
        case NETSOLVE_D:
          ((double *)at)[relpos] = ((double *)a)[j];
          break;
 
        case NETSOLVE_C:
          ((scomplex *)at)[relpos] = ((scomplex *)a)[j];
          break;
 
        case NETSOLVE_Z:
          ((dcomplex *)at)[relpos] = ((dcomplex *)a)[j];
          break;
      }

      ++marker[col];
    }
  }
 
  free(marker);
}

/*
 * Convert a column compressed storage into a row compressed storage.
 */

void CompCol_to_CompRow(int m, int n, int nnz,
            void *a, int *colind, int *rowptr,
            void *at, int *rowind, int *colptr, int datatype)
{
  register int i, j, col, relpos;
  int *marker;
 
  marker = (int *) calloc(n, sizeof(int));
 
  /* Get counts of each column of A, and set up column pointers */
  for (i = 0; i < m; ++i)
    for (j = rowptr[i]; j < rowptr[i+1]; ++j) ++marker[colind[j]];

  colptr[0] = 0;

  for (j = 0; j < n; ++j) {
    colptr[j+1] = colptr[j] + marker[j];
    marker[j] = colptr[j];
  }
 
  /* Transfer the matrix into the compressed column storage. */
  for (i = 0; i < m; ++i) {
    for (j = rowptr[i]; j < rowptr[i+1]; ++j) {
      col = colind[j];
      relpos = marker[col];
      rowind[relpos] = i;

      switch(datatype)
      {
        case NETSOLVE_CHAR:
          ((char *)at)[relpos] = ((char *)a)[j];
          break;
 
        case NETSOLVE_B:
          ((char *)at)[relpos] = ((char *)a)[j];
          break;
 
        case NETSOLVE_I:
          ((int *)at)[relpos] = ((int *)a)[j];
          break;
        case NETSOLVE_S:
          ((float *)at)[relpos] = ((float *)a)[j];
          break;
 
        case NETSOLVE_D:
          ((double *)at)[relpos] = ((double *)a)[j];
          break;
 
        case NETSOLVE_C:
          ((scomplex *)at)[relpos] = ((scomplex *)a)[j];
          break;
 
        case NETSOLVE_Z:
          ((dcomplex *)at)[relpos] = ((dcomplex *)a)[j];
          break;
      }

      ++marker[col];
    }
  }
 
  free(marker);
}
