/*****************************************************************/
/*      netsolveutil.c                                           */
/*      Henri Casanova                                           */
/*---------------------------------------------------------------*/
/*****************************************************************/

#include <ctype.h>
#ifndef WIN32
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <netdb.h>
#include <netinet/in.h>
#endif
#include "core.h"


/*
 * isRemoteFile()
 */
int isRemoteFile(char *filename)
{
  return (strchr(filename,':') != NULL);
}

/*
 * getRemoteFileName()
 *
 * diskname:cap:filename
 */
char *getRemoteFileName(char *filename)
{
  char *ptr;
  char *buf;
  char *result;

  buf = strdup(filename);
  ptr = strtok(buf,":");
  ptr = strtok(NULL,":");
  ptr = strtok(NULL,"\0");
  if (ptr)
    result = strdup(ptr);
  else
    result = NULL;
  free(buf);
  return result;
}

/*
 * getRemoteDisk()
 *
 * diskname:cap:filename
 */
char *getRemoteDisk(char *filename)
{
  char *buf;
  char *ptr;
  char *result;

  buf = strdup(filename);
  ptr = strtok(buf,":");
  if (ptr)
    result = strdup(ptr);
  else
    result = NULL;
  free(buf);
  return result;
}

/*
 * getRemoteCap()
 *
 * disk:cap:filename
 */
char *getRemoteCap(char *filename)
{
  char *buf; 
  char *ptr;
  char *result;

  buf = strdup(filename); 
  ptr = strtok(buf,":");
  ptr = strtok(NULL,":");
  if (ptr)
    result = strdup(ptr);
  else
    result = NULL;
  free(buf);
  return result;
}


/*
 * getMaxNumberFileDesc()
 *
 * in place of getdtablesize() that is not available everywhere :(
 */
int getMaxNumberFileDesc()
{
#ifndef WIN32
  static int max = -1;
  int fd[2000];
  int i;

  if (max != -1)
    return max;

  for (i=0;i<2000;i++)
  {
    fd[i] = open(".",O_RDONLY);
    if (fd[i] == -1)
      break;
  }
  max = i;
  for (i=0;i<max;i++)
    close(fd[i]);

  return max;
#endif
#ifdef WIN32
  return 32;  /* TO CHECK !!!!! JUST A GUESS (UNFOUNDED) */
#endif
}


/*
 * lastInPath()
 *
 *  x/y/r ---> r
 *  r     ---> r
 */
char *lastInPath(char *path)
{
  char *tmp;

  tmp = strrchr(path,'/');
  if (tmp == NULL)
    return path;
  else
    return (char *)(tmp+1);
}

/*
 * readLine()
 *
 * reads a line from a file. Skips lines starting
 * with a '#' or a '\n'. Increments an integer
 * for the line counting.
 */
char *readLine(FILE *f,char *line,int *linecount)
{
  char *tmp = NULL;
  do
  {
    tmp = fgets(line,256,f);
    if (tmp == NULL)
      break;
    (*linecount)++;
  } while((tmp[0] == '\n')||(tmp[0] == '#'));
  return tmp;
}

/*
 * netsolveCaps()
 *
 * netsolveCaps a string
 */
void netsolveCaps(char *s)
{
  char *tmp = s;

  while (*tmp)
  {
    *tmp = toupper(*tmp);
    tmp++;
  }
}

/*
 * isInteger()
 *
 */
int isInteger(char *s)
{
  while ((*s != '\0')&&(*s != '\n'))
  {
    if ((*s<'0')||(*s>'9'))
      return 0;
    s++;
  }
  return 1;
}

/*
 * inList()
 */
int inList(char **list,int nlist,char *s)
{
  int i;
  
  for (i=0;i<nlist;i++)
    if (!strcmp(list[i],s))
      return 1;
  return 0;
}

/*
 * freeCharStarList()
 */
void freeCharStarList(char **list, int nb)
{
  int i;

  for (i=0;i<nb;i++)
    free(list[i]);
  free(list);
  return;
}

/*
 * copyCharStarList()
 */
char **copyCharStarList(char **list,int nb)
{
  char **tmp;
  int i;

  tmp = (char **)calloc(nb,sizeof(char*));
  for (i=0;i<nb;i++)
    tmp[i] = strdup(list[i]);
  return tmp;
}


/*
 * power()
 */
double power(int a,int b)
{
  int i;
  double r=1.0;

  for (i=0;i<b;i++)
    r = r * ((double)a);
  return r;
}

/*
 * getLanguageFromFilename()
 */
int getLanguageFromFilename(char *filename)
{
  int len = strlen(filename);

  if (filename[len-2] != '.')
  {
    ns_errno = NetSolveInvalidUPFFilename;
    return -1;
  }
  if (filename[len-1] == 'f')
    return UPF_LANG_FORTRAN;
  if (filename[len-1] == 'c')
    return UPF_LANG_C;
  ns_errno = NetSolveInvalidUPFFilename;
  return -1;
}

/*
 * getFunctionNameFromFilename()
 */
char *getFunctionNameFromFilename(char *filename)
{
  int len = strlen(filename);
  char *tmp;

  tmp = strdup(filename);
  tmp[len-1] = '\0';
  tmp[len-2] = '\0';
  return tmp; 
}

char *mystrstr(char*s1,char*s2)
{
  int l1 = strlen(s1);
  int l2 = strlen(s2);

  int i,j;

  for (i=0;i<l1-l2;i++)
  {
    int equal = 1;
    for (j=0;j<l2;j++)
    {
      if (s1[i+j] != s2[j])
        equal = 0;
    }
    if (equal == 1)
      return (char *)&s1[i];
  }
  return NULL;
}

/*
 * printIPaddr()
 */
void printIPaddr(NS_IPaddr_type IPaddr)
{
	fprintf(stdout,"%u.%u.%u.%u",
             BYTE(&IPaddr,0),
             BYTE(&IPaddr,1),
             BYTE(&IPaddr,2),
             BYTE(&IPaddr,3));
}

/*
 * generateTmpName()
 */
#ifndef WIN32
char *generateTmpName(char *prefix)
{
  int current = getpid();
  char buffer[256];
  struct stat st;
  
  while(1)
  {
    sprintf(buffer,"%stmp-%d",prefix,current);
    if (!stat(buffer,&st))
      current++;
    else
      break;
  }
  return strdup(buffer);
}
#endif

/*
 * myRealloc()
 *
 * Passing a NULL argument to realloc is not portable :(
 */
void *myRealloc(void *ptr, size_t size)
{
  if (ptr != NULL)
    return realloc(ptr,size);
  else
    return (void*)calloc(1, size);
}

/*
 * getMyIPaddr()
 */
NS_IPaddr_type getMyIPaddr()
{
  char buffer[256];
  struct hostent *hp;
  NS_IPaddr_type IPaddr;

  if (gethostname(buffer,256) == -1)
  {
    perror("gethostname()");
    ns_errno = NetSolveSystemError;
    return -1;
  }
  if ((hp = gethostbyname(buffer)) == NULL)
  {
    perror("gethostbyname()");
    ns_errno = NetSolveSystemError;
    return -1;
  }
  BCOPY(hp->h_addr_list[0],(void*)&IPaddr, sizeof(NS_IPaddr_type));
  return IPaddr;
}
