#ifdef KERBEROS5
#include "core.h"
#include <unistd.h>
#include <sys/socket.h>
#include <signal.h>
#include <time.h>
#include <krb5.h>
#include <com_err.h>
#include "serverglobal.h"

/*
 * send our credentials to the server on the other end of 'comm'
 * return a netsolve response code.
 */

char netsolve_version[] = "NETSOLVE V1.3";

int
SendKerberos5Credentials (char * server_hostname, NS_Communicator * comm)
{
  krb5_context context;
  krb5_auth_context auth_context = 0;
  krb5_error_code retval;
  krb5_principal client, server;
  krb5_ccache cc;

  retval = krb5_init_context (&context);
  if (retval) {
    fprintf(stderr, "krb5_init_context failed: %s\n", error_message(retval));
    return -1;
  } 

  /*
   * XXX we're picking the service name "netsolve" arbitrarily;
   * this should be registered with the appropriate folks.
   */

  retval = krb5_sname_to_principal(context, server_hostname,
				   "netsolve",
				   KRB5_NT_SRV_HST, &server);
  if (retval) {
    fprintf (stderr, "krb5_sname_to_principal failed: %s\n",
	     error_message (retval));
    return -1;
  }

  retval = krb5_cc_default (context, &cc);

  if (retval) {
    fprintf (stderr, "krb5_cc_default failed: %s\n",
	     error_message (retval));
    krb5_free_principal (context, server);
    krb5_cc_close (context, cc);
    return -1;
  }

  retval = krb5_cc_get_principal (context, cc, &client);

  if (retval) {
    fprintf (stderr, "krb5_cc_get_principal failed: %s\n",
	     error_message (retval));
    krb5_free_principal (context, server);
    krb5_cc_close (context, cc);
    return -1;
  }

  retval = krb5_sendauth (context, &auth_context,
			  (krb5_context) &(comm->sock),
			  netsolve_version,
			  client, server,
			  AP_OPTS_MUTUAL_REQUIRED | AP_OPTS_USE_SUBKEY,
			  NULL, NULL, cc, 0, 0, NULL);
  
  if (retval) {
    fprintf (stderr, "krb5_sendauth failed: %s\n",
	     error_message (retval));
    krb5_free_principal (context, client);
    krb5_free_principal (context, server);
    krb5_cc_close (context, cc);
    return -1;
  }

  return 0;
}


int
CheckPrincipalName (name)
char *name;
{
    FILE *fp;
    char buf[1024];
    
    if ((fp = fopen (global.users_file, "r")) == NULL) {
	fprintf (stderr, "cannot open %s: %s\n", global.users_file, strerror (errno));
	return -1;
    }
    while (fgets (buf, sizeof (buf), fp)) {
	char *ptr;
	
	/* if '#' appears in the line, ignore it and everything following */
	if ( (ptr = strchr (buf, '#')) )
	    *ptr = '\0';
	
	/* delete trailing spaces */
	ptr = buf + strlen (buf);
	while (ptr > buf && (ptr[-1] == ' ' || ptr[-1] == '\t' ||
			     ptr[-1] == '\n' || ptr[-1] == '\r'))
	    *--ptr = '\0';
	
	/* skip over leading spaces */
	for (ptr = buf; *ptr == ' ' || *ptr == '\t'; ++ptr);
	
	/* if line is blank, continue */
	if (*ptr == '\0')
	    continue;
	
	/* see if name matches */
	if (strcasecmp (ptr, name) == 0){
	    return 1;
        }
    }
    fclose (fp);
    return -1;
}

int
RecvKerberos5Credentials (NS_Communicator * comm)
{
  krb5_error_code retval;
  krb5_auth_context auth_context = NULL;
  krb5_ticket *ticket;
  char *name;

  fprintf(stderr, "Initializing Kerberos ... "); fflush(stderr);
  if(krb5_init() != 0){
    return -1;
  }
  fprintf(stderr, "success\n"); fflush(stderr);

  fprintf(stderr, "Initializing Authentication Context ... "); fflush(stderr);
  retval = krb5_auth_con_init(global.context, &auth_context);
  if(retval){
    fprintf(stderr, "krb5_auth_con_init() failed: %s\n", error_message(retval));
    return -1;
  }
  fprintf(stderr, "success\n"); fflush(stderr);

  fprintf(stderr, "Receiving authentication info from client ... "); fflush(stderr);
  retval = krb5_recvauth (global.context, &auth_context,
			  (krb5_pointer) &(comm->sock), netsolve_version,
			  global.server, 0, global.keytab, &ticket);
  if (retval) {
      fprintf (stderr, "Authentication failed: %s\n",
	       error_message(retval)); fflush(stderr);
      return -1;
  }
  fprintf(stderr, "success\n"); fflush(stderr);
  
  krb5_unparse_name (global.context, ticket->enc_part2->client, &name);
  krb5_free_ticket (global.context, ticket);
  
  /* it's not enough to know that the ticket is valid;
     (i.e. that the client is who he says he is),
     we also need to check to see whether client has permission */

  fprintf(stderr, "Authenticated, checking if %s is authorized ... ", name); fflush(stderr);
  if (CheckPrincipalName (name) < 0) {
      fprintf (stderr, "permission denied.\n"); fflush(stderr);
      return -1;
  }
  else{
    fprintf (stderr, "yes\n");
    fflush(stderr);
  }

  free (name);

  fprintf(stderr, "Freeing authentication context ... "); fflush(stderr);
  retval = krb5_auth_con_free(global.context, auth_context);
  if(retval){
    fprintf(stderr, "krb5_auth_con_free() failed: %s\n", error_message(retval));
    return -1;
  }
  fprintf(stderr, "success\n"); fflush(stderr);

  return 0;
}

int krb5_init(){
  krb5_error_code retval;
  char * keytab_filename;
  retval = krb5_init_context (&(global.context));

  if (retval)
  {
    fprintf (stderr,
         "krb5_init_context failed: %s\n", error_message(retval));
    return -1;
  }

  global.keytab = NULL;   /* Maybe do something different later */
  keytab_filename = getenv ("NETSOLVE_KEYTAB");

  if (keytab_filename == NULL)
  {
    fprintf (stderr, "Environment variable NETSOLVE_KEYTAB not defined\n");
    return -1;
  }

  retval = krb5_kt_resolve (global.context, keytab_filename, &(global.keytab));
  if(retval){
    fprintf (stderr, "krb5_kt_resolve() failed: %s\n", error_message(retval));
    return -1;
  } 

  global.users_file = getenv("NETSOLVE_USERS");
  if (global.users_file == NULL)
  {
    fprintf (stderr, "Environment variable NETSOLVE_USERS not defined\n");
    return -1;
  }

  retval = krb5_sname_to_principal (global.context, NULL, "netsolve",
                    KRB5_NT_SRV_HST, &(global.server));

  if (retval)
  {
    fprintf (stderr, "krb5_sname_to_principal() failed: %s\n",
         error_message (retval));
    return -1;
  }

  return 0;
}
  
#endif /* KERBEROS5 */
