/*****************************************************************/
/*      code_generator_util.c                                    */
/*      Version 1.0                      Henri Casanova          */
/*****************************************************************/

#include "core.h"
#include "codegenerator.h"
#include "mnemonics.h"
#include "expressions.h"
#include "netsolveutil.h"

/*
 * write_line()
 *
 */

int write_line(char *string)
{
  return (fwrite(string,1,strlen(string),targetfile));
}

/*
 * putTargetHeader
 *
 * put the header at the beginning of the file
 *
 */
int putTargetHeader()
{
  FILE *headerfile;
  char name[256];

  sprintf(name,"%s/src/CodeGenerator/%s",root,HEADER);
  headerfile = fopen(name,"r");
  if (headerfile == NULL)
  {
    fprintf(stderr,"Impossible to open the header file : %s\n",name);
    exit(-1);
  }
  while(fgets(line,MAX_LINE_LENGTH,headerfile) != NULL)
  {
    if (write_line(line) <0)
      return -1;
  }
  return 1;

}

/*
 * putHeaderDef()
 *
 * put the header at the beginning of the file
 *
 */
int putHeaderDef(FILE *f)
{
  FILE *headerfile;
  char name[256];

  sprintf(name,"%s/src/CodeGenerator/%s",root,DEF_HEADER);
  headerfile = fopen(name,"r");
  if (headerfile == NULL)
  {
    fprintf(stderr,"Impossible to open the header file : %s\n",name);
    exit(-1);
  }
  while(fgets(line,MAX_LINE_LENGTH,headerfile) != NULL)
  {
    if ((fwrite(line,1,strlen(line),f)) <0)
      return -1;
  }
  return 1;
}

/*
 *  read_line()
 *
 *  Wrapper around readLine()
 */

char *read_line(FILE *sourcefile)
{
  return readLine(sourcefile,line,&linecount);
}

/*
 * cleaned()
 */
char *cleaned(char *s)
{
  int i;
  int len = strlen(s);
  int nb_returns=0;
  char *new;
  int offset=0;

  for (i=0;i<len;i++)
    if (s[i] == '\n')
      nb_returns++;

  new = (char*)calloc(1+len+nb_returns,sizeof(char));
   
  for (i=0;i<len;i++)
  {
    if (s[i] == '\"')
    {
      new[offset++] = '\'';
    }
    else if (s[i] == '\n')
    {
      new[offset++] = '\\';
      new[offset++] = 'n';
    }
    else
      new[offset++] = s[i];
  }
  return new;
}

/*
 * isCondorServer()
 */
int isCondorServer(char *filename)
{
  FILE *f;
  char line[256];
  int linecount = 0;
 
  f = fopen(filename,"r");
  while(readLine(f,line,&linecount))
  {
    if (!strncmp(line,"@CONDOR:",8))
    {
      fclose(f);
      return 1;
    }
  }
  fclose(f);
  return 0; 
}

/*
 * getDescFileNames()
 */
int getDescFileNames(char *configfilename,char **files)
{
  FILE *configfile;
  char *tmp;
  int count = 0;

  configfile = fopen(configfilename,"r");
  if (configfile == NULL)
  {
    fprintf(stderr,"Cannot open file '%s'\n",configfilename);
    return -1 ;
  }

  fprintf(stderr,"Parsing the configuration file ...\n");

  while(1)
  {
    tmp = fgets(line,256,configfile);
    if (tmp == NULL)
    {
      fclose(configfile);
      break;
    }
    if (strncmp(line,"@PROBLEMS:\n",11) == 0)
      break;
  }
  if (tmp == NULL)
  {
    fprintf(stderr,"The configuration file '%s' does not contain problem files!!\n",
        configfilename);
    return -1;
  }
  while(1)
  {
    tmp = fgets(line,256,configfile);
    if (tmp == NULL)
    {
      fclose(configfile);
      break;
    }
    if (*tmp == '@')
    {
      fclose(configfile);
      break;
    }
    if (line[0] == '#')
      continue;
    line[strlen(line)-1] = '\0';
    strcpy(files[count++]=(char *)malloc(sizeof(char)*(1+strlen(line))),line);
  }

  return count;
}

/*
 * putInitCodeFileHeader()
 */
void putInitCodeFileHeader(FILE *f)
{
  fprintf(f,"/******************************************/\n");
  fprintf(f,"/* File generated by the code_generator   */\n");
  fprintf(f,"/******************************************/\n\n");
  fprintf(f,"#include \"core.h\"\n\n");
  fprintf(f,"#include \"serverglobal.h\"\n\n");
  fprintf(f,"int problemInit(char **list,int nlist)\n");
  fprintf(f,"{\n");
  fprintf(f,"NS_ProblemDesc *pb_desc;\n");
  fprintf(f,"int i;\n");
  fprintf(f,"NS_MatlabMerge matlabtmp;\n");
  fprintf(f,"char buf[256];\n\n");
  fprintf(f,"NS_Object *obj;\n\n");

}

/*
 * reduceAttributes()
 *
 */
int reduceAttributes(NS_DescFileAttributes *d)
{
  int i,j,k,already_there;
  char *current_custom;
  int nb_libs=0;
  char **libs = NULL;
  char *current_lib;
  int nb_includes=0;
  char **includes = NULL;
  char *current_include;
  int nb_dashis=0;
  char **dashis = NULL;
  char *current_dashi;


  /* Check the customizations */
  current_custom = d->pds[0]->customized;
  for (i=1;i<d->nb_pbs;i++)
  {
    if (strcmp(current_custom,d->pds[i]->customized))
    {
      fprintf(stderr,"Customization conflict !!\n");
      return -1;
    }
  }

  if (is_condor_server)
  {
    if (strcmp(current_custom,"CONDOR"))
    {
      fprintf(stderr,"Only CONDOR customization allowed for Condor server\n");
      return -1;
    }
  }
    
  /* Libraries */
  for (i=0;i<d->nb_pbs;i++)
  {
    for (j=0;j<d->pdadds[i]->nb_libs;j++)
    {
      current_lib = d->pdadds[i]->libs[j]; 
      already_there = 0;
      for (k=0;k<nb_libs;k++)
        if (!strcmp(current_lib,libs[k]))
          already_there = 1;
      if (already_there)
        continue;
      /* add the library */
      nb_libs++;
      libs = myRealloc(libs,sizeof(char*)*nb_libs);
      libs[nb_libs-1] = strdup(current_lib);
    }
  }

  /* Includes */
  for (i=0;i<d->nb_pbs;i++)
  {
    for (j=0;j<d->pdadds[i]->nb_includes;j++)
    {
      current_include = d->pdadds[i]->includes[j]; 
      already_there = 0;
      for (k=0;k<nb_includes;k++)
        if (!strcmp(current_include,includes[k]))
          already_there = 1;
      if (already_there)
        continue;
      /* add the include */
      nb_includes++;
      includes = myRealloc(includes,sizeof(char*)*nb_includes);           
      includes[nb_includes-1] = strdup(current_include);
    }
  }

  /* Dashis */
  for (i=0;i<d->nb_pbs;i++)
  {
    for (j=0;j<d->pdadds[i]->nb_dashis;j++)
    {
      current_dashi = d->pdadds[i]->dashis[j];
      already_there = 0;
      for (k=0;k<nb_dashis;k++)
        if (!strcmp(current_dashi,dashis[k]))
          already_there = 1;
      if (already_there)
        continue;
      /* add the dashis */
      nb_dashis++;
      dashis = myRealloc(dashis,sizeof(char*)*nb_dashis);
      dashis[nb_dashis-1] = strdup(current_dashi);
    }
  }

  d->customized = strdup(current_custom);
  d->nb_libs = nb_libs;
  d->libs = libs;
  d->nb_includes = nb_includes;
  d->includes = includes;
  d->nb_dashis = nb_dashis;
  d->dashis = dashis;
  
  return 1;
}

/*
 * splitFile()
 */
int splitFile(char **names,char *filename) {

  int nb_files=0;
  FILE *f=NULL;
  FILE *sourcefile;
  char *name = 0;

  sourcefile = fopen(filename,"r");
  if (sourcefile == NULL)
  {
    fprintf(stderr,"Impossible to open file '%s'\n",filename);
    return -1;
  }

  while(read_line(sourcefile) != NULL)
  {
    /* Generate a bogus name */
    if (!strncmp(line,"@PROBLEM",8))
    {
  
     /* Sudesh: freeing name if not NULL , 
        removes MLK of 420 bytes in codegenerator
      */

     if(name != NULL){
       free(name);
       name = NULL;
     }

      name = generateTmpName("/tmp/");
      f = fopen(name,"w");
      if (f == NULL)
      {
        fprintf(stderr,"Impossible to create file '%s'\n",name);
        return -1;
      }
    }
    else if (!strncmp(line,"@END_CODE",9))
    {
      fwrite(line,1,strlen(line),f);
      names[nb_files] = strdup(name);
      nb_files++;
      fclose(f);
      f = NULL;
    }

    if (f != NULL)
      fwrite(line,1,strlen(line),f);

  }

  free(name);
  return nb_files;
}

/*
 * createDefFile()
 */
int createDefFile(NS_DescFileAttributes **attrs,int nb)
{
  FILE *f;
  char filename[256];
  int i,j,k;
 
  sprintf(filename,"%s/include/%s",root,DEF_FILE);

  f = fopen(filename,"w");
  if (f == NULL)
  {
    fprintf(stderr,"Impossible to open %s\n",filename);
    exit(-1);
  }
  if (putHeaderDef(f)<0)
  {
    perror("putHeaderDef()");
    exit(-1);
  }

  for (i=0;i<nb;i++)
  {
    for (j=0;j<attrs[i]->nb_pbs;j++)
    {
      if (attrs[i]->pdadds[j]->language == C)
        continue;
      for (k=0;k<attrs[i]->pdadds[j]->nb_funcs;k++)
      {
        char *name = attrs[i]->pdadds[j]->funcs[k];
        char caps_name[256];

        if(!strstr(name, "_")){
          fprintf(f,"#if (defined(F2CADD_) || defined (F2CADD__))\n");
          fprintf(f,"#define %s %s_\n",name,name);
          fprintf(f,"#endif\n");
        }
        else{
          fprintf(f,"#if defined(F2CADD__)\n");
          fprintf(f,"#define %s %s__\n",name,name);
          fprintf(f,"#elif defined(F2CADD_)\n");
          fprintf(f,"#define %s %s_\n",name,name);
          fprintf(f,"#endif\n");
        }

        fprintf(f,"#ifdef F2CUPCASE\n");
        strcpy(caps_name,name);
        netsolveCaps(caps_name);
        fprintf(f,"#define %s %s\n",name,caps_name);
        fprintf(f,"#endif\n\n");
      }
    }
  }
  
  fclose(f);

  return 1;
}

/*
 * addIncludes()
 */
int addIncludes(char *target_name, NS_DescFileAttributes *attr)
{
  int i;
  char *filename;
  FILE *f;
  char command[256];
  char *final;

  filename = generateTmpName("/tmp/");
  f = fopen(filename,"a");
  if (f == NULL)
  {
    fprintf(stderr,"Impossible to create file '%s'\n",filename);
    free(filename);
    return -1;
  }
  final =generateTmpName("/tmp/");

  for (i=0;i<attr->nb_includes;i++)
    fprintf(f,"#include %s\n",attr->includes[i]);
  fclose(f);

  sprintf(command,"cat %s %s > %s",filename,target_name,final);
  system(command);
  sprintf(command,"mv -f %s %s",final,target_name);
  system(command);
  sprintf(command,"rm -f %s",filename);
  system(command);
  sprintf(command,"rm -f %s",final);
  system(command);
  
  free(filename);
  free(final);
  return 1;
}
