/*****************************************************************/
/*      check_pseudocode.c                                       */
/*      Version 1.0                      Henri Casanova          */
/*****************************************************************/

#include "core.h"
#include "codegenerator.h"
#include "mnemonics.h"
#include "expressions.h"
#include "netsolveutil.h"

/*
 * checkMnemonicList()
 *
 *   given a problem descriptor and a list of mnemonic, checks if
 *   all problem features are represented and if all mnemonics are
 *   correctly written
 */

int checkMnemonicList(char **mnemonic_list,NS_ProblemDesc *pd)
{
  int i;

  for (i=0;i<pd->nb_input_objects;i++)
  {
    int object_type;
    int status;
    char buf[10];

    object_type = pd->input_objects[i]->object_type;

    sprintf(buf,"I%d",i);
    status = lookup(mnemonic_list,buf);
    if (status == -1)
    {
      fprintf(stderr,"Problem : %s -  One argument should contain 'I%d'\n",
              pd->nickname,i);
      return -1;
    }
    if (status == -2)
    {
      fprintf(stderr,"Problem : %s -  One SINGLE argument should contain 'I%d'\n",
              pd->nickname,i);
      return -1;
    }
    switch (object_type)
    {
      case NETSOLVE_MATRIX :
      {
        sprintf(buf,"mI%d",i);
        status = lookup(mnemonic_list,buf);
        if (status == -1)
        {
          fprintf(stderr,"Problem : %s - One argument should contain 'mI%d'\n",
                  pd->nickname,i);
          return -1;
        }
        if (status == -2)
        {
          fprintf(stderr,"Problem : %s -  One SINGLE argument should contain 'mI%d'\n",
                  pd->nickname,i);
          return -1;
        }
        sprintf(buf,"nI%d",i);
        status = lookup(mnemonic_list,buf);
        if (status == -1)
        {
          fprintf(stderr,"Problem : %s - One argument should contain 'nI%d'\n",
                  pd->nickname,i);
          return -1;
        }
        if (status == -2)
        {
          fprintf(stderr,"Problem : %s - One SINGLE argument should contain 'nI%d'\n",
                  pd->nickname,i);
          return -1;
        }
        sprintf(buf,"lI%d",i);
        status = lookup(mnemonic_list,buf);
        if (status == -1)
        {
          fprintf(stderr,"Problem : %s - One argument should contain 'lI%d'\n",
                  pd->nickname,i);
          return -1;
        }
        if (status == -2)
        {
          fprintf(stderr,"Problem : %s -One SINGLE argument should contain 'lI%d'\n",
                  pd->nickname,i);
          return -1;
        }
        break;
      }
      
      case NETSOLVE_SPARSEMATRIX :
      {
        sprintf(buf,"mI%d",i);
        status = lookup(mnemonic_list,buf);
        if (status == -1)
        {
          fprintf(stderr,"Problem : %s - One argument should contain 'mI%d'\n",
                  pd->nickname,i);
          return -1;
        }
        if (status == -2)
        {
          fprintf(stderr,"Problem : %s -  One SINGLE argument should contain 'mI%d'\n",
                  pd->nickname,i);
          return -1;
        }
        sprintf(buf,"nI%d",i);
        status = lookup(mnemonic_list,buf);
        if (status == -1)
        {
          fprintf(stderr,"Problem : %s - One argument should contain 'nI%d'\n",
                  pd->nickname,i);
          return -1;
        }
        if (status == -2)
        {
          fprintf(stderr,"Problem : %s - One SINGLE argument should contain 'nI%d'\n",
                  pd->nickname,i);
          return -1;
        }

        sprintf(buf,"fI%d",i);
        status = lookup(mnemonic_list,buf);
        if (status == -1)
        {
          fprintf(stderr,"Problem : %s - One argument should contain 'fI%d'\n",
                  pd->nickname,i);
          return -1;
        }
        if (status == -2)
        {
          fprintf(stderr,"Problem : %s -  One SINGLE argument should contain 'fI%d'\n",
                  pd->nickname,i);
          return -1;
        }

        sprintf(buf,"pI%d",i);
        status = lookup(mnemonic_list,buf);
        if (status == -1)
        {
          fprintf(stderr,"Problem : %s - One argument should contain 'pI%d'\n",
                  pd->nickname,i);
          return -1;
        }
        if (status == -2)
        {
          fprintf(stderr,"Problem : %s -  One SINGLE argument should contain 'pI%d'\n",
                  pd->nickname,i);
          return -1;
        }

        sprintf(buf,"iI%d",i);
        status = lookup(mnemonic_list,buf);
        if (status == -1)
        {
          fprintf(stderr,"Problem : %s - One argument should contain 'iI%d'\n",
                  pd->nickname,i);
          return -1;
        }
        if (status == -2)
        {
          fprintf(stderr,"Problem : %s -  One SINGLE argument should contain 'iI%d'\n",
                  pd->nickname,i);
          return -1;
        }
        break;
      }
      case NETSOLVE_VECTOR :
      case NETSOLVE_PACKEDFILES :
      case NETSOLVE_STRINGLIST :
      {
        sprintf(buf,"mI%d",i);
        status = lookup(mnemonic_list,buf);
        if (status == -1)
        {
          fprintf(stderr,"Problem : %s - One argument should contain 'mI%d'\n",
                  pd->nickname,i);
          return -1;
        }
        if (status == -2)
        {
          fprintf(stderr,"Problem : %s -  One SINGLE argument should contain 'mI%d'\n",
                  pd->nickname,i);
          return -1;
        }
        break;
      }
      case NETSOLVE_SCALAR :
      case NETSOLVE_UPF :
      case NETSOLVE_FILE :
      case NETSOLVE_STRING :
      {
        break; /* Nothing special here */
      }
    }
  }

  for (i=0;i<pd->nb_output_objects;i++)
  {
    int object_type;
    int status;
    char buf[10];

    object_type = pd->output_objects[i]->object_type;

    sprintf(buf,"O%d",i);
    status = lookup(mnemonic_list,buf);
    if (status == -1)
    {
      fprintf(stderr,"Problem : %s - One argument should contain 'O%d'\n",
              pd->nickname,i);
      return -1;
    }
    if (status == -2)
    {
      fprintf(stderr,"Problem : %s - One SINGLE argument should contain 'O%d'\n",
              pd->nickname,i);
      return -1;
    }
    switch (object_type)
    {
      case NETSOLVE_MATRIX :
      {
        sprintf(buf,"mO%d",i);
        status = lookup(mnemonic_list,buf);
        if (status == -2)
        {
          fprintf(stderr,"Problem : %s - One SINGLE argument should contain 'mO%d'\n",
                  pd->nickname,i);
          return -1;
        }
        sprintf(buf,"nO%d",i);
        status = lookup(mnemonic_list,buf);
        if (status == -2)
        {
          fprintf(stderr,"Problem : %s - One SINGLE argument should contain 'nO%d'\n",
                  pd->nickname,i);
          return -1;
        }
        sprintf(buf,"lO%d",i);
        status = lookup(mnemonic_list,buf);
        if (status == -2)
        {
          fprintf(stderr,"Problem : %s - One SINGLE argument should contain 'lO%d'\n",
                  pd->nickname,i);
          return -1;
        }
        break;
      }
      case NETSOLVE_SPARSEMATRIX :
      {
        sprintf(buf,"mO%d",i);
        status = lookup(mnemonic_list,buf);
        if (status == -2)
        {
          fprintf(stderr,"Problem : %s - One SINGLE argument should contain 'mO%d'\n",
                  pd->nickname,i);
          return -1;
        }
        sprintf(buf,"nO%d",i);
        status = lookup(mnemonic_list,buf);
        if (status == -2)
        {
          fprintf(stderr,"Problem : %s - One SINGLE argument should contain 'nO%d'\n",
                  pd->nickname,i);
          return -1;
        }
        sprintf(buf,"fO%d",i);
        status = lookup(mnemonic_list,buf);
        if (status == -2)
        {
          fprintf(stderr,"Problem : %s - One SINGLE argument should contain 'fO%d'\n",
                  pd->nickname,i);
          return -1;
        }
        break;
      }
      case NETSOLVE_STRINGLIST :
      {
        fprintf(stderr,"String list in output not supported yet\n");
        return -1;
      }
      case NETSOLVE_PACKEDFILES :
      case NETSOLVE_VECTOR :
      {
        sprintf(buf,"mO%d",i);
        status = lookup(mnemonic_list,buf);
        if (status == -2)
        {
          fprintf(stderr,"Problem : %s - One SINGLE argument should contain 'mO%d'\n",
                  pd->nickname,i);
          return -1;
        }
        break;
      }
      case NETSOLVE_SCALAR :
      case NETSOLVE_STRING :
      case NETSOLVE_FILE :
      case NETSOLVE_UPF :
      {
        break; /* Nothing special here */
      }
    }
  }
        return 1;
}

/*
 * lookup()
 *
 * Returns -1 if the mnemonic is not in the list
 * Returns -2 if the mnemonic is more than once in the list
 */
int lookup(char** list, char *string)
{
  char *token;
  int i=0;
  int result=-1;
  char *buf;

  while(list[i] != NULL)
  {

    buf = strdup(list[i]);

    token = strtok(buf," \t,");
    while(token != NULL)
    {
      if (strcmp(string,token) == 0)
      {
        if (result == -1)
          result = 1;
        else
          result = -2;
      }
      token = strtok(NULL," \t,");
    }
    free(buf);
    i++;
  }
  return result;
}

/*
 * checkMnemonic()
 *
 *   return -1 if a mnemonic is incorrectly specified
 *   return 1 if the mnemonic specifie an integer
 *   return 2 if the mnemonic specifies a pointer
 *
 */
int checkMnemonic(char* string,NS_ProblemDesc *pd)
{

  switch(string[0])
  {
    case '?' :  /* Ignored mnemonic */
    {
      return 1;
    }
    case 'I' :
    {
      int index;

      if (!isInteger((char *)string+1))
      {
        fprintf(stderr,"'%s' : invalid mnemonic at line %d\n",string,linecount);
        return -1;
      }
      index = atoi((char *)string+1);

      if (index >= pd->nb_input_objects)
      {
        fprintf(stderr,
           "Line %d :Only %d objects in input to %s in mnemonic '%s'\n",
           linecount,pd->nb_input_objects,pd->nickname,string);
        return -1;
      }
      return 2;
    }
    case 'O' :
    {
      int index;

      if (!isInteger((char *)string+1))
      {
        fprintf(stderr,"'%s' : invalid mnemonic at line %d\n",string,linecount);
        return -1;
      }
      index = atoi((char *)string+1);

      if (index >= pd->nb_output_objects)
      {
        fprintf(stderr,
           "Line %d :Only %d objects in output to %s in mnemonic '%s'\n",
           linecount,pd->nb_output_objects,pd->nickname,string);
        return -1;
      }
      return 2;
    }
    case 'm' :
    {
      switch (string[1])
      {
        case 'I' :            /* In input */
        {
          int index;
          int object_type;

          if (!isInteger((char *)string+2))
          {
            fprintf(stderr,"'%s' : invalid mnemonic at line %d\n",string,linecount);
            return -1;
          }
          index = atoi((char *)string+2);
          if (index >= pd->nb_input_objects)
          {
            fprintf(stderr,
              "Line %d :Only %d objects in input to %s in mnemonic '%s'\n",
              linecount,pd->nb_input_objects,pd->nickname,string);
            return -1;
          }
          object_type = pd->input_objects[index]->object_type;
          if ((object_type != NETSOLVE_MATRIX) &&
              (object_type != NETSOLVE_SPARSEMATRIX) &&
              (object_type != NETSOLVE_VECTOR) &&
              (object_type != NETSOLVE_STRINGLIST) &&
              (object_type != NETSOLVE_PACKEDFILES))
          {
            fprintf(stderr,
              "Line %d :Object #%d in input to %s does not have a 'm-dimension'\n",
              linecount,index,pd->nickname);
            return -1;
          }
          return 1;
        }
        case 'O' :             /* IN OUTPUT */
        {
          int index;
          int object_type;

          if (!isInteger((char *)string+2))
          {
            fprintf(stderr,"'%s' : invalid mnemonic at line %d\n",string,linecount);
            return -1;
          }
          index = atoi((char *)string+2);
          if (index >= pd->nb_output_objects)
          {
            fprintf(stderr,
              "Line %d :Only %d objects in output to %s in mnemonic '%s'\n",
              linecount,pd->nb_output_objects,pd->nickname,string);
            return -1;
          }
          object_type = pd->output_objects[index]->object_type;
          if ((object_type != NETSOLVE_MATRIX) &&
              (object_type != NETSOLVE_SPARSEMATRIX) &&
              (object_type != NETSOLVE_PACKEDFILES) &&
              (object_type != NETSOLVE_VECTOR))
          {
            fprintf(stderr,
              "Line %d :Object #%d in output to %s does not have a 'm-dimension'\n",
              linecount,index,pd->nickname);
            return -1;
          }
          return 1;
        }
        default :     /* ERROR */
        {
          fprintf(stderr,"Line %d : Invalid mnemonic '%s'\n",
                  linecount,string);
          return -1;
        }
      }
    }
    case 'n' :
    {
      switch (string[1])
      {
        case 'I' :            /* In input */
        {
          int index;
          int object_type;

          if (!isInteger((char *)string+2))
          {
            fprintf(stderr,"'%s' : invalid mnemonic at line %d\n",string,linecount);
            return -1;
          }
          index = atoi((char *)string+2);
          if (index >= pd->nb_input_objects)
          {
            fprintf(stderr,
              "Line %d :Only %d objects in input to %s in mnemonic '%s'\n",
              linecount,pd->nb_input_objects,pd->nickname,string);
            return -1;
          }
          object_type = pd->input_objects[index]->object_type;
          if (object_type != NETSOLVE_MATRIX && object_type != NETSOLVE_SPARSEMATRIX)
          {
            fprintf(stderr,
              "Line %d :Object #%d in input to %s does not have a 'n-dimension'\n",
              linecount,index,pd->nickname);
            return -1;
          }
          return 1;
        }
        case 'O' :             /* IN OUTPUT */
        {
          int index;
          int object_type;

          if (!isInteger((char *)string+2))
          {
            fprintf(stderr,"'%s' : invalid mnemonic at line %d\n",string,linecount);
            return -1;
          }
          index = atoi((char *)string+2);
          if (index >= pd->nb_output_objects)
          {
            fprintf(stderr,
              "Line %d :Only %d objects in output to %s in mnemonic '%s'\n",
              linecount,pd->nb_output_objects,pd->nickname,string);
            return -1;
          }
          object_type = pd->output_objects[index]->object_type;
          if (object_type != NETSOLVE_MATRIX && object_type != NETSOLVE_SPARSEMATRIX)
          {
            fprintf(stderr,
              "Line %d :Object #%d in output to %s does not have a 'n-dimension'\n",
              linecount,index,pd->nickname);
            return -1;
          }
          return 1;
        }
        default :     /* ERROR */
        {
          fprintf(stderr,"Line %d : Invalid mnemonic '%s'\n",
                  linecount,string);
          return -1;
        }
      }
    }
    case 'l' :
    {
      switch (string[1])
      {
        case 'I' :            /* In input */
        {
          int index;
          int object_type;

          if (!isInteger((char *)string+2))
          {
            fprintf(stderr,"'%s' : invalid mnemonic at line %d\n",string,linecount);
            return -1;
          }
          index = atoi((char *)string+2);
          if (index >= pd->nb_input_objects)
          {
            fprintf(stderr,
              "Line %d :Only %d objects in input to %s in mnemonic '%s'\n",
              linecount,pd->nb_input_objects,pd->nickname,string);
            return -1;
          }
          object_type = pd->input_objects[index]->object_type;
          if (object_type != NETSOLVE_MATRIX)
          {
            fprintf(stderr,
              "Line %d :Object #%d in input to %s does not have a 'leading dimension'\n",
              linecount,index,pd->nickname);
            return -1;
          }
          return 1;
        }
        case 'O' :             /* IN OUTPUT */
        {
          int index;
          int object_type;

          if (!isInteger((char *)string+2))
          {
            fprintf(stderr,"'%s' : invalid mnemonic at line %d\n",string,linecount);
            return -1;
          }
          index = atoi((char *)string+2);
          if (index >= pd->nb_output_objects)
          {
            fprintf(stderr,
              "Line %d :Only %d objects in output to %s in mnemonic '%s'\n",
              linecount,pd->nb_output_objects,pd->nickname,string);
            return -1;
          }
          object_type = pd->output_objects[index]->object_type;
          if (object_type != NETSOLVE_MATRIX)
          {
            fprintf(stderr,
              "Line %d :Object #%d in output to %s does not have a 'leading dimension'\n",
              linecount,index,pd->nickname);
            return -1;
          }
          return 1;
        }
        default :     /* ERROR */
        {
          fprintf(stderr,"Line %d : Invalid mnemonic '%s'\n",
                  linecount,string);
          return -1;
        }
      }
    }
    case 'f' :
    {
      switch (string[1])
      {
        case 'I' :            /* In input */
        {
          int index;
          int object_type;
 
          if (!isInteger((char *)string+2))
          {
            fprintf(stderr,"'%s' : invalid mnemonic at line %d\n",string,linecount);
            return -1;
          }
          index = atoi((char *)string+2);
          if (index >= pd->nb_input_objects)
          {
            fprintf(stderr,
              "Line %d :Only %d objects in input to %s in mnemonic '%s'\n",
              linecount,pd->nb_input_objects,pd->nickname,string);
            return -1;
          }
          object_type = pd->input_objects[index]->object_type;
          if (object_type != NETSOLVE_SPARSEMATRIX)
          {
            fprintf(stderr,
              "Line %d :Object #%d in input to %s does not have a 'number on filled positions'\n",
              linecount,index,pd->nickname);
            return -1;
          }
          return 1;
        }
        case 'O' :             /* IN OUTPUT */
        {
          int index;
          int object_type;
 
          if (!isInteger((char *)string+2))
          {
            fprintf(stderr,"'%s' : invalid mnemonic at line %d\n",string,linecount);
            return -1;
          }
          index = atoi((char *)string+2);
          if (index >= pd->nb_output_objects)
          {
            fprintf(stderr,
              "Line %d :Only %d objects in output to %s in mnemonic '%s'\n",
              linecount,pd->nb_output_objects,pd->nickname,string);
            return -1;
          }
          object_type = pd->output_objects[index]->object_type;
          if (object_type != NETSOLVE_SPARSEMATRIX)
          {
            fprintf(stderr,
              "Line %d :Object #%d in output to %s does not have a 'number of filled postions'\n",
              linecount,index,pd->nickname);
            return -1;
          }
          return 1;
        }
        default :     /* ERROR */
        {
          fprintf(stderr,"Line %d : Invalid mnemonic '%s'\n",
                  linecount,string);
          return -1;
        }
      }
    }
    case 'p' :
    {
      switch (string[1])
      {
        case 'I' :            /* In input */
        {
          int index;
          int object_type;
 
          if (!isInteger((char *)string+2))
          {
            fprintf(stderr,"'%s' : invalid mnemonic at line %d\n",string,linecount);
            return -1;
          }
          index = atoi((char *)string+2);
          if (index >= pd->nb_input_objects)
          {
            fprintf(stderr,
              "Line %d :Only %d objects in input to %s in mnemonic '%s'\n",
              linecount,pd->nb_input_objects,pd->nickname,string);
            return -1;
          }
          object_type = pd->input_objects[index]->object_type;
          if (object_type != NETSOLVE_SPARSEMATRIX)
          {
            fprintf(stderr,
              "Line %d :Object #%d in input to %s does not have a 'row/col pointer' field\n",
              linecount,index,pd->nickname);
            return -1;
          }
          return 1;
        }
        case 'O' :             /* IN OUTPUT */
        {
          int index;
          int object_type;
 
          if (!isInteger((char *)string+2))
          {
            fprintf(stderr,"'%s' : invalid mnemonic at line %d\n",string,linecount);
            return -1;
          }
          index = atoi((char *)string+2);
          if (index >= pd->nb_output_objects)
          {
            fprintf(stderr,
              "Line %d :Only %d objects in output to %s in mnemonic '%s'\n",
              linecount,pd->nb_output_objects,pd->nickname,string);
            return -1;
          }
          object_type = pd->output_objects[index]->object_type;
          if (object_type != NETSOLVE_SPARSEMATRIX)
          {
            fprintf(stderr,
              "Line %d :Object #%d in output to %s does not have a 'row/col pointer' field\n",
              linecount,index,pd->nickname);
            return -1;
          }
          return 1;
        }
        default :     /* ERROR */
        {
          fprintf(stderr,"Line %d : Invalid mnemonic '%s'\n",
                  linecount,string);
          return -1;
        }
      }
    }
    case 'i' :
    {
      switch (string[1])
      {
        case 'I' :            /* In input */
        {
          int index;
          int object_type;
 
          if (!isInteger((char *)string+2))
          {
            fprintf(stderr,"'%s' : invalid mnemonic at line %d\n",string,linecount);
            return -1;
          }
          index = atoi((char *)string+2);
          if (index >= pd->nb_input_objects)
          {
            fprintf(stderr,
              "Line %d :Only %d objects in input to %s in mnemonic '%s'\n",
              linecount,pd->nb_input_objects,pd->nickname,string);
            return -1;
          }
          object_type = pd->input_objects[index]->object_type;
          if (object_type != NETSOLVE_SPARSEMATRIX)
          {
            fprintf(stderr,
              "Line %d :Object #%d in input to %s does not have 'row/col indices'\n",
              linecount,index,pd->nickname);
            return -1;
          }
          return 1;
        }
        case 'O' :             /* IN OUTPUT */
        {
          int index;
          int object_type;
 
          if (!isInteger((char *)string+2))
          {
            fprintf(stderr,"'%s' : invalid mnemonic at line %d\n",string,linecount);
            return -1;
          }
          index = atoi((char *)string+2);
          if (index >= pd->nb_output_objects)
          {
            fprintf(stderr,
              "Line %d :Only %d objects in output to %s in mnemonic '%s'\n",
              linecount,pd->nb_output_objects,pd->nickname,string);
            return -1;
          }
          object_type = pd->output_objects[index]->object_type;
          if (object_type != NETSOLVE_SPARSEMATRIX)
          {
            fprintf(stderr,
              "Line %d :Object #%d in output to %s does not have a 'row/col indicess'\n",
              linecount,index,pd->nickname);
            return -1;
          }
          return 1;
        }
        default :     /* ERROR */
        {
          fprintf(stderr,"Line %d : Invalid mnemonic '%s'\n",
                  linecount,string);
          return -1;
        }
      }
    }
    default :
    {
      fprintf(stderr,"'%s' : invalid mnemonic at line %d\n",string,linecount);
      return -1;
    }
  }
}

/*
 * checkExpression()
 *
 *  Checks that :
 *    1. The expression is syntacticly correct
 *    2. The mnemonics used have been seen BEFORE
 *
 */

int checkExpression(char *string,char **mnemonic_list,int mnemonic_index,NS_ProblemDesc *pd)
{
  if (strncmp(string,"op(",3) == 0) /* OPERATOR */
  {
    char *op;
    char *arg1,*arg2;

    getOpArg1Arg2(string,&op,&arg1,&arg2,1);
    if (op == NULL)
    {
      fprintf(stderr,"Invalid operator in 'op' expression at %d\n",linecount);
      return -1;
    }
    if (arg1 == NULL)
    {
      fprintf(stderr,"Invalid first argument in 'op' expression at %d\n",
              linecount);
      return -1;
    }
    if (arg2 == NULL)
    {
      fprintf(stderr,"arg2 = NULL\n");
      fprintf(stderr,"Invalid second argument in 'op' expression at %d\n",
              linecount);
      return -1;
    }
    /* Examine the operator */
    if ((strcmp(op,"+") != 0) && 
        (strcmp(op,"-") != 0) &&
        (strcmp(op,"*") != 0))
    {
      fprintf(stderr,"Invalid operator in 'op' expression at %d\n",linecount);
      return -1;
    }
    if (checkExpression(arg1,mnemonic_list,mnemonic_index,pd) != 1)
    {
      fprintf(stderr,"Invalid first argument in 'op' expression at %d\n",
              linecount);
      return -1;
    }
    if (checkExpression(arg2,mnemonic_list,mnemonic_index,pd) != 1)
    {
      fprintf(stderr,"Invalid second argument in 'op' expression at %d\n",
              linecount);
      return -1;
    }
    free(op);free(arg1);free(arg2);
    return 1;

  }
  else if (strncmp(string,"array(",6) == 0) /* ARRAY ACCESS */
  {
    char *array;
    char *index;
    int number;
    
    getArrayIndex(string,&array,&index,1);
    if (array == NULL)
    {
      fprintf(stderr,"Invalid array in 'array' expression at %d\n",linecount);
      return -1;
    }
    if (index == NULL)
    {
      fprintf(stderr,"Invalid index in 'array' expression at %d\n",linecount);
      return -1;
    }
    /* Check the array */
    if (checkMnemonic(array,pd) < 0)
    {
      fprintf(stderr,"Invalid mnemonic '%s' at line %d\n",
                     array,linecount);
      return -1;
    }
    if (checkMnemonic(array,pd) != 2)
    {
      fprintf(stderr,"'%s' should be an array at line %d\n",
                     array,linecount);
      return -1;
    }
    if (array[0] == 'O')
    {
      fprintf(stderr,
              "Only input parameters can be used in expressions at %d\n",
              linecount);
      return -1;
    }
    number = atoi(array+1);
    if ((pd->input_objects[number]->data_type != NETSOLVE_I)&&
        (pd->input_objects[number]->data_type != NETSOLVE_CHAR))
    {
      fprintf(stderr,"The array accessed at %d should be of",linecount);
      fprintf(stderr," integers or chars\n");
      return -1;
    }
    if (lookup(mnemonic_list,array) ==  -1)
    {
      fprintf(stderr,"First time mnemonic '%s' seen at line %d\n",
                     array,linecount);
      return -1;
    }
    /* -2 is impossible as a returned value of lookup() */
    /* Therefore the array is okay here                 */

    /* Check the index */
    if (checkExpression(index,mnemonic_list,mnemonic_index,pd) != 1)
    {
      fprintf(stderr,"'%s' is an invalid index expression at line %d\n",
                     index,linecount);
      return -1;
    }
    /* Everything os OK */
    free(array);free(index);
    return 1;
  }
  else if (strncmp(string,"'",1) == 0) /* CHARACTER CONSTANT */
  {
    if (strlen(string) != 3)
    {
      fprintf(stderr,"Invalid character constant %s at line %d\n",
              string,linecount);
      return -1;
    }
    if (string[2] != '\'')
    {
      fprintf(stderr,"Invalid character constant %s at line %d\n",
              string,linecount);
      return -1;
    }
    return 1;
  }
  else if (strncmp(string,"if(",3) == 0) /* CONDITIONAL */
  {
    char *lhs;
    char *rhs;
    char *ifstmt;
    char *elsestmt;
    char *start = string + 3;


    if (getIfThenElse(start,&lhs,&rhs,&ifstmt,&elsestmt,1) == -1)
      return -1;

    if (checkExpression(lhs,mnemonic_list,mnemonic_index,pd) != 1)
    {
      fprintf(stderr," The Left Hand side of the expression is invalid on line %d\n",linecount);
      return -1;
    }
    if (checkExpression(rhs,mnemonic_list,mnemonic_index,pd) != 1)
    {
      fprintf(stderr," The Right hand side of the  expression is invalid on line %d\n",linecount);
      return -1;
    }
    if (checkExpression(ifstmt,mnemonic_list,mnemonic_index,pd) != 1)
    {
      fprintf(stderr," The If clause is invalid on line %d\n",linecount);
      return -1;
    }
    if (checkExpression(elsestmt,mnemonic_list,mnemonic_index,pd) != 1)
    {
      fprintf(stderr," The Else clause is invalid on line %d\n",linecount);
      return -1;
    }
    free(lhs);free(rhs);free(ifstmt);free(elsestmt);
    return 1;
  }
  else if (isInteger(string))
  {
    return 1;
  }  
  else /* MNEMONIC : must be integer */
  {
    if (checkMnemonic(string,pd) < 0)
    {
      fprintf(stderr,"Invalid mnemonic '%s' at line %d\n",string,linecount);
      return -1;
    }
    if (checkMnemonic(string,pd) != 1)
    {
      fprintf(stderr,"mnemonic '%s' at line %d is not an integer\n",
              string,linecount);
      return -1;
    }
    if ((string[0] == 'O')||(string[1] == 'O'))
    {
      fprintf(stderr,
              "Only input parameters can be used in expressions at %d\n",
              linecount);
      return -1;
    }
    if (lookup(mnemonic_list,string) == -1)
    {
      fprintf(stderr,"First time mnemonic '%s' seen at line %d\n",
                     string,linecount);
      return -1;
    }
    return 1;
  }
   
}

