/*****************************************************************/
/* netslX.c                                                      */
/* Version 1.0                                   Henri Casanova  */
/* sequencing                                    Dieter Bachmann */
/* now called from all C and Fortran API calls     Dorian Arnold */
/*---------------------------------------------------------------*/
/*  netslX()                                                     */
/*****************************************************************/

/*
 * netslX() : vararg C Interface to the NetSolve Client
 */

#include <stdarg.h>
#include "core.h"
#include "client.h"
#include "mnemonics.h"
#include "callingsequence.h"
#include "cfortranclient.h"
#include "sequences.h"

int netslX(char *buf, va_list arglist,
           int language, int blocking, int assignment)
{
  int i, len;
  int request_id;
  int status;
  void **tmp_output_ptr;
  int elapsed;
  NS_Node *act_node = 0;
  NS_ProblemDesc *pd;
  NS_RequestDesc *rd;
  char *nickname, *serverhostname;

#if defined(DEBUG)
  ns_printinfo();
  fprintf(stderr, "Processing a Client Request: language = %d, "
                  "blocking = %d, assignment = %d\n", language,
          blocking, assignment);
#endif

  /* find a slot for the request */
  request_id = 0;
  while(requests[request_id] != NULL)
  {
    request_id++;
    if (request_id == NB_MAX_REQUESTS)
    {
      ns_errno = NetSolveTooManyPendingRequests;
      return ns_errno;
    }
  }
 
  /*== proxy needs to be started ==*/
  if(!proxy_pid){
    if(netslinit(NULL) < 0){
      return ns_errno;
    }
  }

#if defined(DEBUG)
  ns_printinfo();
  fprintf(stderr, "Setting Default Client Major ...\n");
#endif
  /*== Initializing the major ==*/
  if(language == NS_CALL_FROM_C)
    setMajorDefault("Row");
  else
    setMajorDefault("Col");

#if defined(DEBUG)
  ns_printinfo();
  fprintf(stderr, "Stripping `(` and `)` from problemname %s ...\n",
          buf);
#endif
  /*== Stripping off the parenthesis ==*/
  len = strlen(buf);
  if ((buf[len-2] != '(' ) ||
      (buf[len-1] != ')' ) )
  {
#if defined(DEBUG)
  ns_printinfo();
  fprintf(stderr, "There is a bad problem name...\n"); fflush(stderr);
#endif
    free(buf);
    ns_errno = NetSolveBadProblemName;
    return ns_errno;
  }
  buf[len-2] = '\0';

#if defined(DEBUG)
  ns_printinfo();
  fprintf(stderr, "Extracting the problem name [and serverhostname] ...\n");
  fflush(stderr);
#endif
  if(assignment){
    /* Extracting the nickname, serverhostname */
    serverhostname = strtok(buf,":");
    nickname = strtok(NULL,"\0");

    if ((serverhostname == NULL)||
        (nickname == NULL))
    {
      free(buf);
      return NetSolveBadProblemName;
    }
  }
  else{
    nickname = buf;
    serverhostname = NULL;
  }

#if defined(DEBUG)
  ns_printinfo();
  fprintf(stderr, "Problem name: %s\n", nickname);
  if(assignment){
    ns_printinfo();
    fprintf(stderr, "Assigned Server: %s\n", serverhostname);
  }
#endif

#if defined(WIN32)
  /*== Initialize WinSock ==*/
  if ((!WinSockInit()))
  {
    ns_errno = NetSolveNetworkError;
    return ns_errno;
  }
#endif /* WIN32 */

#if defined(DEBUG)
  ns_printinfo();
  fprintf (stderr, "Getting Problem Descriptor via netsolveInfo()\n");
#endif

  /*== Getting the problem descriptor ==*/
  status = netsolveInfo(nickname, &pd);
  free(nickname);
  if (status == -1)
    return ns_errno;

#if defined(DEBUG)
  ns_printinfo();
  fprintf (stderr, "Initializing NetSolve objects: %d in input and %d"
                    "in output\n",
                    pd->nb_input_objects, pd->nb_output_objects);
#endif

  /*== Initializing the objects and set Major==*/
  for (i=0;i<pd->nb_input_objects;i++)
  {
    int object_type,data_type;
    object_type = pd->input_objects[i]->object_type;
    data_type = pd->input_objects[i]->data_type;
    freeObject(pd->input_objects[i]);
    pd->input_objects[i] = createEmptyObject(object_type,data_type);

    if (pd->input_objects[i]->object_type == NETSOLVE_MATRIX)
      pd->input_objects[i]->attributes.matrix_attributes.major = my_major;
    else if(pd->input_objects[i]->object_type == NETSOLVE_SPARSEMATRIX)
      pd->input_objects[i]->attributes.sparsematrix_attributes.major = my_major;
    else
      continue;
  }
  for (i=0;i<pd->nb_output_objects;i++)
  {
    int object_type,data_type;
    object_type = pd->output_objects[i]->object_type;
    data_type = pd->output_objects[i]->data_type;
    freeObject(pd->output_objects[i]);
    pd->output_objects[i] = createEmptyObject(object_type,data_type);

    if (pd->output_objects[i]->object_type == NETSOLVE_MATRIX)
      pd->output_objects[i]->attributes.matrix_attributes.major = my_major;
    else if(pd->output_objects[i]->object_type == NETSOLVE_SPARSEMATRIX)
      pd->output_objects[i]->attributes.sparsematrix_attributes.major = my_major;
    else 
      continue;
  }

#if defined(DEBUG)
  ns_printinfo();
  fprintf (stderr, "Parsing the call stack... call StackToObjects()\n");
#endif

  /*== Parsing the stack ==*/
  if (StackToObjects(language, pd, arglist,
                     pd->input_objects, pd->output_objects) == -1)
  {
    freeProblemDesc(pd);
    return ns_errno;
  }

#if defined(DEBUG)
  ns_printinfo();
  fprintf (stderr, "Assigning the Constant Values ... call "
                   "assignConstantValues()\n");
#endif

  /*== Processing the constants ==*/
  status = assignConstantValues(pd,pd->input_objects,pd->output_objects);
  if (status == -1)
  {
    freeProblemDesc(pd);
    return ns_errno;
  }

#if defined(DEBUG)
  ns_printinfo();
  fprintf (stderr, "Assigning the Formula Values ... call "
                   "assignFormulaValues()\n");
#endif

  /*== Processing the formulae ==*/
  status = assignFormulaValues(pd,pd->input_objects,pd->output_objects);
  if (status == -1)
  {
    freeProblemDesc(pd);
    return ns_errno;
  }

#if defined(DEBUG)
  ns_printinfo();
  fprintf (stderr, "Special Preprocessing for packed files ... \n");
#endif

  /*== Special preprocessing for the packed file type ==*/
  for (i=0;i<pd->nb_input_objects;i++)
  {
    if (pd->input_objects[i]->object_type == NETSOLVE_PACKEDFILES){
      pd->input_objects[i]->attributes.packedfiles_attributes.filenames =
      copyCharStarList(
        pd->input_objects[i]->attributes.packedfiles_attributes.filenames,
        pd->input_objects[i]->attributes.packedfiles_attributes.m);
    }
  }

  for (i=0;i<pd->nb_output_objects;i++)
  {
    if (pd->output_objects[i]->object_type == NETSOLVE_PACKEDFILES)
    {
      pd->output_objects[i]->attributes.packedfiles_attributes.defaultprefix =
        (char *)(pd->output_objects[i]->attributes.packedfiles_attributes.filenames);
      pd->output_objects[i]->attributes.packedfiles_attributes.filenames = NULL;
    }
  }

#if defined(DEBUG)
  ns_printinfo();
  fprintf (stderr, "Saving the Output pointers for strings ... \n");
#endif

  /*== Saving the tmp_output_ptr ==*/
  tmp_output_ptr = (void **)calloc(pd->nb_output_objects,sizeof(void*));
  for (i=0;i<pd->nb_output_objects;i++)
  {
    if (pd->output_objects[i]->object_type == NETSOLVE_STRING)
      tmp_output_ptr[i] = 
        pd->output_objects[i]->attributes.string_attributes.ptr;
  }

  if(netsl_sequence_status() == FALSE)  {
#if defined(DEBUG)
  ns_printinfo();
  fprintf (stderr, "Finally, about to submit the request ... \n");
#endif
    status = submit_problem(blocking, assignment, serverhostname, pd,
                            pd->input_objects, pd->output_objects, &elapsed, request_id);

    if(blocking){
#if defined(DEBUG)
      ns_printinfo();
      fprintf(stderr, "Blocking Call Returned, copying string outputs "
                      "...\n");
#endif
      if (status != -1) {
        /*== take care of the strings in output ==*/
        for (i=0; i<pd->nb_output_objects; i++) {
          if (pd->output_objects[i]->object_type != NETSOLVE_STRING)
            continue;
          strcpy(tmp_output_ptr[i],
              pd->output_objects[i]->attributes.string_attributes.ptr);
          free(pd->output_objects[i]->attributes.string_attributes.ptr);
        }
        ns_errno = NetSolveOK;
      }

      free(tmp_output_ptr);
      freeProblemDesc(pd);
      return ns_errno;
    }
    else{   /*== NOBLOCK ==*/
#if defined(DEBUG)
      ns_printinfo();
      fprintf(stderr, "Non-Blocking Call Returned, saving Request "
                      "Descriptor ...\n");
#endif
      if (status < 0) {
        free(tmp_output_ptr);
        freeProblemDesc(pd);
        return ns_errno;
      }

      output_ptrs[request_id] = tmp_output_ptr;
      return NetSolveOK;
    }
  }
  else {
#if defined(DEBUG)
      ns_printinfo();
      fprintf(stderr, "In Sequencing mode, not yet scheduling request\n");
#endif
    /*== if neccessary generate a NS_Node and insert into structure ==*/
    act_node = netsl_new_Node_Dep(pd, pd->nb_input_objects, pd->nb_output_objects);
    netsl_insert_Node (act_node, netsl_get_Sequence());

    /*== Analyzing dependencies if sequencing mode ==*/
    if (netsl_analyze_Graph(act_node,*netsl_get_Sequence())!=TRUE){
      fprintf(stderr,"Failed to analyze dependencies, giving up\n");
      return ns_errno;
    }

    act_node->tmp_output_ptr = tmp_output_ptr;   /* store the string pointers */

    /*== do nothing, until netsl_sequence_end is reached ==*/
    return ns_errno;
  }

}
