/*****************************************************************/
/*      scheduler.c                                              */
/*      Henri Casanova                                           */
/*      modified by Karine Heydeman                              */
/*      modified by Susan Blackford and Patrick Geoffray         */
/*---------------------------------------------------------------*/
/*****************************************************************/

#include <sys/time.h>
#include "core.h"
#include "agentglobal.h"

#ifdef NWS
#include "nws_api.h"
#include "forecast_api.h"
#include "nwsutils.h"
#endif

#include "scheduler.h"
#include "netsolveutil.h"


/*
 * assignServer()
 */
void assignServer(NS_ProblemDesc *pd,int input_size, int output_size,
                  int problem_size, NS_ServerDesc ***chosen, 
                  int **predictionlist, int *nchosen)
{
  NS_MappingDesc **mlist;
  int nb,i;
  NS_ScoredServerDesc *scorelist;
  int count;
  time_t date;
  int idate;
 
  mlist = lookupMappingProblem(pd->nickname);

  nb = 0;
  while(mlist[nb] != NULL)
    nb++;
  
  if (nb == 0)
  {
#ifdef VIEW
    fprintf(stderr,"No server for '%s'\n",pd->nickname);
#endif
    *nchosen = 0;
    *chosen = NULL;
    return;
  }
  scorelist = (NS_ScoredServerDesc *)calloc(nb,sizeof(NS_ScoredServerDesc));
  for (i=0;i<nb;i++)
  {
    scorelist[i].d = mlist[i]->server_desc;
  }

  /* Weed out the servers that are "down" */
  for (i=0;i<nb;i++)
  {
    if ( (((NS_ServerDesc *)(mlist[i]->server_desc))->server_status == SERVER_DOWN) ){
      scorelist[i].score = -1;
      fprintf(stdout, "Weeding out %s (host error reported): status = %d, failures = %d\n", 
      ((NS_ServerDesc *)(mlist[i]->server_desc))->host_desc->hostname,
      ((NS_ServerDesc *)(mlist[i]->server_desc))->server_status,
      mlist[i]->nb_failures);
    }
  }

  free(mlist);

  /* compute the actual score */
  for (i=0;i<nb;i++)
  {
    if (scorelist[i].score != -1)
      scorelist[i].score = computeScore(pd,input_size,
              output_size,problem_size,scorelist[i].d,scorelist+i);
  }
 
  /* Sort */
  qsort(scorelist,nb,sizeof(NS_ScoredServerDesc),scoreCompare);

  /* Remove the ones with -1 score and send back the rest */
  count = 0;
  for (i=0;i<nb;i++)
    if (scorelist[i].score != -1)
      count++;

  *chosen = (NS_ServerDesc **)calloc(count,sizeof(NS_ServerDesc*));
  *predictionlist = (int *)calloc(count,sizeof(int));
  count = 0;
  for (i=0;i<nb;i++)
  { 
    if (scorelist[i].score != -1)
    {
      (*chosen)[count] = scorelist[i].d;
      (*predictionlist)[count] = scorelist[i].score;
      count++;
    }
  }

  /* HEURISTIC: Increase the workload of the first one by 100 */
  time(&date);
  idate = (int)date;
  
  if (count != 0)
  {
    addToWorkloadHistory((*chosen)[0]->workload_history,
			 (int)(scorelist[0].forecast*100/ 
(100+scorelist[0].forecast/(*chosen)[0]->host_desc->number_proc)),
idate+10+scorelist[0].sendtime);
  }

  /* free some memory */
  free(scorelist);
  /* fprintf(stderr, " date : %d, score :%d,server : %s \n",idate,(*predictionlist)[0],(*chosen)[0]->host_desc->hostname);  */
  *nchosen = count;
 
  return;
}

/*
 * ScoreCompare()
 */
#ifdef sunos
int scoreCompare(void *s1, void *s2)
#else
int scoreCompare(const void *s1, const void *s2)
#endif
{
  NS_ScoredServerDesc *d1=(NS_ScoredServerDesc *)s1;
  NS_ScoredServerDesc *d2=(NS_ScoredServerDesc *)s2;

  if (d1->score > d2->score)
    return 1;
  if (d1->score < d2->score)
    return -1;
  return 0;
}

/*
 * computeScore()
 */
int computeScore(NS_ProblemDesc *pd,int input_size,int output_size,
                  int problem_size,NS_ServerDesc *sd,NS_ScoredServerDesc *score)
{ int ForeCast;
  int communication;
  int computation;

  communication = computeCommunicationCost(input_size,output_size,sd);
  computation   = computeComputationCost(pd,problem_size,sd,&ForeCast);

  /* fprintf(stderr,"%s: %d s.\n",sd->host_desc->hostname,
	  communication+computation); */
#ifdef NWS
  score->sendtime=computeCommunicationCost(input_size,0,sd);
  score->forecast=ForeCast;
#else
 score->sendtime=communication;
 score->forecast=sd->workload_history->w;
#endif  
 return (communication+computation);
}

/*
 * computeCommunicationCost()
 *
 * returns an estimate (in usec) of the time necessary
 * to send input and retrieve output from a server
 */
int computeCommunicationCost(int input_size,int output_size,NS_ServerDesc *sd)
{ 
  int bandwidth,latency;
  int result;

  /* We need to modify here to receive latency/bandwidth from NWS
     and use that for the calculation of the CommunicationCost */

  bandwidth = sd->network_history->bandwidth;
  latency = sd->network_history->latency;
 
  if ((bandwidth == -1)||(latency == -1))
  {
#ifdef VIEW
    fprintf(stderr,"Network statistics not available yet for '%s'\n",
                      sd->host_desc->hostname);
#endif
    return 10000;
  }

  if (bandwidth != 0){
    result = (input_size+output_size)/bandwidth+latency/1000000;
  }
  else
    result = 10000;

  return result;
}

/*
 * computeComputationCost()
 *
 * returns an estimate (in usec) of the time required to
 * perform a computation on a server.
 */
int computeComputationCost (NS_ProblemDesc *pd, int problem_size,
                            NS_ServerDesc *sd,int * ForeCast)
{ 
  int speed;
  int workload;
  int nb_proc;
  double nb_flops;
  int  effective_speed;

  speed = sd->host_desc->speed;
#ifdef NWS
 workload=extractNWSworkload(sd);
 fprintf(stderr,"NWS workload : %d\n", workload);
 fprintf(stderr,"NetSolve workload : %d\n",sd->workload_history->w);
 *ForeCast=workload;
#else  
 workload = sd->workload_history->w;
 
#endif
  nb_proc = sd->host_desc->number_proc;
  nb_flops = evalFlops(pd,problem_size)/(double)(1000);

  /* calculation per eijkhout - mm */
  effective_speed = speed*nb_proc/((workload/100)+1);

#ifdef DEBUG
  fprintf(stderr,"speed = %d\t",speed);
  fprintf(stderr,"workload = %d\n",workload);
  fprintf(stderr,"effective_speed = %d\t",effective_speed);
  fprintf(stderr,"nb_flops = %.0f\n",nb_flops);
  fprintf(stderr,"computation cost = %f\n",(nb_flops/effective_speed)); 
#endif

  return (int)(nb_flops/effective_speed);
}

/*
 * evalFlops()
 */
double evalFlops(NS_ProblemDesc *pd, int problem_size)
{
  int a,b;
  char *s;
  char *token1,*token2;

  s = strdup(pd->complexity);
  token1 = strtok(s,",");
  token2 = strtok(NULL,"\0");

  a = atoi(token1);
  b = atoi(token2);
  free(s);
  
  return ((double)a)*power(problem_size,b);
}

#ifdef NWS

int extractNWSworkload (NS_ServerDesc* sd)
{
 NWSAPI_ForecastCollection ForeCast;
 const NWSAPI_SeriesSpec * AvailCPUServer;
 int ok;
    
/* which experiment */
 AvailCPUServer= NWSAPI_MakeSeriesSpec(sd->host_desc->hostname,NULL,NWSAPI_DEFAULT_AVAILABLE_CPU_RESOURCE);
    
    if (AvailCPUServer == NULL){
      printf("NWSAPI_MakeSeriesSpec\n");
      return -1;
 }
 /* get forecast for the server */ 
 ok=NWSAPI_GetForecast(AvailCPUServer->resourceName,&ForeCast);
 if (ok==0){
   printf("\n NWS GetForecast for the server ...\n");
   /*
   AdminNWSFore(global.my_self->host_desc->hostname,what,"test");
    if (strncmp(*what,"dead",strlen("dead"))==0)
   AdminNWSFore(global.my_self->host_desc->hostname,what,"revive");
   */
   return sd->workload_history->w;
 }
 if ((sd->workload_history->time) <= (ForeCast.measurement.timeStamp))
   {
     return Bestforecast(&ForeCast);
   }
 /*NWSAdmin(sd,"cycle");*/  
   return sd->workload_history->w;
 
}




/* Bestforecast return the forecast of Forecast  whose error is the smallest */

int Bestforecast(NWSAPI_ForecastCollection  *Forecast)
{ 
  if ((*Forecast).forecasts[0].error < (* Forecast).forecasts[1].error )
    {
      fprintf(stderr,"NWS Fore : %f\n",(*Forecast).forecasts[0].forecast*100);
      return (int) ((*Forecast).forecasts[0].forecast * 100);
    }
  else
    {
      fprintf(stderr,"NWS Fore : %f\n",(*Forecast).forecasts[1].forecast*100);
      return (int) ((*Forecast).forecasts[1].forecast *100);
    }
}


int extractNWSbandwidth (NS_ServerDesc* sd)
{
  NWSAPI_ForecastCollection ForeCast;
 const NWSAPI_SeriesSpec *ServerAgent; 
 int ok;
    
/* which experiment */
 ServerAgent=NWSAPI_MakeSeriesSpec(sd->host_desc->hostname,global.my_self->host_desc->hostname,NWSAPI_DEFAULT_BANDWIDTH_RESOURCE);
    
    if (ServerAgent == NULL){
      printf("NWSAPI_MakeSeriesSpec\n");
      return -1;
 }
 /* get forecast for the server */ 
 ok=NWSAPI_GetForecast(ServerAgent->resourceName,&ForeCast);
 if (ok==0){
   printf("getforecast\n");
   return -1;
 }
  return Bestforecast(&ForeCast)/100;

}

int extractNWSlatency (NS_ServerDesc* sd)
{
 NWSAPI_ForecastCollection ForeCast;
 const NWSAPI_SeriesSpec * ServerAgent; 
 int ok;
    
/* which experiment */
 ServerAgent= NWSAPI_MakeSeriesSpec(sd->host_desc->hostname,global.my_self->host_desc->hostname,NWSAPI_DEFAULT_LATENCY_RESOURCE);
    
    if (ServerAgent == NULL){
      printf("NWSAPI_MakeSeriesSpec\n");
      return -1;
 }
 /* get forecast for the server */ 
 ok=NWSAPI_GetForecast(ServerAgent->resourceName,&ForeCast);
 if (ok==0){
   printf("getforecast\n");
   return -1;
 }
  return Bestforecast(&ForeCast)/100;

}



#endif
