/*****************************************************************/
/*      agent_process_message.c                                  */
/*      Henri Casanova, Dorian Arnold                            */
/*---------------------------------------------------------------*/
/*  processMessage()                                             */
/*****************************************************************/

#include "core.h"
#include "agentglobal.h"
#include "agentprocessmessage.h"
#include "exchangemappings.h"
#ifdef NWS
# include "nws_api.h"
#endif
#include "scheduler.h"
#include <unistd.h>
#include "networkmeasurement.h"
#include <sys/types.h>
#include <netinet/in.h>
#include "workloadmanager.h"
#include "abort.h"

#ifdef INFOSERVER
# include "infoserver.h"
#endif

/*
 *  processMessage()
 *
 *  A message has been received. Let's see what it's about
 */
void processMessage(int sock)
{
  NS_Communicator *comm;
  int tag;

  SET_TIMEOUT(CONNECT_TIMEOUT);

  comm = acceptTransaction(sock);
  if (comm == NULL)
  {
    netsolvePerror("acceptTransaction()");
    UNSET_TIMEOUT();
    return;  
  }
  
  if (recvInt(comm,&tag) == -1)
  {
    netsolvePerror("recvInt()");
    endTransaction(comm);
    UNSET_TIMEOUT();
    return;
  }

  UNSET_TIMEOUT();

  fflush(stderr);
  switch(tag)
  {
    case NS_PROT_ARE_YOU_THERE: 
      processAreYouThere(comm);
      endTransaction(comm);
      break;
    case NS_PROT_NETWORK_REPORT:
      processNetworkReport(comm);
      endTransaction(comm);
      break;
    case NS_PROT_SV_REGISTER:
      processServerRegistration(comm);
      endTransaction(comm);
      break;
    case NS_PROT_AG_REGISTER:
      processAgentRegistration(comm);
      endTransaction(comm);
      break;
    case NS_PROT_WORKLOAD_REPORT:
      processWorkloadReport(comm);
      endTransaction(comm);
      break;
    case NS_PROT_NB_SERVERS:
      processNumberOfServers(comm);
      endTransaction(comm);
      break;
    case NS_PROT_SV_INFO:
      processServerInfo(comm);
      endTransaction(comm);
      break;
    case NS_PROT_PROBLEM_INFO:
      processProblemInfo(comm);
      endTransaction(comm);
      break;
    case NS_PROT_STORE_HANDLE:
      processStoreHandle(comm);
      endTransaction(comm);
	break;
    case NS_PROT_GET_HANDLE:
      processGetHandle(comm);
      endTransaction(comm);
	break;
    case NS_PROT_KILL_SERVER:
      processKillServer(comm);
      endTransaction(comm);
      break;
    case NS_PROT_KILL_AGENT:
      processKillAgent(comm);
      endTransaction(comm);
      break;
    case NS_PROT_AGENT_LIST:
      processAgentList(comm);
      endTransaction(comm);
      break;
    case NS_PROT_SERVER_LIST:
      processServerList(comm);
      endTransaction(comm);
      break;
    case NS_PROT_PROBLEM_LIST:
      processProblemList(comm);
      endTransaction(comm);
      break;
    case NS_PROT_PROBLEM_SUBMIT:
      processProblemSubmit(comm);
      endTransaction(comm);
      break;
    case NS_PROT_SV_FAILURE:
      processServerFailure(comm);
      endTransaction(comm);
      break;
    case NS_PROT_JOB_COMPLETED:
      processJobCompleted(comm);
      endTransaction(comm);
      break;
    case NS_PROT_PONG_LATENCY:
      /* Do nothing */
      endTransaction(comm);
      break;
    case NS_PROT_PONG_BANDWIDTH: /* fork */
      processPongBandwidth(comm);
      endTransaction(comm);
      break;
    default:
      if (sendInt(comm,NS_PROT_PROTOCOL_ERROR) == -1)
        netsolvePerror("sendInt()");
      endTransaction(comm);
      break;
  }
  return;
}

/*
 * processNumberOfServers()
 */
void processNumberOfServers(NS_Communicator *comm)
{
  char *nickname;
  NS_MappingDesc **mlist;
  int i;
  int nb;

  if(fork())
    return;

  SET_TIMEOUT(TRANS_TIMEOUT);
  close(global.sock);

  if (recvString(comm,&nickname) == -1)
  {
    netsolvePerror("recvString()");
    free(nickname);
    endTransaction(comm);
    exit(-1);
  }

  mlist = lookupMappingProblem(nickname);
  free(nickname);

  i = 0;
  nb = 0;
  while(mlist[i] != NULL)
  {
    if (mlist[i]->nb_failures == 0)
      nb++;
    i++;
  }
  free(mlist);
  if (sendInt(comm,nb) == -1)
  {
    netsolvePerror("sendInt()");
    endTransaction(comm);
    exit(-1);
  }

  exit(0);
}

/*
 * processNetworkReport()
 */
void processNetworkReport(NS_Communicator *comm)
{
  NS_IPaddr_type IPaddr1,IPaddr2;
  int latency,bandwidth;
  int time;
  NS_ServerDesc **slist;

  SET_TIMEOUT(TRANS_TIMEOUT);

  if ((recvIPaddr(comm,&IPaddr1) == -1) ||
      (recvIPaddr(comm,&IPaddr2) == -1) ||
      (recvInt(comm,&latency) == -1) ||
      (recvInt(comm,&bandwidth) == -1) ||
      (recvInt(comm,&time) == -1))
  {
    netsolvePerror("");
    UNSET_TIMEOUT();
    return;
  }
  
  /* For now, I am only interested in me-server distance */
  if (IPaddr1 == global.my_self->host_desc->IPaddr)
  {
    slist = lookupServer(&IPaddr2);
  }
  else if (IPaddr2 == global.my_self->host_desc->IPaddr)
  {
     slist = lookupServer(&IPaddr1);
  }
  else
  {
    UNSET_TIMEOUT();
    return;
  }

  if (*slist == NULL)
  {
#ifdef DEBUG
    fprintf(stderr,"WARNING: Received network report for an unknown server\n");
#endif
    free(slist);
    UNSET_TIMEOUT();
    return;
  }

  /*
   * Tell the InfoServer what's going on
   */
#ifdef INFOSERVER
  info_add_keyval("HOST", (*slist)->host_desc->hostname);
  info_add_keyval_int("LATENCY", latency);
  info_add_keyval_int("BANDWIDTH", bandwidth);
  info_submit();
#endif

#ifndef NO_OUTPUT
  fprintf(stderr,"Server %s: latency: %d   bandwidth: %d\n",
          (*slist)->host_desc->hostname,latency,bandwidth);
#endif
      
  addToNetworkHistory((*slist)->network_history,latency,bandwidth,time);
  free(slist);

  UNSET_TIMEOUT();
  return;
}

/*
 * processAreYouThere()
 */
void processAreYouThere(NS_Communicator *comm)
{
  if(fork())
    return;
  close(global.sock);

  SET_TIMEOUT(TRANS_TIMEOUT);
  if (sendInt(comm,NS_PROT_I_AM_THERE) == -1)
  {
    netsolvePerror("sendInt()");
    endTransaction(comm);
    exit(-1);
  }
  UNSET_TIMEOUT();

  endTransaction(comm);
  exit(0);
}


/*
 * processServerRegistration
 */
void processServerRegistration(NS_Communicator *comm)
{
  NS_ServerDesc *sd;
  NS_ProblemDesc **pds;
  int i,nb_prob;
  NS_ProblemDesc **plist;
  NS_ServerDesc **slist;
  int ok = 1;
  int nb_agents;
  NS_LinkItem *link;
  int *new;
  time_t cur_time;
  
  SET_TIMEOUT(TRANS_TIMEOUT);

  cur_time = time(0);
  sd = recvServerDesc(comm);
  if (sd == NULL)
  {
    netsolvePerror("recvServerDesc()");
    UNSET_TIMEOUT();
    return;
  }

  sd->server_status = SERVER_UP;
  sd->workload_history->time = (int)cur_time;
  fprintf(stderr, "NS_PROT_SV_REGISTER: Time %ld (%s), Hostname %s, "
                  "IP %u.%u.%u.%u\n", cur_time, Ctime(&cur_time),
                   sd->host_desc->hostname,
                   BYTE(&(sd->host_desc->IPaddr),0),
                   BYTE(&(sd->host_desc->IPaddr),1),
                   BYTE(&(sd->host_desc->IPaddr),2),
                   BYTE(&(sd->host_desc->IPaddr),3));

  /*
   * Tell the InfoServer
   */
#ifdef INFOSERVER
  info_add_keyval("HOST", sd->host_desc->hostname);
  info_add_keyval_ulong("IP", htonl(sd->host_desc->IPaddr));
  info_add_keyval("EVENT", "NS_PROT_SV_REGISTER");
  info_submit();
#endif

  /* Register problems */

  if (recvInt(comm,&nb_prob) == -1)
  {
    netsolvePerror("recvInt()");
    UNSET_TIMEOUT();
    return;
  }

  pds = (NS_ProblemDesc **)calloc(nb_prob,sizeof(NS_ProblemDesc *));
  new = (int *)calloc(nb_prob,sizeof(int));
  
  for(i=0;i<nb_prob;i++)
  {
    pds[i] = recvProblemDesc(comm);    
    if (pds[i] == NULL)
    {
      netsolvePerror("recvProblemDesc()");
      UNSET_TIMEOUT();
      return;
    }
    plist = lookupProblem(pds[i]->nickname);
    if (*plist == NULL)  /* New problem */
    {
      new[i] = 1;
      free(plist);
      continue;
    }
    else
    {
      if (!compareProblemDesc(*plist,pds[i]))
        ok = 0;
      free(pds[i]);
      pds[i] = *plist;
      free(plist);
    }
  }

  /* If a server is already running on this machine, deny registration */

  slist = lookupServer(&(sd->host_desc->IPaddr));
  if (*slist != NULL)
  {
    free(slist);
    fprintf(stdout, "NS_PROT_NOT_ALLOWED server register of %s\n",
            sd->host_desc->hostname);
#ifdef INFOSERVER
    info_add_keyval("HOST", sd->host_desc->hostname);
    info_add_keyval("EVENT", "NS_PROT_NOT_ALLOWED");
    info_submit();
#endif

    if (sendInt(comm,NS_PROT_NOT_ALLOWED) == -1)
    {
      netsolvePerror("sendInt()");
      UNSET_TIMEOUT();
      return;
    }
    UNSET_TIMEOUT();
    return;
  }

  /* If problem descriptions don't match, refuse registration */

  if (ok == 0)
  {
    fprintf(stdout, "NS_PROT_REGISTRATION_REFUSED\n");

#ifdef INFOSERVER
    info_add_keyval("HOST", sd->host_desc->hostname);
    info_add_keyval("EVENT", "NS_PROT_REGISTRATION_REFUSED");
    info_submit();
#endif

    if (sendInt(comm,NS_PROT_REGISTRATION_REFUSED) == -1)
    {
      netsolvePerror("sendInt()");
      UNSET_TIMEOUT();
      return;
    }
    UNSET_TIMEOUT();
    return;
  }

  addServer(sd);
  for (i=0;i<nb_prob;i++)
  {
    NS_MappingDesc *t;
    if (new[i])
    {
      fprintf(stderr,"New problem: %s\n",pds[i]->nickname);

      /*
       * Tell the inforserver what's going on
       */
#ifdef INFOSERVER
      info_add_keyval("HOST", sd->host_desc->hostname);
      info_add_keyval("EVENT", "NEW_PROBLEM");
      info_add_keyval("NICKNAME", pds[i]->nickname);
      info_submit();
#endif
      
      addProblem(pds[i]);
    }
    t = newMappingDesc(pds[i],sd);
    addMapping(t);
  }
  if (sendInt(comm,NS_PROT_REGISTRATION_ACCEPTED) == -1)
  {
    netsolvePerror("sendInt()");
    UNSET_TIMEOUT();
    return;
  }
  
  nb_agents = NScountList(global.agents);
  if (sendInt(comm,nb_agents) == -1)
  {
    netsolvePerror("sendInt()");
    UNSET_TIMEOUT();
    return;
  }
  link = global.agents->start;
  while(link != NULL)
  {
    if (sendAgentDesc(comm,(NS_AgentDesc *)(link->content)) == -1)
    {
      netsolvePerror("sendAgentDesc");
      UNSET_TIMEOUT();
      return;
    }
    link = link->next;
  }

  fprintf(stderr,"NS_PROT_REGISTRATION_ACCEPTED: %s\n",
	  sd->host_desc->hostname);

  /*
   * Tell the InfoServer too
   */
#ifdef INFOSERVER
  info_add_keyval("HOST", sd->host_desc->hostname);
  info_add_keyval("EVENT", "NS_PROT_REGISTRATION_ACCEPTED");
  info_submit();
#endif

  free(pds);
  UNSET_TIMEOUT();
  return; 
}

/*
 * processAgentRegistration()
 */
void processAgentRegistration(NS_Communicator *comm)
{
  NS_AgentDesc *ad;
  NS_AgentDesc **alist;
  int nb_agents,nb_servers,nb_problems;
  NS_LinkItem *link;
  time_t cur_time = time(0);
  
  SET_TIMEOUT(TRANS_TIMEOUT);
  ad = recvAgentDesc(comm);
  if (ad == NULL)
  {
    netsolvePerror("recvAgentDesc");
    UNSET_TIMEOUT();
    return;
  }

  fprintf(stdout, "NS_PROT_AG_REGISTER: Time %ld (%s), Hostname %s, "
                  "IP %u.%u.%u.%u\n", cur_time, Ctime(&cur_time),
                   ad->host_desc->hostname,
                   BYTE(&(ad->host_desc->IPaddr),0),
                   BYTE(&(ad->host_desc->IPaddr),1),
                   BYTE(&(ad->host_desc->IPaddr),2),
                   BYTE(&(ad->host_desc->IPaddr),3));

  /* If the agent is in the list, remove it */
  alist = lookupAgent(&(ad->host_desc->IPaddr));
  if (*alist != NULL)
    removeAgent(&(ad->host_desc->IPaddr));
  free(alist);

  addAgent(ad);

  if (sendInt(comm,NS_PROT_REGISTRATION_ACCEPTED) == -1)
  {
    netsolvePerror("sendInt()");
    UNSET_TIMEOUT();
    return;
  }

  /* send all the agents I know of */
  nb_agents = NScountList(global.agents);
  if (sendInt(comm,nb_agents) == -1)
  {
    netsolvePerror("");
    UNSET_TIMEOUT();
    return;
  }
  link = global.agents->start;
  while(link != NULL)
  {
    if (sendAgentDesc(comm,(NS_AgentDesc *)(link->content)) == -1)
    {
      netsolvePerror("");
      UNSET_TIMEOUT();
      return;
    }
    link = link->next;
  }

  /* send all the servers I know  if they authorized  */
  /* me to broadcast.                                 */
  nb_servers = 0;
  link = global.servers->start;
  while(link != NULL)
  {
    if (((NS_ServerDesc *)(link->content))->broadcast == 1)
      nb_servers++;  
    link = link->next;
  } 
  if (sendInt(comm,nb_servers) == -1)
  {
    netsolvePerror("");
    UNSET_TIMEOUT();
    return;
  }
  link = global.servers->start;
  while(link != NULL)
  {
    if (((NS_ServerDesc *)(link->content))->broadcast == 1)
    {
      if (sendServerDesc(comm,(NS_ServerDesc *)(link->content)) == -1)
      {
        netsolvePerror("");
        UNSET_TIMEOUT();
        return;
      }
    }
    link = link->next;
  }

  /* Send all the problems */
  nb_problems = 0;
  link = global.problems->start;
  while (link != NULL)
  {
    nb_problems++;
    link = link->next;
  }
  if (sendInt(comm,nb_problems) == -1)
  {
    netsolvePerror("");
    UNSET_TIMEOUT();
    return;
  }
  link = global.problems->start;
  while (link != NULL)
  {
    if (sendProblemDesc(comm,(NS_ProblemDesc *)(link->content)) == -1)
    {
      netsolvePerror("");
    UNSET_TIMEOUT();
      return;
    }
    link = link->next;
  }

  /* Send all the mapping */
  if (sendMappings(comm) == -1)
  {
    netsolvePerror("");
    UNSET_TIMEOUT();
    return;
  }

  fprintf(stdout,"NS_PROT_REGISTRATION_ACCEPTED: %s",ad->host_desc->hostname);

  UNSET_TIMEOUT();
  return; 
}

/*
 * processProblemList()
 */
void processProblemList(NS_Communicator *comm)
{
  int nb_problems=0;
  NS_LinkItem *link;

  if(fork())
    return;

  SET_TIMEOUT(TRANS_TIMEOUT);
  close(global.sock);

  link = global.problems->start;

  while(link != NULL)
  {
    nb_problems++;
    link=link->next;
  }

  if (sendInt(comm,nb_problems) == -1)
  {
    netsolvePerror("");
    endTransaction(comm);
    exit(-1);
  }

  link = global.problems->start;

  while(link != NULL)
  {
    if (sendProblemDesc(comm,(NS_ProblemDesc *)(link->content)) == -1)
    {
      netsolvePerror("");
      endTransaction(comm);
      exit(-1);
    }
    link=link->next;
  }

  endTransaction(comm);
  exit(0);
}

/*
 * processKillAgent()
 */
void processKillAgent(NS_Communicator *comm)
{
  char *username = getUserName();
  char *hostname;
  char *s;
  time_t cur_time = time(0);

  SET_TIMEOUT(TRANS_TIMEOUT);
  if (recvString(comm,&s) == -1)
  {
    netsolvePerror("");
    UNSET_TIMEOUT();
    return;
  }

  if (recvString(comm,&hostname) == -1)
  {
    netsolvePerror("");
    UNSET_TIMEOUT();
    return;
  }

  if (username == NULL)
  {
#ifdef VIEW
    fprintf(stderr,"Killable by anyone.\n");
#endif
    username = s;
  }

  fprintf(stderr, "NS_PROT_KILL_AGENT: Time %ld (%s), ID %s@%s\n",
	  cur_time, Ctime(&cur_time), s, hostname);

  /*
   * Tell the InfoServer
   */
#ifdef INFOSERVER
  info_add_keyval("HOST", hostname);
  info_add_keyval("EVENT", "NS_PROT_KILL_AGENT");
  info_add_keyval("USERNAME", s);
  info_submit();
#endif

  if (strcmp(s,username))
  {
    if (sendInt(comm,NS_PROT_NOT_ALLOWED) == -1)
      netsolvePerror("");
    UNSET_TIMEOUT();
    free(username);
    fprintf(stdout, "NS_PROT_NOT_ALLOWED\n");
    return;
  }
  else
  {
    if (sendInt(comm,NS_PROT_KILLED) == -1)
      netsolvePerror("");

    free(username);
    fprintf(stdout, "NS_PROT_KILLED\n");
    endTransaction(comm);
    close(global.sock);
    Abort();
    exit(0);
  }  
}

/*    
 * processKillServer()
 */   
void processKillServer(NS_Communicator *comm)
{
  NS_Communicator *server_comm;
  NS_IPaddr_type IPaddr;
  int port;
  NS_ServerDesc **slist;
  char * username;
  char * hostname;
  char * servername;
  int sock;
  int status;
  time_t cur_time = time(0);
      
  SET_TIMEOUT(TRANS_TIMEOUT);
  if (recvIPaddr(comm,&IPaddr) == -1)
  {
    netsolvePerror("");
    UNSET_TIMEOUT();
    return;
  }

  if (recvString(comm,&username) == -1)
  {
    netsolvePerror("");
    UNSET_TIMEOUT();
    return;
  }

  if (recvString(comm,&hostname) == -1)
  {
    netsolvePerror("");
    UNSET_TIMEOUT();
    return;
  }

  slist = lookupServer(&IPaddr);
  if (*slist == NULL)
  {
    free(slist);
    port = -1;

    if (sendInt(comm,NS_PROT_NO_SERVER) == -1)
      netsolvePerror("");

    UNSET_TIMEOUT();
    return;
  }

  port = (*slist)->port;
  servername = (*slist)->host_desc->hostname;

  fprintf(stderr, "NS_PROT_KILL_SERVER: Time %ld (%s), Server %s, ID %s@%s\n",
	  cur_time, Ctime(&cur_time), servername, username, hostname);
  
  /*
   * Tell the InfoServer about this
   */
#ifdef INFOSERVER
  info_add_keyval("HOST", hostname);
  info_add_keyval("EVENT", "NS_PROT_KILL_SERVER");
  info_add_keyval("USERNAME", username);
  info_add_keyval("SERVER", servername);
  info_submit();
#endif

  /* connecting to the server */
  sock = connectToSocket(servername,IPaddr,port);
  if (sock == -1)
  {
    netsolvePerror("");
    free(username);
    if (sendInt(comm,NetSolveServerError) == -1)
      netsolvePerror("");
    UNSET_TIMEOUT();
    return;
  }

  server_comm = initTransaction(sock,DATA_XDR);
  if (server_comm == NULL)
  {
    netsolvePerror("");
    free(username);
    if (sendInt(comm,NetSolveServerError) == -1)
      netsolvePerror("");
    UNSET_TIMEOUT();
    return;
  }
 
  if (sendInt(server_comm,NS_PROT_KILL_SERVER) == -1)
  {
    netsolvePerror("");
    free(username);
    if (sendInt(comm,NetSolveServerError) == -1)
      netsolvePerror("");
    UNSET_TIMEOUT();
    return;
  }
  if (sendString(server_comm,username) == -1)
  {
    netsolvePerror("");
    if (sendInt(comm,NetSolveServerError) == -1)
      netsolvePerror("");
    free(username);
    UNSET_TIMEOUT();
    return;
  }
 
  if (recvInt(server_comm,&status) == -1)
  {
    netsolvePerror("");
    if (sendInt(comm,NetSolveServerError) == -1)
      netsolvePerror("");
    free(username);
    UNSET_TIMEOUT();
    return;
  }
 
  endTransaction(server_comm);

  if(status == NS_PROT_KILLED){
    NS_MappingDesc **mlist, **temp_mlist;
    int i;
    int nb_probs=0;
    int nb_servers=0;

    mlist = lookupMappingServer(&IPaddr);
    while(mlist[nb_probs] != NULL)
      nb_probs++;

    fprintf(stdout, "NS_PROT_KILLED\n");
  /* remove problem from list if this was only server to service that problem */
    for(i=0; i<nb_probs; i++){
      nb_servers=0;

      temp_mlist = lookupMappingProblem(((NS_ProblemDesc *)(mlist[i]->problem_desc))->nickname);
      while(temp_mlist[nb_servers] != NULL)
        nb_servers++;

      if(nb_servers == 1){
#ifndef NO_OUTPUT
        fprintf(stderr, "Removing problem %s\n", ((NS_ProblemDesc *)
                                (mlist[i]->problem_desc))->nickname);
#endif
        removeProblem(((NS_ProblemDesc *)(mlist[i]->problem_desc))->nickname);
        removeMappingProblem(((NS_ProblemDesc *)(mlist[i]->problem_desc))->nickname);
      }
    }
    removeMappingServer(&IPaddr);
    removeServer(&IPaddr);
  }
  else
  {
    fprintf(stdout, "NS_PROT_NOT_ALLOWED\n");
  }

  /** send results to back to requesting process **/
  if (sendInt(comm,status) == -1)
  {
    netsolvePerror("");
    free(username);
    UNSET_TIMEOUT();
    return;
  }
 
  UNSET_TIMEOUT();
  return;
}


/*
 * processAgentList()
 */
void processAgentList(NS_Communicator *comm)
{
  int nb_agents=0;
  NS_LinkItem *link;

  if(fork())
    return;

  close(global.sock);

  SET_TIMEOUT(TRANS_TIMEOUT);
  link = global.agents->start;

  while(link != NULL)
  {
    nb_agents++;
    link=link->next;
  }

  nb_agents++;  /* For myself */
  if (sendInt(comm,nb_agents) == -1)
  {
    netsolvePerror("");
    endTransaction(comm);
    exit(-1);
  }

  if (sendAgentDesc(comm,global.my_self) == -1)
  {
    netsolvePerror("");
    endTransaction(comm);
    exit(-1);
  }

  fflush(stderr);
  link = global.agents->start;
  while(link != NULL)
  {
    if (sendAgentDesc(comm,(NS_AgentDesc *)(link->content)) == -1)
    {
      netsolvePerror("");
      endTransaction(comm);
      exit(-1);
    }
    link=link->next;
  }
  endTransaction(comm);
  exit(0);
}

/*
 * processServerList()
 */
void processServerList(NS_Communicator *comm)
{
  int nb_servers=0;
  NS_LinkItem *link;

  if(fork())
    return;

  close(global.sock);

  SET_TIMEOUT(TRANS_TIMEOUT);
  link = global.servers->start;

  while(link != NULL)
  {
    nb_servers++;
    link=link->next;
  }

  if (sendInt(comm,nb_servers) == -1)
  {
    netsolvePerror("");
    endTransaction(comm);
    exit(-1);
  }


  link = global.servers->start;

  while(link != NULL)
  {
    NS_ServerDesc *sd;

    sd = (NS_ServerDesc *)(link->content);
    if (sendServerDesc(comm,sd) == -1)
    {
      netsolvePerror("");
      endTransaction(comm);
      exit(-1);
    }

    link=link->next;
  }

  endTransaction(comm);
  exit(0);
}

/*
 * processProblemInfo()
 */
void processProblemInfo(NS_Communicator *comm)
{
  char *nickname;
  NS_ProblemDesc **plist;

  if(fork())
    return;
  close(global.sock);

  SET_TIMEOUT(TRANS_TIMEOUT);
  if (recvString(comm,&nickname) == -1)
  {
    netsolvePerror("");
    endTransaction(comm);
    exit(-1);
  }

  plist = lookupProblem(nickname);
  if (*plist == NULL)
  {
    if (sendInt(comm,NS_PROT_PROBLEM_NOT_FOUND) == -1)
      netsolvePerror("sendInt()");
    endTransaction(comm);
    exit(-1);
  }

  if (sendInt(comm,NS_PROT_PROBLEM_DESC) == -1)
  {
    netsolvePerror("");
    endTransaction(comm);
    exit(-1);
  }
  if (sendProblemDesc(comm,*plist) == -1)
  {
    netsolvePerror("");
    endTransaction(comm);
    exit(-1);
  }
  free(plist);

  endTransaction(comm);
  exit(0);
}

/*
 * processServerInfo()
 */
void processServerInfo(NS_Communicator *comm)
{
  NS_IPaddr_type IPaddr;
  NS_ServerDesc **slist;

  if(fork())
    return;
  close(global.sock);

  SET_TIMEOUT(TRANS_TIMEOUT);
  if (recvIPaddr(comm,&IPaddr) == -1)
  {
    netsolvePerror("");
    endTransaction(comm);
    exit(-1);
  }

  slist = lookupServer(&IPaddr);
  if (*slist == NULL)
  {
    if (sendInt(comm,NS_PROT_SV_NOT_FOUND) == -1)
      netsolvePerror("");
    endTransaction(comm);
    exit(-1);
  }

  if (sendInt(comm,NS_PROT_SV_DESC) == -1)
  {
    netsolvePerror("");
    endTransaction(comm);
    exit(-1);
  }
  if (sendServerDesc(comm,*slist) == -1)
  {
    netsolvePerror("");
    endTransaction(comm);
    exit(-1);
  }
  free(slist);

  exit(0);
}

/*
 * processProblemSubmit()
 */
void processProblemSubmit(NS_Communicator *comm)
{
  NS_ProblemDesc **plist;
  int nb_server;
  NS_ServerDesc  **slist;
  int *predictionlist;
  char *nickname;
  char *hostname;
  char *username;
  int i;
  int input_size,output_size,problem_size;
  time_t cur_time = time(0);

  SET_TIMEOUT(TRANS_TIMEOUT);
  if ((recvString(comm,&nickname) == -1) ||
      (recvInt(comm,&input_size) == -1) ||
      (recvInt(comm,&output_size) == -1) ||
      (recvInt(comm,&problem_size) == -1) ||
      (recvString(comm,&username) == -1) ||
      (recvString(comm,&hostname) == -1) )
  {
    netsolvePerror("");
    UNSET_TIMEOUT();
    return;
  }

  fprintf(stderr, "NS_PROT_PROBLEM_SUBMIT: Time %ld (%s), Nickname %s, "
                  "Input size %d, Output Size %d, Problem Size %d, ID %s@%s\n",
                   cur_time, Ctime(&cur_time), nickname, input_size, output_size,
                   problem_size, username, hostname);

  /*
   * Tell the InfoServer
   */
#ifdef INFOSERVER
  info_add_keyval("NICKNAME", nickname);
  info_add_keyval_int("INPUTSIZE", input_size);
  info_add_keyval_int("OUTPUTSIZE", output_size);
  info_add_keyval_int("PROBLEMSIZE", problem_size);
  info_add_keyval("USERNAME", username);
  info_add_keyval("HOST", hostname);
  info_add_keyval("EVENT", "NS_PROT_PROBLEM_SUBMIT");
  info_submit();
#endif

  plist = lookupProblem(nickname);
  if (*plist == NULL)
  {
    if (sendInt(comm,NS_PROT_PROBLEM_NOT_FOUND) == -1)
      netsolvePerror("");
    fprintf(stdout, "NS_PROT_PROBLEM_NOT_FOUND\n");
    free(nickname);
    free(plist);
    UNSET_TIMEOUT();
    return;
  }
  
  assignServer(*plist,input_size,output_size,problem_size,
               &slist,&predictionlist,&nb_server);

  if (sendInt(comm,NS_PROT_OK) == -1)
  {
    netsolvePerror("");
    UNSET_TIMEOUT();
    return;
  }
  if (sendInt(comm,nb_server) == -1)
  {
    netsolvePerror("");
    UNSET_TIMEOUT();
    return;
  }

  /*
   * Tell the InfoServer   (Coral)
   */
#ifdef INFOSERVER
  info_add_keyval("NICKNAME", nickname);
  info_add_keyval("HOST", slist[0]->host_desc->hostname);
  info_add_keyval("EVENT", "SERVER_CHOSEN");
  info_add_keyval("USERNAME", "x");
  info_submit();
#endif

  fprintf(stdout, "Server List for problem %s:\n", nickname);
  for (i=0;i<nb_server;i++)
  {
    if ((sendString(comm,slist[i]->host_desc->hostname) == -1) ||
        (sendIPaddr(comm,&(slist[i]->host_desc->IPaddr)) == -1) ||
        (sendInt(comm,slist[i]->port) == -1) ||
        (sendInt(comm,slist[i]->host_desc->data_format) == -1) ||
        (sendInt(comm,predictionlist[i]) == -1))
    {
      free(plist);
      free(slist);
      free(predictionlist);
      free(nickname);
      netsolvePerror("");
      UNSET_TIMEOUT();
      return;
    }
    fprintf(stdout, "%s\n", slist[i]->host_desc->hostname);
  }

  free(plist);
  free(slist);
  free(predictionlist);
  free(nickname);
  UNSET_TIMEOUT();
  return;
}

/*
 * processServerFailure()
 */
void processServerFailure(NS_Communicator *comm)
{
  NS_IPaddr_type IPaddr;
  int error_type;
  char *nickname;
  NS_MappingDesc *m;
  NS_ProblemDesc **plist;
  NS_ServerDesc **slist;
  NS_MappingDesc **mlist, **temp_mlist;
  int i;
  int nb_probs=0;
  int nb_servers=0;

  SET_TIMEOUT(TRANS_TIMEOUT);

  if(recvIPaddr(comm,&IPaddr) == -1 ||
     recvString(comm,&nickname) == -1 ||
     recvInt(comm,&error_type) == -1){
    netsolvePerror("");
    UNSET_TIMEOUT();
    return;
  }
  
  switch(error_type)
  {
    case HOST_ERROR:
      slist = lookupServer(&IPaddr);
      if (*slist == NULL)
      {
#ifndef NO_OUTPUT
        fprintf(stderr,"Warning: received an error report for an unknown server\n");
#endif
        free(slist);
        free(nickname);
        return;
      }
      fprintf(stderr, "HOST_ERROR: %s (%s)\n", 
	      (*slist)->host_desc->hostname, nickname);
      fprintf(stderr, "Removing server %s\n", (*slist)->host_desc->hostname);

      /*
       * Tell the InfoServer
       */
#ifdef INFOSERVER
      info_add_keyval("HOST", (*slist)->host_desc->hostname);
      info_add_keyval("EVENT", "HOST_ERROR");
      info_add_keyval("NICKNAME", nickname);
      info_submit();
#endif

      mlist = lookupMappingServer(&IPaddr);
      while(mlist[nb_probs] != NULL)
        nb_probs++;

      /* remove problem from list if this was only server to service that problem */
      for(i=0; i<nb_probs; i++){
        nb_servers=0;

        temp_mlist = lookupMappingProblem
                     (((NS_ProblemDesc *)(mlist[i]->problem_desc))->nickname);

        while(temp_mlist[nb_servers] != NULL)
          nb_servers++;

        if(nb_servers == 1){
          fprintf(stderr, "Removing problem %s\n", ((NS_ProblemDesc *)
                                  (mlist[i]->problem_desc))->nickname);
          removeProblem(((NS_ProblemDesc *)(mlist[i]->problem_desc))->nickname);
          removeMappingProblem(((NS_ProblemDesc *)(mlist[i]->problem_desc))->nickname);
        }
      }
      removeMappingServer(&IPaddr);
      removeServer(&IPaddr);
      free(mlist);
      free(nickname);

      break;
  
    case SERVER_ERROR:
      plist = lookupProblem(nickname);
      slist = lookupServer(&IPaddr);
    
      if (*plist == NULL)
      {
#ifndef NO_OUTPUT
        fprintf(stderr,"Warning: received an error report for an unknown problem\n");
#endif
        free(plist);free(slist);
        UNSET_TIMEOUT();
        return;
      }
      if (*slist == NULL) 
      {
#ifndef NO_OUTPUT
        fprintf(stderr,"Warning: received an error report for an unknown server\n");
#endif  
        free(plist);free(slist);
        UNSET_TIMEOUT();
        return;
      }
      m = newMappingDesc(*plist,*slist);
      free(plist);
    
      mlist = lookupMappingAll(m);
      if (*mlist == NULL)
      {
#ifndef NO_OUTPUT
        fprintf(stderr,"Warning: received an error report for an unmatching server/problem pair\n");
#endif
        free(mlist);free(slist);
        freeMappingDesc(m);
        UNSET_TIMEOUT();
        return;
      }
      fprintf(stderr, "SERVER_ERROR: %s (%s)\n", 
	      (*slist)->host_desc->hostname, nickname);

      /*
       * Tell the InfoServer about this
       */
#ifdef INFOSERVER
      info_add_keyval("HOST", (*slist)->host_desc->hostname);
      info_add_keyval("NICKNAME", nickname);
      info_add_keyval("EVENT", "SERVER_ERROR");
      info_submit();
#endif

      ((*mlist)->nb_failures)++;
      free(mlist);
      freeMappingDesc(m);
      (*slist)->nb_failures++;
      free(slist);
      break;

    default:
      break;
  }
  UNSET_TIMEOUT();
  return;
}

/*
 * processWorkloadReport()
 */
void processWorkloadReport(NS_Communicator *comm)
{
  NS_IPaddr_type IPaddr;
  int port;
  int workload;
  NS_ServerDesc **slist;
  int idate;
  
  SET_TIMEOUT(TRANS_TIMEOUT);
  if ((recvIPaddr(comm,&IPaddr) == -1) ||
      (recvInt(comm,&port) == -1) ||
      (recvInt(comm,&workload) == -1) ||
      (recvInt(comm,&idate) == -1))
  {
    netsolvePerror("");
    UNSET_TIMEOUT();
    return;
  }

  slist = lookupServer(&IPaddr);
  if (*slist == NULL)
  {
#ifndef NO_OUTPUT
    fprintf(stderr,"Warning: Received a workload report from unknown server\n");
    fprintf(stderr,"         trying to find out more .....\n");
#endif
    free(slist);
    if (!fork())
    {
      close(global.sock);
      registerAgain(IPaddr,port);
      exit(0);
    }
    UNSET_TIMEOUT();
    return;
  }

  /*
   * Tell the InfoServer
   */
#ifdef INFOSERVER
  info_add_keyval("HOST", (*slist)->host_desc->hostname);
  info_add_keyval_int("WORKLOAD", workload);
  info_submit();
#endif

#ifndef NO_OUTPUT
  fprintf(stderr, "Server %s: workload = %d\n",(*slist)->host_desc->hostname,
	  workload); 
#endif

  /* should I get new network measurements */
  if (idate - (*slist)->network_history->time > TTL)
  {
    if (!fork())
    {
      close(global.sock);
      getNetworkMeasurement((*slist)->host_desc,(*slist)->port,
                          global.my_self->host_desc,
                          global.my_self->port);
      exit(0);
    }
  }

  addToWorkloadHistory((*slist)->workload_history,workload,idate);
  free(slist);
  UNSET_TIMEOUT();
  return;
}

/*
 * processPongBandwidth()
 */
void processPongBandwidth(NS_Communicator *comm)
{
  char *data;
  int size;

  if(fork())
    return;

  SET_TIMEOUT(TRANS_TIMEOUT);
  close(global.sock);

  if (recvInt(comm,&size) == -1)
  {
    netsolvePerror("");
    endTransaction(comm);
    exit(-1);
  }
  data = (char *)malloc(size*sizeof(char));
  if (recvArray(comm,NETSOLVE_CHAR,data,size) == -1)
  {
    endTransaction(comm);
    netsolvePerror("");
    exit(-1);
  }
  if (sendArray(comm,NETSOLVE_CHAR,data,size) == -1){
    netsolvePerror("");
    endTransaction(comm);
    exit(-1);
  }
  free(data);

  endTransaction(comm);
  exit(0);
}

/*
 * registerAgain()
 */
void registerAgain(NS_IPaddr_type IPaddr, int port)
{
  int sock;
  NS_Communicator *comm;

  SET_TIMEOUT(TRANS_TIMEOUT);

  sock = connectToSocket(NULL,IPaddr,port);
  if (sock < 0){
    UNSET_TIMEOUT();
    return;
  }
  
  comm = initTransaction(sock,DATA_XDR);
  if (comm == NULL)
  {
    netsolvePerror("");
    close(sock);
    UNSET_TIMEOUT();
    return;
  }
  if (sendInt(comm,NS_PROT_REGISTER_AGAIN) == -1)
  {
    netsolvePerror("");
    endTransaction(comm);
    UNSET_TIMEOUT();
    return;
  }
  if (sendIPaddr(comm,&(global.my_self->host_desc->IPaddr)) == -1)
  {
    netsolvePerror("");
    endTransaction(comm);
    UNSET_TIMEOUT();
    return;
  }
  endTransaction(comm);
  UNSET_TIMEOUT();
  return;
}

/*
 * processJobCompleted()
 */
void processJobCompleted(NS_Communicator *comm)
{
  NS_IPaddr_type IPaddr;
  NS_ServerDesc **slist;

  SET_TIMEOUT(TRANS_TIMEOUT);

  if (recvIPaddr(comm,&IPaddr) == -1)
  {
    netsolvePerror("");
    UNSET_TIMEOUT();
    return;
  }

  slist = lookupServer(&IPaddr);
  if (*slist == NULL)
  {
#ifndef NO_OUTPUT
    fprintf(stderr,"Warning: received job completion for unknown server\n");
#endif
    free(slist);
    UNSET_TIMEOUT();
    return;
  }

  /*
   * Tell the InfoServer what's happening
   */
#ifdef INFOSERVER
  info_add_keyval("HOST", (*slist)->host_desc->hostname);
  info_add_keyval("EVENT", "NS_PROT_JOB_COMPLETED");
  info_submit();
#endif

  /* HEURISTIC */
  if ((*slist)->workload_history->w > 100)
    (*slist)->workload_history->w = (*slist)->workload_history->w - 99;
  free(slist);
  UNSET_TIMEOUT();
  return;
}

void check_servers(time_t cur_time){
  NS_LinkItem *link, *tmp_link;
  time_t last_check;

  link = global.servers->start;

  while(link != NULL)
  {
    NS_ServerDesc *sd;

    sd = (NS_ServerDesc *)(link->content);

    /* remove server if we haven't heard from it in a while */
    last_check = cur_time - sd->workload_history->time;

    if(last_check > FORCE_BROADCAST + 60){ /* extra 60 secs delay JIC */
      NS_MappingDesc **mlist, **temp_mlist;
      int i;
      int nb_probs=0;
      int nb_servers=0;

      mlist = lookupMappingServer(&(sd->host_desc->IPaddr));
      while(mlist[nb_probs] != NULL)
        nb_probs++;

      fprintf(stdout, "REMOVING SERVER (stale workload: %ld secs): "
	      "Hostname = %s, Time = %ld (%s)\n",
	      last_check, sd->host_desc->hostname,
	      cur_time, Ctime(&cur_time));

      /*
       * Tell the InfoServer
       */
#ifdef INFOSERVER
      info_add_keyval("HOST", "@AGENT@");
      info_add_keyval("EVENT", "NS_PROT_KILL_SERVER");
      info_add_keyval("USERNAME", "@AGENT@");
      info_add_keyval("SERVER", sd->host_desc->hostname);
      info_submit();
#endif

    /* remove problem from list if this was only server to 
              service that problem */
      for(i=0; i<nb_probs; i++){
        nb_servers=0;

        temp_mlist = lookupMappingProblem(((NS_ProblemDesc *)
                                    (mlist[i]->problem_desc))->nickname);
        while(temp_mlist[nb_servers] != NULL)
          nb_servers++;

        if(nb_servers == 1){
#ifndef NO_OUTPUT
          fprintf(stderr, "Removing problem %s\n", ((NS_ProblemDesc *)
                                  (mlist[i]->problem_desc))->nickname);
#endif
          removeProblem(((NS_ProblemDesc *)
                                (mlist[i]->problem_desc))->nickname);
          removeMappingProblem(((NS_ProblemDesc *)
                                (mlist[i]->problem_desc))->nickname);
        }
      }

      removeMappingServer(&(sd->host_desc->IPaddr));

      /* remove server from list */
      tmp_link = link->next;

      if (link->prev == NULL)
        global.servers->start = link->next;
      else
        link->prev->next = link->next;

      if (link->next != NULL)
        link->next->prev = link->prev;

      freeServerDesc(sd);
      free(link);
      link = tmp_link;
    }
    else{
      link=link->next;
    }
  }
}

/*
 * processStoreHandle()
 */
void processStoreHandle(NS_Communicator *comm)
{
  DSI_OBJECT* dsi_obj;

  dsi_obj = (DSI_OBJECT *)malloc(sizeof(DSI_OBJECT));
  dsi_obj->dsi_file = (DSI_FILE *)malloc(sizeof(DSI_FILE));
  if(recvDsiObj(comm, dsi_obj) == -1)
  {
    netsolvePerror("");
    endTransaction(comm);
    exit(-1);
  }

  addHandle(dsi_obj);

  fprintf(stderr, "Store Handle - %s\n", dsi_obj->name);

  return;
}

/*
 * processGetHandle()
 */
void processGetHandle(NS_Communicator *comm)
{
char* name;
DSI_OBJECT** dsi_obj_list = NULL;

  if(fork())
    return;
  close(global.sock);

  if(recvString(comm, &name) == -1)
  {
    netsolvePerror("");
    endTransaction(comm);
    exit(-1);
  }


  dsi_obj_list = lookupHandle(name);
  if (*dsi_obj_list == NULL)
  {
    if (sendInt(comm,NS_PROT_HANDLE_NOT_FOUND) == -1)
	netsolvePerror("");
    fprintf(stdout, "NS_PROT_HANDLE_NOT_FOUND\n");
    free(name);
    free(dsi_obj_list);
    UNSET_TIMEOUT();
    return;
  }

  if (sendInt(comm,NS_PROT_HANDLE) == -1)
  {
    netsolvePerror("");
    endTransaction(comm);
    exit(-1);
  }

  if(sendDsiObj(comm, *dsi_obj_list) == -1)
  {
    netsolvePerror("");
    endTransaction(comm);
    exit(-1);
  }

  fprintf(stderr, "Get Handle - %s\n", name);

  endTransaction(comm);
  exit(0);
}
