/*****************************************************************/
/*      agent_init.c                                             */
/*      Henri Casanova, Dorian Arnold                            */
/*---------------------------------------------------------------*/
/*  agentInit()                                                 */
/*****************************************************************/

#include <signal.h>
#include <fcntl.h>
#include <netdb.h>
#include <sys/stat.h>
#include "core.h"
#include "agentglobal.h"
#include "agentinit.h"

#ifdef NWS
#include "nws_api.h"
#include "nwsutils.h"
#endif

#include "agentwakeup.h"
#include "exchangemappings.h"
#include "netsolvesignals.h"
#include <unistd.h>
extern int errno;

/*
 *  agentInit()
 *
 *  Performs all the initializations required by the agent
 */
int agent_init(int argc,char **argv)
{
  /* initialize the NETSOLVE_ROOT path */
  global.netsolve_root_path=getNetSolveRoot();

  set_defaults();
  check_args(argc, argv);

  /* Initialize the NSLinked Lists */
  global.problems = newNSLinkedList();
  global.handles  = newNSLinkedList();
  global.agents   = newNSLinkedList();
  global.servers  = newNSLinkedList();
  global.mappings = newNSLinkedList();


#if (defined(DEBUG) || defined(VIEW))
  fprintf(stderr,"Initializing Network connections...\n");
#endif

  /* Initialize my_self */
  if (initMySelf() == -1){
    netsolvePerror( "initMySelf()");
    return -1;
  }

  /* Signals */
  if (setSignals() == -1){
    netsolvePerror( "setSignals()");
    return -1;
  }

#ifdef NWS
  if (setNWS()==-1){
    netsolvePerror( "setNWS()");
    return -1;
  }
#endif

  /* Register to the NetSolve system */
  if (registerToNetSolve() == -1){
    netsolvePerror( "registerToNetSolve()");
    return -1;
  }

  if(daemon_init() < 0 ){
    netsolvePerror( "daemon_init()");
    return -1;
  }

#ifdef INFOSERVER
#ifdef STANDALONEISERV
  /*
   * If we want to use the infoserver, AND we want
   * to run it locally, then fork it off, otherwise,
   * either remember where it is, or don't do a thing.
   */
#if (defined(DEBUG) || defined(VIEW))
  fprintf(stderr, "NS: Agent will use standalone InfoServer\n");
#endif
  remote_infoserver("localhost", 0);
#else
#if (defined(DEBUG) || defined(VIEW))
  fprintf(stderr, "NS: Agent will spawn the InfoServer\n");
#endif
  spawn_infoserver();
#endif
#endif

  fflush(stderr);
  fflush(stdout);
  return 0;
}

/*
 * initMySelf()
 *
 * Performs the initialization of the host descriptor
 * the IP port and the listening socket
 */
int initMySelf()
{
  struct hostent *hp;
  char buffer[256];
  
  global.my_self = newAgentDesc();
  global.my_self->host_desc = newHostDesc();

  /* Getting my host name and IPaddr */
  if (gethostname(buffer,256) == -1)
  {
    ns_printinfo();
    perror("gethostname()");
    ns_errno = NetSolveSystemError;
    return -1;
  }

  if ((hp = gethostbyname(buffer)) == NULL)
  {
    ns_printinfo();
    perror("gethostbyname()");
    ns_errno = NetSolveSystemError;
    return -1;
  }

  BCOPY(hp->h_addr_list[0],
        &(global.my_self->host_desc->IPaddr),sizeof(NS_IPaddr_type));
  global.my_self->host_desc->hostname = strdup(hp->h_name);

  global.my_self->port=AGENT_PORT;
  global.sock=establishSocket(&(global.my_self->port), 1);

  if (global.sock < 0)
  {
    netsolvePerror("establishSocket()");
    return -1; 
  }

#if (defined(VIEW) || defined(DEBUG))
  fprintf(stderr, "Binding to port %d...\n",global.my_self->port);
#endif

  global.my_self->host_desc->host_status = HOST_UP;
  global.my_self->host_desc->data_format = getArch();
  if (global.my_self->host_desc->data_format == -1)
  {
    netsolvePerror("getArch()");
    ns_errno = NetSolveInternalError;
    return -1;
  }
  return 0;
}

/*
 * setMaster()
 *
 * Build the agent descriptor
 */
int setMaster(char *mastername)
{
  struct hostent *hp;
  NS_AgentDesc *agentdesc;

  /* Get the full name and IPaddr of the agent */
  hp = gethostbyname(mastername);
  if (hp == NULL)
  {
    ns_printinfo();
    perror("gethostbyname()");
    ns_errno = NetSolveSystemError;
    return -1;
  } 
  agentdesc = newAgentDesc();
  agentdesc->host_desc = newHostDesc();
  agentdesc->port = AGENT_PORT;
  agentdesc->host_desc->hostname= 
      strdup(hp->h_name);
  BCOPY(hp->h_addr_list[0],
       &(agentdesc->host_desc->IPaddr),
       sizeof(NS_IPaddr_type));

  addAgent(agentdesc);
  global.master = agentdesc;
  return 0;
}

/*
 * registerToNetSolve
 *
 * contacts the master agent and try to get registered
 */
int registerToNetSolve()
{
  int sock;
  NS_Communicator *comm;
  NS_AgentDesc *ad;
  int nb_problems;
  NS_ServerDesc *sd;
  int nb_servers,nb_agents,i;
  NS_LinkItem *link;
  NS_ProblemDesc *pd;
  int tag;

  /* Nothing to do if I am stand-alone */
  if (global.master == NULL)
    return 0;

#if (defined(DEBUG) || defined(VIEW))
  fprintf(stderr, "NS: Registering with agent %s.\n",
                            global.master->host_desc->hostname);
#endif

  sock = connectToSocket(global.master->host_desc->hostname,
                           global.master->host_desc->IPaddr,
                           global.master->port);
  if (sock <0)
  {
    char buf[256];
    ns_printinfo();
    sprintf(buf, "Impossible to contact agent on '%s'\n",
                   global.master->host_desc->hostname);
    netsolvePerror(buf);
    return -1;
  }

  comm=initTransaction(sock,DATA_XDR);
  if (comm == NULL)
  {
    netsolvePerror("initTransaction()");
    endTransaction(comm);
    return -1;
  }
  if (sendInt(comm,NS_PROT_AG_REGISTER) == -1)
  {
    netsolvePerror("sendInt()");
    endTransaction(comm);
    return -1;
  }
  if (sendAgentDesc(comm,global.my_self) == -1)
  {
    netsolvePerror("sendAgentDesc()");
    endTransaction(comm);
    return -1;
  }

  if (recvInt(comm,&tag) == -1)
  {
    netsolvePerror("recvInt()");
    endTransaction(comm);
    return -1;
  }
  if (tag == NS_PROT_REGISTRATION_REFUSED)
  {
    fprintf(stderr, "Registration rejected by the agent\n");
    endTransaction(comm);
    return -1;
  }
  if (tag != NS_PROT_REGISTRATION_ACCEPTED)
  {
    fprintf(stderr, "Registration failed\n");
    ns_errno = NetSolveProtocolError;
    endTransaction(comm);
  }
#if (defined(VIEW) || defined(DEBUG))
  fprintf(stderr,"Registration accepted\n");
#endif

  if (recvInt(comm,&nb_agents) == -1)
  {
    netsolvePerror( "recvInt()");
    endTransaction(comm);
    return -1;
  }

  for (i=0;i<nb_agents;i++)
  {
    ad = recvAgentDesc(comm);
    if (ad == NULL)
    {
      char buf[128];
      sprintf(buf, "Error in recvAgentDesc() for agent %d\n", i);
      netsolvePerror(buf);
      endTransaction(comm);
      return -1;
    }
    if (ad->host_desc->IPaddr == global.my_self->host_desc->IPaddr)
      freeAgentDesc(ad);
    else
      addAgent(ad);
  }
  if (recvInt(comm,&nb_servers) == -1)
  {
    netsolvePerror( "Error in recvInt()");
    endTransaction(comm);
    return -1;
  }
  for (i=0;i<nb_servers;i++)
  {
    sd = recvServerDesc(comm);
    if (sd == NULL)
    {
      char buf[128];
      sprintf(buf, "Error in recvServerDesc() for agent %d\n", i);
      netsolvePerror(buf);
      endTransaction(comm);
      return -1;
    }
    addServer(sd);
  }
  
  /* Recv all the problems */
  if (recvInt(comm,&nb_problems) == -1)
  {
    netsolvePerror( "recvInt()");
    endTransaction(comm);
    return -1;
  }
  for (i=0;i<nb_problems;i++)
  {
    pd = recvProblemDesc(comm);
    if (pd == NULL)
    {
      netsolvePerror( "recvProblemDesc()");
      endTransaction(comm);
      return -1;
    }
    addProblem(pd);
  }
 

  /* Recv the mappings */
  if (recvMappings(comm) == -1)
  {
    netsolvePerror( "recvMappings()");
    endTransaction(comm);
    return -1;
  }

  /* Notify my existence to the servers */
  link = global.servers->start;
  while(link != NULL)
  {
    sd = (NS_ServerDesc *)(link->content);
#if( defined(VIEW) || defined (DEBUG))
    fprintf(stderr,"Reporting to %s\n",sd->host_desc->hostname);
#endif
    reportToServer(sd);
    link = link->next;
  }

  endTransaction(comm);
  return 0; 
}

/*
 * reportToServer()
 *
 * report my existence to all the servers that I have 
 * received from my master agent.
 */
void reportToServer(NS_ServerDesc *sd)
{
  int sock;
  NS_Communicator *comm;

  sock = connectToSocket(sd->host_desc->hostname,
                         sd->host_desc->IPaddr,
                         sd->port);
  if (sock < 0)
  {
#if (defined(VIEW) || defined(DEBUG))
    ns_printinfo();
    fprintf(stderr,"WARNING:Impossible to contact server on '%s'\n",
            sd->host_desc->hostname);
    return;
#endif
  }
  comm = initTransaction(sock,DATA_XDR);
  if (comm == NULL)
  {
    netsolvePerror( "initTransaction()");
    close(sock);
    return;
  }
  if (sendInt(comm,NS_PROT_NEW_AGENT) == -1)
  {
    netsolvePerror( "sendInt()");
    endTransaction(comm);
    return;
  }
  if (sendAgentDesc(comm,global.my_self) == -1)
  {
    netsolvePerror( "sendAgentDesc()");
    endTransaction(comm);
    return;
  }
  endTransaction(comm);
  return;
}

void print_usage(char *command){
  fprintf(stderr,"Usage: %s [ -a agent_name ] [ -l logfile ]\n"
                  , command);
  exit(-1);
}

int check_args(int argc, char **argv){
  int i, retval=0;

  for(i=1; i<argc; i++){
    if(strcmp(argv[i], "-a") == 0){
      if(i == argc-1){   /* last argument? */
        print_usage(argv[0]);
        return -1;
      }
      else{
        retval = setMaster(argv[i+1]);
        i++;
        if(retval < 0)
          return -1;
      }
    }

    else if(strcmp(argv[i], "-l") == 0){
      if(i == argc-1){   /* last argument? */
        print_usage(argv[0]);
        return -1;
      }
      else{
        global.log_file = (char *)strdup(argv[i+1]);
        i++;
      }
    }

    else{
      print_usage(argv[0]);
      return -1;
    }
  }

  return 0;
}

int set_defaults(){

  /*** set logfile to $NETSOLVE_ROOT/AGENT_LOGFILE **/
  global.log_file=(char *)calloc(strlen(global.netsolve_root_path) +
                                   strlen(AGENT_LOGFILE) + 2, sizeof(char));
  strcpy(global.log_file, global.netsolve_root_path);
  strcat(global.log_file, "/");
  strcat(global.log_file, AGENT_LOGFILE);

  global.master = NULL;

  return 0;
}

int daemon_init(){
  pid_t pid;
  int fd;

  pid = fork();

  if(pid < 0){
    ns_printinfo();
    perror("fork()");
    return -1;
  }

  if(pid != 0) /* parent exits */
    exit(0);

  setsid();   /* become session leader */

  pid = fork();
 
  if(pid < 0){
    ns_printinfo();
    perror("fork()");
    return -1;
  }
 
  if(pid != 0) /* 1st child exits */
    exit(0);

  chdir(global.netsolve_root_path);
  umask(0);

  fd = open("/dev/null", O_RDWR|O_CREAT);
  if(fd < 0){
    ns_printinfo();
    perror("open()");
    return -1;
  }
  close(0);
  if(dup2(fd, 0) < 0){
    ns_printinfo();
    perror("dup2()");
    return -1;
  }
  close(fd);

#if (defined(VIEW) || defined(DEBUG))
  fprintf(stderr, 
    "\n"
    "==============================================================\n"
    "|       -- Agent giving up terminal control! --\n|\n"
    "| All further terminal output will be sent to the log file:\n"
    "|\t\"%s\"\n"
    "=============================================================="
    "\n\n", global.log_file);
#endif

  remove(global.log_file);
  fd = open(global.log_file, O_RDWR|O_CREAT, 0644);
  if(fd < 0){
    ns_printinfo();
    perror("open()");
    return -1;
  }
  close(1);
  if(dup2(fd, 1) < 0){
    ns_printinfo();
    perror("dup2()");
    return -1;
  }
  close(2);
  if(dup2(fd, 2) < 0){
    ns_printinfo();
    perror("dup2()");
    return -1;
  }
  close(fd);

  return 0;
}

#ifdef NWS
int  initforecasting1()
{ 
  int ok;

  /* forcaster initialisation */ 
  ok=NWSAPI_UseForecaster(NWSAPI_MakeHostSpec(
       global.my_self->host_desc->hostname,NWS_DEFAULT_FORECASTER_PORT));

  if (ok==0){
    ns_printinfo();
    fprintf(stderr,"ERROR: NWSUSeForecast ...\n");
    return -1;
  }

  /* which nameserver */
  ok=NWSAPI_UseNameServer(NWSAPI_MakeHostSpec(
      global.my_self->host_desc->hostname, NWS_DEFAULT_NAMESERVER_PORT));

  if (ok==0){
    ns_printinfo();
    fprintf(stderr,"ERROR: NWSnameserver ...\n");
    return -1;
  }

  /*which memory */
  ok=NWSAPI_UseMemory(NWSAPI_MakeHostSpec(
          global.my_self->host_desc->hostname, NWS_DEFAULT_MEMORY_PORT));

  if (ok==0){
    ns_printinfo();
    fprintf(stderr,"ERROR: NWSmem ...\n");
    return -1;
  }
 
  return 0;
}

int  initforecasting2()
{ 
  int ok;

  /* forcaster initialisation */ 
  ok=NWSAPI_UseForecaster(NWSAPI_MakeHostSpec(
      global.my_self->host_desc->hostname,NWS_DEFAULT_FORECASTER_PORT));

  if (ok==0){
    ns_printinfo();
    fprintf(stderr,"ERROR: NWSAPIUSeForecast ...\n");
    return -1;
  }

  /* which nameserver */
  ok=NWSAPI_UseNameServer(NWSAPI_MakeHostSpec(
      global.master->host_desc->hostname, NWS_DEFAULT_NAMESERVER_PORT));

  if (ok==0){
    ns_printinfo();
    fprintf(stderr,"ERROR: NWSnameserver ...\n");
    return -1;
  }

  /*which memory */
  ok=NWSAPI_UseMemory(NWSAPI_MakeHostSpec(
          global.master->host_desc->hostname, NWS_DEFAULT_MEMORY_PORT));

  if (ok==0){
    ns_printinfo();
    fprintf(stderr,"ERROR: NWSmem ...\n");
    return -1;
  }

  return 0;
}

/* setNWS() */
int setNWS()
{
  int pid;

#if (defined(DEBUG) || defined(VIEW))
  fprintf(stderr, "Starting NWS ...\n");
#endif

  if (global.master==NULL) /* I am the master agent, start nameserver, */
                              /* a memory and a forecaster*/
  {
#if (defined(DEBUG) || defined(VIEW))
    fprintf(stderr, "Starting NWS name server ...\n");
#endif

    pid=startNWSNameServer(global.my_self->host_desc->hostname);
    if ( pid < 0){
      ns_printinfo();
      fprintf(stderr, "Error starting nameserver\n");
      return -1;
    }

#if (defined(DEBUG) || defined(VIEW))
    fprintf(stderr, "Starting NWS memory ...\n");
#endif
    pid=startNWSMemory(
         global.my_self->host_desc->hostname,NWS_DEFAULT_NAMESERVER_PORT,
         global.my_self->host_desc->hostname);

    if (pid < 0){
      ns_printinfo();
      fprintf(stderr, "Error starting memory\n");
      return -1;
    }

#if (defined(DEBUG) || defined(VIEW))
    fprintf(stderr, "Starting NWS forecaster ...\n");
#endif
    pid=startNWSForecaster(global.my_self->host_desc->hostname,
                           NWS_DEFAULT_NAMESERVER_PORT);

    if (pid < 0) {
      ns_printinfo();
      fprintf(stderr, "Error starting forecaster\n");
      return -1;
    }

    if(initforecasting1()==-1){
      ns_printinfo();
      fprintf(stderr, "Error initializing forecast\n");
      return -1;
    }

    return 0;
  }

  /* I start only my own forecaster */
#if (defined(DEBUG) || defined(VIEW))
  fprintf(stderr, "Starting NWS forecaster ...\n");
#endif
  pid=startNWSForecaster(global.master->host_desc->hostname,
                         NWS_DEFAULT_NAMESERVER_PORT);

  if (pid < 0){ 
    ns_printinfo();
    fprintf(stderr, "Error starting forecaster\n");
    return -1;
  }

  if(initforecasting2()==-1){
    ns_printinfo();
    fprintf(stderr, "Error initializing forecast\n");
    return -1;
  }

#if (defined(DEBUG) || defined(VIEW))
  fprintf(stderr,"NWS (forecaster) running ...\n");
#endif

  return 0;
}
#endif
