/*
 * Include file for regression tests
 * 2000 - Jakob Ostergaard
 *
 * $Id:
 */

#ifndef __NS_REGRESSION_H
#define __NS_REGRESSION_H

#include <sys/time.h>
#include <unistd.h>


/*
 * Structure of a typical test-program:
 *
 * test_init("my_test");
 *
 * ... generate input data ...
 * subtest_submit("problem #1");
 * netsl(...);
 * subtest_complete();
 * ... test results ...
 * if(bad result) subtest_fail("reason...");
 * else subtest_success();
 *
 * .... generate new input data ....
 * subtest_submit("problem #2");
 * netsl(...);
 * subtest_complete();
 * ... test results ...
 * if(bad result) subtest_fail("reason...");
 * else subtest_success();
 *
 * (eventually repeat with more subtests...)
 *
 * text_exit();
 */


/*
 * This routine must be called immediately when the
 * test program is started (should be one of the first
 * lines in main()
 */
void test_init(const char*);

/*
 * This routine should be called before the test program 
 * submits the first request to NetSolve that is being tested,
 * and before any subsequent NetSolve request is being submitted
 */
void subtest_submit(const char*);

/*
 * This routine should be called immediately after the
 * NetSolve call returns, marking the end of distributed
 * computation in that subtest
 */
void subtest_complete(void);

/*
 * This routine should be called on error, eventually
 * with a text string description of the nature of the error
 */
void subtest_fail(const char*);

/*
 * Call this routine once the actual results passed back
 * from a NetSolve request has been validated and found
 * to be good
 */
void subtest_success(void);

/*
 * Call this routine once the actual NetSolve call has
 * completed, if you cannot check the result and just
 * assume that it's ok (or will check it in a later test)
 *
 * Note:  This call should be used if subtest_success()
 *        or subtest_fail() is not used.
 */
void subtest_unchecked(void);

/*
 * This routine should be called on successful exit 
 */
void test_exit(void);

/*
 * Below are helper routines for test matrix/vector generation.
 * The routines will either return a valid matrix/vector, or
 * exit the program with a failure.  Do not concern yourself
 * with null pointer checking.
 */

double * tgen_matrix_random(unsigned,unsigned);
double * tgen_matrix_copy(const double*,unsigned,unsigned);

/* 0 => equality,  1 => difference */
int tgen_matrix_compare(const double*,const double*,unsigned,unsigned);

#endif
