/*****************************************************************/
/*      problemdesc.h                                            */
/*	Henri Casanova					         */
/*---------------------------------------------------------------*/

#ifndef __NS_PROBLEMDESC_H
#define __NS_PROBLEMDESC_H

/**
 **  MACRO DEFINITIONS
 **/

/* Languages supported by our UPF mechanism */
#define UPF_LANG_C       0
#define UPF_LANG_FORTRAN 1

/* object types */
#define NETSOLVE_MATRIX 	1
#define NETSOLVE_VECTOR 	2
#define NETSOLVE_SCALAR 	3
#define NETSOLVE_UPF    	4
#define NETSOLVE_FILE   	5
#define NETSOLVE_STRING 	6   /* Null terminated !!! */
#define NETSOLVE_PACKEDFILES   	7
#define NETSOLVE_STRINGLIST   	8
#define NETSOLVE_SPARSEMATRIX 	9

/* majors */
#define COL_MAJOR 0
#define ROW_MAJOR 1

/* packed file stuff */
#define PREFIX		0
#define FILENAMES	1

/**
 **  STRUCTURE DEFINITIONS
 **/

typedef struct NS_ProblemDesc_        	NS_ProblemDesc;
typedef struct NS_ProblemItem_        	NS_ProblemItem;
typedef struct NS_Object_             	NS_Object;
typedef struct NS_MatlabMerge_        	NS_MatlabMerge;
typedef struct NS_MatrixAttributes_   	NS_MatrixAttributes;
typedef struct NS_SparseMatrixAttributes_  	NS_SparseMatrixAttributes;
typedef struct NS_VectorAttributes_   	NS_VectorAttributes;
typedef struct NS_ScalarAttributes_   	NS_ScalarAttributes;
typedef struct NS_FileAttributes_     	NS_FileAttributes;
typedef struct NS_PackedFilesAttributes_   NS_PackedFilesAttributes;
typedef struct NS_StringListAttributes_    NS_StringListAttributes;
typedef struct NS_UPFAttributes_      	NS_UPFAttributes;
typedef struct NS_StringAttributes_   	NS_StringAttributes;
typedef struct NS_CallingSequence_    	NS_CallingSequence;
typedef struct NS_Formula_            	NS_Formula;
typedef struct NS_Constant_           	NS_Constant;

/*  ProblemDesc 
 *
 *  This structure contains all there is to know about
 *  a problem that can be solved by a server.
 */

struct NS_ProblemDesc_ {
  /* Problem ID */  
  char *name;         		/* Problem full name            */
  char *nickname;     		/* short version of the name    */
  char *description;  		/* Textual description          */
  char *file;         		/* Problem description filename */

  /* Problem Specifics */
  int ftp;                           /* Is ftp on the fly required ? */
  int major;          		     /* Row or Column major          */
  char *customized;   		     /* Customized NetSolve server ? */
  int nb_matlab_merge;               /* Number of Matlab merge       */
  struct NS_MatlabMerge_ *matlab_merge; /* Matlab merges                */
  char *complexity;                  /* Problem complexity           */
  
  /* Calling sequence from C or Fortran */
  struct NS_CallingSequence_ *calling_sequence;   
 
  /* Inputs and Outputs */
  int nb_input_objects;
  struct NS_Object_ **input_objects;
  int nb_output_objects;
  struct NS_Object_ **output_objects;

  /* Server side filenames for sequences */
  char **input_names;     /* file names of the input objects */
  char **output_names;    /* file names of the input objects */
};

/*
 *
 * Linked-list element
 */
struct NS_ProblemItem_ {
  struct NS_ProblemDesc_ *problem_desc;
  struct NS_ProblemItem_ *next; 
};

/*  MatrixAttributes
 *
 *  Attributes for a Matrix.
 */

struct NS_MatrixAttributes_ {
  void *ptr;    /* pointer to the data           */
  int major;    /* row or column major           */
  int m;	/* number of rows                */
  int n;	/* number of columns             */
  int l;        /* leading dimension             */
  int *d;       /* possible ScaLAPACK descriptor */
                /* only used at the Server side  */
};
 
/*  MatrixAttributes
 *
 *  Attributes for a Matrix.
 */
 
struct NS_SparseMatrixAttributes_ {
  void *ptr;    /* pointer to the data           */
  int major;    /* row or column major           */
  int m;        /* number of rows                */
  int n;        /* number of columns             */
  int f;		/* number of filled positions */

  /* row/col indices/ptrs depending on major */
  int *rc_ptr;   
  int *rc_index; 

  int *d;       /* possible ScaLAPACK descriptor */
                /* only used at the Server side  */
};

/*  VectorAttributes
 *
 *  Attributes for a Vector.
 */

struct NS_VectorAttributes_ {
  void *ptr;    /* pointer to the data           */
  int m;        /* number of rows                */
  int *d;       /* Possible PETSC descriptor     */
                /* only used at the Server side  */
};

/*  ScalarAttributes
 *
 *  Attributes for a Scalar.
 */

struct NS_ScalarAttributes_{
  void *ptr;    /* pointer to the data */
};

/*  FileAttributes
 *
 *  Attributes for a File.
 */

struct NS_FileAttributes_ {
  char *filename; /* file name         */
};

/*  PackedFilesAttributes
 *
 *  Attributes for a PackedFiles.
 */

struct NS_PackedFilesAttributes_ {
  char *defaultprefix;        /* default prefix     */
  char **filenames;           /* list of file names */
  int m; /* number of files in the list   */
};

/*  StringListAttributes
 *
 *  Attributes for a StringList.
 */
   
struct NS_StringListAttributes_ {
  char **strings;             /* list of file names */
  int m; /* number of strings in the list   */
};



/*  UPFAttributes
 *
 *  Attributes for a User Provided Function.
 */

struct NS_UPFAttributes_ {
  int language;   /* Language (C/Fortran) */
  char *filename; /* file name            */
  char *funcname; /* function name        */
};

struct NS_StringAttributes_ {
  char *ptr;
};


/*  Object
 *
 *  This structure contains the what characterizes an Object
 *  such as his type, its data type, and a pointer to its
 *  attributes;
 */

struct NS_Object_ {
  int object_type;      	/* matrix,vector,file,url,etc... */
  int data_type;        	/* double,single,ascii,etc...    */
  int distributed;           /* 1 if distributed, 0 if not */
  char *description;    	/* textual description 	         */
  char *name;			/* Name				 */
  int  pass_back;               /* pass back the object over the network ? */
  DSI_OBJECT *dsi_obj;      /* pointer used if object is "distributed" */
  union {
    struct NS_MatrixAttributes_ 		matrix_attributes;
    struct NS_SparseMatrixAttributes_ 	sparsematrix_attributes;
    struct NS_VectorAttributes_ 		vector_attributes;
    struct NS_ScalarAttributes_ 		scalar_attributes;
    struct NS_FileAttributes_   		file_attributes;
    struct NS_PackedFilesAttributes_ 	packedfiles_attributes;
    struct NS_StringListAttributes_ 	stringlist_attributes;
    struct NS_UPFAttributes_    		upf_attributes;
    struct NS_StringAttributes_ 		string_attributes;
  } attributes;
}; 

/*  Formula
 *
 *  formal description of a formula:  
 *     mnemonic = expression
 *     e.g. mI0 = op(+,mI1,2)
 */

struct NS_Formula_ {
  char *mnemonic;
  char *expression;
};

/*  Constant
 *
 *  formal description of a constant:
 *     mnemonic = constant
 *     e.g. mI0 = 12
 *     e.g. mI0 = 'a'
 */

struct NS_Constant_ {
  char *mnemonic;
  char *value;
};

/*  CallingSequence
 *
 *  Formal description of the calling sequence that must
 *  be used to call a problem from C or Fortran.
 */

struct NS_CallingSequence_ {
  int nb_args;                 /* number of arguments */
  char **arg_to_mnemonic;      /* what each means     */
  int nb_constants;            /* number of constants */
  struct NS_Constant_ *constants; /* what they are       */
  int nb_formulae;             /* number of formulae  */
  struct NS_Formula_ *formulae;  /* what they are       */
};

/*  MatlabMerge
 *
 *  This structure contains the indices of two output
 *  objects than can be merged in Matlab in order to
 *  return a complex (as opposed to real) object to
 *  the user. This structure is used only by the
 *  client libraries which (i) receives the output objects
 *  and (ii) aggregates the ones that need to be merged
 */

struct NS_MatlabMerge_ {
  int index1;
  int index2;
};

/**
 **  FUNCTION PROTOTYPES
 **/

/* freeing memory */
void freeProblemDesc(void*);
void freeObject(NS_Object*);
void freeObjectData(NS_Object*);
void freeHandle(void*);

/* comparison */
int compProblemDescKey(void*,void*);
int compareProblemDesc(NS_ProblemDesc*,NS_ProblemDesc*);
int compareCallingSequence(NS_CallingSequence*,NS_CallingSequence*);
int compHandleName(void*, void*);

/* misc. */
int containsUPF(NS_ProblemDesc*);
int getObjectByteSize(NS_Object*);
#ifdef sunos
int problemOrder(void*,void*);
#else
int problemOrder(const void*,const void*);
#endif


/* allocating memory */
NS_ProblemDesc *newProblemDesc();
NS_Object *newObject();
NS_Object *createMatrixObject(int,void*,int,int,int,int);
NS_Object *createSparseMatrixObject(int,void*,int,int,int,int,int*,int*);
NS_Object *createVectorObject(int,void*,int);
NS_Object *createScalarObject(int,void*);
NS_Object *createFileObject(char*);
NS_Object *createPackedFilesObject(char*,char**,int);
NS_Object *createStringListObject(char**,int);
NS_Object *createUPFObject(int,char*,char*);
NS_Object *createStringObject(char*);
NS_Object *createEmptyObject(int,int);

/* ProblemDesc network/files */
int writeProblemDescToFile(int,NS_ProblemDesc*);
NS_ProblemDesc *readProblemDescFromFile(int);
int sendProblemDesc(NS_Communicator*,NS_ProblemDesc*);
NS_ProblemDesc* recvProblemDesc(NS_Communicator*);

/* Object level routines */
int sendObject(NS_Communicator*,NS_Object*);
int recvObject(NS_Communicator*,NS_Object*);
int recvObjectToFile(NS_Communicator*,int,NS_Object*,void *misc);
int sendObjectFromFile(NS_Communicator*,int,NS_Object*);
int readObjectFromFile(int,NS_Object*);
int writeObjectToFile(int,NS_Object*);

/* groups of objects */
int recvInputObjectsToFiles(NS_Communicator*,NS_ProblemDesc*);
int sendOutputObjectsFromFiles(NS_Communicator*,NS_ProblemDesc*);
int readInputObjectsFromFiles(NS_ProblemDesc*);
int writeOutputObjectsToFiles(NS_ProblemDesc*);


/* Calling Sequence */
int sendCallingSequence(NS_Communicator*,NS_CallingSequence*);
NS_CallingSequence *recvCallingSequence(NS_Communicator*);
void freeCallingSequence(NS_CallingSequence*);

/* TRANSPOSE */
int transposeMatrix(NS_Object*);
int transposeSparseMatrix(NS_Object*);
void CompRow_to_CompCol(int, int, int, void *, int *, int *,
                         void *, int *, int *, int );
void CompCol_to_CompRow(int, int, int, void *, int *, int *,
                         void *, int *, int *, int );

#endif

