/*****************************************************************/
/*      infoserver_int.h                                         */
/*      Jakob Oestergaard                                        */
/*---------------------------------------------------------------*/
/*  header for use by the infoserver itself                      */
/*****************************************************************/

#ifndef __NS_INFOSERVER_INT_H
#define __NS_INFOSERVER_INT_H

#include <time.h>
#include <stdio.h>

/*
 * Global definitions for use by the InfoServer process
 * please note: An InfoServer client is *NOT* a NetSolve
 *              client. The InfoServer clients are users
 *              of the InfoServer information services.
 *              Both the agents and actual clients are
 *              managed in the is_clients list and represented
 *              by the NS_IS_ClientConnection structure.
 *              Sure, it's confusing, but hey if this was
 *              all obvious it would be no fun.  ;)
 */

typedef struct _NS_IS_BufLine {
  unsigned len;
  char * line;
  struct _NS_IS_BufLine * next;
} NS_IS_BufLine;

typedef struct _NS_IS_ClientConnection {
  int fd;
  NS_IS_BufLine * first_pending_line;
  NS_IS_BufLine * last_pending_line;
  struct _NS_IS_ClientConnection * next;
} NS_IS_ClientConnection;

/*
 * The core database structures
 */

typedef struct _NS_IS_Record {
  char ** value;     /* value array indexed by KeyNameIdx */
  unsigned valuelen; /* # of entries allocated in value[] array */
  struct _NS_IS_Record * prev, * next;
} NS_IS_Record;

typedef struct _NS_IS_KeyNameIdx {
  char * keyname;
  int index;
  struct _NS_IS_KeyNameIdx * next;
} NS_IS_KeyNameIdx;


/*
 * Request management stuff
 */

extern int infoserver_input_fd;
extern int infoserver_client_fd;
extern NS_IS_ClientConnection * is_clients;

#define NS_IS_MAX_REQLEN 1024

/*
 * A public InfoServer uses TCP for receiving agent data,
 * a private InfoServer uses a UN*X socket
 */
typedef enum { NS_IS_PUBLIC, NS_IS_PRIVATE } iserv_access_t;

int spawn_infoserver_int(iserv_access_t accesst);
void iserv_main_loop(void);
void iserv_new_input(void);
void iserv_new_client(void);
void iserv_receive_request(int fd);
void iserv_feed_client(NS_IS_ClientConnection*);

void chain_client(int fd);
void unchain_client(int fd);
void push_to_client(int fd, const char* line);

void print_text(int, const char**);
void treat_get_help(int fd, const char* req);

int dbms_initialize(void);
unsigned dbms_stat_records(void);
double dbms_stat_keyrec(void);
unsigned dbms_stat_ukeys(void);
time_t req_stat_lastput(void);
time_t req_stat_lastget(void);
unsigned req_stat_puts(void);
unsigned req_stat_gets(void);
unsigned req_stat_clients(void);

void search_server_name(int fd, const char * hostname);
void search_server_ueid(int fd, unsigned ueid);
void search_server_time(int fd, time_t t);
void search_job_id(int fd, unsigned id);
void search_job_ueid(int fd, unsigned ueid);
void search_job_time(int fd, time_t t);

int find_key_index(const char * key);
const char * find_index_key(unsigned index);
void iserv_putrecord(const NS_IS_Record * rec);
void iserv_readdb(void);

/*
 * Parser stuff
 */
typedef enum { NS_IS_Equality, NS_IS_Succeeds } NS_IS_Exp_RelOp_t;
typedef enum { NS_IS_And, NS_IS_Or} NS_IS_Exp_CombOp_t;

typedef enum { NS_IS_PMode_Interactive, NS_IS_PMode_LogFile } NS_IS_PMode_t;
extern NS_IS_PMode_t parser_mode;

typedef struct _NS_IS_Expression {
  enum { NS_IS_Exp_KeyValue, 
	 NS_IS_Exp_KeyReq,
	 NS_IS_Exp_Combination } etype;
  /* For key <relation> value */
  NS_IS_Exp_RelOp_t rel_op;
  char * key, * value;
  int key_ndx;
  /* For exp <combination> exp */
  NS_IS_Exp_CombOp_t comb_op;
  struct _NS_IS_Expression * lexp, * rexp;
} NS_IS_Expression;

typedef struct _NS_IS_Insertion {
  char * key, * value;
  struct _NS_IS_Insertion * next;
} NS_IS_Insertion;

int buf_yyinput(char * buf, int max);
int yyerror(char * err);
int yylex(void);
int yyparse(void);
void yyrestart(FILE*);

void dbms_process_query(int, const NS_IS_Expression*);
void dbms_process_insertion(int, const NS_IS_Insertion*);
void treat_quit(int);
void treat_stats(int);

/*
 * Global variables for use by the input routines called
 * by the agnet
 */
extern const char * infoserver_input_path;
extern const char * infoserver_input_host;
extern int infoserver_input_port;

extern int parse_client;
extern char * input_parser_line;
extern int input_parser_pos;
extern int input_parser_len;

/*
 * Reference monitor stuff
 */
typedef enum {NS_IS_RM_Export, NS_IS_RM_Import} NS_IS_RM_Operation_t;

int refmon_grant(NS_IS_RM_Operation_t op, NS_IS_Record * rec, NS_IS_ClientConnection * peer);

#endif
