c Use INSTAB to test numerical stability of
c Gaussian elimination without pivoting
c Coded by Tom Rowan		
c
c constants
c     See instab comments for storage requirements.
c
      integer niwmax,nwmax,nxmax
c
      parameter (niwmax=300,nwmax=300,nxmax=300)
c
c local variables
c
      integer i,iflag,istab,iwork(niwmax),iy,iy1,iy2,iyt,
     *        maxnfe,mdabs,mdcmp,mdcont,mdsing,mduser,mode,
     *        nf1,nf2,nfe,nfeval,nfinc,normx,normy,nx,ny
      real    bechek,bestop,bex,scale(nxmax),scl,stp1,stp2,
     *        stpfac,tol,tol1,tol2,tolfac,x(nwmax),xbound,
     *        xhi,xlo,xrng
      logical ranx
      character*26 flgdsc(-2:2)
c
c subroutines and functions
c
      real    urand
      external beaopt,instab,urand
c
c data
c
      data flgdsc /'invalid input',
     *             'maxnfe exceeded',
     *             'tol satisfied',
     *             'limit of machine precision',
     *             'bestop reached'/
c
c-----------------------------------------------------------
c
      print *,'********************************************'
      print *,'*  INSTAB stability test of                *'
      print *,'*  Gaussian Elimination without pivoting   *'
      print *,'********************************************'
c 
c For descriptions of instab arguments see instab comments.
c
      read *, nx
      read *, ny
c
c Define sense of stability to be used.
c See instab comments for description of istab.
c
c Using relative input perturbations so absolute 
c perturbations are off.
c
      mdabs = 0
c
c Using vector sense to measure distance in the output
c space so component sense is off.  Using the vector sense
c (mdcmp = 0) is the normal choice although also the
c component sense would be appropriate in applications
c like eigenvalue solvers or root finders where it may be
c important for even the smallest component of the solution
c to be computed to high accuracy.
c
      read *, mdcmp
      istab = 2*mdabs + mdcmp
c
      read *, normx
      read *, normy
      read *, xbound
c
c The following three read statements determine when instab
c is interrupted so the user can examine intermediate 
c results.Instab can be interrupted and then continued as if 
c no interrupt had occured when an optimization tolerance is
c satisfied and/or when the maximum number of objective
c function evaluations is reached and/or when the estimate
c of the lower bound on the backward error exceeds a 
c specified threshold.
c 
c Variables that define the sequence of threshold values.
c See instab comments for description of bestop.
c
      read *, stp1,stp2,stpfac
c Variables that define the sequence of tolerances.
c See instab comments for description of tol.
c
      read *, tol1,tol2,tolfac
c
c Variables that define the sequence of maximum number of
c function evaluations.
c See instab comments for description of maxnfe.
c
      read *, nf1,nf2,nfinc
c
c Set initial stepsizes for optimization.
c See instab comments for description of scale.
c
      read *, scl
      scale(1) = -abs(scl)
c
c Each value in the range [iy1, ..., iy2] is used as a 
c seed value to generate a vector of algorithm inputs 
c at which to start instab's search for instability.
c
      read *, iy1,iy2
c
c Examples with backward error estimates .ge. bechek
c will be checked with a more detailed analysis to
c try to verify instability.
c
      read *, bechek
c
c If ranx is true, read limits within which to 
c generate random starting points for each search.
c
      read *, ranx
      if (ranx) then
        read *, xlo,xhi
      end if
c
      print *, 'nx =    ',nx
      print *, 'ny =    ',ny
      print *, 'mdcmp = ',mdcmp
      print *, 'normx =  ',normx
      print *, 'normy =  ',normy
      print *, 'xbound = ',xbound
      print *, 'stp1,stp2,stpfac=',stp1,stp2,stpfac
      print *, 'tol1,tol2,tolfac=',tol1,tol2,tolfac
      print *, 'nf1,nf2,nfinc=',nf1,nf2,nfinc
      print *, 'scale =',scale(1)
      print *, 'iy1,iy2 =',iy1,iy2
      print *, 'bechek =',bechek
      print *, 'ranx =',ranx
      if (ranx) then
        print *, xlo,xhi
      end if
c
c Each iteration of the this loop starts a search for
c instability from a new starting point.
c
      do 50 iyt = iy1,iy2
        print *, '****************************************'
        print *, '****************************************'
        print *, '****** new starting point **************'
        print *, ' '
        maxnfe = nf1
        tol = tol1
        bestop = stp1
        iy = iyt
        print *, 'iy0 =',iy
c
c Generate random starting point or read user-supplied
c starting point.
c
        if (ranx) then
          xrng = xhi-xlo
          do 10 i = 1,nx
            x(i) = xrng*urand(iy)+xlo
   10     continue
        else
          read *, (x(i),i=1,nx)
        end if
        call psvec ('x0  ',nx,x) 
c
c Print output headers.
c
        write (6,1010)
 1010   format (///t5,'maxnfe',t15,'tol',t30,'bestop',t45,
     *          'bex',t60,'nfe',t70,'iflag'/)
c
c Set instab's operating mode.
c See instab comments for description of mode.
c 
c First call to instab so continuation mode is off.
c
        mdcont = 0
c
c Using default options so user options mode is off.
c
        mduser = 0
c
c Using optimization so single-step mode is off.
c
        mdsing = 0
c
   20   continue
c
        mode = 4*mdsing + 2*mduser + mdcont
c
c Test the numerical algorithm implemented in subroutine
c approx for instability.  The subroutine exact computes
c estimates of the true solutions, usually by implementing
c the numerical algorithm in higher-precision.
c
        call instab (nx,ny,mode,istab,normx,normy,xbound,
     *              bestop,tol,maxnfe,scale,x,bex,nfe,iwork,
     *              iflag)
c
c Print intermediate results for this search.
c
        write (6,1020) maxnfe,tol,bestop,bex,nfe,iflag
 1020   format (t5,i6,t15,e10.2,t30,e10.2,t45,e13.5,t60,
     *          i6,t70,i5)
c
c Check iflag to see if done or which termination
c test needs to be reset before resuming search.
c
        if (iflag .eq. -1) then
          if (maxnfe .ge. nf2) go to 30
          maxnfe = maxnfe+nfinc
        else if (iflag .eq. 0) then
          if (tol .le. tol2) go to 30
          tol = tol*tolfac
        else if (iflag .eq. 2) then
          if (bestop .ge. stp2) go to 30
        bestop = bestop*stpfac
        else
          go to 30
        end if
c 
c Resume search for instability in continuation mode.
c
        mdcont = 1
        go to 20
c
c Print results for this search.
c
   30   continue
        print *, '****************************************'
        print *, '****************************************'
        print *, 'search results for this initial x *******'
	write (6,1025) iflag,flgdsc(iflag)
 1025   format (' iflag =',i3,5x,'(',a26,')')
        write (6,1030) nfe,bex,iyt
 1030   format (/1x,'nfe=',i7,5x,'bex=',e16.8,5x,'iy=',
     *          i10/)
        call psvec ('x =',nx,x)
c
c The value of bex returned by instab after the search for
c instability may be a severe overestimate of the backward
c error because it is based on a small number of
c perturbations to estimate the problem's condition. If this
c point appears to be an example of instability, check it
c more closely in single-step mode making at least 1000
c more perturbations to estimate the problem's condition.
c
        if (bex .ge. bechek) then
          print *, '****************************************'
          print *, '********* checking *********************'
          mdsing = 1
          mode = 4*mdsing + 2*mduser + mdcont
c
c Refine the current estimate of lower bound on backward 
c error 1000 times.
c
          do 40 nfeval = 1,1000
            call instab (nx,ny,mode,istab,normx,normy,
     *                   xbound,bestop,tol,maxnfe,scale,x,
     *                   bex,nfe,iwork,iflag)
   40     continue
c 
c The computed estimate of the lower bound on the backward
c error corresponds roughly to the relative error in a
c computed solution to a well-conditioned problem.
c
          print *, '****************************************'
          write (6,1040) bex
 1040     format (t5,'estimated lower bound on backward ',
     *            'error =',e20.8)
	  if (bex .ge. 1.) then
	    write (6,1050)
 1050       format (//t5,'Tested numerical algorithm ',
     *              'appears to be UNSTABLE.'//t5,
     *              'The degree of instability ',
     *              'exhibited for this input x corresponds to'/t5,
     *              'getting garbage answers on ',
     *              'well-conditioned problems.'/)
	    print *, '   The matrix A and right-hand ',
     *               'side b corresponding to x are:'
            call psmat ('A =',ny,ny,ny,x)
	    call psvec ('b =',ny,x(ny*ny+1))
            print *, '****************************************'
	  end if
        end if
  50  continue
      stop
      end
      subroutine approx (nx,x,ny,y,feas)
c
      integer nx,ny
      real x(nx),y(ny)
      logical feas
c
      integer lda,job
      parameter (lda=10,job=0)
c
      integer info,ipvt(lda),j
      real a(lda,lda)
c
c gaussian elimination without pivoting
c
      do 10 j = 1,ny
        call scopy (ny,x((j-1)*ny+1),1,a(1,j),1)
   10 continue
      call xsgefa (a,lda,ny,ipvt,info)
      feas = info .eq. 0
      if (.not. feas) return
      call scopy (ny,x(ny*ny+1),1,y,1)
      call sgesl (a,lda,ny,ipvt,y,job)
      return
      end
      subroutine exact (nx,x,ny,y,feas)
c
      integer nx,ny
      real x(nx),y(ny)
      logical feas
c
      integer lda,job
      parameter (lda=10,job=0)
c
      integer info,ipvt(lda),j
      double precision da(lda,lda),db(lda)
c
c gaussian elimination without pivoting
c
      do 10 j = 1,ny
        call sdcopy (ny,x((j-1)*ny+1),1,da(1,j),1)
   10 continue
      call xdgefa (da,lda,ny,ipvt,info)
      feas = info .eq. 0
      if (.not. feas) return
      call sdcopy (ny,x(ny*ny+1),1,db,1)
      call dgesl (da,lda,ny,ipvt,db,job)
      call dscopy (ny,db,1,y,1)
      return
      end
      subroutine xsgefa(a,lda,n,ipvt,info)
      integer lda,n,ipvt(1),info
      real a(lda,1)
c
c     xsgefa factors a real matrix by gaussian elimination.
c
c*************************************************************
c     xsgefa is linpack routine sgefa but without pivoting 
c     all code modifications are surrounded by asterisks
c*************************************************************
c
c     on entry
c
c        a       real(lda, n)
c                the matrix to be factored.
c
c        lda     integer
c                the leading dimension of the array  a .
c
c        n       integer
c                the order of the matrix  a .
c
c     on return
c
c        a       an upper triangular matrix and the multipliers
c                which were used to obtain it.
c                the factorization can be written  a = l*u  where
c                l  is a product of permutation and unit lower
c                triangular matrices and  u  is upper triangular.
c
c        ipvt    integer(n)
c                an integer vector of pivot indices.
c
c        info    integer
c                = 0  normal value.
c                = k  if  u(k,k) .eq. 0.0 .  this is not an error
c                     condition for this subroutine, but it does
c                     indicate that sgesl or sgedi will divide by zero
c                     if called.  use  rcond  in sgeco for a reliable
c                     indication of singularity.
c
c     subroutines and functions
c
c     blas saxpy,sscal,isamax
c
c     internal variables
c
      real t
      integer isamax,j,k,kp1,l,nm1
c
c
c     gaussian elimination without pivoting
c
      info = 0
      nm1 = n - 1
      if (nm1 .lt. 1) go to 70
      do 60 k = 1, nm1
         kp1 = k + 1
c
c        diagonal element is pivot
c
c****************************************************************
c**      l = isamax(n-k+1,a(k,k),1) + k - 1
         l = k
c****************************************************************
         ipvt(k) = l
c
c        zero pivot implies back solve will fail
c
         if (a(l,k) .eq. 0.0e0) go to 40
c
c           disallow interchanges
c
c****************************************************************
c**         if (l .eq. k) go to 10
c**            t = a(l,k)
c**            a(l,k) = a(k,k)
c**            a(k,k) = t
c**10       continue
c****************************************************************
c
c           compute multipliers
c
            t = -1.0e0/a(k,k)
            call sscal(n-k,t,a(k+1,k),1)
c
c           row elimination with column indexing
c
            do 30 j = kp1, n
               t = a(l,j)
               if (l .eq. k) go to 20
                  a(l,j) = a(k,j)
                  a(k,j) = t
   20          continue
               call saxpy(n-k,t,a(k+1,k),1,a(k+1,j),1)
   30       continue
         go to 50
   40    continue
            info = k
   50    continue
   60 continue
   70 continue
      ipvt(n) = n
      if (a(n,n) .eq. 0.0e0) info = n
      return
      end
      subroutine xdgefa(a,lda,n,ipvt,info)
      integer lda,n,ipvt(1),info
      double precision a(lda,1)
c
c     xdgefa factors a double precision matrix by gaussian elimination.
c
c**********************************************************************
c     xdgefa is linpack routine dgefa but without pivoting.
c     all code modifications are surrounded by asterisks.
c**********************************************************************
c
c     on entry
c
c        a       double precision(lda, n)
c                the matrix to be factored.
c
c        lda     integer
c                the leading dimension of the array  a .
c
c        n       integer
c                the order of the matrix  a .
c
c     on return
c
c        a       an upper triangular matrix and the multipliers
c                which were used to obtain it.
c                the factorization can be written  a = l*u  where
c                l  is a product of permutation and unit lower
c                triangular matrices and  u  is upper triangular.
c
c        ipvt    integer(n)
c                an integer vector of pivot indices.
c
c        info    integer
c                = 0  normal value.
c                = k  if  u(k,k) .eq. 0.0 .  this is not an error
c                     condition for this subroutine, but it does
c                     indicate that dgesl or dgedi will divide by zero
c                     if called.  use  rcond  in dgeco for a reliable
c                     indication of singularity.
c
c     subroutines and functions
c
c     blas daxpy,dscal,idamax
c
c     internal variables
c
      double precision t
      integer idamax,j,k,kp1,l,nm1
c
c
c     gaussian elimination without pivoting
c
      info = 0
      nm1 = n - 1
      if (nm1 .lt. 1) go to 70
      do 60 k = 1, nm1
         kp1 = k + 1
c
c        diagonal element is pivot
c
c***************************************************************
c**      l = idamax(n-k+1,a(k,k),1) + k - 1
         l = k
c***************************************************************
         ipvt(k) = l
c
c        zero pivot implies this column already triangularized
c
         if (a(l,k) .eq. 0.0d0) go to 40
c
c           disallow interchanges   
c
c***************************************************************
c**         if (l .eq. k) go to 10
c**            t = a(l,k)
c**            a(l,k) = a(k,k)
c**            a(k,k) = t
c**10       continue
c***************************************************************
c
c           compute multipliers
c
            t = -1.0d0/a(k,k)
            call dscal(n-k,t,a(k+1,k),1)
c
c           row elimination with column indexing
c
            do 30 j = kp1, n
               t = a(l,j)
               if (l .eq. k) go to 20
                  a(l,j) = a(k,j)
                  a(k,j) = t
   20          continue
               call daxpy(n-k,t,a(k+1,k),1,a(k+1,j),1)
   30       continue
         go to 50
   40    continue
            info = k
   50    continue
   60 continue
   70 continue
      ipvt(n) = n
      if (a(n,n) .eq. 0.0d0) info = n
      return
      end
      subroutine sgesl(a,lda,n,ipvt,b,job)
      integer lda,n,ipvt(1),job
      real a(lda,1),b(1)
c
c     sgesl solves the real system
c     a * x = b  or  trans(a) * x = b
c     using the factors computed by sgeco or sgefa.
c
c     on entry
c
c        a       real(lda, n)
c                the output from sgeco or sgefa.
c
c        lda     integer
c                the leading dimension of the array  a .
c
c        n       integer
c                the order of the matrix  a .
c
c        ipvt    integer(n)
c                the pivot vector from sgeco or sgefa.
c
c        b       real(n)
c                the right hand side vector.
c
c        job     integer
c                = 0         to solve  a*x = b ,
c                = nonzero   to solve  trans(a)*x = b  where
c                            trans(a)  is the transpose.
c
c     on return
c
c        b       the solution vector  x .
c
c     error condition
c
c        a division by zero will occur if the input factor contains a
c        zero on the diagonal.  technically this indicates singularity
c        but it is often caused by improper arguments or improper
c        setting of lda .  it will not occur if the subroutines are
c        called correctly and if sgeco has set rcond .gt. 0.0
c        or sgefa has set info .eq. 0 .
c
c     to compute  inverse(a) * c  where  c  is a matrix
c     with  p  columns
c           call sgeco(a,lda,n,ipvt,rcond,z)
c           if (rcond is too small) go to ...
c           do 10 j = 1, p
c              call sgesl(a,lda,n,ipvt,c(1,j),0)
c        10 continue
c
c     linpack. this version dated 08/14/78 .
c     cleve moler, university of new mexico, argonne national lab.
c
c     subroutines and functions
c
c     blas saxpy,sdot
c
c     internal variables
c
      real sdot,t
      integer k,kb,l,nm1
c
      nm1 = n - 1
      if (job .ne. 0) go to 50
c
c        job = 0 , solve  a * x = b
c        first solve  l*y = b
c
         if (nm1 .lt. 1) go to 30
         do 20 k = 1, nm1
            l = ipvt(k)
            t = b(l)
            if (l .eq. k) go to 10
               b(l) = b(k)
               b(k) = t
   10       continue
            call saxpy(n-k,t,a(k+1,k),1,b(k+1),1)
   20    continue
   30    continue
c
c        now solve  u*x = y
c
         do 40 kb = 1, n
            k = n + 1 - kb
            b(k) = b(k)/a(k,k)
            t = -b(k)
            call saxpy(k-1,t,a(1,k),1,b(1),1)
   40    continue
      go to 100
   50 continue
c
c        job = nonzero, solve  trans(a) * x = b
c        first solve  trans(u)*y = b
c
         do 60 k = 1, n
            t = sdot(k-1,a(1,k),1,b(1),1)
            b(k) = (b(k) - t)/a(k,k)
   60    continue
c
c        now solve trans(l)*x = y
c
         if (nm1 .lt. 1) go to 90
         do 80 kb = 1, nm1
            k = n - kb
            b(k) = b(k) + sdot(n-k,a(k+1,k),1,b(k+1),1)
            l = ipvt(k)
            if (l .eq. k) go to 70
               t = b(l)
               b(l) = b(k)
               b(k) = t
   70       continue
   80    continue
   90    continue
  100 continue
      return
      end
      subroutine dgesl(a,lda,n,ipvt,b,job)
      integer lda,n,ipvt(1),job
      double precision a(lda,1),b(1)
c
c     dgesl solves the double precision system
c     a * x = b  or  trans(a) * x = b
c     using the factors computed by dgeco or dgefa.
c
c     on entry
c
c        a       double precision(lda, n)
c                the output from dgeco or dgefa.
c
c        lda     integer
c                the leading dimension of the array  a .
c
c        n       integer
c                the order of the matrix  a .
c
c        ipvt    integer(n)
c                the pivot vector from dgeco or dgefa.
c
c        b       double precision(n)
c                the right hand side vector.
c
c        job     integer
c                = 0         to solve  a*x = b ,
c                = nonzero   to solve  trans(a)*x = b  where
c                            trans(a)  is the transpose.
c
c     on return
c
c        b       the solution vector  x .
c
c     error condition
c
c        a division by zero will occur if the input factor contains a
c        zero on the diagonal.  technically this indicates singularity
c        but it is often caused by improper arguments or improper
c        setting of lda .  it will not occur if the subroutines are
c        called correctly and if dgeco has set rcond .gt. 0.0
c        or dgefa has set info .eq. 0 .
c
c     to compute  inverse(a) * c  where  c  is a matrix
c     with  p  columns
c           call dgeco(a,lda,n,ipvt,rcond,z)
c           if (rcond is too small) go to ...
c           do 10 j = 1, p
c              call dgesl(a,lda,n,ipvt,c(1,j),0)
c        10 continue
c
c     linpack. this version dated 08/14/78 .
c     cleve moler, university of new mexico, argonne national lab.
c
c     subroutines and functions
c
c     blas daxpy,ddot
c
c     internal variables
c
      double precision ddot,t
      integer k,kb,l,nm1
c
      nm1 = n - 1
      if (job .ne. 0) go to 50
c
c        job = 0 , solve  a * x = b
c        first solve  l*y = b
c
         if (nm1 .lt. 1) go to 30
         do 20 k = 1, nm1
            l = ipvt(k)
            t = b(l)
            if (l .eq. k) go to 10
               b(l) = b(k)
               b(k) = t
   10       continue
            call daxpy(n-k,t,a(k+1,k),1,b(k+1),1)
   20    continue
   30    continue
c
c        now solve  u*x = y
c
         do 40 kb = 1, n
            k = n + 1 - kb
            b(k) = b(k)/a(k,k)
            t = -b(k)
            call daxpy(k-1,t,a(1,k),1,b(1),1)
   40    continue
      go to 100
   50 continue
c
c        job = nonzero, solve  trans(a) * x = b
c        first solve  trans(u)*y = b
c
         do 60 k = 1, n
            t = ddot(k-1,a(1,k),1,b(1),1)
            b(k) = (b(k) - t)/a(k,k)
   60    continue
c
c        now solve trans(l)*x = y
c
         if (nm1 .lt. 1) go to 90
         do 80 kb = 1, nm1
            k = n - kb
            b(k) = b(k) + ddot(n-k,a(k+1,k),1,b(k+1),1)
            l = ipvt(k)
            if (l .eq. k) go to 70
               t = b(l)
               b(l) = b(k)
               b(k) = t
   70       continue
   80    continue
   90    continue
  100 continue
      return
      end
      subroutine daxpy(n,da,dx,incx,dy,incy)
c
c     constant times a vector plus a vector.
c     uses unrolled loops for increments equal to one.
c     jack dongarra, linpack, 3/11/78.
c
      double precision dx(1),dy(1),da
      integer i,incx,incy,ix,iy,m,mp1,n
c
      if(n.le.0)return
      if (da .eq. 0.0d0) return
      if(incx.eq.1.and.incy.eq.1)go to 20
c
c        code for unequal increments or equal increments
c          not equal to 1
c
      ix = 1
      iy = 1
      if(incx.lt.0)ix = (-n+1)*incx + 1
      if(incy.lt.0)iy = (-n+1)*incy + 1
      do 10 i = 1,n
        dy(iy) = dy(iy) + da*dx(ix)
        ix = ix + incx
        iy = iy + incy
   10 continue
      return
c
c        code for both increments equal to 1
c
c
c        clean-up loop
c
   20 m = mod(n,4)
      if( m .eq. 0 ) go to 40
      do 30 i = 1,m
        dy(i) = dy(i) + da*dx(i)
   30 continue
      if( n .lt. 4 ) return
   40 mp1 = m + 1
      do 50 i = mp1,n,4
        dy(i) = dy(i) + da*dx(i)
        dy(i + 1) = dy(i + 1) + da*dx(i + 1)
        dy(i + 2) = dy(i + 2) + da*dx(i + 2)
        dy(i + 3) = dy(i + 3) + da*dx(i + 3)
   50 continue
      return
      end
      subroutine  dscal(n,da,dx,incx)
c
c     scales a vector by a constant.
c     uses unrolled loops for increment equal to one.
c     jack dongarra, linpack, 3/11/78.
c
      double precision da,dx(1)
      integer i,incx,m,mp1,n,nincx
c
      if(n.le.0)return
      if(incx.eq.1)go to 20
c
c        code for increment not equal to 1
c
      nincx = n*incx
      do 10 i = 1,nincx,incx
        dx(i) = da*dx(i)
   10 continue
      return
c
c        code for increment equal to 1
c
c
c        clean-up loop
c
   20 m = mod(n,5)
      if( m .eq. 0 ) go to 40
      do 30 i = 1,m
        dx(i) = da*dx(i)
   30 continue
      if( n .lt. 5 ) return
   40 mp1 = m + 1
      do 50 i = mp1,n,5
        dx(i) = da*dx(i)
        dx(i + 1) = da*dx(i + 1)
        dx(i + 2) = da*dx(i + 2)
        dx(i + 3) = da*dx(i + 3)
        dx(i + 4) = da*dx(i + 4)
   50 continue
      return
      end
      double precision function ddot(n,dx,incx,dy,incy)
c
c     forms the dot product of two vectors.
c     uses unrolled loops for increments equal to one.
c     jack dongarra, linpack, 3/11/78.
c
      double precision dx(1),dy(1),dtemp
      integer i,incx,incy,ix,iy,m,mp1,n
c
      ddot = 0.0d0
      dtemp = 0.0d0
      if(n.le.0)return
      if(incx.eq.1.and.incy.eq.1)go to 20
c
c        code for unequal increments or equal increments
c          not equal to 1
c
      ix = 1
      iy = 1
      if(incx.lt.0)ix = (-n+1)*incx + 1
      if(incy.lt.0)iy = (-n+1)*incy + 1
      do 10 i = 1,n
        dtemp = dtemp + dx(ix)*dy(iy)
        ix = ix + incx
        iy = iy + incy
   10 continue
      ddot = dtemp
      return
c
c        code for both increments equal to 1
c
c
c        clean-up loop
c
   20 m = mod(n,5)
      if( m .eq. 0 ) go to 40
      do 30 i = 1,m
        dtemp = dtemp + dx(i)*dy(i)
   30 continue
      if( n .lt. 5 ) go to 60
   40 mp1 = m + 1
      do 50 i = mp1,n,5
        dtemp = dtemp + dx(i)*dy(i) + dx(i + 1)*dy(i + 1) +
     *   dx(i + 2)*dy(i + 2) + dx(i + 3)*dy(i + 3) + dx(i + 4)*dy(i + 4)
   50 continue
   60 ddot = dtemp
      return
      end
      real function sdot(n,sx,incx,sy,incy)
c
c     forms the dot product of two vectors.
c     uses unrolled loops for increments equal to one.
c     jack dongarra, linpack, 3/11/78.
c
      real sx(1),sy(1),stemp
      integer i,incx,incy,ix,iy,m,mp1,n
c
      stemp = 0.0e0
      sdot = 0.0e0
      if(n.le.0)return
      if(incx.eq.1.and.incy.eq.1)go to 20
c
c        code for unequal increments or equal increments
c          not equal to 1
c
      ix = 1
      iy = 1
      if(incx.lt.0)ix = (-n+1)*incx + 1
      if(incy.lt.0)iy = (-n+1)*incy + 1
      do 10 i = 1,n
        stemp = stemp + sx(ix)*sy(iy)
        ix = ix + incx
        iy = iy + incy
   10 continue
      sdot = stemp
      return
c
c        code for both increments equal to 1
c
c
c        clean-up loop
c
   20 m = mod(n,5)
      if( m .eq. 0 ) go to 40
      do 30 i = 1,m
        stemp = stemp + sx(i)*sy(i)
   30 continue
      if( n .lt. 5 ) go to 60
   40 mp1 = m + 1
      do 50 i = mp1,n,5
        stemp = stemp + sx(i)*sy(i) + sx(i + 1)*sy(i + 1) +
     *   sx(i + 2)*sy(i + 2) + sx(i + 3)*sy(i + 3) + sx(i + 4)*sy(i + 4)
   50 continue
   60 sdot = stemp
      return
      end
