import AT.Ac.univie.imp.loeffler.pde.threeD.fd.*;
import AT.Ac.univie.imp.loeffler.util.*;
import AT.Ac.univie.imp.loeffler.parallel.*;

/**
 * Representation of the Poisson equation (laplace phi = -4 pi rho) with a point charge as the source term rho.
 * <p>
 * The PDE is solved on a box centered at the origin, where also the point charge is located.
 *
 * @author Gerald Loeffler (Gerald.Loeffler@univie.ac.at)
 */
public final class PoissonPointCharge extends CachingPDE {
     /**
      * construct given the mapping from physical space to grid space and the amount of the charge at the center.
      * <p>
      * The box is always centered at the origin and is a cube.
      *
      * @param boxLength length of the box ( > 0)
      * @param charge    amount of the charge to be put in the center of the box ( != 0)
      */
     public PoissonPointCharge(double boxLength, double charge) {
          Contract.pre(boxLength > 0,"boxLength > 0");
          Contract.pre(charge != 0,"charge != 0");
          
          this.boxLength = boxLength;
          this.charge    = charge;
     }

     /**
      * implements method from PDE
      *
      * @see PDE#getGridSpacing
      */
     public double getGridSpacing(int size) {
          Contract.pre(size > 0,"size > 0");
          
          return (boxLength/(size - 1.0));
     }
     
     /**
      * implements method from PDE
      *
      * @see PDE#evaluateLHS
      */
     public double evaluateLHS(ConstBoundaryGrid u, int x, int y, int z) {
          Contract.pre(u != null,"u not null-object");
          
          int size = u.size();
          
          Contract.pre(x > 0 && x < (size - 1),"0 < x < (grid size - 1)");
          Contract.pre(y > 0 && y < (size - 1),"0 < y < (grid size - 1)");
          Contract.pre(z > 0 && z < (size - 1),"0 < z < (grid size - 1)");
          
          double h = getGridSpacing(size);
          return ((1.0/(h*h))*(u.get(x + 1,y,z) + u.get(x - 1,y,z) + u.get(x,y + 1,z) + u.get(x,y - 1,z) +
                               u.get(x,y,z + 1) + u.get(x,y,z - 1) - 6*u.get(x,y,z)));
     }
     
     /**
      * implements method from PDE
      *
      * @see PDE#evaluate
      */
     public double evaluate(ConstBoundaryGrid u, ConstNoBoundaryGrid f, int x, int y, int z) {
          Contract.pre(u != null && f != null,"all objects not null-objects");
          
          int size = u.size();
          
          Contract.pre(f.size() == size,"size of u equal size of f");
          Contract.pre(x > 0 && x < (size - 1),"0 < x < (grid size - 1)");
          Contract.pre(y > 0 && y < (size - 1),"0 < y < (grid size - 1)");
          Contract.pre(z > 0 && z < (size - 1),"0 < z < (grid size - 1)");
          
          double h = getGridSpacing(size);
          return ((1.0/6.0)*(u.get(x + 1,y,z) + u.get(x - 1,y,z) + u.get(x,y + 1,z) + u.get(x,y - 1,z) +
                             u.get(x,y,z + 1) + u.get(x,y,z - 1) - h*h*f.get(x,y,z)));
     }
     
     /**
      * implements method from CachingPDE
      *
      * @see CachingPDE#actuallySampleRHS
      */
     protected NoBoundaryGrid actuallySampleRHS(int size) {
          Contract.pre(size > 0,"size > 0");
          
          NoBoundaryGrid f = new NoBoundaryGrid(size,0);
          int center = size/2;
          f.set(center,center,center,-4*Math.PI*charge/Math.pow(getGridSpacing(size),3));
          return f;
     }

     /**
      * implements method from CachingPDE
      *
      * @see CachingPDE#sampleMatrix
      */
     protected Object actuallySampleMatrix(int size) {
          Contract.pre(size > 0,"size > 0");
          
          return (new Double(getGridSpacing(size)));
     }

     private double boxLength;
     private double charge;

     private BoundaryGrid exactSolution(int size) {
          BoundaryGrid phi = new FixedBoundaryGrid(size,0,0);
          double h = boxLength/(size - 1.0);
          int center = size/2 + 1;
          for (int x = 1; x < (size - 1); x++) {
               double dx = (x - center)*h;
               for (int y = 1; y < (size - 1); y++) {
                    double dy = (y - center)*h;
                    for (int z = 1; z < (size - 1); z++) {
                         double dz = (z - center)*h;
                         if ((dx != 0) && (dy != 0) && (dz != 0)) {
                              phi.set(x,y,z,charge/Math.sqrt(dx*dx + dy*dy + dz*dz));
                         }
                    }
               }
          }
          return phi;
     }
/*
     static native void nativeTSetconc();
     static {
          System.loadLibrary("NativeThreads");
     }
*/
     public static void main(String[] args) {
          if (args.length != 5) {
               throw new IllegalArgumentException("Give num of threads, finest level, num of smoothing steps, " +
                    "cycling strategy, and num of multigrid steps on the command line.");
          }
          int numThreads = Integer.valueOf(args[0]).intValue();
          int finest = Integer.valueOf(args[1]).intValue();
          int numSmooth = Integer.valueOf(args[2]).intValue();
          int cycStrategy = Integer.valueOf(args[3]).intValue();
          int numMG = Integer.valueOf(args[4]).intValue();
/*
          nativeTSetconc();
*/
          Parallelizer.setDefaultNumberOfThreads(numThreads);
          
          PoissonPointCharge pde = new PoissonPointCharge(100,0.001);
          Smoother sm = new RedBlackGaussSeidel(pde);
          //Smoother sm = new DampedJacobi(pde,DampedJacobi.IDEAL_WEIGHTING_FACTOR);
          Restrictor res = new RestrictorByStencil(RestrictorByStencil.FULL_WEIGHTING);
          Interpolator in = new InterpolatorByStencil(InterpolatorByStencil.TRILINEAR);
          Solver s = new ArithmeticSolver(pde);
          //Solver s = new SolverThroughSmoother(sm,0.00001);
          ConstBoundaryGrid p = new FixedBoundaryGrid(3,0,0);
          FMG fmg = new FMG(pde,sm,res,in,s,p,p);
          fmg.addObserver(new TestObserver());
          try {
               fmg.fmg(1,finest,numSmooth,numSmooth,cycStrategy,numMG);
          } catch (FMGAlreadyExecutingException e) {
               System.err.println("Shit!");
               return;
          }
          ConstBoundaryGrid result;
          try {
               result = fmg.waitForResult();
          } catch (FMGNotExecutingException e) {
               System.err.println("Shit again!");
               return;
          }
          
          System.out.println("0  0  0:  " + result.get(0,0,0));
          System.out.println("1  2  3:  " + result.get(1,2,3));
          System.out.println("30 20 10: " + result.get(30,20,10));
/*
          int xMaxErr = 0,yMaxErr = 0,zMaxErr = 0,xMinErr = 0,yMinErr = 0,zMinErr = 0;
          double maxErr = 0,minErr = Double.MAX_VALUE,avgErr = 0;
          int nErr = 0;
          ConstBoundaryGrid phi = pde.exactSolution(FMG.sizeFromLevel(finest));
          for (int x = 1; x < (FMG.sizeFromLevel(finest) - 1); x++) {
               for (int y = 1; y < (FMG.sizeFromLevel(finest) - 1); y++) {
                    for (int z = 1; z < (FMG.sizeFromLevel(finest) - 1); z++) {
                         double exact = phi.get(x,y,z);
                         if (exact != 0) {
                              double thisError = Math.abs((exact - result.get(x,y,z))/exact);
                              avgErr += thisError; nErr++;
                              if (thisError > maxErr) {
                                   maxErr = thisError;
                                   xMaxErr = x; yMaxErr = y; zMaxErr = z;
                              }
                              if (thisError < minErr) {
                                   minErr = thisError;
                                   xMinErr = x; yMinErr = y; zMinErr = z;
                              }
                         }
                    }
               }
          }
          System.out.println("Largest error:  " + maxErr + " at " + xMaxErr + "," + yMaxErr + "," + zMaxErr +
                             ", true value = " + phi.get(xMaxErr,yMaxErr,zMaxErr) +
                             ", approx. value = " + result.get(xMaxErr,yMaxErr,zMaxErr));
          System.out.println("Smallest error: " + minErr + " at " + xMinErr + "," + yMinErr + "," + zMinErr+
                             ", true value = " + phi.get(xMinErr,yMinErr,zMinErr) +
                             ", approx. value = " + result.get(xMinErr,yMinErr,zMinErr));
          System.out.println("Average error:  " + avgErr/nErr);
*/
     }
}
