package AT.Ac.univie.imp.loeffler.pde.threeD.fd;
import  AT.Ac.univie.imp.loeffler.util.*;
import  AT.Ac.univie.imp.loeffler.parallel.*;

/**
 * Solve a PDE by iteratively applying a smoothing (i.e. relaxation) algorithm.
 *
 * @author Gerald Loeffler (Gerald.Loeffler@univie.ac.at)
 */
public final class SolverThroughSmoother implements Solver {
     /**
      * construct from smoother and allowed error.
      *
      * @param smoother the object supplying the smoothing algorithm
      * @param error    the allowed error for the solution as a fraction of the true solution (0 < error < 1)
      */
     public SolverThroughSmoother(Smoother smoother, double error) {
          Contract.pre(smoother != null,"smoother not null-object");
          Contract.pre(error > 0 && error < 1,"0 < error < 1");
          
          this.smoother = smoother;
          this.error    = error;
     }

     /**
      * implements method from Solver.
      *
      * @see Solver#solve
      */
     public BoundaryGrid solve(ConstBoundaryGrid u, ConstNoBoundaryGrid f) {
          Contract.pre(u != null && f != null,"all objects not null-objects");
          
          size = u.size();
          
          Contract.pre(f.size() == size,"size of u equal size of f");
          
          totalRange             = new IntRange1D(1,size - 2);
          ConstBoundaryGrid gOld = u;
          do {
               ConstBoundaryGrid gNew = smoother.smooth(gOld,f);
               goOn = false;
               new SolveThroughSmootherParallelizer(this,gOld,gNew);
               gOld = gNew;
          } while (goOn);

          return ((BoundaryGrid) gOld);
     }
     
     /**
      * implements the parallel part of solve().
      */
     void solveProper(ConstBoundaryGrid gOld, ConstBoundaryGrid gNew, int myNum, int totalNum) {
          IntRange1D range = Parallelizer.partition(totalRange,myNum,totalNum);
          for (int x = range.from(); (x <= range.to()) && !goOn; x++) {
                    for (int y = 1; (y <= (size - 2)) && !goOn; y++) {
                         for (int z = 1; (z <= (size - 2)) && !goOn; z++) {
                              double n = gNew.get(x,y,z);
                              double o = gOld.get(x,y,z);
                              if ((n != 0) && (Math.abs((o - n)/n) > error)) goOn = true;
                         }
                    }
               }
     }

     private Smoother   smoother;
     private double     error;
     private int        size;
     private IntRange1D totalRange;
     private boolean    goOn;
}
