package AT.Ac.univie.imp.loeffler.pde.threeD.fd;
import  AT.Ac.univie.imp.loeffler.util.*;
import  AT.Ac.univie.imp.loeffler.parallel.*;

/**
 * A restrictor that operates according to a restriction stencil.
 * <p>
 * A stencil is a means of defining the relationship of a grid element and its 26 nearest neighbors. In this case the
 * restriction stencil defines how to contract the values of a fine grid to the values of a coarse grid during the
 * process of restriction.
 * 
 * @see Stencil
 *
 * @author Gerald Loeffler (Gerald.Loeffler@univie.ac.at)
 */
public final class RestrictorByStencil implements Restrictor {
     /**
      * the stencil that characterises straight injection where elements of those elements of the fine grid that are 
      * also present in the coarse grid are simply copied from the fine grid to the coarse grid. This method is not 
      * always stable.
      */
     public static final Stencil STRAIGHT_INJECTION = new StencilImpl(
          0,0,0, 0,0,0, 0,0,0,
          0,0,0, 0,1,0, 0,0,0,
          0,0,0, 0,0,0, 0,0,0);

     /**
      * the stencil that characterises full weighting. The stencil for full weighting is the the transpose of the 
      * stencil for trilinear interpolation, which is of theoretical importance.
      */
     public static final Stencil FULL_WEIGHTING = new StencilImpl(
          1./64.,1./32.,1./64., 1./32.,1./16.,1./32., 1./64.,1./32.,1./64.,
          1./32.,1./16.,1./32., 1./16.,1./8. ,1./16., 1./32.,1./16.,1./32.,
          1./64.,1./32.,1./64., 1./32.,1./16.,1./32., 1./64.,1./32.,1./64.);

     /**
      * the stencil that characterises half weighting which is halfway between full weighting and straigh injection.
      */
     public static final Stencil HALF_WEIGHTING = new StencilImpl(
            0   ,1./32.,  0   , 1./32.,1./16.,1./32.,   0   ,1./32.,  0   ,
          1./32.,1./16.,1./32., 1./16.,1./4. ,1./16., 1./32.,1./16.,1./32.,
            0   ,1./32.,  0   , 1./32.,1./16.,1./32.,   0   ,1./32.,  0   );

     /**
      * construct from restriction stencil.
      *
      * @param stencil the stencil that defines restriction
      */
     public RestrictorByStencil(Stencil stencil) {
          Contract.pre(stencil != null,"stencil not null-object");
          
          s = stencil;
     }

     /**
      * implements method from Restrictor.
      *
      * @see Restrictor#restrict
      */
     public Grid restrict(ConstGrid grid) {
          Contract.pre(grid != null,"grid not null-object");
          
          int fineSize   = grid.size();
          int fineLevel  = FMG.levelFromSize(fineSize);
          
          Contract.pre(fineLevel > 1,"level of grid > 1");
          
          coarseSize = FMG.sizeFromLevel(fineLevel - 1);
          totalRange = new IntRange1D(1,coarseSize - 2);
          Grid coarse = grid.newInstance(coarseSize,0); // create coarse grid
          new RestrictByStencilParallelizer(this,coarse,grid);
          
          return coarse;
     }
     
     /**
      * implements the parallel part of restrict().
      */
     void restrictProper(Grid coarse, ConstGrid fine, int myNum, int totalNum) {
          IntRange1D range = Parallelizer.partition(totalRange,myNum,totalNum);
          for (int i = range.from(); i <= range.to(); i++) {
               int iFine = FMG.fineIndexFromCoarseIndex(i);
               for (int j = 1; j <= (coarseSize - 2); j++) {
                    int jFine = FMG.fineIndexFromCoarseIndex(j);
                    for (int k = 1; k <= (coarseSize - 2); k++) {
                         int kFine = FMG.fineIndexFromCoarseIndex(k);
                         // calc coarse grid element from stencil and fine grid
                         double v = 0;
                         for (int l = -1; l <= +1; l++) {
                              int lFine = iFine + l;
                              for (int m = -1; m <= +1; m++) {
                                   int mFine = jFine + m;
                                   for (int n = -1; n <= +1; n++) {
                                        int nFine = kFine + n;
                                        v += s.get(l,m,n)*fine.get(lFine,mFine,nFine);
                                   }
                              }
                         }
                         coarse.set(i,j,k,v);
                    }
               }
          }
     }

     private Stencil    s;
     private int        coarseSize;
     private IntRange1D totalRange;
}
