package AT.Ac.univie.imp.loeffler.pde.threeD.fd;
import  AT.Ac.univie.imp.loeffler.util.*;
import  AT.Ac.univie.imp.loeffler.parallel.*;

/**
 * An interpolator that operates according to an interpolation stencil.
 * <p>
 * A stencil is a means of defining the relationship of a grid element and its 26 nearest neighbors. In this case the
 * interpolation stencil defines how to distribute the values of a coarse grid to the values of a fine grid during the
 * process of interpolation.
 * 
 * @see Stencil
 *
 * @author Gerald Loeffler (Gerald.Loeffler@univie.ac.at)
 */
public final class InterpolatorByStencil implements Interpolator {
     /**
      * the stencil that characterises trilinear interpolation where every element of the fine grid that does not 
      * coincide with an element of the coarse grid is calculated by linear interpolation between its nearest neighbors.
      */
     public static final Stencil TRILINEAR = new StencilImpl(
          1./8.,1./4.,1./8., 1./4.,1./2.,1./4., 1./8.,1./4.,1./8.,
          1./4.,1./2.,1./4., 1./2.,1./1.,1./2., 1./4.,1./2.,1./4.,
          1./8.,1./4.,1./8., 1./4.,1./2.,1./4., 1./8.,1./4.,1./8.);

     /**
      * construct from interpolation stencil.
      *
      * @param stencil the stencil that defines interpolation
      */
     public InterpolatorByStencil(Stencil stencil) {
          Contract.pre(stencil != null,"stencil not null-object");
          
          s = stencil;
     }

     /**
      * implements method from Interpolator.
      *
      * @see Interpolator#interpolate
      */
     public BoundaryGrid interpolate(ConstBoundaryGrid grid) {
          Contract.pre(grid != null,"grid not null-object");
          
          coarseSize = grid.size();
          fineSize = FMG.sizeFromLevel(FMG.levelFromSize(coarseSize) + 1);
          totalRange = new IntRange1D(0,coarseSize - 1);
          BoundaryGrid fine = (BoundaryGrid) grid.newInstance(fineSize,0); // create fine grid
          new InterpolateByStencilParallelizer(this,fine,grid);
          return fine;
     }
     
     /**
      * implements the parallel part of interpolate().
      */
     void interpolateProper(BoundaryGrid fine, ConstBoundaryGrid coarse, int myNum, int totalNum) {
          IntRange1D range = Parallelizer.partition(totalRange,myNum,totalNum);
          for (int i = range.from(); i <= range.to(); i++) {
               int iFine = FMG.fineIndexFromCoarseIndex(i);
               for (int j = 0; j < coarseSize; j++) {
                    int jFine = FMG.fineIndexFromCoarseIndex(j);
                    for (int k = 0; k < coarseSize; k++) {
                         int kFine = FMG.fineIndexFromCoarseIndex(k);
                         // add stencil to fine grid
                         double v = coarse.get(i,j,k);
                         for (int l = -1; l <= +1; l++) {
                              int lFine = iFine + l;
                              if (lFine >= 1 && lFine < (fineSize - 1)) {
                                   for (int m = -1; m <= +1; m++) {
                                        int mFine = jFine + m;
                                        if (mFine >= 1 && mFine < (fineSize - 1)) {
                                             for (int n = -1; n <= +1; n++) {
                                                  int nFine = kFine + n;
                                                  if (nFine >= 1 && nFine < (fineSize - 1)) {
                                                       fine.add(lFine,mFine,nFine,s.get(l,m,n)*v);
                                                  }
                                             }
                                        }
                                   }
                              }
                         }
                    }
               }
          }
     }

     private Stencil    s;
     private int        coarseSize;
     private int        fineSize;
     private IntRange1D totalRange;
}
