package AT.Ac.univie.imp.loeffler.pde.threeD.fd;
import  AT.Ac.univie.imp.loeffler.parallel.*;
import  AT.Ac.univie.imp.loeffler.util.*;
import  java.util.*;

/**
 * A class that implements the Full Multigrid (FMG) algorithm for the solution of a linear elliptic partial differential
 * equation (PDE) on a regular cubic grid in 3D.
 * <p>
 * The FMG algorithm solves a PDE by discretizing it on several grids of different coarseness corresponding to different
 * levels. The size of a grid at a certain level is 2^level + 1, including the boundary of the grid. The grids at
 * different levels are positioned in such a way that the corners of their boundaries are superimposed. This has the
 * effect that every new grid level shares some grid elements with the next coarser grid and introduces new grid 
 * elements that lie exactly between the grid elements of the next coarser grid.
 * <p>
 * The FMG algorithm starts by solving the PDE on the coarsest grid and then decends to the finest grid by interpolating
 * stepwise from a coarse grid to the next finer grid. At each grid level (except the coarsest) the FMG algorithm
 * employs the Multigrid (MG) algorithm possibly several times to improve the solution on this level.
 * <p>
 * The MG algorithm improves an approximate solution of the PDE on a certain grid level by pre-smoothing the solution
 * with a conventional relaxation algorithm, restricting the problem (in principle) to the next coarser grid level, 
 * obtaining an improved solution to this coarse grid problem by (recursively) calling the MG algorithm possibly
 * several times at that level, interpolating the solution at the coarse grid to the fine grid thereby correcting the 
 * fine grid solution, and finally post-smoothing the solution.
 *
 * @author Gerald Loeffler (Gerald.Loeffler@univie.ac.at)
 */
public class FMG extends Observable {
     /**
      * construct an FMG object that will be able to solve a linear elliptic PDE using the FMG algorithm on a regular 
      * cubic grid in 3D.
      * <p>
      * The FMG algorithm employs several sub-algorithms that must be specified as arguments to the constructor and
      * can not be changed afterwards. Thus a specific FMG object will allways employ these sub-algorithms. However,
      * the details of the FMG algorithm itself must be specified when calling the fmg() method and can thus be changed
      * with every invocation of this method.
      *
      * @param pde                an object representing the PDE to solve
      * @param smoother           an object encapsulating the smoothing algorithm to use
      * @param restrictor         an object encapsulating the restriction algorithm to use
      * @param interpolator       an object encapsulating the interpolation method to use
      * @param solver             an object encapsulating the algorithm to use for solving the PDE on the coarsest level
      * @param solutionPrototype  a prototype object from which the exact type of the desired solution will be deduced.
      *                           By selecting a proper subclass of BoundaryGrid for this prototype object, the boundary
      *                           conditions that the solution must satisfy are selected. The state of the prototype
      *                           object is irrelevant because only its type is used.
      * @param correctionProtype  a prototype object from which the exact type of the correction is deduced. The
      *                           correction is the error of the solution and its type is thus closely linked to the 
      *                           type of the solution. Choose boundary conditions for the correction that are identical
      *                           in nature to the boundary conditions for the solution. E.g. match periodic boundary 
      *                           conditions (PBCs) with PBCs and fixed boundary conditions (FBCs) with FBCs. However,
      *                           for fixed boundary conditions choose zero as the boundary value for the correction 
      *                           such that it will not alter the boundary of the solution when added to it.
      */
     public FMG(PDE pde, Smoother smoother, Restrictor restrictor, Interpolator interpolator, Solver solver, 
                ConstBoundaryGrid solutionPrototype, ConstBoundaryGrid correctionPrototype) {
          Contract.pre(pde != null && smoother != null && restrictor != null && interpolator != null && 
                       solver != null && solutionPrototype != null && correctionPrototype != null,
                       "all objects not null-objects");
          
          this.pde          = pde;
          this.smoother     = smoother;
          this.restrictor   = restrictor;
          this.interpolator = interpolator;
          this.solver       = solver;
          this.uPrototype   = solutionPrototype;
          this.vPrototype   = correctionPrototype;
     }

     /**
      * starts the FMG Algorithm with the given parameters in a separate thread and returns immediately.
      * <p>
      * At any time there can be only one instance of the FMG algorithm executing. Hence if this method is called while
      * the FMG algorithm is already running, an exception is thrown.
      * <p>
      * There are two independent ways of receiving the result of the FMG algorithm that this method produces.
      * The first is to call waitForResult() which blocks the thread that is calling it until the result is available 
      * and then returns it. 
      * The second is by registering one or more java.util.Observer objects with this object (via addObserver()) which 
      * will be notified as soon as the result is available (their update() method will be called with the result
      * of type BoundaryGrid as the second (the arg) argument).
      *
      * @see java.util.Observer
      *
      * @param coarsestLevel   the coarsest level at which the FMG algorithm should operate ( > 0). Note that 
      *                        the Solver object with which the PDE is solved at the coarsest level must be able to 
      *                        handle the level given here.
      * @param finestLevel     the finest level at which the FMG algorithm should operate and hence the level at which
      *                        the solution will be returned (finestLevel >= coarsestLevel)
      * @param numPresmooth    the number of pre-smoothing steps to perform ( > 0)
      * @param numPostsmooth   the number of post-smoothing steps to perform ( >= 0)
      * @param cyclingStrategy the cycling strategy to use ( > 0). A value of 1 results in V-cycles, a value of 2 in 
      *                        W-cycles. This gives the number of times the MG algorithm recursively calls itself to
      *                        solve the coarse grid problem.
      * @param numMultiGrid    the number of invocations of the MG algorithm per level of the FMG algorithm ( > 0)
      * @exception FMGAlreadyExecutingException fmg() has already been called for this object and is still executing
      */
     public synchronized void fmg(int coarsestLevel, int finestLevel, int numPresmooth, int numPostsmooth,
                                  int cyclingStrategy, int numMultiGrid) throws FMGAlreadyExecutingException {
          Contract.pre(coarsestLevel > 0,"coarsestLevel > 0");
          Contract.pre(finestLevel >= coarsestLevel,"finestLevel >= coarsestLevel");
          Contract.pre(numPresmooth > 0,"numPresmooth > 0");
          Contract.pre(numPostsmooth >= 0,"numPostsmooth >= 0");
          Contract.pre(cyclingStrategy > 0,"cyclingStrategy > 0");
          Contract.pre(numMultiGrid > 0,"numMultiGrid > 0");
          
          if (fmgIsExecuting) throw new FMGAlreadyExecutingException();
          fmgIsExecuting = true;
          
          // let's register the default observer with this observable object so that waitForResult() can get the result
          defaultObserver = new FMGDefaultObserver();
          addObserver(defaultObserver);
          
          // now start the FMG algorithm (implemented via fmgProper()) in its own thread
          new FMGThread(this,coarsestLevel,finestLevel,numPresmooth,numPostsmooth,cyclingStrategy,numMultiGrid);
     }
     
     /**
      * blocks the current thread until the FMG algorithm which is executing in a different thread has finished and 
      * then returns the solution of the PDE.
      * <p>
      * If the FMG algorithm is not currently executing or has not just delivered a result an exception is thrown.
      * In other words, call waitForResult() only after calling fmg().
      * <p>
      * This method can be called only once for every invocation of fmg().
      *
      * @return the solution of the PDE
      * @exception FMGNotExecutingException method fmg() has not been called before and hence the FMG algorithm is not
      *                                     executing or has not just delivered a result
      */
     public BoundaryGrid waitForResult() throws FMGNotExecutingException {
          if (defaultObserver == null) throw new FMGNotExecutingException();
          BoundaryGrid result = defaultObserver.getResult(); // blocks until result is available
          deleteObserver(defaultObserver);
          defaultObserver = null;
          return result;
     }

     /**
      * calculate the level of a grid in the FMG algorithm from its size such that size = 2^level + 1.
      *
      * @param size the size of the grid including its boundary ( > 0 and of the form 2^level + 1)
      * @return the level corresponding to the given size such that size = 2^level + 1
      */
     public static int levelFromSize(int size) {
          Contract.pre(size > 0,"size > 0");
          
          int s     = size - 1;
          int level = 0;
          while ((s /= 2) >= 1) level++;
          
          Contract.pre(sizeFromLevel(level) == size,"size must be of the form 2^level + 1");
          
          return level;
     }

     /**
      * calculate the size of a grid at the given level of the FMG algorithm such that size = 2^level + 1.
      *
      * @param level the level of the grid in the FMG algorithm ( > 0)
      * @return the size of the grid such that size = 2^level + 1
      */
     public static int sizeFromLevel(int level) {
          Contract.pre(level > 0,"level > 0");
          
          return ((1<<level) + 1);
     }
     
     /**
      * given the index in one dimension of a grid element on a grid calculate the index of the corresponding grid
      * element on the next finer grid.
      *
      * @param index the index of a grid element in one arbitrary dimension on a grid of a certain level ( >= 0)
      * @return the index of the corresponding grid element on a grid of the next finer (higher) level
      */
     public static int fineIndexFromCoarseIndex(int index) throws IllegalArgumentException {
          Contract.pre(index >= 0,"index >= 0");
          
          return (2*index);
     }

     /**
      * the body fo the FMG Algorithm that executes in its own thread.
      * <p>
      * This method is called from FMGThread.run() which in turn is indirectly called from FMG.fmg().
      * <p>
      * The parameters of this method are identical to the parameters of method fmg(). The result of this method (which
      * is the result of the FMG algorithm and hence a grid of type BoundaryGrid) is returned via the notification
      * mechanism to all observers that have been registered with this object.
      *
      * @see FMG#fmg
      */
     protected void fmgProper(int coarsestLevel, int finestLevel, int numPresmooth, int numPostsmooth,
                              int cyclingStrategy, int numMultiGrid) {
          // don't check preconditions here, because they were already checked in fmg()
          
          ConstBoundaryGrid u       = null;
          ConstBoundaryGrid uCoarse = null;
          for (int level = coarsestLevel; level <= finestLevel; level++) {
               int size              = sizeFromLevel(level);
               ConstNoBoundaryGrid f = pde.sampleRHS(size);
               if (level == coarsestLevel) {
                    // compute the exact solution on coarsest level
                    u = solver.solve((BoundaryGrid) uPrototype.newInstance(size,0),f);
               } else {
                    u = interpolator.interpolate(uCoarse);
                    for (int i = 1; i <= numMultiGrid; i++) {
                         u = mg(u,f,coarsestLevel,numPresmooth,numPostsmooth,cyclingStrategy);
                    }
               }
               uCoarse = u;
          }
          
          // notify Observers of the availability of the result by sending it to them
          setChanged();
          notifyObservers(u);
          fmgIsExecuting = false;
     }
     
     /**
      * performs the Multigrid algorithm which is at the heart of the FMG algorithm.
      * <p>
      * Given an approximate solution of the PDE on a grid the Multigrid algorithm finds a better solution on that
      * grid by pre-smoothing, going to the next coarser grid and approximately solving there, going back to the
      * original grid, and finally post-smoothing.
      *
      * @param u               the approximate solution an a grid at a certain level
      * @param f               the right hand side of the PDE at a grid at the same level as u
      * @param coarsestLevel   the coarsest level to which the Multigrid algorithm should recursively go
      *                        (0 < coarsestLevel <= level of u and f)
      * @param numPresmooth    the number of pre-smoothing steps to perform ( > 0)
      * @param numPostsmooth   the number of post-smoothing steps to perform ( >= 0)
      * @param cyclingStrategy the cycling strategy to use ( > 0). A value of 1 results in V-cycles, a value of 2 in 
      *                        W-cycles. This gives the number of times the MG algorithm recursively calls itself to
      *                        solve the coarse grid problem.
      * @return an improved approximate solution on a grid of the same size as u
      */
     protected BoundaryGrid mg(ConstBoundaryGrid u, ConstNoBoundaryGrid f, int coarsestLevel, int numPresmooth, 
                               int numPostsmooth, int cyclingStrategy) {
          Contract.pre(u != null && f != null,"all objects not null-objects");
          Contract.pre(u.size() == f.size(),"u and f of same size");
          
          int level = levelFromSize(u.size());
          
          Contract.pre(coarsestLevel > 0 && coarsestLevel <= level,"0 < coarsestLevel <= level of u and f");
          Contract.pre(numPresmooth > 0,"numPresmooth > 0");
          Contract.pre(numPostsmooth >= 0,"numPostsmooth >= 0");
          Contract.pre(cyclingStrategy > 0,"cyclingStrategy > 0");

          if (level == coarsestLevel) {
               // solve the coarse grid equation
               u = solver.solve(u,f);
          } else {
               // find a better solution at this level by using an approx. solution from the next higher level
               for (int i = 1; i <= numPresmooth; i++) {
                    u = smoother.smooth(u,f);
               }
               ConstNoBoundaryGrid rCoarse = (NoBoundaryGrid) restrictor.restrict(calcResidual(u,f));
               ConstBoundaryGrid   vCoarse = (BoundaryGrid) vPrototype.newInstance(sizeFromLevel(level - 1),0);
               for (int cycle = 1; cycle <= cyclingStrategy; cycle++) {
                    vCoarse = mg(vCoarse,rCoarse,coarsestLevel,numPresmooth,numPostsmooth,cyclingStrategy);
               }
               rCoarse = null;
               u = (BoundaryGrid) u.add(interpolator.interpolate(vCoarse));
               vCoarse = null;
               for (int i = 1; i <= numPostsmooth; i++) {
                    u = smoother.smooth(u,f);
               }
          }
          return ((BoundaryGrid) u);
     }
     
     /**
      * calculate the residual of an approximate solution to a PDE.
      * <p>
      * If Ax = f is the PDE and u is an approximation to the true solution x, then the residual r is defined as
      * r = f - Au and is hence a 3-dimensional function sampled on a regular cubic grid without a boundary (because
      * f is in our scheme without a boundary, too).
      *
      * @param u the approximate solution to the PDE whose residual is to be calculated
      * @param f the right hand side (source term) of the PDE
      * @return the residual of the given approximate solution on a grid of the size in question
      */
     protected NoBoundaryGrid calcResidual(ConstBoundaryGrid u, ConstNoBoundaryGrid f) {
          Contract.pre(u != null && f != null,"all objects not null-objects");
          
          int size = u.size();
          
          Contract.pre(f.size() == size,"size of u equal size of f");

          NoBoundaryGrid r = (NoBoundaryGrid) f.newInstance(size,0);
          new CalcResidualParallelizer(this,r,u,f);
          return r;
     }
     
     /**
      * implements the parallel part of calcResidual().
      */
     void calcResidualProper(NoBoundaryGrid r, ConstBoundaryGrid u, ConstNoBoundaryGrid f, int myNum, int totalNum) {
          int size = u.size();
          IntRange1D range = Parallelizer.partition(new IntRange1D(1,size - 2),myNum,totalNum);
          for (int x = range.from(); x <= range.to(); x++) {
               for (int y = 1; y <= (size - 2); y++) {
                    for (int z = 1; z <= (size - 2); z++) {
                         r.set(x,y,z,f.get(x,y,z) - pde.evaluateLHS(u,x,y,z));
                    }
               }
          }
     }

     private Smoother           smoother;
     private Restrictor         restrictor;
     private Interpolator       interpolator;
     private Solver             solver;
     private PDE                pde;
     private ConstBoundaryGrid  uPrototype;
     private ConstBoundaryGrid  vPrototype;
     private boolean            fmgIsExecuting = false;
     private FMGDefaultObserver defaultObserver = null;
}
