/*
 * $Source: /a/thud/chalk/homes/moore/src/hence/master/RCS/cost.c,v $
 * $Revision: 1.1 $
 * $Date: 1992/04/08 05:51:45 $
 * $Author: moore $
 */

/* Cost.c defines all routines involved with reading and using the cost 
 * matrix. The cost matrix is maintained in G in the two rb_trees 
 * G->machines and G->subroutines. G->machines is keyed by machine name, 
 * and points to a struct containing the machine's name, and its current 
 * load (all loads start at 0).  G->subroutines is keyed by subroutine 
 * name, and points to a struct containing the subroutine's name, and a 
 * rb_tree of costs. This tree is keyed on the machine name of 
 * cost/machine pairs, and points to a struct containing a pointer to the
 * machine's struct in G->machine, as well as the cost for that sub on 
 * that machine, and a count of times that the machine has been used for 
 * that subroutine. 
 * 
 * That's basically it -- once you know whta those structs do, 
 * initializing them, getting a machine for a subroutine, logging the 
 * subroutine as over, and printing out the cost matrix are all 
 * straightforward. 
 */

#include <stdio.h>
#include "std.h"
#include "rb.h"
#include "dlist.h"
#include "hence.h"

/* ---------------------------------------------------------------------- */
/* The structs */

typedef struct {
  char *name;			/* Machine name */
  int load;			/* Hence load on the machine */
} machine;
typedef machine *Machine;

typedef struct {
  char *name;			/* Subroutine name */
  Rb_node costs;		/* Rb_tree: key = machine name, val = Cost */
} sub;
typedef sub *Sub;
  
typedef struct {
  Machine m;
  int c;
  int num;
} cost;
typedef cost *Cost;

/* ---------------------------------------------------------------------- */
/* Red-black junk -- this is for type-safety & lint */

static Rb_node rb_insert_b_sub(r, s)
Rb_node r;
Sub s;
{
  return rb_insert_b(r, s->name, (char *) s);
}

static Rb_node rb_insert_b_mach(r, m)
Rb_node r;
Machine m;
{
  return rb_insert_b(r, m->name, (char *) m);
}

static Rb_node rb_insert_b_cst(r, s, c)
Rb_node r;
char *s;
Cost c;
{
  return rb_insert_b(r, s, (char *) c);
}

/* ---------------------------------------------------------------------- */
/* Read_cm reads a cost-matrix from a file.  It doesn't ignore comments
 * yet.  It should */

read_cm(G)
Graph G;
{
  FILE *f;
  int nm, c;
  char mname[256], sname[256];
  Rb_node rm, rs;
  Machine m;
  Sub s;
  Cost cst;
  int fnd;

  /* Open the cost matrix */

  if (G->cost_mat == CNULL) G->cost_mat = "cost.mat";

  f = fopen(G->cost_mat, "r");
  if (f == NULL) {
    fprintf(stderr, "ERROR: Can't open Cost matrix %s\n", G->cost_mat);
    bail(CNULL);
  }

  for (;;) {

    /* Read a line in the cost matrix: <mach-name> <sub-name> <cost> */

    nm = fscanf(f, "%s %s %d\n", mname, sname, &c);
    if (nm == EOF) {
      return;
    } else if (nm != 3) {
      fprintf(stderr, "ERROR in reading cost matrix.\n");
      fprintf(stderr, "      Format is <machine-name> <subroutine-name>");
      fprintf(stderr, " <positive-integer-cost>\n");
      bail(CNULL);
    }

    /* Find (or create) the machine in G->machines. */ 

    rm = rb_find_key_n(G->machines, mname, &fnd);
    if (!fnd) {
      m = talloc(machine, 1);
      m->name = copy_string(mname);
      m->load = 0;
      (void) rb_insert_b_mach(rm, m);
    } else {
      m = (Machine) rm->v.val;
    }

    /* Find (or create) the subroutine in G->subroutines */

    rs = rb_find_key_n(G->subroutines, sname, &fnd);
    if (!fnd) {
      s = talloc(sub, 1);
      s->name = copy_string(sname);
      s->costs = make_rb();
      (void) rb_insert_b_sub(rs, s);
    } else {
      s = (Sub) rs->v.val;
    }

    /* Insert the machine and cost into the subroutine struct */

    rm = rb_find_key_n(s->costs, mname, &fnd);
    if (!fnd) {
      cst = talloc(cost, 1);
      cst->m = m;
      cst->c = c;
      cst->num = 0;
      (void) rb_insert_b_cst(rm, m->name, cst);
    } else {
      fprintf(stderr, "ERROR in cost matrix: <%s %s> pair defined twice\n",
              mname, sname);
      bail(CNULL);
    }

  }
}

/* ---------------------------------------------------------------------- */
/* Print_costs prints out the contents of G->subroutines */

print_costs(G)
Graph G;
{
  Rb_node rs, rc;
  Cost c;
  Sub s;

  rb_traverse(rs, G->subroutines) {
    s = (Sub) rs->v.val;
    fprintf(stdout, "Subroutine: %s\n", s->name);
    rb_traverse(rc, s->costs) {
      c = (Cost) rc->v.val;
      if (c->num != 0) {
        fprintf(stdout, "  Machine: %s, Cost: %d   Run %d times\n", 
               c->m->name, c->c, c->num);
      }
    }
    fprintf(stdout, "\n");
  }
}

/* ---------------------------------------------------------------------- */
/* Get machine returns the machine which will be the least loaded down
 * after running subr.  It returns CNULL if there is a problem and no
 * such machine can be found */

char *get_machine(subr, G)
char *subr;
Graph G;
{
  Rb_node r;
  Sub s;
  Cost c, minc;
  int fnd, minload, ld;

  /* Get the subroutine struct */

  r = rb_find_key_n(G->subroutines, subr, &fnd);
  if (!fnd) return CNULL;
  s = (Sub) r->v.val;
  
  if (rb_empty(s->costs)) return CNULL;

#if 1
  minload = 0x7fffffff;
  minc = NULL;

  rb_traverse (r, s->costs) {
      c = (Cost) r->v.val;
      if (c && c->m) {
	  ld = c->m->load + c->c;
	  if (ld < minload) {
	      minc = c;
	      minload = ld;
	  }
      }
  }

  if (minc == NULL) {
      fprintf (stderr, ">>> can't find a node to run %s\n", subr);
      return CNULL;
  }
      
#else
  r = rb_first(s->costs);
  c = (Cost) r->v.val;
  minc = c;
  minload = c->m->load + c->c;

  /* Find the minimum load */

  rb_traverse(r, s->costs) {
    c = (Cost) r->v.val;
    ld = c->m->load + c->c;
    if (ld < minload) {
      minc = c;
      minload = ld;
    }
  }
#endif
  
  /* Set the new load value, and return the machine */

  minc->m->load = minload;
  minc->num++;
  return minc->m->name;
}

/* ---------------------------------------------------------------------- */
/* Subroutine_over is called when a subroutine is finished -- it removes 
 * the load induced by subr on the machine. */ 

subroutine_over(subr, mach, G)
char *subr;
char *mach;
Graph G;
{
  Rb_node r;
  Sub s;
  Cost c;
  int fnd;

  r = rb_find_key_n(G->subroutines, subr, &fnd);
  if (!fnd) return;
  s = (Sub) r->v.val;

  if (rb_empty(s->costs)) return;

  r = rb_find_key_n(s->costs, mach, &fnd);
  if (!fnd) return;
  c = (Cost)r->v.val;
  c->m->load -= c->c;
}
  
