/*
 * Functions for building sub.defs files and other things that grok
 * function-call information from the HeNCE graph.
 *
 * XXX eventually, just make the wrappers directly, instead of building
 * sub.defs and calling mkwrap.
 */

#ifndef FuncPr
#if defined(__STDC__)
#define FuncPr 1
#else
#define FuncPr 0
#endif
#endif

#include <stdio.h>
#include "rb.h"
#include "param.h"
#include "exp.h"
#include "graph.h"
#include "graphP.h"
#include "misc.h"

/*
 * utility function used to compare two string keys,
 * passed to rb_Find() and rb_Insert() routines for symbol table
 * lookup.
 */

static int
nameCompare (k1, k2)
Key *k1, *k2;
{
    return (strcmp ((char *) k1->pkey, (char *) k2->pkey));
}

/****************************************************************************
 * routines to manipulate parameter symbol table:
 *
 * since there can be more than one parameter with the same name, the
 * symbol table entry is actually a stack with the most recent binding
 * to that name on top.
 *
 * the symbol table is built from the top of the tree down through
 * each possible path to each node.
 ****************************************************************************/

static Tree paramSymTable;

struct paramSym {
    Param p;
    struct paramSym next;
};

/*
 * declare a parameter
 *
 * XXX should this push a new stack entry if a parameter by that
 * name already exists?
 */

void
declareParameter (p)
Param p;
{
    Key k;
    int found;
    TreeNode tn;
    struct paramSym *ps;

    ps = talloc (1, struct paramSym);
    ps->p = p;
    ps->next = NULL;

    k.pkey = (void *) p->name;
    tn = rb_Find (paramSymTable, &k, nameCompare, &found);
    if (found) {
	ps->next = (struct paramSym *) rb_Value (tn);
	rb_SetValue (tn, ps);
    }
    else {
	rb_InsertBefore (tn, &k, (void *) ps);
    }
}

/*
 * remove a parameter's (most recent) declaration
 */

void
undeclareParameter (name)
char *name;
{
    Key k;
    TreeNode tn;
    int found = 0;
    struct paramSym *ps;

    k.pkey = (void *) name;
    tn = rb_Find (paramSymTable, &k, nameCompare, &found);
    if (found) {

	/*
	 * if there are multiple instances of this parameter,
	 * pop the most recently declared one.  Otherwise,
	 * delete the symbol table entry.
	 */
	ps = rb_Value (tn);

	if (ps->next) {
	    rb_SetValue (tn, ps->next);
	    ps->next = NULL;
	}
	else
	    rb_DeleteNode (tn);

	free (ps);
    }
}

/*
 * find the most recent declaration of a parameter
 */

struct paramSym *
lookupParameter (name)
char *name;
{
    TreeNode tn;
    Key k;
    int found = 0;

    k.pkey = name;
    tn = rb_Find (paramSymTable, &k, nameCompare, &found);
    if (found)
	return (struct paramSym *) rb_Value (tn);
    return NULL;
}


/****************************************************************************
 * routines to manipulate the function symbol table.
 * unlike parameters, there can be only one function with a particular
 * name in a HeNCE program.
 ****************************************************************************/

static Tree funcSymTable;

struct funcSym {
    char *name;			/* function name */
    int retParamType;		/* type of returned parameter */
    int nargs;			/* number of arguments */
    int *argParamTypes;		/* type of each argument */
#define ARRAY_FLAG 0x80		/* if array, this bit is ORed in */
};

/*
 * Given a node structure, allocate and build a function symbol table entry
 */

struct funcSym *
buildFunctionDecl (n)
Node n;
{
    struct funcSym *f;

    f = talloc (1, struct funcSym);
    f->name = strsave (n->sub_name);
    f->retParamType = n->ret_param ? n->ret_param->type : TYPE_UNSPECIFIED;
    f->nargs = n->nargs;
    f->argParamTypes = talloc (n->nargs, int);
    for (i = 0; i < n->nargs; ++i) {
	if (n->nargs[i]->type == TYPE_ARRAY)
	    f->argParamTypes[i] = ARRAY_FLAG | n->nargs[i]->a->type;
	else
	    f->argParamTypes[i] = n->nargs[i]->type;
    }
    return f;
}

/*
 * free a function symbol table entry that we don't need anymore
 */

void
freeFuncSym (fs)
struct funcSym *fs;
{
    if (fs) {
	if (fs->argParamTypes) {
	    free (fs->argParamTypes);
	    fs->argParamTypes = NULL;
	}
	if (fs->name) {
	    free (fs->name);
	    fs->name = NULL;
	}
	free (fs);
    }
}

/*
 * write a sub.defs-style function prototype
 */

void
writeFunctionDecl (fp, fs)
FILE *fp;
struct funcSym *fs;
{
    if (fs->retParamType == TYPE_UNSPECIFIED)
	fprintf (fp, "void ");
    else
	fprintf (fp, "%s ", paramTypeName (fs->retParamType));
    fprintf (fp, "%s(", fs->name);
    for (i = 0; i < fs->nargs; ++fs) {
	fprintf (fp, "%s%s%s",
		 paramTypeName (fs->argParamTypes[i] & ~ARRAY_FLAG);
		 fs->argParamTypes[i] & ARRAY_FLAG ? " *" : "",
		 i == fs->nargs - 1 ? ");\n" : ", ");
    }
}

/*
 * add a function to the function symbol table
 */

void
declareFunction (fs)
struct funcSym *fs;
{
    Key k;

    k.pkey = fs->name;
    rb_Insert (funcSymTable, &k, (void *) fs, nameCompare);
}

/*
 * remove a function from the function symbol table, and
 * free the storage associated with it.
 */

void
undeclareFunction (name)
char *name;
{
    Key k;
    TreeNode tn;
    int found = 0;

    k.pkey = (void *) name;
    tn = rb_Find (funcSymTable, &k, nameCompare, &found);
    if (found) {
	struct funcSym *f = rb_Value (tn);
	freeFuncSym (f);
	rb_DeleteNode (tn);
    }
}

/*
 * find a function in the function symbol table
 */

struct funcSym *
lookupFunction (name)
char *name;
{
    TreeNode tn;
    Key k;
    int found = 0;

    k.pkey = name;
    tn = rb_Find (funcSymTable, &k, nameCompare, &found);
    if (found)
	return (struct funcSym *) rb_Value (tn);
    return NULL;
}


/*
 * given two entries in the function symbol table, see if they're
 * the same
 */

int
compareFunctions (f1, f2)
struct funcSym *f1, *f2;
{
    int x, i;

    if (f1 == NULL || f2 == NULL)
	return -1;
    if (f1 == f2)
	return 0;

    if ((x = strcmp (f1->name, f2->name)) != 0)
	return x;
    if ((x = (f1->retParamType - f2->retParamType)) != 0)
	return x;
    if ((x = (f1->nargs - f2->nargs)) != 0)
	return x;
    for (i = 0; i < f1->nargs; ++i)
	if ((x = (f1->argParamTypes[i] - f2->argParamTypes[i])) != 0)
	    return x;
    return 0;
}

/*
 * traverse a HeNCE subgraph.  Make sure parameter and function declarations
 * are consistent, and collect definitions for functions.
 */

static int
scanSubGraph (n)
Node n;
{
    TreeNode tn;
    int errors = 0;


    /*
     * special case for non-normal nodes, since we don't have to check
     * parameters or function names (is this true?  how about loop vars?)
     */

    if (n->type != NODE_NORMAL) {
	TreeNode tn;

	if (rb_Empty (n->children))
	    return 0;
	for (tn = rb_First (n->children); tn != n->children;
	     tn = rb_Next (tn)){
	    errors += scanSubGraph ((Node) rb_Value (tn));
	}
	return errors;
    }

    /*
     * 1.  for each named parameter:
     * if it's new, declare it; else, find its declaration
     */
    for (tn = rb_First (n->params); tn != n->params; tn = rb_Next (tn)) {
	struct paramSym *ps;
	Param p0, p1;

	p1 = rb_Value (tn);
	if (p1->io.new) {
	    /*
	     * new parameter.  ignore any previous declarations, and
	     * push this declaration on the stack
	     */
	    p1->io.used = 1;
	    p0 = p1;
	    p1->nodeWhereDeclared = n;
	    declareParameter (p1);
	}
	else {
	    /*
	     * not declared with NEW keyword; find in symbol table
	     */
	    ps = lookupParamater (p1->name);
	    p0 = ps ? ps->p : (Param) NULL;

	    /*
	     * Not declared as new, and no previous declaration
	     */
	    if (p0 == (Param) NULL) {
		msg_Format ("Node %d: no NEW declaration for parameter %s.\n",
			    n->nk.id, p1->name);
		++errors;
		continue;
	    }		
	    /*
	     * (simple) check for type mismatch
	     *
	     * XXX should check for equivalence of array types
	     */
	    else if (p1->type != TYPE_INHERITED && p1->type != p0->type) {
		msg_Format ("Node %d: type mismatch for parameter %s.\n",
			    n->nk.id, p1->name);
		msg_Format ("(originally declared @ node %d)\n",
			    p0->nodeWhereDeclared->nk.id);	
		++errors;
		continue;
	    }
	    /*
	     * if we've been here before, check to make sure that there
	     * isn't a declaration through this path of ancestors that
	     * conflicts with another declaration through another path.
	     */
	    else if (p1->io.used && p1 != p0) {
		msg_Format ("Node %d: multiple declarations for %s %s\n",
			    n->nk.id, "parameter", p1->name);
		msg_Format ("(originally declared @ node %d)\n",
			    p0->nodeWhereDeclared->nk.id);
		++errors;
		continue;
	    }
	    /*
	     * okay.  we've found the declaration for this parameter.
	     * replace this parameter with its original declaration.
	     */
	    if (p0 != p1) {
		rb_SetValue (tn, (void *) p0);
		/* free (p1); */
	    }
	}
    }

    /*
     * 2. Now we should know the type of each param.  Build a descriptor
     * for this function call.  If there are no previously declared
     * instances of this function call, declare one; else, check
     * to make sure this use of the function matches all others.
     */
    sc1 = buildFuncDecl (n);
    sc0 = lookupFunction (n->sub_name);
    if (sc0 == NULL) {
	/* no previous declaration */
	declareFunction (sc1);
    }
    else if (compareFunctions (sc0, sc1) != 0) {
	msg_Format ("Node %d: use of function %s does not match\n",
		    n->nk.id, n->sub_name);
	msg_Format ("use of that function in node %d\n",
		    sc0->node_id);
	++errors;
	free (sc1);
    }
    else {
	/* the identical function was already declared */
	free (sc1);
    }

    /*
     * 3. scan all of our children looking for similar declaration conflicts
     */
    for (tn = rb_First (n->children); tn != n->children; tn = rb_Next (tn))
	errors += scanSubGraph ((Node) rb_Value (tn));

    /*
     * 4. for any parameters that we declared here, undeclare them now
     */

    for (tn = rb_First (n->params); tn != n->params; tn = rb_Next (tn)) {
	Param p1;

	p1 = rb_Value (tn);
	if (p1->nodeWhereDeclared == n)
	    undeclareParameter (p1);
    }
    return errors;
}

/*
 * do any initialization that needs to be done before collecting
 * parameters and function names
 */

static int
zapNodes (n)
Node n;
{
    n->flags &= ~NF_BEEN_HERE;
    return 0;
}

/*
 * do any cleanup that needs to be done after collecting parameters
 * and function names
 */

static int
unzapNodes (n)
Node n;
{
    n->flags &= ~NF_BEEN_HERE;
    return 0;
}

/*
 * Given a graph, generate a sub.defs file.  Return zero if successful,
 * nonzero (error count) on errors.
 */

int
buildSubDefs (g, fp)
Graph g;
FILE *fp;
{
    TreeNode tn;
    int errors = 0;

    paramSymTable = rb_MakeTree ();
    funcSymTable = rb_MakeTree ();

    if (rb_Empty (g->heads) || rb_Empty (g->nlist)) {
	msg_Format ("graph is empty\n");
	return 1;
    }

    /* initialize all nodes as necessary */
    for (tn = rb_First (g->nlist); tn != g->nlist; tn = rb_Next (tn))
	zapNodes ((Node) rb_Value (tn));

    for (tn = rb_First (g->heads); tn != g->heads; tn = rb_Next (tn))
	errors += scanSubGraph ((Node) rb_Value (tn));

    /* cleanup all nodes as necessary */
    for (tn = rb_First (g->nlist); tn != g->nlist; tn = rb_Next (tn))
	unzapNodes ((Node) rb_Value (tn));

    /*
     * if there are no errors, write out a sub.defs file
     */

    if (errors == 0) {
	for (tn = rb_First (funcSymTable); tn != funcSymTable;
	     tn = rb_Next (tn))
	    writeFunctionDecl (fp, (struct funcSym *) rb_Value (tn));
    }

    /* XXX free up paramSymTable */
    /* XXX free up funcSymTable */

    return errors;
}
