/*
 * functions to
 * (a) parse the function definitions file
 * (b) check the functions in the HeNCE graph against the declarations
 *     in that file.
 * (c) write out a func.defs file to be fed to mkwrap.
 *
 * XXX eventually, fold the code from mkwrap into this module, and build
 * a version of mkwrap that shares code with htool for standalone (non-X)
 * usage of HeNCE.
 */

%{
#define TYPECHECKING

#include <stdio.h>
#include <errno.h>
#include "rb.h"
#include "param.h"
#include "exp.h"
#include "expP.h"
#include "graph.h"
#include "graphP.h"
#include "misc.h"

#ifndef errno
extern int errno;		/* 4.3bsd needs this */
#endif

/****************************************************************************
 * routines to manipulate the function symbol table.
 * unlike parameters, there can be only one function with a particular
 * name in a HeNCE program.
 ****************************************************************************/

static Tree funcSymTable = 0;

struct funcSym {
    char *name;			/* name of function */
    int retType;		/* type of returned parameter */
    int isUsed;			/* true if used in graph */
    int nargs;			/* number of arguments */
    int nodeWhereDeclared;	/* node # where function was declared,
				 * or -1 if declared in sub.defs file */
    struct argType {
	int baseType;		/* base type of each argument */
	int indirs;		/* number of indirections (*'s) */
	/* XXX 200 args max is ugly, but easier to do storage management */
    } argTypes[200];		/* type of each argument */
} currFuncSym;

struct argType currArgTypes[200]; /* XXX should go away */
int currNumArgs;
int errorCount;
char *currFileName;

%}

%start decl_list

%union {
    long integer;
    char *id;
}

%type <id> name
%type <integer> indirs stars type
%token TK_TYPE TK_ID
%%

decl_list:	decl
    	 |	decl_list decl
	 ;

decl	:	type indirs name { currNumArgs = 0; } '(' arglist ')' ';' {
	    	    int i;

		    if ($2 != 0) {
			msg_Format ("%s%s%d: return type must be scalar\n",
				    currFileName, ", line ", yylineno);
			++errorCount;
		    }
		    else {
			currFuncSym.retType = $1;
			currFuncSym.name = $3;
			currFuncSym.nodeWhereDeclared = -1;
			currFuncSym.nargs = currNumArgs;
			for (i = 0; i < currNumArgs; ++i)
			    currFuncSym.argTypes[i] = currArgTypes[i];
			currFuncSym.isUsed = 0;
			addFuncSym (&currFuncSym, currFileName, yylineno);
		    }
		}
	;

name	:	TK_ID	{
    		    char *strsave ();
    		    $$ = strsave (yytext);
		}
	;

arglist	:	/* empty */
    	|	args
	;

args	:	arg
	|	args ',' arg
	;

arg	:	type indirs	{
    		    currArgTypes[currNumArgs].baseType = $1;
		    currArgTypes[currNumArgs].indirs = $2;
		    currNumArgs++;
		}
	;

indirs	:	/* empty */	{
		    $$ = 0;
		}
	|	stars	{
    		    $$ = $1;
		}
	;

stars	:	'*'		{
		    $$ = 1;
		}
	|	stars '*'	{
		    $$ = $1 + 1;
		}
	;

type	:	TK_TYPE	{
		    if (strcmp ("int", yytext) == 0)
			$$ = TYPE_INT;
		    else if (strcmp ("float", yytext) == 0)
			$$ = TYPE_FLOAT;
		    else if (strcmp ("double", yytext) == 0)
			$$ = TYPE_DOUBLE;
		    else if (strcmp ("char", yytext) == 0)
			$$ = TYPE_CHAR;
		    else if (strcmp ("void", yytext) == 0)
			$$ = TYPE_VOID;
		}
	;

%%
#include "funcdefs.lex.out"
/*
 * flex defines a fake yywrap function, which we undef so we can
 * define our own.
 */
#ifdef yywrap
#undef yywrap
#endif

/* helper function used by addFuncSym() and lookupFuncDecl () */

static int
nameCompare (k1, k2)
Key *k1, *k2;
{
    return (strcmp ((char *) k1->pkey, (char *) k2->pkey));
}


/*
 * add a new function declaration to the symbol table.
 * filename and lineno are optional. If filename is NULL, no error
 * message will be printed if the function is already declared.
 */

static void
addFuncSym (fs, filename, lineno)
struct funcSym *fs;
char *filename;
int lineno;
{
    TreeNode tn;
    Key k;
    struct funcSym *p;
    int fnd = 0;

    if (funcSymTable == NULL)
	funcSymTable = rb_MakeTree ();

    k.pkey = fs->name;
    tn = rb_Find (funcSymTable, &k, nameCompare, &fnd);
    if (fnd) {
	if (filename)
	    msg_Format ("%s, line %d: function %s already declared\n",
			filename, lineno, fs->name);
	else
	    msg_Format ("\
Node %d: [htool bug] function %s is already declared\n",
			lineno, fs->name);
	++errorCount;
	return;
    }
    p = talloc (1, struct funcSym);
    *p = *fs;
    k.pkey = (void *) strsave (fs->name);
#if 0
    if (fs->nodeWhereDeclared == -1)
	fprintf (stderr, "%s: defining function %s\n", filename, fs->name);
    else
	fprintf (stderr, "node %d: defining function %s\n",
		 fs->nodeWhereDeclared, fs->name);
#endif
    rb_InsertBefore (tn, &k, (void *) p);
#if 0
    fprintf (stderr, "writing decls\n");
    subdefs_WriteDecls (stderr);
    fprintf (stderr, "finished writing decls\n");
#endif
}

/*
 * given a function's name, find its declaration
 */

static struct funcSym *
lookupFuncDecl (name)
char *name;
{
    TreeNode tn;
    Key k;
    int fnd = 0;

    if (funcSymTable == NULL || name == NULL)
	return NULL;

#ifdef DEBUG
    fprintf (stderr, "lookupFuncDecl (%s)", name);
#endif
    k.pkey = name;
    tn = rb_Find (funcSymTable, &k, nameCompare, &fnd);
    if (!fnd) {
#ifdef DEBUG
	fprintf (stderr, " => (not found)\n");
#endif
	return NULL;
    }
#ifdef DEBUG
    fprintf (stderr, " => (found)\n");
#endif
    return (struct funcSym *) rb_Value (tn);
}

static char *
typeNameX (baseType, indirs)
int baseType, indirs;
{
    static char buf[40];
    char *ptr;

    switch (baseType) {
    case TYPE_PARAM:
	ptr = "[param]";
	break;
    case TYPE_UNSPECIFIED:
	ptr = "[unspecified]";
	break;
    case TYPE_INHERITED:
	ptr = "[inherited]" ;
	break;
    case TYPE_INT:
	ptr = "int";
	break;
    case TYPE_CHAR:
	ptr = "char";
	break;
    case TYPE_FLOAT:
	ptr = "float";
	break;
    case TYPE_DOUBLE:
	ptr = "double";
	break;
    case TYPE_ARRAY:
	ptr = "array";
	break;
    case TYPE_STRING:
	ptr = "string";
	break;
    case TYPE_VOID:
	ptr = "void";
	break;
    default:
	sprintf (buf, "[type = %d, indirs = %d]", baseType, indirs);
	return buf;
	break;
    }
    
    sprintf (buf, "%s%s", ptr, indirs ? " *" : "");
    return buf;
}

/*
 * compare two function declarations.  If they differ, print out a message
 * that indicates how they are inconsistent.  Return 0 if the two are the
 * same, or the error count if they are somehow different.
 *
 * "n" is the node at which the function "f2" is being declared.
 * If non-NULL, "filename" is the name of the sub.defs file.
 */

static int
compareFunctions (n, f1, f2, filename)
Node n;
struct funcSym *f1, *f2;
char *filename;
{
    int i;
    int nodenum = n->nk.id;
    int errcount = 0;

    if (f1 == NULL || f2 == NULL)
	return -1;
    if (f1 == f2)
	return 0;

    if (strcmp (f1->name, f2->name) != 0) {
	msg_Format ("node %d: function %s != declared function %s\n",
		    nodenum, f1->name, f2->name);
	return 1;
    }
#ifdef TYPECHECKING
    if (f1->retType != f2->retType) {
	if (f1->nodeWhereDeclared == -1)
	    msg_Format ("\
Node %d: result type of function %s [%s] does not match declaration in %s\n",
			nodenum, f1->name, typeNameX (f2->retType, 0),
			filename);
	else
	    msg_Format ("\
Node %d: result type of function %s [%s] does not match use of %s at node %d\n",
			nodenum, f1->name, typeNameX (f2->retType, 0),
			f1->name, f1->nodeWhereDeclared);
	return 1;
    }
#endif
    if (f1->nargs != f2->nargs) {
	if (f1->nodeWhereDeclared == -1)
	    msg_Format ("\
Node %d: function %s: arg count does not match declaration in \"%s\"\n",
			nodenum, f1->name, filename);
	else
	    msg_Format ("\
Node %d: function %s: arg count does not match use of \"%s\" at node %d\n",
			nodenum, f1->name, f1->name, f1->nodeWhereDeclared);
	return 1;
    }
#ifdef TYPECHECKING
    for (i = 0; i < f1->nargs; ++i) {
	if ((f1->argTypes[i].baseType != f2->argTypes[i].baseType) ||
	    (f1->argTypes[i].indirs != f2->argTypes[i].indirs)) {
	    if (f1->nodeWhereDeclared == -1)
		msg_Format ("\
Node %d: type of arg #%d (%s) does not match declaration in \"%s\"\n",
			    nodenum, i + 1,
			    typeNameX (f2->argTypes[i].baseType,
				       f2->argTypes[i].indirs),
			    filename);
	    else {
		char argtype1[40], argtype2[40]; /* XXX yuk! */

		strcpy (argtype1, typeNameX (f1->argTypes[i].baseType,
					     f1->argTypes[i].indirs));
		strcpy (argtype2, typeNameX (f2->argTypes[i].baseType,
					     f2->argTypes[i].indirs));
		msg_Format ("\
Node %d: type of arg #%d (%s) does not match type used at node %d (%s)\n",
			    nodenum, i + 1, argtype2,
			    f1->nodeWhereDeclared, argtype1);
#if 0
		subdefs_WriteDecls (stderr);
#endif
	    }
	    ++errcount;
	}
    }
#endif
    return errcount;
}

/*
 * given a parameter p, return its type
 *
 * XXX this should be in param.c, but it can't be because param.c
 * doesn't know about htool's parameter symbol tables.
 */

int
param_Type (p, nodenum)
Param p;
int nodenum;			/* for error messages */
{
    Param findParamDecl();
    int type;

    if (p == NULL)
	type = TYPE_INHERITED;
    else if (p->type == TYPE_INHERITED) {
	type = param_Type (findParamDecl (p, nodenum), nodenum);
    }
    else if (p->type == TYPE_ARRAY) {
	if (p->a->type == TYPE_INHERITED) {
	    /*
	     * for inherited parameters that appear with a dimension list,
	     * find the most recent declaration for that parameter.  If
	     * the number of dimensions of this use is < the number of
	     * dimensions in the declaration, this is an array.  If the
	     * number of dimensions is equal to the number of dims in the
	     * declaration, then this is a scalar.  Otherwise it's an error.
	     */
	    Param p0 = findParamDecl (p, nodenum);
	    if (p0 == NULL)
		type = TYPE_INHERITED;
	    else {
		int i;
		int dim_count = 0;

		/*
		 * count the number of dimensions that are a scalar.
		 * if the dimension is an array section, it doesn't
		 * count.  Same for an empty '[]' - this is equivalent
		 * to a array section that goes all the way through
		 * an array in that dimension.
		 *
		 * XXX get this out of here and into exp.c
		 * XXX replace all of these ridiculous data structures
		 *     with something more reasonable
		 * XXX come up with some better terminology to describe
		 *     what's going on
		 * XXX ensure world peace
		 */

		for (i = 0; i < p->a->ndims; ++i)
		    switch (p->a->dims[i]->elt_type) {
		    case OP_DIM:
			dim_count++;
			break;
		    case OP_ARR_SECTION:
		    case OP_EMPTY_DIM:
			break;
		    default:
			break;
		    }
#ifdef DEBUG
		fprintf (stderr,
			 "param_Type(%s) p->a->ndims=%d, p0->a->ndims=%d\n",
			 p->name, p->a->ndims, p0->a->ndims);
#endif
		type = MAKE_EXPR_TYPE(p0->a->type, p0->a->ndims - dim_count);
	    }
	}
	else {
	    /* XXX does this ever happen? */
	    type = MAKE_EXPR_TYPE(p->a->type, p->a->ndims);
	}
    }
    else
	type = p->type;

#ifdef DEBUG
    fprintf (stderr, "param_Type (%s) => %d [%s]\n",
	     p ? p->name : "NULL",
	     type,  typeNameX(EXPR_BASE_TYPE(type), EXPR_NUM_DIMS(type)));
#endif
    return (type);
}

/*
 * Return the type of the result that you would get if you evaluated
 * expr e.  This is used to do crude type-checking before generating
 * wrappers.
 *
 * The prefix '*' and '&' probably aren't handled correctly, but these
 * should be nuked from the whole HeNCE language.
 *
 * XXX somewhere (perhaps in parse.y) we ought to check to make sure
 * that we aren't doing bogus operations (e.g. left shift on double,
 * arithmetic on strings)
 *
 * XXX this should be in exp.c
 */


int
expr_Type (e, nodenum)
Expr e;
int nodenum;			/* for error messages */
{
    int promote ();
    int x;

    switch (e->elt_type) {
    case OP_COND_EXPR:
	x = promote (e->args[1], e->args[2], nodenum);
	break;
    case OP_LSHIFT:
    case OP_RSHIFT:
    case OP_NEG:
	x = expr_Type (e->args[0], nodenum);
	break;
    case OP_LOGICAL_OR:
    case OP_LOGICAL_AND:
    case OP_LOGICAL_CMPL:
    case OP_COMP_NE:
    case OP_COMP_EQ:
    case OP_COMP_LE:
    case OP_COMP_GE:
    case OP_COMP_LT:
    case OP_COMP_GT:
	x = TYPE_INT;
	break;
    case OP_BIT_OR:
    case OP_BIT_AND:
    case OP_BIT_XOR:
    case OP_BIT_CMPL:
    case OP_MULT:
    case OP_DIV:
    case OP_MOD:
    case OP_ADD:
    case OP_SUBTRACT:
	x = promote (e->args[0], e->args[1], nodenum);
	break;
    case OP_CONST:
	x = e->type;
	break;
    case OP_PARAM:
	x = param_Type (e->val.p, nodenum);
	break;
    case OP_DIM:
    case OP_INDIR:		/* XXX this op should go away */
	x = expr_Type (e->args[0], nodenum);
	if (EXPR_IS_ARRAY(x))
	    x = MAKE_EXPR_TYPE(EXPR_BASE_TYPE(x),EXPR_NUM_DIMS(x)-1);
#if 0
	fprintf (stderr, "OP_DIM input type = %d, result = %d\n",
		 expr_Type (e->args[0], nodenum), x);
#endif
	break;
    case OP_EMPTY_DIM:
    case OP_ARR_SECTION:
	return expr_Type (e->args[0], nodenum);
    case OP_ADDRESS:		/* XXX this operator should go away */
	x = expr_Type (e->args[0], nodenum);
	if (EXPR_IS_ARRAY(x))
	    ;
	else
	    x = MAKE_EXPR_TYPE(EXPR_BASE_TYPE(x),1);
	break;
    default:
#ifdef DEBUG
	fprintf (stderr, "expr_Type (%x op = %c) => unknown type %d\n",
		 e, e->elt_type, x);
#endif
	return TYPE_UNSPECIFIED;
    }
#ifdef DEBUG
    fprintf (stderr, "expr_Type (%x op = %c) => %d\n", e, e->elt_type, x);
#endif
    return x;
}


int
promote (e1, e2, nodenum)
Expr e1, e2;
int nodenum;			/* for error messages */
{
    int t1, t2;

    t1 = expr_Type (e1, nodenum);
    t2 = expr_Type (e2, nodenum);
    if (t1 == t2)
	return t1;
    if (t1 > t2)
	return t1;
    return t2;
}

static struct funcSym *
buildFuncDecl (n)
Node n;
{
    struct funcSym *f;
    int i;

    f = talloc (1, struct funcSym);
    f->name = n->sub_name;
    f->nargs = n->nargs;
    f->nodeWhereDeclared = n->nk.id;
    f->isUsed = 1;

    if (n->ret_param) {
#if 1
	f->retType = param_Type (n->ret_param, n->nk.id);
#else
	if (n->ret_param->type == TYPE_INHERITED && n->ret_param->name)
	    f->retType = param_Type (n->ret_param, n->nk.id);
	else
	    f->retType = n->ret_param->type;
#endif
    }
    else
	f->retType = TYPE_VOID;

#ifdef DEBUG
    fprintf (stderr, "buildFuncDecl (node %d)\n", n->nk.id);
#endif

    for (i = 0; i < n->nargs; ++i) {
	int x;

#ifdef DEBUG
	fprintf (stderr, "node %d: checking type of arg %d\n",
		 n->nk.id, i+1);
#endif
	x = expr_Type (n->args[i], n->nk.id);
	if (EXPR_IS_ARRAY(x)) {
	    f->argTypes[i].indirs = EXPR_NUM_DIMS(x);
	    f->argTypes[i].baseType = EXPR_BASE_TYPE(x);
	}
	else {
	    f->argTypes[i].indirs = 0;
	    f->argTypes[i].baseType = EXPR_BASE_TYPE(x);
	}
#ifdef DEBUG
	fprintf (stderr, "node %d: function %s, arg %d, type %s\n",
		 n->nk.id,
		 f->name, i + 1, typeNameX (f->argTypes[i].baseType,
					    f->argTypes[i].indirs));
#endif
    }
    return f;
}

static void
nukeFuncDecls ()
{
    TreeNode tn, tn2;

    if (funcSymTable == NULL)
	return;

    for (tn = rb_First (funcSymTable); tn != funcSymTable; tn = tn2) {
	free (rb_Value (tn));
	tn2 = rb_Next (tn);
	rb_DeleteNode (tn);
    }
}

static int
yywrap ()
{
    return 1;
}

static int
subdefs_ReadDecls (filename)
char *filename;
{

    if (funcSymTable)
	nukeFuncDecls ();
    else
	funcSymTable = rb_MakeTree ();
    
    if ((yyin = fopen (filename, "r")) == NULL) {
	msg_Format ("can't open %s (%s)\n", filename, strerror (errno));
	return 1;
    }

    /*
     * this is a horrible hack to reset the lexical analyzer's state
     * back to its initial state.  This probably won't work with anything
     * except Lex Classic.  If this code ever has to run with anything
     * else we should probably rewrite the lexical analyzer in bare C.
     */

    yylineno = 1;
#ifndef FLEX_SCANNER
    yysptr = yysbuf;
    yytchar = 0;
    yyprevious = YYNEWLINE;
#else
    /* XXX how to reset FLEX's scanner? */
#endif
    BEGIN 0;

    currFileName = filename;
    errorCount = 0;
    yyparse ();
    fclose (yyin);
    return errorCount;
}

static char *
stars (n)
int n;
{
    static char buf[20];
    char *ptr = buf;

    if (n == 0)
	return "";
    else if (n > 18)
	return "???";

    *ptr++ = ' ';
    while (n > 0) {
	*ptr++ = '*';
	--n;
    }
    *ptr = '\0';
    return buf;
}


static char *
typeName (t)
int t;
{
    switch (t) {
    case TYPE_VOID: return "void";
    case TYPE_INT:  return "int";
    case TYPE_FLOAT: return "float";
    case TYPE_DOUBLE: return "double";
    case TYPE_CHAR: return "char";
    default: return "???";
    }
}

/*
 * write out all the declarations to the indicated file
 */

int
subdefs_WriteDecls (fp)
FILE *fp;
{
    TreeNode tn;
    int i;
    char *ptr;

    for (tn = rb_First (funcSymTable); tn != funcSymTable; tn = rb_Next (tn)) {
	struct funcSym *x = rb_Value (tn);

	/*
	 * if a function was declared in the sub.defs file but was not
	 * used in the graph, don't put it in the temp sub.defs file
	 */

	if (x->isUsed == 0) {
	    if (fp == stderr) {
		fprintf (stderr, "warning: %s (%x) is not used\n", x->name, x);
	    }
	    continue;
	}

	/*
	 * write out the function declaration.  "pointer" types are special.
	 * If the original sub.defs file defined a function as being something
	 * like "char **" (i.e. more than one star), then we respect that
	 * and copy it to the file we are writing.  On the other hand, if
	 * we are inferring the type of a parameter from its usage in the
	 * HeNCE graph, then we always use either one star or none at all.
	 */

	fprintf (fp, "%s %s (", typeName (x->retType), x->name);
	for (i = 0; i < x->nargs; ++i) {
	    if (x->nodeWhereDeclared == -1)
		ptr = stars (x->argTypes[i].indirs);
	    else
		ptr = x->argTypes[i].indirs > 0 ? " *" : "";
	    fprintf (fp, "%s%s%s",
		     typeName (x->argTypes[i].baseType), ptr, 
		     i == x->nargs - 1 ? "" : ", ");
	}
	fprintf (fp, ");\n");
    }
    return 0;
}

static int
yyerror ()
{
    extern int yylineno;

    char *foo = (yychar == TK_ID) ? "identifier" : "symbol";

    if (currFileName != (char *) NULL)
	msg_Format ("%s, line %d: unexpected %s \"%s\"\n",
		    currFileName, yylineno, foo, yytext);
    else
	msg_Format ("line %d: unexpected %s \"%s\"\n", yylineno, foo, yytext);
    ++errorCount;
}

/****************************************************************************
 * routines to manipulate parameter symbol table:
 *
 * since there can be more than one parameter with the same name, the
 * symbol table entry is actually a stack with the most recent binding
 * to that name on top.
 *
 * the symbol table is built from the top of the tree down through
 * each possible path to each node.
 ****************************************************************************/

static Tree paramSymTable;

struct paramSym {
    Param p;
    int nodeWhereDeclared;	/* if of node where param was declared */
    struct paramSym *next;
};

/*
 * declare a parameter
 *
 * XXX should this push a new stack entry if a parameter by that
 * name already exists?
 */

static void
declareParameter (p, n)
Param p;
Node n;
{
    Key k;
    int found;
    TreeNode tn;
    struct paramSym *ps;

#ifdef DEBUG
    fprintf (stderr, "declareParameter (%x \"%s\", %d)\n", p, p->name,
	     n->nk.id);
#endif

    ps = talloc (1, struct paramSym);
    ps->p = p;
    ps->nodeWhereDeclared = n->nk.id;
    ps->next = NULL;

    k.pkey = (void *) p->name;

    if (paramSymTable == NULL)
	paramSymTable = rb_MakeTree ();

    tn = rb_Find (paramSymTable, &k, nameCompare, &found);
    if (found) {
	ps->next = (struct paramSym *) rb_Value (tn);
	rb_SetValue (tn, ps);
    }
    else {
	rb_InsertBefore (tn, &k, (void *) ps);
    }
}

/*
 * remove a parameter's (most recent) declaration, if declared at
 * this node.
 */

static void
undeclareParameter (p, n)
Param p;
Node n;
{
    Key k;
    TreeNode tn;
    int found = 0;
    struct paramSym *ps;

#ifdef DEBUG
    fprintf (stderr, "undeclareParameter (%s, %d)\n", p->name, n->nk.id);
#endif

    if (paramSymTable == NULL)
	return;

    k.pkey = (void *) p->name;
    tn = rb_Find (paramSymTable, &k, nameCompare, &found);
    if (found) {

	/*
	 * if there are multiple instances of this parameter,
	 * pop the most recently declared one.  Otherwise,
	 * delete the symbol table entry.
	 */
	ps = rb_Value (tn);

	if (ps->nodeWhereDeclared != n->nk.id)
	    return;

	if (ps->next) {
	    rb_SetValue (tn, ps->next);
	    ps->next = NULL;
	}
	else
	    rb_DeleteNode (tn);

	free (ps);
    }
}

/*
 * find the most recent declaration of a parameter
 */

static struct paramSym *
lookupParameter (name)
char *name;
{
    TreeNode tn;
    Key k;
    int found = 0;

#ifdef DEBUG
    fprintf (stderr, "lookupParameter (%s)", name);
#endif

    if (name == NULL || paramSymTable == NULL)
	return NULL;

    k.pkey = name;
    tn = rb_Find (paramSymTable, &k, nameCompare, &found);
    if (found) {
#ifdef DEBUG
	fprintf (stderr, " [found]\n");
#endif
	return (struct paramSym *) rb_Value (tn);
    }
#ifdef DEBUG
    fprintf (stderr, " [not found]\n");
#endif
    return NULL;
}

static void
nukeParamDecls ()
{
    TreeNode tn, tn2;

    if (paramSymTable == NULL)
	return;

    for (tn = rb_First (paramSymTable); tn != paramSymTable; tn = tn2) {
	free (rb_Value (tn));
	tn2 = rb_Next (tn);
	rb_DeleteNode (tn);
    }
}

/*
 * find and return the original definition of parameter p
 */

Param
findParamDecl (p, nodenum)
Param p;
int nodenum;
{
    struct paramSym *ps;

#ifdef DEBUG
    fprintf (stderr, "findParamDecl (%s,%d)\n", p->name, nodenum);
#endif

    if ((ps = lookupParameter (p->name)) == NULL) {
	msg_Format ("Node %d: %s: parameter not declared\n",
		    nodenum, p->name);
	return NULL;
    }
#ifdef DEBUG
    fprintf (stderr, "original declaration of %s found at node %d\n",
	     p->name, ps->nodeWhereDeclared);
#endif
    return ps->p;
}

#if 0
static void
fixExpression (e, nodenum)
Expr e;
int nodenum;
{
    int i;

    if (e->elt_type == OP_PARAM) {
	e->val.p = findParamDecl (e->val.p, nodenum);
	return;
    }
    for (i = 0; i < e->nargs; ++i)
	fixExpression (e->args[i]);
}
#endif


/*
 * traverse a HeNCE subgraph.  Make sure parameter and function declarations
 * are consistent, and collect definitions for functions.
 */

static int
scanSubGraph (n, filename)
Node n;
char *filename;			/* for error messages */
{
    TreeNode tn;
    int errors = 0;
    struct funcSym *sc0, *sc1;
    struct paramSym *ps;
    int i;

    /*
     * special case for non-normal nodes, since we don't have to check
     * parameters or function names (is this true?  how about loop vars?)
     */

#ifdef DEBUG
    fprintf (stderr, "ScanSubGraph (node %d)\n", n->nk.id);
#endif

    switch (n->node_type) {

    case NODE_NORMAL:
	/* handle below */
	break;
    case NODE_LOOP:
	/*
	 * special code to infer the type of a loop parameter from
	 * the initializing expression.  This code probably won't get
	 * used...it appears that the current HeNCE executioner always
	 * treats loop parameters as integers, so htool's parser now
	 * has it wired in to declare them to be integers.
	 */
	{
	    int oldtype = n->ret_param->type;
	    int newtype = loop_param_type (n);

	    if (newtype == TYPE_INHERITED) {
		msg_Format ("node %d: cannot determine type of variable %s\n",
			    n->nk.id, n->ret_param->name);
		++errors;
	    }
	    else {
		n->ret_param->type = newtype;
		declareParameter (n->ret_param, n);
		for (tn = rb_First (n->children); tn != n->children;
		     tn = rb_Next (tn)){
		    errors += scanSubGraph ((Node) rb_Value (tn), filename);
		}
		undeclareParameter (n->ret_param, n);
		n->ret_param->type = oldtype;
	    }
	    return errors;
	}
    case NODE_PIPE:
    case NODE_FANOUT:
	/* declare control parameter as a variable */
	declareParameter (n->ret_param, n);
	for (tn = rb_First (n->children); tn != n->children;
	     tn = rb_Next (tn)){
	    errors += scanSubGraph ((Node) rb_Value (tn), filename);
	}
	undeclareParameter (n->ret_param, n);
	return errors;
    default:
	if (n->nchildren == 0 || rb_Empty (n->children))
	    return 0;
	for (tn = rb_First (n->children); tn != n->children;
	     tn = rb_Next (tn)){
	    errors += scanSubGraph ((Node) rb_Value (tn), filename);
	}
	return errors;
    }

    /*
     * 1.  for each named parameter:
     * if it's new, declare it; else, find its declaration
     */
    for (tn = rb_First (n->params); tn != n->params; tn = rb_Next (tn)) {
	struct paramSym *ps;
	Param p0, p1;

	p1 = rb_Value (tn);
	if (p1->io.new) {
	    /*
	     * new parameter.  ignore any previous declarations, and
	     * push this declaration on the stack
	     */
	    p1->io.used = 1;	/* XXX this is probably bogus */
	    p0 = p1;
	    declareParameter (p1, n);
	}
	else {
	    /*
	     * not declared with NEW keyword; find in symbol table
	     */
	    ps = lookupParameter (p1->name);
	    p0 = ps ? ps->p : (Param) NULL;

	    /*
	     * Not declared as new, and no previous declaration
	     */
	    if (p0 == (Param) NULL) {
		if (p1->type == TYPE_INHERITED) {
		    msg_Format ("Node %d: parameter %s is not declared.\n",
				n->nk.id, p1->name);
		    ++errors;
		    continue;
		}
		else {
		    msg_Format ("Node %d: warning: first declaration of \"%s\" lacks NEW keyword\n", n->nk.id, p1->name);
		    declareParameter (p1, n);
		}
	    }		
	    /*
	     * (simple) check for type mismatch
	     *
	     * XXX should check for equivalence of array types
	     */
	    else if (p1->type != TYPE_INHERITED && p1->type != p0->type) {
		msg_Format ("Node %d: type mismatch for parameter %s.\n",
			    n->nk.id, p1->name);
		msg_Format ("(originally declared @ node %d)\n",
			    ps->nodeWhereDeclared);	
		++errors;
		continue;
	    }
	    /*
	     * if we've been here before, check to make sure that there
	     * isn't a declaration through this path of ancestors that
	     * conflicts with another declaration through another path.
	     *
	     * XXX the use of p1->io.used is questionable
	     */
	    else if (p1->io.used && p1 != p0) {
		msg_Format ("Node %d: multiple declarations for %s %s\n",
			    n->nk.id, "parameter", p1->name);
		msg_Format ("(originally declared @ node %d)\n",
			    ps->nodeWhereDeclared);
		++errors;
		continue;
	    }
#if 0
	    /*
	     * okay.  we've found the declaration for this parameter.
	     * replace this parameter with its original declaration.
	     */
	    if (p0 != p1) {
		rb_SetValue (tn, (void *) p0);
		/* free (p1); */
	    }
#endif
	}
    }

#if 0
    /*
     * 1a. We may have parameters in the expression trees that are still
     * of unknown type.  Fix these.
     */

    for (i = 0; i < n->nargs; ++i) {
	fprintf (stderr, "fixExpression [arg %d]\n", i + 1);
	fixExpression (n->args[i]);
    }
#endif

    /*
     * 2. Now we should know the type of each param.  Build a descriptor
     * for this function call.  If there are no previously declared
     * instances of this function call, declare one; else, check
     * to make sure this use of the function matches all others.
     */
    if (n->node_type == NODE_NORMAL && n->sub_name != NULL) {
	sc0 = lookupFuncDecl (n->sub_name);
	sc1 = buildFuncDecl (n);
	if (sc0 == NULL) {
	    /*
	     * first use of this function.  If we have a sub.defs file,
	     * and the function is not declared in the sub.defs file,
	     * issue a warning.  Either way, declare the function based
	     * on the apparent number and types of its parameters.
	     */
	    if (filename)
		msg_Format ("\
Node %d: warning: no declaration for function %s in %s\n",
			    n->nk.id, n->sub_name, filename);
	    addFuncSym (sc1, NULL, n->nk.id);
	}
	else if (compareFunctions (n, sc0, sc1, filename) != 0) {
	    ++errors;
	}
	else {
#ifdef DEBUG
	    fprintf (stderr, "marking %s (%x) as used\n", sc0->name, sc0);
#endif
	    sc0->isUsed = 1;		/* mark function as being used */
	}
	free (sc1);
    }

    /*
     * 3. scan all of our children looking for similar declaration conflicts
     */
    if (n->nchildren != 0 && n->children != NULL) {
	for (tn = rb_First (n->children); tn != n->children; tn = rb_Next (tn))
	    errors += scanSubGraph ((Node) rb_Value (tn), filename);
    }

    /*
     * 4. for any parameters that we declared here, undeclare them now
     */

    for (tn = rb_First (n->params); tn != n->params; tn = rb_Next (tn)) {
	Param p1;

	p1 = rb_Value (tn);
	undeclareParameter (p1, n); /* happens only if we declared param */
    }
    return errors;
}

#if 0
/*
 * do any initialization that needs to be done before collecting
 * parameters and function names
 */

static int
zapNodes (n)
Node n;
{
    n->flags &= ~NF_BEEN_HERE;
    return 0;
}

/*
 * do any cleanup that needs to be done after collecting parameters
 * and function names
 */

static int
unzapNodes (n)
Node n;
{
    n->flags &= ~NF_BEEN_HERE;
    return 0;
}
#endif

/*
 * read declarations from the sub.defs file (if there is one), and
 * check the graph for consistency (both with the declarations
 * in the sub.defs file and between declarations).  Leave the symbol
 * tables lying around so that a subsequent call to subdefs_WriteDecls()
 * can write out a file to be used by mkwrap.
 */

int
subdefs_CheckGraph (g, filename)
Graph g;
char *filename;
{
    TreeNode tn;
    int errors = 0;

#ifdef DEBUG
    fprintf (stderr, "checking graph\n");
#endif
    if (filename != NULL) {
	if (subdefs_ReadDecls (filename) != 0)
	    return 1;
    }
#ifdef DEBUG
    subdefs_WriteDecls (stderr);
#endif
    paramSymTable = rb_MakeTree ();

    if (!rb_Empty (g->heads)) {
	for (tn = rb_First (g->heads); tn != g->heads; tn = rb_Next (tn))
	    errors += scanSubGraph (rb_Value (tn), filename);
    }
#ifdef DEBUG
    fprintf (stderr, "%d error%s\n", errors, errors == 1 ? "" : "s");
#endif

    return errors;
}

/*
 * clean out the symbol tables built by subdefs_CheckGraph ()
 */


void
subdefs_NukeDecls ()
{
    nukeFuncDecls ();
    nukeParamDecls ();
}


static int
loop_param_type (n)
Node n;
{
    int type;

    type = n->ret_param->type;
    if (type == TYPE_INHERITED) {
	if (lookupParameter (n->ret_param->name) != NULL)
	    type = param_Type (n->ret_param, n->nk.id);
    }
    if (type == TYPE_INHERITED)
	type = expr_Type (n->args[0]);
    return type;
}

/*
 * Local variables:
 * mode:c
 * End:
 */
