      SUBROUTINE CAREX(NO, N, M, P, NPAR, DPARAM, DATAF, A, LDA, B,
     1                 LDB, C, LDC, G, LDG, Q, LDQ, X, LDX, NOTE, STORE,
     2                 FORM, RWORK, IERR)
C     
C     PURPOSE   
C
C     To generate the benchmark examples for the numerical solution of 
C     continuous-time algebraic Riccati equations as presented in [1]
C
C       0 = Q + A'X + XA - XGX
C
C     corresponding to the Hamiltonian matrix
C        
C            (  A  -G  ) 
C        H = (       T ).
C            ( -Q  -A  ) 
C
C     A,G,Q,X are real N-by-N matrices, Q and G are symmetric and may  
C     be given in factored form
C
C                   -1 T                         T
C      (I)   G = B R  B  ,           (II)   Q = C Q0 C .
C    
C     Here, C is P-by-N, Q0 P-by-P, B N-by-M, and R M-by-M, where Q0
C     and R are symmetric. In linear-quadratic control problems, 
C     usually Q0 is positive semidefinite and R positive definite. 
C
C     ARGUMENT LIST 
C       ARGUMENTS IN
C 
C         NO - INTEGER.
C              The number of the benchmark example to generate according  
C              to [1]. 
C 
C          N - INTEGER.
C              This integer determines the actual state dimension, i.e., 
C              the order of the matrix A as follows:  
C              N = number of vehicles for Example 15.
C              N = order of matrix A for Examples 16-18.
C              N = dimension of second-order system, i.e., order of
C                  stiffness matrix for Examples 19 and 20. The order of
C                  the output matrix A is 2*N for Example 19 and 2*N-1
C                  for Example 20.  
C              N is fixed for the examples of Sections 2 and 3 of [1],
C              i.e., currently Examples 1-14.
C              NOTE that N is overwritten for Examples 1-14 and for the
C              other examples if N is set by default.
C
C       M, P - INTEGER.
C              M is the number of columns in the matrix B from (I) (in
C              control problems, the number of inputs of the system). 
C              P is the number of rows in the matrix C from (II) (in
C              control problems, the number of outputs of the system).
C              Currently, M and P are fixed or determined by N for all
C              examples and thus are not referenced on input.  
C              NOTE that M and P are overwritten.
C
C       NPAR - INTEGER.
C              Number of input parameters supplied by the user.
C              Examples 1-6 (Section 3 of [1]) have no parameters.
C              Examples 7-14 (Section 4 of [1]) each have one DOUBLE
C              PRECISION parameter which may be supplied in DPARAM(1).
C              Examples 15,16 have one INTEGER parameter which determines
C              the size of the problem. This parameter may be supplied in 
C              the input argument N.
C              Examples 17-19 have one INTEGER (supplied in N) and 
C              several DOUBLE PRECISION parameters (supplied in DPARAM). 
C              If for Example 20 user supplied data is to be used, i.e.,
C              NPAR > 0, the INTEGER input argument N must contain an
C              INTEGER l (as described in [1]) and the CHARACTER input
C              argument DATAF must contain the name of a data file. 
C              If the input value of NPAR is less than the number of
C              parameters of the Example NO (according to [1]), the
C              missing parameters are set by default.
C
C     DPARAM - DOUBLE PRECISION array of DIMENSION at least 7.
C              Double precision parameter vector. For explanation of the
C              parameters see [1].
C              DPARAM(1) defines the parameters 'epsilon' for the
C              examples in Section 3 (NO = 7,...,14), the parameter 'q' 
C              for NO = 17, 'a' for NO = 18, and 'mu' for NO = 19. 
C              DPARAM(2) defines parameters 'r' for NO = 17, 'b' for 
C              NO = 18, and 'delta' for NO = 19.
C              DPARAM(3) defines 'c' for NO = 18 and 'kappa' for NO = 19.
C              DPARAM(4) - DPARAM(7) are only used to generate Example 
C              18 and define in consecutive order the intervals 
C              ['beta_1', 'beta_2'],  ['gamma_1', 'gamma_2'].
C              If NPAR is smaller than the number of used parameters in
C              Example NO (as described in [1]), default values are
C              used and returned in corresponding components of DPARAM.
C              NOTE that those entries of DPARAM are overwritten which
C              are used to generate the example but were not supplied by
C              the user.
C
C      DATAF - CHARACTER*255.
C              The name of a data file supplied by the user. In the
C              current version, only Example 20 allows a user-defined
C              data file. This file must contain consecutively DOUBLE
C              PRECISION vectors mu, delta, gamma, and kappa. The length
C              of these vectors is determined by the input value for N.
C              If on entry N = l, then mu, delta must each contain l,
C              gamma, kappa each l-1 DOUBLE PRECISION values.
C
C        LDA - INTEGER.
C              The leading dimension of array A as declared in the 
C              calling program.
C              LDA .GE. N  where N is the order of the matrix A, i.e.,
C              the output value of the integer N.  
C        
C        LDB - INTEGER.
C              The leading dimension of array B as declared in the 
C              calling program.
C              LDB .GE. N  (output value of N).
C
C        LDC - INTEGER.
C              The leading dimension of array C as declared in the 
C              calling program.
C              LDC .GE. P  where P is either defined by default or
C              depends upon N. (For all examples, P .LE. N, where N is
C              the output value of the argument N.)   
C
C        LDG - INTEGER.
C              If full storage mode is used for G, i.e., STORE = 'F' 
C              or 'f', then G is stored like a 2-dimensional array
C              with leading dimension LDG. If packed symmetric storage
C              mode is used, then LDG is not referenced.
C              LDG .GE. N  if STORE = 'F' or 'f'.
C
C        LDQ - INTEGER.
C              If full storage mode is used for Q, i.e., STORE = 'F' 
C              or 'f', then Q is stored like a 2-dimensional array
C              with leading dimension LDQ. If packed symmetric storage
C              mode is used, then LDQ is not referenced.
C              LDQ .GE. N  if STORE = 'F' or 'f'.
C
C        LDX - INTEGER.
C              The leading dimension of array X as declared in the 
C              calling program.
C              LDX .GE. N. 
C
C       ARGUMENTS OUT
C
C          N - INTEGER.
C              The order of matrix A. 
C
C          M - INTEGER.
C              The number of columns of matrix B from (I), rank(G) <= M.
C
C          P - INTEGER.
C              The number of rows of matrix C from (II), rank(Q) <= P.
C 
C     DPARAM - DOUBLE PRECISION array of DIMENSION at least 7.
C              Double precision parameter vector. For explanation of the
C              parameters see [1].
C              DPARAM(1) defines the parameters 'epsilon' for the
C              examples in Section 3 (NO = 7,...,14), the parameter 'q' 
C              for NO = 17, 'a' for NO = 18, and 'mu' for NO = 19. 
C              DPARAM(2) defines 'r' for NO = 17, 'b' for NO = 18, and
C              'delta' for NO = 19.
C              DPARAM(3) defines 'c' for NO = 18 and 'kappa' for NO = 19.
C              DPARAM(4) - DPARAM(7) are only used to generate Example
C              18 and define in consecutive order the intervals 
C              ['beta_1', 'beta_2'],  ['gamma_1', 'gamma_2'].
C
C          A - DOUBLE PRECISION array of DIMENSION (LDA,N).
C              The leading N by N part of this array contains the
C              coefficient matrix A of the ARE.
C        
C          B - DOUBLE PRECISION array of DIMENSION (LDB,M).
C              If (FORM .EQ. 'F' or 'f' or 'G' or 'g') then array B
C              contains the matrix B of the factored form (I) of G. 
C              Otherwise, B is used as workspace. 
C
C          C - DOUBLE PRECISION array of DIMENSION (LDC,N).
C              If (FORM .EQ. 'F' or 'f' or 'Q' or 'q') then array C
C              contains the matrix C of the factored form (II) of Q. 
C              Otherwise, C is used as workspace. 
C
C          G - DOUBLE PRECISION array of DIMENSION at least ng.
C              If STORE = 'F' or 'f'               then ng = LDG*N.
C              If STORE = 'U' or 'u' or 'L' or 'l' then ng = N*(N+1)/2.
C              If (FORM .EQ. 'P' or 'p' or 'Q' or 'q'), then array G
C              contains the coefficient matrix G of the ARE.  
C              If (FORM .EQ. 'F' or 'f' or 'G' or 'g'), then array G
C              contains the 'control weighting matrix' R of G's factored 
C              form as in (I).  
C              The symmetric matrix contained in array G is stored
C              according to MODE PARAMETER STORE. 
C
C          Q - DOUBLE PRECISION array of DIMENSION at least nq.
C              If STORE = 'F' or 'f'               then nq = LDQ*N.
C              If STORE = 'U' or 'u' or 'L' or 'l' then nq = N*(N+1)/2.
C              If (FORM .EQ. 'P' or 'p' or 'G' or 'g'), then array Q
C              contains the coefficient matrix Q of the ARE.  
C              If (FORM .EQ. 'F' or 'f' or 'Q' or 'q'), then array Q
C              contains the 'output weighting matrix' Q0 of Q's factored   
C              form as in (II).  
C              The symmetric matrix contained in array Q is stored
C              according to MODE PARAMETER STORE. 
C
C          X - DOUBLE PRECISION array of DIMENSION (LDX,N).
C              If an exact solution is available (NO = 1,2,7,9-12,16),
C              then the leading N-by-N part of this array contains
C              the solution matrix X. Otherwise, X is not referenced.
C
C       NOTE - CHARACTER*70.
C              String containing short information about the chosen 
C              example.
C
C     WORK SPACE
C 
C      RWORK - DOUBLE PRECISION array of DIMENSION at least N*MAX(4,N).
C
C     MODE PARAMETERS
C 
C       FORM - CHARACTER.
C              Specifies the output format of the examples, i.e., if Q
C              and G are returned in factored form (I),(II), or not.
C              FORM = 'P' or 'p': The matrices Q and G are returned.
C              FORM = 'G' or 'g': G is returned in factored form, i.e., 
C                                 B and R from (I) are returned, array Q  
C                                 contains the coefficient matrix Q. 
C              FORM = 'Q' or 'q': Q is returned in factored form, i.e., 
C                                 C and Q0 from (II) are returned, array 
C                                 G contains the coefficient matrix G. 
C              FORM = 'F' or 'f': Q and G are given in factored form,  
C                                 i.e., B, R, C, and Q0 from (I) and (II)
C                                 are returned.   
C              Otherwise, CAREX returns with an error.
C              NOTE that for factored forms, output array G contains R
C              from (I) whereas output array Q contains Q0 from (II).
C
C      STORE - CHARACTER.
C              Specifies the storage mode for arrays G and Q.
C              STORE = 'F' or 'f': Full symmetric matrices are stored in 
C                                  G and Q, i.e., the leading N-by-N
C                                  part of these arrays each contain a 
C                                  symmetric matrix.
C              STORE = 'L' or 'l': Matrices contained in arrays G and Q
C                                  are stored in lower packed mode,
C                                  i.e., the lower triangle of a
C                                  symmetric n-by-n matrix is stored by
C                                  columns, e.g., the matrix entry
C                                  G(i,j) is stored in the array entry 
C                                  G(i+(2*n-j)*(j-1)/2)  for j <= i. 
C              STORE = 'U' or 'u': Matrices contained in arrays G and Q
C                                  are stored in upper packed mode,
C                                  i.e., the upper triangle of a
C                                  symmetric n-by-n matrix is stored by
C                                  columns, e.g., the matrix entry
C                                  G(i,j) is stored in the array entry
C                                  G(i+j*(j-1)/2)  for i <= j.
C              Otherwise, CAREX returns with an error.
C
C     ERROR INDICATOR
C     
C       IERR - INTEGER. 
C              Unless the routine detects an error (see next section),
C              IERR contains 0 on exit.
C
C
C     WARNINGS AND ERRORS DETECTED BY THE ROUTINE
C
C     IERR = 1 : (NO .LT. 1) or (NO .GT. NEX).  
C                (NEX = number of available examples.)
C     IERR = 2 : (N .LT. 1) or (N .GT. LDA) or (N .GT. LDB) or 
C                (N. GT. LDX) or (P .GT. LDC).      
C     IERR = 3 : MODE PARAMETER STORE had an illegal value on input.
C     IERR = 4 : MODE PARAMETER FORM had an illegal value on input.
C     IERR = 5 : Data file could not be opened or had wrong format.
C     IERR = 6 : Division by zero.  
C     IERR = 7 : G can not be computed as in (I) due to a singular R
C                matrix.  
C
C     REFERENCE
C
C     [1] P. BENNER, A.J. LAUB and V. MEHRMANN
C         A Collection of Benchmark Examples for the Numerical Solution
C         of Algebraic Riccati Equations I: Continuous-Time Case.
C         Technical Report SPC 95_22, Fak. f. Mathematik, 
C         TU Chemnitz-Zwickau (Germany), October 1995.
C     [2] E. ANDERSON ET AL.
C         LAPACK Users' Guide, second edition.
C         SIAM, Philadelphia, PA (1994).
C
C     CONTRIBUTOR
C     
C     Peter Benner and Volker Mehrmann (TU Chemnitz-Zwickau)
C     Alan J. Laub (University of California, Santa Barbara) 
C
C     KEYWORDS
C    
C     algebraic Riccati equation, Hamiltonian matrix
C
C     REVISIONS
C
C     1995, January 18.
C     1995, October 12.
C
C***********************************************************************    
C
C     .. Parameters ..
C     . # of examples available , # of examples with fixed size. .
      INTEGER          NEX, NFSEX
      PARAMETER        (NEX = 20, NFSEX = 14)
      DOUBLE PRECISION ZERO, ONE, TWO, THREE, FOUR, PI
      PARAMETER        (ZERO = .0D0, ONE = .1D1, TWO = .2D1, 
     1                  THREE = .3D1, FOUR = .4D1, 
     2                  PI = .3141592653589793D1) 
C
C     .. Scalar Arguments ..
      INTEGER          NO, N, M, P, NPAR, LDA, LDB, LDC, LDG, LDQ, LDX,
     1                 IERR 
      CHARACTER*80     DATAF
      CHARACTER        FORM, STORE

C
C     .. Array Arguments ..
      DOUBLE PRECISION A(LDA,*), B(LDB,*), C(LDC,*), G(*), Q(*), 
     1                 X(LDX,*), DPARAM(*), RWORK(*)
      CHARACTER        NOTE*70
C      
C     .. Local Scalars ..
      INTEGER          GDIMM, I, ISYMM, INFO, J, L, MSYMM, NSYMM, POS, 
     1                 PSYMM, QDIMM, STATUS   
      DOUBLE PRECISION APPIND, B1, B2, C1, C2, TEMP, TTEMP, SUM
C
C     ..Local Arrays ..
      INTEGER          NDEF(NEX), MDEF(NFSEX), PDEF(NFSEX)
      DOUBLE PRECISION PARDEF(NEX)
      CHARACTER        IDENT*4
      CHARACTER*70     NOTES(NEX)
C      
C     .. External Functions ..
C     . BLAS, LAPACK .
      DOUBLE PRECISION DDOT, DLAPY2
      EXTERNAL         DDOT, DLAPY2
C     . LAPACK .
      LOGICAL          LSAME
      EXTERNAL         LSAME
C
C     .. External Subroutines ..
C     . BLAS .
      EXTERNAL         DCOPY, DGEMV, DSCAL, DSPMV, DSPR, DSYMM
C     . LAPACK .
      EXTERNAL         DLASET, DPPTRF, DPPTRI, DPTTRF, DPTTRS, DSYRK 
C     . CAREX .
      EXTERNAL         SP2SY, SY2SP
C     
C     .. Intrinsic Functions ..
      INTRINSIC        COS, MAX, MIN, MOD, SQRT
C      
C     .. Data Statements ..
C     . default values for dimensions .
      DATA NDEF /2, 2, 4, 8, 9, 30, 2, 2, 2, 2, 2, 3, 4, 4, 20, 64,
     1           21, 100, 30, 211/  
      DATA MDEF /1, 1, 2, 2, 3, 3, 1, 2, 1, 2, 1, 3, 1, 1/
      DATA PDEF /2, 2, 4, 8, 9, 5, 1, 1, 2, 2, 2, 3, 2, 1/
C     . default values for parameters .
      DATA PARDEF /ZERO, ZERO, ZERO, ZERO, ZERO, ZERO, .1D-5,
     1             .1D-7, .1D7, .1D-6, ZERO, .1D7, .1D-5, .1D-5, ZERO,
     2             ZERO, ONE, .1D-1, FOUR, ZERO/
C     . comments on examples .
      DATA NOTES /
     1'Laub 1979, Ex.1', 'Laub 1979, Ex.2: uncontrollable-unobservable d
     2ata', 'Beale/Shafai 1989: model of L-1011 aircraft', 'Bhattacharyy
     3a et al. 1983: binary distillation column', 'Patnaik et al. 1980: 
     4tubular ammonia reactor', 'Davison/Gesing 1978: J-100 jet engine',
     5'Arnold/Laub 1984, Ex.1: (A,B) unstabilizable as EPS -> 0',
     6'Arnold/Laub 1984, Ex.3: control weighting matrix singular as EPS 
     7-> 0', 'Kenney/Laub/Wette 1989, Ex.2: ARE ill conditioned for EPS
     8-> oo', 'Bai/Qian 1994: ill-conditioned Hamiltonian for EPS -> 0',
     9'Laub 1992: H-infinity problem, eigenvalues  +/- EPS +/- i',
     A'Petkov et al. 1987: increasingly badly scaled Hamiltonian as EPS
     B-> oo', 'Chow/Kokotovic 1976: magnetic tape control system', 
     C'Arnold/Laub 1984, Ex.2: poor sep. of closed-loop spectrum as EPS
     D-> 0', 'Laub 1979, Ex.4: string of high speed vehicles', 'Laub 197
     E9, Ex.5: circulant matrices', 'Laub 1979, Ex.6: ill-conditioned Ri
     Fccati equation', 'Rosen/Wang 1992: lq control of 1-dimensional hea
     Gt flow','Hench et al. 1995: coupled springs, dashpots and masses',
     F'Lang/Penzl 1994: rotating axle' /
C
C     .. Executable Statements ..
C
      IERR = 0
C
      IF ((NO .LT. 1) .OR. (NO .GT. NEX)) THEN
	IERR = 1
      ELSE IF (NO .GT. NFSEX) THEN
        IF (NPAR .LE. 0)  N = NDEF(NO)
        IF (NO .EQ. NFSEX+1) THEN
          M = N
          P = N - 1
          N = 2*N - 1
        ELSE IF (NO .EQ. NFSEX+2) THEN
          M = N
          P = N
        ELSE IF (NO .EQ. NEX-1) THEN
          L = N
          M = 2
          P = 2*L
          N = 2*L
        ELSE IF (NO .EQ. NEX) THEN
          L = N
          M = L
          P = L
          N = 2*L-1
        ELSE 
          M = 1
          P = 1
        END IF
      ELSE IF ((NO .GT. 0) .AND. (NO .LT. NFSEX)) THEN
        N = NDEF(NO)  
        M = MDEF(NO)
        P = PDEF(NO) 
      END IF
      IF (IERR .NE. 0)  GOTO 2001
      IF ((N .LT. 1) .OR. (N .GT. LDA) .OR. (N .GT. LDB) .OR. 
     1    (N .GT. LDX) .OR. (P .GT. LDC)) THEN 
        IERR = 2  
      ELSE IF (.NOT. (LSAME(STORE,'F') .OR. LSAME(STORE,'L') .OR. 
     1    LSAME(STORE,'U'))) THEN
        IERR = 3
      ELSE IF (.NOT. (LSAME(FORM,'G') .OR. LSAME(FORM,'F') .OR.
     1    LSAME(FORM,'P') .OR. LSAME(FORM,'Q'))) THEN
        IERR = 4
      END IF
      IF (IERR .NE. 0)  GOTO 2001
C      
      NOTE  = NOTES(NO)
      NSYMM = N*(N+1)/2
      MSYMM = M*(M+1)/2
      PSYMM = P*(P+1)/2
      IF (NPAR .LE. 0)  DPARAM(1) = PARDEF(NO)
C     
      CALL DLASET('A', N, N, ZERO, ZERO, A, LDA)
      CALL DLASET('A', N, M, ZERO, ZERO, B, LDB)
      CALL DLASET('A', P, N, ZERO, ZERO, C, LDC)
      CALL DLASET('L', MSYMM, 1, ZERO, ZERO, G, 1)
      CALL DLASET('L', PSYMM, 1, ZERO, ZERO, Q, 1)
C
      IF (NO .EQ. 1) THEN
        A(1,2) = ONE
        B(2,1) = ONE
        Q(1)   = ONE
        Q(3)   = TWO
        IDENT  = '0101'
        CALL DLASET('A', N, N, ONE, TWO, X, LDX)
C
      ELSE IF (NO. EQ. 2) THEN
        A(1,1) = FOUR
        A(2,1) = -.45D1
        A(1,2) = THREE
        A(2,2) = -.35D1
        CALL DLASET('A', N, M, -ONE, ONE, B, LDB)
        Q(1)  = .9D1
        Q(2)  = .6D1
        Q(3)  = FOUR
        IDENT = '0101'
        TEMP  = ONE + SQRT(TWO)
        CALL DLASET('A', N, N, .6D1*TEMP, FOUR*TEMP, X, LDX)
        X(1,1) = .9D1*TEMP
C
      ELSE IF ((NO .GE. 3) .AND. (NO .LE. 6)) THEN
        WRITE (DATAF(1:10), '(A,I1,A)') 'CAREX', NO, '.DAT'
        IF ((NO .EQ. 3) .OR. (NO .EQ. 4)) THEN
          IDENT = '0101'
        ELSE IF (NO .EQ. 5) THEN
          IDENT = '0111'
        ELSE IF (NO .EQ. 6) THEN
          IDENT = '0011'
        END IF
        OPEN(1, IOSTAT = STATUS, STATUS = 'OLD', FILE = DATAF)
        IF (STATUS .NE. 0) THEN
          IERR = 5
        ELSE
          DO 10  I = 1, N
            READ (1, FMT = *, IOSTAT = STATUS) (A(I,J), J = 1, N)
            IF (STATUS .NE. 0)  IERR = 5
10        CONTINUE
          DO 20  I = 1, N
            READ (1, FMT = *, IOSTAT = STATUS) (B(I,J), J = 1, M)
            IF (STATUS .NE. 0)  IERR = 5
20        CONTINUE
          IF (NO. LE. 4) THEN 
            DO 30  I = 1, N
              POS = (I-1)*N
              READ (1, FMT = *, IOSTAT = STATUS) (RWORK(POS+J), J = 1,N)
30          CONTINUE
            IF (STATUS .NE. 0) THEN               
              IERR = 5
            ELSE
              CALL SY2SP(N, RWORK, N, 'L', 'L', INFO)
              CALL DCOPY(NSYMM, RWORK, 1, Q, 1)
            END IF
          ELSE IF (NO .EQ. 6) THEN
            DO 35  I = 1, P
              READ (1, FMT = *, IOSTAT = STATUS) (C(I,J), J = 1, N)
              IF (STATUS .NE. 0)  IERR = 5
35          CONTINUE
          END IF
        END IF
        CLOSE(1)
C
      ELSE IF (NO .EQ. 7) THEN
        A(1,1) =  ONE
	A(2,2) = -TWO
 	B(1,1) = DPARAM(1)
        CALL DLASET('U', P, N, ONE, ONE, C, LDC) 
        IDENT  = '0011'
        IF (DPARAM(1) .NE. ZERO) THEN
          TEMP   = DLAPY2(ONE, DPARAM(1))
          X(1,1) = (ONE + TEMP)/DPARAM(1)/DPARAM(1)
          X(2,1) = ONE/(TWO + TEMP)
          X(1,2) = X(2,1)
          TTEMP  = DPARAM(1)*X(1,2)
          TEMP   = (ONE - TTEMP) * (ONE + TTEMP)
          X(2,2) = TEMP / FOUR
        ELSE
          IERR = 6
        END IF
C
      ELSE IF (NO .EQ. 8) THEN
        A(1,1) = -.1D0
	A(2,2) = -.2D-1
	B(1,1) =  .1D0
	B(2,1) =  .1D-2
	B(2,2) =  .1D-1
        CALL DLASET('L', MSYMM, 1, ONE, ONE, G, MSYMM) 
        G(1)   = G(1) + DPARAM(1)
        C(1,1) = .1D2
        C(1,2) = .1D3
        IDENT  = '0010'
C
      ELSE IF (NO .EQ. 9) THEN 
        A(1,2) = DPARAM(1)
        B(2,1) = ONE
        IDENT  = '0111'
        IF (DPARAM(1) .NE. ZERO) THEN 
          TEMP   = SQRT(ONE + TWO*DPARAM(1))
          CALL DLASET('A', N, N, ONE, TEMP, X, LDX)
          X(1,1) = X(1,1)/DPARAM(1)
        ELSE
          IERR = 6
        END IF
C
      ELSE IF (NO .EQ. 10) THEN
        TEMP = DPARAM(1) + ONE
        CALL DLASET('A', N, N, ONE, TEMP, A, LDA)
        Q(1) = DPARAM(1)**2
        Q(3) = Q(1)
        IDENT = '1101'
        X(1,1) = TWO*TEMP + SQRT(2*TEMP**2 + TWO) + SQRT(TWO)*DPARAM(1) 
        X(1,1) = X(1,1)/TWO 
        X(2,2) = X(1,1)
        TTEMP  = X(1,1) - TEMP
        IF (TTEMP .NE. ZERO) THEN
          X(2,1) = X(1,1) / TTEMP
          X(1,2) = X(2,1)
        ELSE
          IERR = 6
        END IF
C
      ELSE IF (NO .EQ. 11) THEN
        A(1,1) = THREE - DPARAM(1) 
        A(2,1) = FOUR
        A(1,2) = ONE
        A(2,2) = TWO - DPARAM(1)
        CALL DLASET('L', N, M, ONE, ONE, B, LDB)
        Q(1)   = FOUR*DPARAM(1) - .11D2
        Q(2)   = TWO*DPARAM(1)  - .5D1 
        Q(3)   = TWO*DPARAM(1)  - TWO 
        IDENT  = '0101'
        CALL DLASET('A', N, N, ONE, ONE, X, LDX)
        X(1,1) = TWO
C
      ELSE IF (NO .EQ. 12) THEN
        IF (DPARAM(1) .NE. ZERO) THEN
          A(1,1) = DPARAM(1)
          A(2,2) = DPARAM(1)*TWO
          A(3,3) = DPARAM(1)*THREE
C       .. set C = V ..
          TEMP   = TWO/THREE
          CALL DLASET('A', P, N, -TEMP, ONE - TEMP, C, LDC)
          CALL DSYMM('L','L', N, N, ONE, C, LDC, A, LDA, ZERO, RWORK, N)    
          CALL DSYMM('R','L', N, N, ONE, C, LDC, RWORK, N, ZERO, A, LDA) 
C       .. G = R ! ..
          G(1) = DPARAM(1)
          G(4) = DPARAM(1)
          G(6) = DPARAM(1)
          Q(1) = ONE/DPARAM(1)
          Q(4) = ONE
          Q(6) = DPARAM(1)
          IDENT = '1000'
          CALL DLASET('A', N, N, ZERO, ZERO, X, LDX)
          TEMP   = DPARAM(1)**2
          X(1,1) = TEMP + SQRT(TEMP**2 + ONE)
          X(2,2) = TEMP*TWO + SQRT(FOUR*TEMP**2 + DPARAM(1))     
          X(3,3) = TEMP*THREE + DPARAM(1)*SQRT(.9D1*TEMP + ONE)
          CALL DSYMM('L','L', N, N, ONE, C, LDC, X, LDX, ZERO, RWORK, N)    
          CALL DSYMM('R','L', N, N, ONE, C, LDC, RWORK, N, ZERO, X, LDX)
        ELSE
          IERR = 6
        END IF
C 
      ELSE IF (NO .EQ. 13) THEN
        IF (DPARAM(1) .NE. ZERO) THEN
          A(1,2) =  .400D0
          A(2,3) =  .345D0
          A(3,2) = -.524D0/DPARAM(1)
          A(3,3) = -.465D0/DPARAM(1)
          A(3,4) =  .262D0/DPARAM(1)
          A(4,4) = -ONE/DPARAM(1)
          B(4,1) =  ONE/DPARAM(1)
          C(1,1) =  ONE
          C(2,3) =  ONE
          IDENT  = '0011'
        ELSE
          IERR = 6
        END IF
C
      ELSE IF (NO .EQ. 14) THEN
        A(1,1) = -DPARAM(1)
	A(2,1) = -ONE
	A(1,2) =  ONE
 	A(2,2) = -DPARAM(1)
	A(3,3) =  DPARAM(1)
	A(4,3) = -ONE
	A(3,4) =  ONE
	A(4,4) =  DPARAM(1)
        CALL DLASET('L', N, M, ONE, ONE, B, LDB)
        CALL DLASET('U', P, N, ONE, ONE, C, LDC)
        IDENT = '0011' 
C
      ELSE IF (NO .EQ. 15) THEN
        DO 40  I = 1, N
	  IF (MOD(I,2) .EQ. 1) THEN
	    A(I,I)       = -ONE
	    B(I,(I+1)/2) =  ONE
	  ELSE
	    A(I,I-1) =  ONE
	    A(I,I+1) = -ONE
	    C(I/2,I) =  ONE
	  END IF
40      CONTINUE       
        ISYMM = 1
        DO 50  I = P, 1, -1
          Q(ISYMM) = .1D2
          ISYMM    = ISYMM + I
50      CONTINUE 
        IDENT = '0001' 
C     
      ELSE IF (NO .EQ. 16) THEN
	DO 60  I = 1, N
          A(I,I) = -TWO
	  IF (I .LT. N) THEN 
	    A(I,I+1) = ONE
	    A(I+1,I) = ONE
	  END IF
60	CONTINUE
	A(1,N) = ONE
	A(N,1) = ONE
        IDENT = '1111'
        TEMP = TWO * PI / DBLE(N)
        DO 70  I = 1, N 
          RWORK(I)   = COS(TEMP*DBLE(I-1))
          RWORK(N+I) = -TWO + TWO*RWORK(I)
          RWORK(N+I) = RWORK(N+I) + 
     1                 SQRT(.5D1 + FOUR*RWORK(I)*(RWORK(I) - TWO)) 
70      CONTINUE 
        DO 90  J = 1, N
          DO 80  I = 1, N 
             RWORK(2*N+I) = COS(TEMP*DBLE(I-1)*DBLE(J-1))
80        CONTINUE 
          X(J,1) = DDOT(N, RWORK(N+1), 1, RWORK(2*N+1), 1)/DBLE(N)
90      CONTINUE
C       .. set up circulant solution matrix ..
        DO 100  I = 2, N
          CALL DCOPY(N-I+1, X(1,1),     1, X(I,I), 1)
          CALL DCOPY(I-1,   X(N-I+2,1), 1, X(1,I), 1)
100     CONTINUE 
C
      ELSE IF (NO .EQ. 17) THEN
C     .. set up remaining parameter ..
	IF (NPAR .LT. 2)  DPARAM(1) = ONE
	IF (NPAR .LT. 3)  DPARAM(2) = ONE
        CALL DLASET('A', N-1, N-1, ZERO, ONE, A(1,2), LDA)
        B(N,1) = ONE
	C(1,1) = ONE
        Q(1)   = DPARAM(1)
        G(1)   = DPARAM(2)
        IDENT  = '0000'   
C     
      ELSE IF (NO .EQ. 18) THEN
C       .. set up remaining parameters ..
        APPIND = DBLE(N + 1)
	IF (NPAR .LT. 2)  DPARAM(1) = PARDEF(NO)
        IF (NPAR .LT. 3)  DPARAM(2) = ONE
        IF (NPAR .LT. 4)  DPARAM(3) = ONE
        IF (NPAR .LT. 5)  DPARAM(4) = .2D0
        IF (NPAR .LT. 6)  DPARAM(5) = .3D0
        IF (NPAR .LT. 7)  DPARAM(6) = .2D0
        IF (NPAR .LT. 8)  DPARAM(7) = .3D0
C       .. set up stiffness matrix ..
        TEMP = -DPARAM(1)*APPIND
        CALL DLASET('A', N, N, ZERO, TWO*TEMP, A, LDA)
        DO 110  I = 1, N - 1
          A(I+1,I) = -TEMP
          A(I,I+1) = -TEMP
110     CONTINUE
C       .. set up Gramian, stored by diagonals ..
        TEMP = ONE/(.6D1*APPIND)
        CALL DLASET('L', N, 1, FOUR*TEMP, FOUR*TEMP, RWORK, N)
        CALL DLASET('L', N-1, 1, TEMP, TEMP, RWORK(N+1), N)
        CALL DPTTRF(N, RWORK(1), RWORK(N+1), INFO)
C       .. A = (inverse of Gramian) * (stiffness matrix) ..
        CALL DPTTRS(N, N, RWORK(1), RWORK(N+1), A, LDA, INFO)
C       .. compute B, C ..
        DO 120  I = 1, N
          B1 = MAX(DBLE(I-1)/APPIND, DPARAM(4))
          B2 = MIN(DBLE(I+1)/APPIND, DPARAM(5))
          C1 = MAX(DBLE(I-1)/APPIND, DPARAM(6))
          C2 = MIN(DBLE(I+1)/APPIND, DPARAM(7))
          IF (B1 .GE. B2) THEN
            B(I,1) = ZERO
          ELSE
            B(I,1) = B2 - B1 
            TEMP   = MIN(B2, DBLE(I)/APPIND)
            IF (B1 .LT. TEMP) THEN   
              B(I,1) = B(I,1) + APPIND*(TEMP**2 - B1**2)/TWO 
              B(I,1) = B(I,1) + DBLE(I)*(B1 - TEMP)
            END IF
            TEMP = MAX(B1, DBLE(I)/APPIND)
            IF (TEMP .LT. B2) THEN
              B(I,1) = B(I,1) - APPIND*(B2**2 - TEMP**2)/TWO
              B(I,1) = B(I,1) - DBLE(I)*(TEMP - B2)
            END IF
          END IF
          IF (C1 .GE. C2) THEN
            C(1,I) = ZERO
          ELSE
            C(1,I) = C2 - C1 
            TEMP   = MIN(C2, DBLE(I)/APPIND)
            IF (C1 .LT. TEMP) THEN   
              C(1,I) = C(1,I) + APPIND*(TEMP**2 - C1**2)/TWO 
              C(1,I) = C(1,I) + DBLE(I)*(C1 - TEMP)
            END IF
            TEMP = MAX(C1, DBLE(I)/APPIND)
            IF (TEMP .LT. C2) THEN
              C(1,I) = C(1,I) - APPIND*(C2**2 - TEMP**2)/TWO
              C(1,I) = C(1,I) - DBLE(I)*(TEMP - C2)
            END IF
          END IF
120     CONTINUE
        CALL DSCAL(N, DPARAM(2), B(1,1), 1)
        CALL DSCAL(N, DPARAM(3), C(1,1), LDC)
        CALL DPTTRS(N, 1, RWORK(1), RWORK(N+1), B, LDB, INFO)
        IDENT = '0011'
C
      ELSE IF (NO .EQ. 19) THEN
C       .. set up remaining parameters ..
        IF (NPAR .LT. 2)  DPARAM(1) = PARDEF(NO)
        IF (NPAR .LT. 3)  DPARAM(2) = FOUR
        IF (NPAR .LT. 4)  DPARAM(3) = ONE
        IF (DPARAM(1) . NE. 0) THEN 
          CALL DLASET('A', L, L, ZERO, ONE, A(1,L+1), LDA) 
          TEMP  = DPARAM(3) / DPARAM(1)
          A(L+1,1) = -TEMP
          A(L+1,2) =  TEMP
          A(N,L-1) =  TEMP
          A(N,L)   = -TEMP
          TTEMP = TWO*TEMP
          DO 130  I = 2, N-1
            A(L+I,I)   = -TTEMP
            A(L+I,I+1) =  TEMP
            A(L+I,I-1) =  TEMP
130       CONTINUE
          CALL DLASET('A', L, L, ZERO, -DPARAM(2)/DPARAM(1), A(L+1,L+1),
     1                LDA)
          B(L+1,1) =  ONE / DPARAM(1) 
          B(N,M)   = -ONE / DPARAM(1)
          IDENT = '0111' 
        ELSE
          IERR = 6
        END IF
C
      ELSE IF (NO .EQ. 20) THEN
        IF (NPAR .LT. 1)  WRITE (DATAF(1:11), '(A,I2,A)') 'CAREX', NO,
     1                          '.DAT' 
        OPEN(1, IOSTAT = STATUS, STATUS = 'OLD', FILE = DATAF)
        IF (STATUS .NE. 0) THEN
          IERR = 5
        ELSE
          READ (1, FMT = *, IOSTAT = STATUS) (RWORK(I), I = 1, 4*L-2)
          IF (STATUS .NE. 0)  IERR = 5
        END IF
        CLOSE(1)
        IF (IERR .EQ. 0) THEN
          CALL DLASET('A', L-1, L-1, ZERO, ONE, A(L+1,2), LDA) 
          POS    = 2*L + 1           
          A(1,2) = - RWORK(POS) / RWORK(1)
          DO 140  I = 2, L
            TEMP  = RWORK(POS) / RWORK(I-1)
            TTEMP = RWORK(POS) / RWORK(I)
            IF (I .GT. 2)  A(I-1,I) = TEMP
            A(I,I)   = -(TEMP + TTEMP)
            IF (I .LT. L)  A(I+1,I) = TTEMP            
            POS = POS + 1
140       CONTINUE
          POS    = L  
          TEMP   = RWORK(POS+1) / RWORK(1)
          A(1,1) = -TEMP
          DO 160  I = 2, L
            TTEMP  = TEMP
            TEMP   = RWORK(POS+I) / RWORK(I)
            SUM = TTEMP - TEMP
            A(I,1) = -SUM
            A(I,I) = A(I,I) - TEMP
            DO 150  J = 2, I-2
              A(I,J) = SUM
150         CONTINUE
            IF (I .GT. 2)  A(I,I-1) = A(I,I-1) + SUM
160       CONTINUE
          POS      = 3*L
          A(1,L+1) = -RWORK(3*L)/RWORK(1)
          DO 170  I = 2, L
            TEMP  = RWORK(POS) / RWORK(I-1)
            TTEMP = RWORK(POS) / RWORK(I)
            IF (I .GT. 2)  A(I-1,L+I-1) = TEMP
            A(I,L+I-1)   = -(TEMP + TTEMP)
            IF (I .LT. L)  A(I+1,L+I-1) = TTEMP            
            POS = POS + 1
170       CONTINUE
          B(1,1) = ONE/RWORK(1)
          DO 180  I = 1, L
            TEMP = ONE/RWORK(I)
            IF (I .GT. 1)  B(I,I)   = -TEMP
            IF (I .LT. L)  B(I+1,I) =  TEMP
180       CONTINUE
          C(1,1) = ONE
          Q(1)   = ONE
          POS    = 2*L - 1
          ISYMM  = L + 1
          DO 190  I = 2, L
            TEMP       = RWORK(POS+I)
            TTEMP      = RWORK(POS+L+I-1)
            C(I,I)     = TEMP
            C(I,L+I-1) = TTEMP
            Q(ISYMM)   = ONE / (TEMP*TEMP + TTEMP*TTEMP)
            ISYMM      = ISYMM + L - I + 1
190       CONTINUE
          IDENT = '0001'
        END IF

      END IF
C
      IF (IERR .NE. 0)  GOTO 2001
C     .. set up data in required format ..
C
      IF (LSAME(FORM,'P') .OR. LSAME(FORM,'Q')) THEN
C     .. G is to be returned in product form ..
        GDIMM = N
        IF (IDENT(4:4) .EQ. '0') THEN
C       .. invert R using Cholesky factorization, store in G ..
          CALL DPPTRF('L', M, G, INFO)
          IF (INFO .EQ. 0) THEN
            CALL DPPTRI('L', M, G, INFO)
            IF (IDENT(1:1) .EQ. '0') THEN
C         .. B not identity matrix ..
              DO 200  I = 1, N
                CALL DSPMV('L', M, ONE, G, B(I,1), LDB, ZERO, 
     1                     RWORK((I-1)*N+1), 1)  
200           CONTINUE 
              CALL DGEMV('T', M, N, ONE, RWORK, N, B(1,1), LDB, ZERO,
     1                   G, 1) 
              ISYMM = N + 1
              DO 210  I = 2, N 
                CALL DGEMV('T', M, N, ONE, RWORK, N, B(I,1), LDB, ZERO,
     1                     B(1,1), LDB)
                CALL DCOPY(N - I + 1, B(1,I), LDB, G(ISYMM), 1)
                ISYMM = ISYMM + (N - I + 1)
210           CONTINUE 
            END IF
          ELSE
            IF (INFO .GT. 0)  IERR = 7
          END IF
        ELSE
C       .. R = identity ..
          IF (IDENT(1:1) .EQ. '0') THEN
C         .. B not identity matrix ..
            IF (M .EQ. 1) THEN
              CALL DLASET('L', NSYMM, 1, ZERO, ZERO, G, 1)
              CALL DSPR('L', N, ONE, B, 1, G)
            ELSE
              CALL DSYRK('L', 'N', N, M, ONE, B, LDB, ZERO, RWORK, N) 
              CALL SY2SP(N, RWORK, N, 'L', 'L', INFO)
              CALL DCOPY(NSYMM, RWORK, 1, G, 1)
            END IF
          ELSE
C         .. B = R = identity ..
            ISYMM = 1
            DO 220  I = N, 1, -1
              G(ISYMM) = ONE
              ISYMM = ISYMM + I
220         CONTINUE 
          END IF 
        END IF         
      ELSE
        GDIMM = M
        IF (IDENT(1:1) .EQ. '1') 
     1    CALL DLASET('A', N, M, ZERO, ONE, B, LDB) 
        IF (IDENT(4:4) .EQ. '1') THEN
          ISYMM = 1
          DO 230  I = M, 1, -1
            G(ISYMM) = ONE
            ISYMM = ISYMM + I
230       CONTINUE
        END IF 
      END IF
C 
      IF (LSAME(FORM,'P') .OR. LSAME(FORM,'G')) THEN
C     .. Q is to be returned in product form ..
        QDIMM = N
        IF (IDENT(3:3) .EQ. '0') THEN
          IF (IDENT(2:2) .EQ. '0') THEN
C         .. C not identity matrix ..
            DO 240  I = 1, N
              CALL DSPMV('L', P, ONE, Q, C(1,I), 1, ZERO, 
     1                   RWORK((I-1)*N+1), 1)  
240         CONTINUE 
C         .. use Q(1:N) as workspace and compute the first column of Q
C            in the end .. 
            ISYMM = N + 1
            DO 250  I = 2, N 
              CALL DGEMV('T', P, N, ONE, RWORK, N, C(1,I), 1, ZERO,
     1                   Q(1), 1)
              CALL DCOPY(N - I + 1, Q(I), 1, Q(ISYMM), 1)
              ISYMM = ISYMM + (N - I + 1)
250         CONTINUE 
            CALL DGEMV('T', P, N, ONE, RWORK, N, C(1,1), 1, ZERO, Q, 1) 
          END IF 
        ELSE
C       .. Q = identity ..
          IF (IDENT(2:2) .EQ. '0') THEN
C         .. C not identity matrix ..
            IF (P .EQ. 1) THEN
              CALL DLASET('L', NSYMM, 1, ZERO, ZERO, Q, 1)
              CALL DSPR('L', N, ONE, C, LDC, Q)
            ELSE
              CALL DSYRK('L', 'T', N, P, ONE, C, LDC, ZERO, RWORK, N)
              CALL SY2SP(N, RWORK, N, 'L', 'L', INFO)
              CALL DCOPY(NSYMM, RWORK, 1, Q, 1)
            END IF
          ELSE
C         .. C = Q = identity ..
            ISYMM = 1
            DO 260  I = N, 1, -1
              Q(ISYMM) = ONE
              ISYMM    = ISYMM + I
260         CONTINUE 
          END IF 
        END IF
      ELSE
        QDIMM = P
        IF (IDENT(2:2) .EQ. '1')
     1    CALL DLASET('A', P, N, ZERO, ONE, C, LDC)
        IF (IDENT(3:3) .EQ. '1') THEN
          ISYMM = 1
          DO 270  I = P, 1, -1
            Q(ISYMM) = ONE
            ISYMM    = ISYMM + I
270       CONTINUE 
        END IF 
      END IF
C
C     .. unpack symmetric matrices if desired ..
      IF (LSAME(STORE,'F')) THEN
        CALL SP2SY(GDIMM, G, LDG, 'L', INFO)
        CALL SP2SY(QDIMM, Q, LDQ, 'L', INFO)
      ELSE IF (LSAME(STORE,'U')) THEN
        ISYMM = GDIMM * (GDIMM + 1) / 2
        CALL DCOPY(ISYMM, G, 1, RWORK, 1)
        CALL SP2SY(GDIMM, RWORK, GDIMM, 'L', INFO)
        CALL SY2SP(GDIMM, RWORK, GDIMM, 'L', 'U', INFO)
        CALL DCOPY(ISYMM, RWORK, 1, G, 1)
        ISYMM = QDIMM * (QDIMM + 1) / 2
        CALL DCOPY(ISYMM, Q, 1, RWORK, 1)
        CALL SP2SY(QDIMM, RWORK, QDIMM, 'L', INFO)
        CALL SY2SP(QDIMM, RWORK, QDIMM, 'L', 'U', INFO)
        CALL DCOPY(ISYMM, RWORK, 1, Q, 1)
      END IF
C 
2001  CONTINUE
      RETURN
C *** Last Line of CAREX ***
      END
 

