      SUBROUTINE DTRMM( SIDE, UPLO, TRANSA, DIAG, M, N, ALPHA, A, LDA,
     $                   C, LDC )
*     .. Scalar Arguments ..
      CHARACTER*1        SIDE, UPLO, TRANSA, DIAG
      INTEGER            M, N, LDA, LDC
      DOUBLE PRECISION   ALPHA
*     .. Array Arguments ..
      DOUBLE PRECISION   A( LDA, * ), C( LDC, * )
*     ..
*
*  Purpose
*  =======
*
*  DTRMM  performs one of the matrix-matrix operations
*
*     C := alpha*op( A )*C,   or   C := alpha*C*op( A ),
*
*  where  alpha  is a scalar,  C  is an m by n matrix,  A  is a unit, or
*  non-unit,  upper or lower triangular matrix  and  op( A )  is one  of
*
*     op( A ) = A   or   op( A ) = A'.
*
*  Parameters
*  ==========
*
*  SIDE   - CHARACTER*1.
*           On entry,  SIDE specifies whether  op( A ) multiplies C from
*           the left or right as follows:
*
*              SIDE = 'L' or 'l'   C := alpha*op( A )*C.
*
*              SIDE = 'R' or 'r'   C := alpha*C*op( A ).
*
*           Unchanged on exit.
*
*  UPLO   - CHARACTER*1.
*           On entry, UPLO specifies whether the matrix A is an upper or
*           lower triangular matrix as follows:
*
*              UPLO = 'U' or 'u'   A is an upper triangular matrix.
*
*              UPLO = 'L' or 'l'   A is a lower triangular matrix.
*
*           Unchanged on exit.
*
*  TRANSA - CHARACTER*1.
*           On entry, TRANSA specifies the form of op( A ) to be used in
*           the matrix multiplication as follows:
*
*              TRANSA = 'N' or 'n'   op( A ) = A.
*
*              TRANSA = 'T' or 't'   op( A ) = A'.
*
*              TRANSA = 'C' or 'c'   op( A ) = A'.
*
*           Unchanged on exit.
*
*  DIAG   - CHARACTER*1.
*           On entry, DIAG specifies whether or not A is unit triangular
*           as follows:
*
*              DIAG = 'U' or 'u'   A is assumed to be unit triangular.
*
*              DIAG = 'N' or 'n'   A is not assumed to be unit
*                                  triangular.
*
*           Unchanged on exit.
*
*  M      - INTEGER.
*           On entry, M specifies the number of rows of C. M must be at
*           least zero.
*           Unchanged on exit.
*
*  N      - INTEGER.
*           On entry, N specifies the number of columns of C.  N must be
*           at least zero.
*           Unchanged on exit.
*
*  ALPHA  - DOUBLE PRECISION.
*           On entry,  ALPHA specifies the scalar  alpha. When  alpha is
*           zero then  A is not referenced and  C need not be set before
*           entry.
*           Unchanged on exit.
*
*  A      - DOUBLE PRECISION array of DIMENSION ( LDA, k ), where k is m
*           when  SIDE = 'L' or 'l'  and is  n  when  SIDE = 'R' or 'r'.
*           Before entry  with  UPLO = 'U' or 'u',  the  leading  k by k
*           upper triangular part of the array  A must contain the upper
*           triangular matrix  and the strictly lower triangular part of
*           A is not referenced.
*           Before entry  with  UPLO = 'L' or 'l',  the  leading  k by k
*           lower triangular part of the array  A must contain the lower
*           triangular matrix  and the strictly upper triangular part of
*           A is not referenced.
*           Note that when  DIAG = 'U' or 'u',  the diagonal elements of
*           A  are not referenced either,  but are assumed to be  unity.
*           Unchanged on exit.
*
*  LDA    - INTEGER.
*           On entry, LDA specifies the first dimension of A as declared
*           in the calling (sub) program.  When  SIDE = 'L' or 'l'  then
*           LDA  must be at least  max( 1, m ),  when  SIDE = 'R' or 'r'
*           then LDA must be at least max( 1, n ).
*           Unchanged on exit.
*
*  C      - DOUBLE PRECISION array of DIMENSION ( LDC, n ).
*           Before entry,  the leading  m by n part of the array  C must
*           contain the matrix  C,  and  on exit  is overwritten  by the
*           transformed matrix.
*
*  LDC    - INTEGER.
*           On entry, LDC specifies the first dimension of C as declared
*           in  the  calling  (sub)  program.   LDC  must  be  at  least
*           max( 1, m ).
*           Unchanged on exit.
*
*
*  Level 3 Blas routine.
*
*  -- Written on 8-February-1989.
*     Jack Dongarra, Argonne National Laboratory.
*     Iain Duff, AERE Harwell.
*     Jeremy Du Croz, Numerical Algorithms Group Ltd.
*     Sven Hammarling, Numerical Algorithms Group Ltd.
*
*  -- Rewritten in December-1993.
*     GEMM-Based Level 3 BLAS.
*     Per Ling, Institute of Information Processing,
*     University of Umea, Sweden.
*
*  -- Rewritten in Mars-1995.
*     Superscalar GEMM-Based Level 3 BLAS (Version 0.1).
*     Per Ling, Department of Computing Science,
*     University of Umea, Sweden.
*
*
*     .. Local Scalars ..
      INTEGER            INFO, NROWA
      LOGICAL            LSIDE, UPPER, NOTR, NOUNIT
      INTEGER            I, II, IX, ISEC, J, JJ, JX, JY, JSEC
      INTEGER            L, UJ, UJSEC, UISEC, RI, RISEC, RJSEC
      DOUBLE PRECISION   FR1, FR2, FR3, FR4, FR5, FR6, FR7, FR8
*     .. Intrinsic Functions ..
      INTRINSIC          MAX, MIN, MOD
*     .. External Functions ..
      LOGICAL            LSAME
      EXTERNAL           LSAME
*     .. External Subroutines ..
      EXTERNAL           XERBLA
      EXTERNAL           DGEMM
*     .. Parameters ..
      DOUBLE PRECISION   ZERO, ONE
      PARAMETER        ( ZERO = 0.0D+0, ONE = 1.0D+0 )
*     .. User specified parameters for DTRMM ..
      INTEGER            RCB, RB, CB
      PARAMETER        ( RCB = 32, RB = 32, CB = 32 )
      DOUBLE PRECISION   T1( RB, CB ), T2( RB, RB ), T3( RCB, RCB )
*     ..
*     .. Executable Statements ..
*
*     Test the input parameters.
*
      LSIDE = LSAME( SIDE, 'L' )
      UPPER = LSAME( UPLO, 'U' )
      NOTR = LSAME( TRANSA, 'N' )
      NOUNIT = LSAME( DIAG, 'N' )
      IF( LSIDE )THEN
         NROWA = M
      ELSE
         NROWA = N
      END IF
      INFO = 0
      IF( ( .NOT.LSIDE ).AND.( .NOT.LSAME( SIDE, 'R' ) ) )THEN
         INFO = 1
      ELSE IF( ( .NOT.UPPER ).AND.( .NOT.LSAME( UPLO, 'L' ) ) )THEN
         INFO = 2
      ELSE IF( ( .NOT.NOTR ).AND.( .NOT.LSAME( TRANSA, 'T' ) ).AND.
     $                               ( .NOT.LSAME( TRANSA, 'C' ) ) )THEN
         INFO = 3
      ELSE IF( ( .NOT.NOUNIT ).AND.( .NOT.LSAME( DIAG, 'U' ) ) )THEN
         INFO = 4
      ELSE IF( M.LT.0 )THEN
         INFO = 5
      ELSE IF( N.LT.0 )THEN
         INFO = 6
      ELSE IF( LDA.LT.MAX( 1, NROWA ) )THEN
         INFO = 9
      ELSE IF( LDC.LT.MAX( 1, M ) )THEN
         INFO = 11
      END IF
      IF( INFO.NE.0 )THEN
         CALL XERBLA( 'DTRMM ', INFO )
         RETURN
      END IF
*
*     Quick return if possible.
*
      IF( ( M.EQ.0 ).OR.( N.EQ.0 ) )
     $   RETURN
*
*     And when alpha.eq.zero.
*
      IF( ALPHA.EQ.ZERO )THEN
*
*        C := zero, set the elements of the rectangular M*N matrix C
*        to zero.
*
         UISEC = M-MOD( M, 4 )
         DO 30, J = 1, N
            DO 10, I = 1, UISEC, 4
               C( I, J ) = ZERO
               C( I+1, J ) = ZERO
               C( I+2, J ) = ZERO
               C( I+3, J ) = ZERO
   10       CONTINUE
            DO 20, I = UISEC+1, M
               C( I, J ) = ZERO
   20       CONTINUE
   30    CONTINUE
         RETURN
      END IF
*
*     Start the operations.
*
      IF( LSIDE )THEN
         IF( UPPER )THEN
            IF( NOTR )THEN
*
*              Form  C := alpha*A*C. Left, Upper, No transpose.
*
               DO 150, II = 1, M, RCB
                  ISEC = MIN( RCB, M-II+1 )
*
*                 T3 := alpha*A', copy the transpose of a upper
*                 triangular diagonal block of alpha*A to T3.
*
                  IF( NOUNIT )THEN
                     DO 50, L = II+ISEC-2, II-1, -2
                        UISEC = L-II+1-MOD( L-II+1, 2 )
                        DO 40, I = II, II+UISEC-1, 2
                           T3( L-II+1, I-II+1 ) = ALPHA*A( I, L )
                           T3( L-II+2, I-II+1 ) = ALPHA*A( I, L+1 )
                           T3( L-II+1, I-II+2 ) = ALPHA*A( I+1, L )
                           T3( L-II+2, I-II+2 ) = ALPHA*A( I+1, L+1 )
   40                   CONTINUE
                        IF( MOD( L-II+1, 2 ).EQ.1 )THEN
                           T3( L-II+1, L-II+1 ) = ALPHA*A( L, L )
                           T3( L-II+2, L-II+1 ) = ALPHA*A( L, L+1 )
                        END IF
                        T3( L-II+2, L-II+2 ) = ALPHA*A( L+1, L+1 )
   50                CONTINUE
                  ELSE
                     DO 70, L = II+ISEC-2, II-1, -2
                        UISEC = L-II-MOD( L-II, 2 )
                        DO 60, I = II, II+UISEC-1, 2
                           T3( L-II+1, I-II+1 ) = ALPHA*A( I, L )
                           T3( L-II+2, I-II+1 ) = ALPHA*A( I, L+1 )
                           T3( L-II+1, I-II+2 ) = ALPHA*A( I+1, L )
                           T3( L-II+2, I-II+2 ) = ALPHA*A( I+1, L+1 )
   60                   CONTINUE
                        IF( MOD( L-II, 2 ).EQ.1 )THEN
                           T3( L-II+1, L-II ) = ALPHA*A( L-1, L )
                           T3( L-II+2, L-II ) = ALPHA*A( L-1, L+1 )
                        END IF
                        IF( L-II.GE.0 )THEN
                           T3( L-II+1, L-II+1 ) = ALPHA
                           T3( L-II+2, L-II+1 ) = ALPHA*A( L, L+1 )
                        END IF
                        T3( L-II+2, L-II+2 ) = ALPHA
   70                CONTINUE
                  END IF
*
*                 C := alpha*T3'*C, update a rectangular block of C
*                 using the transpose of the upper triangular diagonal
*                 block of A stored in T3.
*
                  UISEC = ISEC-MOD( ISEC, 4 )
                  DO 140, JX = N, 1, -2
                     UJ = MAX( 1, JX-2+1 )
                     UJSEC = JX-UJ+1
*
*                    Four by two unrolling.
*
                     IF( UJSEC.EQ.2 )THEN
                        DO 90, I = II, II+UISEC-1, 4
                           FR1 = ZERO
                           FR2 = ZERO
                           FR3 = ZERO
                           FR4 = ZERO
                           FR5 = ZERO
                           FR6 = ZERO
                           FR7 = ZERO
                           FR8 = ZERO
                           DO 80, L = I+4, II+ISEC-1
                              FR1 = FR1 + T3( L-II+1, I-II+1 )*
     $                                                        C( L, UJ )
                              FR2 = FR2 + T3( L-II+1, I-II+1 )*
     $                                                      C( L, UJ+1 )
                              FR3 = FR3 + T3( L-II+1, I-II+2 )*
     $                                                        C( L, UJ )
                              FR4 = FR4 + T3( L-II+1, I-II+2 )*
     $                                                      C( L, UJ+1 )
                              FR5 = FR5 + T3( L-II+1, I-II+3 )*
     $                                                        C( L, UJ )
                              FR6 = FR6 + T3( L-II+1, I-II+3 )*
     $                                                      C( L, UJ+1 )
                              FR7 = FR7 + T3( L-II+1, I-II+4 )*
     $                                                        C( L, UJ )
                              FR8 = FR8 + T3( L-II+1, I-II+4 )*
     $                                                      C( L, UJ+1 )
   80                      CONTINUE
                           FR1 = FR1 + T3( I-II+1, I-II+1 )*C( I, UJ )
                           FR2 = FR2 + T3( I-II+1, I-II+1 )*C( I, UJ+1 )
                           FR1 = FR1 + T3( I-II+3, I-II+1 )*C( I+2, UJ )
                           FR2 = FR2 + T3( I-II+3, I-II+1 )*
     $                                                    C( I+2, UJ+1 )
                           FR1 = FR1 + T3( I-II+4, I-II+1 )*C( I+3, UJ )
                           FR2 = FR2 + T3( I-II+4, I-II+1 )*
     $                                                    C( I+3, UJ+1 )
                           FR1 = FR1 + T3( I-II+2, I-II+1 )*C( I+1, UJ )
                           FR2 = FR2 + T3( I-II+2, I-II+1 )*
     $                                                    C( I+1, UJ+1 )
                           C( I, UJ ) = FR1
                           C( I, UJ+1 ) = FR2
                           FR3 = FR3 + T3( I-II+2, I-II+2 )*C( I+1, UJ )
                           FR4 = FR4 + T3( I-II+2, I-II+2 )*
     $                                                    C( I+1, UJ+1 )
                           FR3 = FR3 + T3( I-II+4, I-II+2 )*C( I+3, UJ )
                           FR4 = FR4 + T3( I-II+4, I-II+2 )*
     $                                                    C( I+3, UJ+1 )
                           FR3 = FR3 + T3( I-II+3, I-II+2 )*C( I+2, UJ )
                           FR4 = FR4 + T3( I-II+3, I-II+2 )*
     $                                                    C( I+2, UJ+1 )
                           C( I+1, UJ ) = FR3
                           C( I+1, UJ+1 ) = FR4
                           FR5 = FR5 + T3( I-II+3, I-II+3 )*C( I+2, UJ )
                           FR6 = FR6 + T3( I-II+3, I-II+3 )*
     $                                                    C( I+2, UJ+1 )
                           FR5 = FR5 + T3( I-II+4, I-II+3 )*C( I+3, UJ )
                           FR6 = FR6 + T3( I-II+4, I-II+3 )*
     $                                                    C( I+3, UJ+1 )
                           C( I+2, UJ ) = FR5
                           C( I+2, UJ+1 ) = FR6
                           FR7 = FR7 + T3( I-II+4, I-II+4 )*C( I+3, UJ )
                           FR8 = FR8 + T3( I-II+4, I-II+4 )*
     $                                                    C( I+3, UJ+1 )
                           C( I+3, UJ ) = FR7
                           C( I+3, UJ+1 ) = FR8
   90                   CONTINUE
                        DO 110, RI = II+UISEC, II+ISEC-1
                           FR1 = ZERO
                           FR2 = ZERO
                           DO 100, L = RI, II+ISEC-1
                              FR1 = FR1 + T3( L-II+1, RI-II+1 )*
     $                                                        C( L, UJ )
                              FR2 = FR2 + T3( L-II+1, RI-II+1 )*
     $                                                      C( L, UJ+1 )
  100                      CONTINUE
                           C( RI, UJ ) = FR1
                           C( RI, UJ+1 ) = FR2
  110                   CONTINUE
                     ELSE IF( UJSEC.EQ.1 )THEN
                        DO 130, RI = II, II+ISEC-1
                           FR1 = ZERO
                           DO 120, L = RI, II+ISEC-1
                              FR1 = FR1 + T3( L-II+1, RI-II+1 )*
     $                                                        C( L, UJ )
  120                      CONTINUE
                           C( RI, UJ ) = FR1
  130                   CONTINUE
                     END IF
  140             CONTINUE
*
*                 C := alpha*A*C + C, general matrix multiply
*                 involving a rectangular block of A.
*
                  IF( II+ISEC.LE.M )THEN
                     CALL DGEMM ( 'N', 'N', ISEC, N, M-II-ISEC+1,
     $                                     ALPHA, A( II, II+ISEC ), LDA,
     $                                        C( II+ISEC, 1 ), LDC, ONE,
     $                                                 C( II, 1 ), LDC )
                  END IF
  150          CONTINUE
            ELSE
*
*              Form  C := alpha*A'*C. Left, Upper, Transpose.
*
               DO 270, II = M-MOD( M-1, RCB ), 1, -RCB
                  ISEC = MIN( RCB, M-II+1 )
*
*                 T3 := alpha*A, copy a upper triangular diagonal block
*                 of alpha*A to T3.
*
                  IF( NOUNIT )THEN
                     DO 170, L = II+ISEC-2, II-1, -2
                        UISEC = L-II+1-MOD( L-II+1, 2 )
                        DO 160, I = II, II+UISEC-1, 2
                           T3( I-II+1, L-II+1 ) = ALPHA*A( I, L )
                           T3( I-II+1, L-II+2 ) = ALPHA*A( I, L+1 )
                           T3( I-II+2, L-II+1 ) = ALPHA*A( I+1, L )
                           T3( I-II+2, L-II+2 ) = ALPHA*A( I+1, L+1 )
  160                   CONTINUE
                        IF( MOD( L-II+1, 2 ).EQ.1 )THEN
                           T3( L-II+1, L-II+1 ) = ALPHA*A( L, L )
                           T3( L-II+1, L-II+2 ) = ALPHA*A( L, L+1 )
                        END IF
                        T3( L-II+2, L-II+2 ) = ALPHA*A( L+1, L+1 )
  170                CONTINUE
                  ELSE
                     DO 190, L = II+ISEC-2, II-1, -2
                        UISEC = L-II-MOD( L-II, 2 )
                        DO 180, I = II, II+UISEC-1, 2
                           T3( I-II+1, L-II+1 ) = ALPHA*A( I, L )
                           T3( I-II+1, L-II+2 ) = ALPHA*A( I, L+1 )
                           T3( I-II+2, L-II+1 ) = ALPHA*A( I+1, L )
                           T3( I-II+2, L-II+2 ) = ALPHA*A( I+1, L+1 )
  180                   CONTINUE
                        IF( MOD( L-II, 2 ).EQ.1 )THEN
                           T3( L-II, L-II+1 ) = ALPHA*A( L-1, L )
                           T3( L-II, L-II+2 ) = ALPHA*A( L-1, L+1 )
                        END IF
                        IF( L-II.GE.0 )THEN
                           T3( L-II+1, L-II+1 ) = ALPHA
                           T3( L-II+1, L-II+2 ) = ALPHA*A( L, L+1 )
                        END IF
                        T3( L-II+2, L-II+2 ) = ALPHA
  190                CONTINUE
                  END IF
*
*                 C := alpha*T3'*C, update a rectangular block of C
*                 using the upper triangular diagonal block of A
*                 stored in T3'.
*
                  RISEC = MOD( ISEC, 4 )
                  DO 260, JX = N, 1, -2
                     UJ = MAX( 1, JX-2+1 )
                     UJSEC = JX-UJ+1
*
*                    Four by two unrolling.
*
                     IF( UJSEC.EQ.2 )THEN
                        DO 210, I = II+ISEC-4, II+RISEC, -4
                           FR1 = ZERO
                           FR2 = ZERO
                           FR3 = ZERO
                           FR4 = ZERO
                           FR5 = ZERO
                           FR6 = ZERO
                           FR7 = ZERO
                           FR8 = ZERO
                           DO 200, L = II, I-1
                              FR1 = FR1 + T3( L-II+1, I-II+1 )*
     $                                                        C( L, UJ )
                              FR2 = FR2 + T3( L-II+1, I-II+1 )*
     $                                                      C( L, UJ+1 )
                              FR3 = FR3 + T3( L-II+1, I-II+2 )*
     $                                                        C( L, UJ )
                              FR4 = FR4 + T3( L-II+1, I-II+2 )*
     $                                                      C( L, UJ+1 )
                              FR5 = FR5 + T3( L-II+1, I-II+3 )*
     $                                                        C( L, UJ )
                              FR6 = FR6 + T3( L-II+1, I-II+3 )*
     $                                                      C( L, UJ+1 )
                              FR7 = FR7 + T3( L-II+1, I-II+4 )*
     $                                                        C( L, UJ )
                              FR8 = FR8 + T3( L-II+1, I-II+4 )*
     $                                                      C( L, UJ+1 )
  200                      CONTINUE
                           FR7 = FR7 + T3( I-II+4, I-II+4 )*C( I+3, UJ )
                           FR8 = FR8 + T3( I-II+4, I-II+4 )*
     $                                                    C( I+3, UJ+1 )
                           FR7 = FR7 + T3( I-II+3, I-II+4 )*C( I+2, UJ )
                           FR8 = FR8 + T3( I-II+3, I-II+4 )*
     $                                                    C( I+2, UJ+1 )
                           FR7 = FR7 + T3( I-II+2, I-II+4 )*C( I+1, UJ )
                           FR8 = FR8 + T3( I-II+2, I-II+4 )*
     $                                                    C( I+1, UJ+1 )
                           FR7 = FR7 + T3( I-II+1, I-II+4 )*C( I, UJ )
                           FR8 = FR8 + T3( I-II+1, I-II+4 )*C( I, UJ+1 )
                           C( I+3, UJ ) = FR7
                           C( I+3, UJ+1 ) = FR8
                           FR5 = FR5 + T3( I-II+3, I-II+3 )*C( I+2, UJ )
                           FR6 = FR6 + T3( I-II+3, I-II+3 )*
     $                                                    C( I+2, UJ+1 )
                           FR5 = FR5 + T3( I-II+2, I-II+3 )*C( I+1, UJ )
                           FR6 = FR6 + T3( I-II+2, I-II+3 )*
     $                                                    C( I+1, UJ+1 )
                           FR5 = FR5 + T3( I-II+1, I-II+3 )*C( I, UJ )
                           FR6 = FR6 + T3( I-II+1, I-II+3 )*C( I, UJ+1 )
                           C( I+2, UJ ) = FR5
                           C( I+2, UJ+1 ) = FR6
                           FR3 = FR3 + T3( I-II+2, I-II+2 )*C( I+1, UJ )
                           FR4 = FR4 + T3( I-II+2, I-II+2 )*
     $                                                    C( I+1, UJ+1 )
                           FR3 = FR3 + T3( I-II+1, I-II+2 )*C( I, UJ )
                           FR4 = FR4 + T3( I-II+1, I-II+2 )*C( I, UJ+1 )
                           C( I+1, UJ ) = FR3
                           C( I+1, UJ+1 ) = FR4
                           FR1 = FR1 + T3( I-II+1, I-II+1 )*C( I, UJ )
                           FR2 = FR2 + T3( I-II+1, I-II+1 )*C( I, UJ+1 )
                           C( I, UJ ) = FR1
                           C( I, UJ+1 ) = FR2
  210                   CONTINUE
                        DO 230, RI = II+RISEC-1, II, -1
                           FR1 = ZERO
                           FR2 = ZERO
                           DO 220, L = II, RI
                              FR1 = FR1 + T3( L-II+1, RI-II+1 )*
     $                                                        C( L, UJ )
                              FR2 = FR2 + T3( L-II+1, RI-II+1 )*
     $                                                      C( L, UJ+1 )
  220                      CONTINUE
                           C( RI, UJ ) = FR1
                           C( RI, UJ+1 ) = FR2
  230                   CONTINUE
                     ELSE IF( UJSEC.EQ.1 )THEN
                        DO 250, RI = II+ISEC-1, II, -1
                           FR1 = ZERO
                           DO 240, L = II, RI
                              FR1 = FR1 + T3( L-II+1, RI-II+1 )*
     $                                                        C( L, UJ )
  240                      CONTINUE
                           C( RI, UJ ) = FR1
  250                   CONTINUE
                     END IF
  260             CONTINUE
*
*                 C := alpha*A'*C + C, general matrix multiply
*                 involving the transpose of a rectangular block
*                 of A.
*
                  IF( II.GT.1 )THEN
                     CALL DGEMM ( 'T', 'N', ISEC, N, II-1, ALPHA,
     $                                  A( 1, II ), LDA, C( 1, 1 ), LDC,
     $                                            ONE, C( II, 1 ), LDC )
                  END IF
  270          CONTINUE
            END IF
         ELSE
            IF( NOTR )THEN
*
*              Form  C := alpha*A'*C. Left, Lower, No transpose.
*
               DO 390, IX = M, 1, -RCB
                  II = MAX( 1, IX-RCB+1 )
                  ISEC = IX-II+1
*
*                 T3 := alpha*A', copy the transpose of a lower
*                 triangular diagonal block of alpha*A to T3.
*
                  IF( NOUNIT )THEN
                     DO 290, L = II, II+ISEC-1, 2
                        T3( L-II+1, L-II+1 ) = ALPHA*A( L, L )
                        RISEC = MOD( II+ISEC-L-1, 2 )+1
                        IF( RISEC.EQ.2 )THEN
                           T3( L-II+1, L-II+2 ) = ALPHA*A( L+1, L )
                           T3( L-II+2, L-II+2 ) = ALPHA*A( L+1, L+1 )
                        END IF
                        DO 280, I = L+RISEC, II+ISEC-1, 2
                           T3( L-II+1, I-II+1 ) = ALPHA*A( I, L )
                           T3( L-II+2, I-II+1 ) = ALPHA*A( I, L+1 )
                           T3( L-II+1, I-II+2 ) = ALPHA*A( I+1, L )
                           T3( L-II+2, I-II+2 ) = ALPHA*A( I+1, L+1 )
  280                   CONTINUE
  290                CONTINUE
                  ELSE
                     DO 310, L = II, II+ISEC-1, 2
                        T3( L-II+1, L-II+1 ) = ALPHA
                        RISEC = MOD( II+ISEC-L-2, 2 )+2
                        IF( RISEC.GE.2 )THEN
                           T3( L-II+1, L-II+2 ) = ALPHA*A( L+1, L )
                           T3( L-II+2, L-II+2 ) = ALPHA
                        END IF
                        IF( RISEC.EQ.3 )THEN
                           T3( L-II+1, L-II+3 ) = ALPHA*A( L+2, L )
                           T3( L-II+2, L-II+3 ) = ALPHA*A( L+2, L+1 )
                        END IF
                        DO 300, I = L+RISEC, II+ISEC-1, 2
                           T3( L-II+1, I-II+1 ) = ALPHA*A( I, L )
                           T3( L-II+2, I-II+1 ) = ALPHA*A( I, L+1 )
                           T3( L-II+1, I-II+2 ) = ALPHA*A( I+1, L )
                           T3( L-II+2, I-II+2 ) = ALPHA*A( I+1, L+1 )
  300                   CONTINUE
  310                CONTINUE
                  END IF
*
*                 C := alpha*T3*C, update a rectangular block of C
*                 using the transpose of the lower triangular diagonal
*                 block of A stored in T3.
*
                  RISEC = MOD( ISEC, 4 )
                  DO 380, JX = N, 1, -2
                     UJ = MAX( 1, JX-2+1 )
                     UJSEC = JX-UJ+1
*
*                    Four by two unrolling.
*
                     IF( UJSEC.EQ.2 )THEN
                        DO 330, I = II+ISEC-4, II+RISEC, -4
                           FR1 = ZERO
                           FR2 = ZERO
                           FR3 = ZERO
                           FR4 = ZERO
                           FR5 = ZERO
                           FR6 = ZERO
                           FR7 = ZERO
                           FR8 = ZERO
                           DO 320, L = II, I-1
                              FR1 = FR1 + T3( L-II+1, I-II+1 )*
     $                                                        C( L, UJ )
                              FR2 = FR2 + T3( L-II+1, I-II+1 )*
     $                                                      C( L, UJ+1 )
                              FR3 = FR3 + T3( L-II+1, I-II+2 )*
     $                                                        C( L, UJ )
                              FR4 = FR4 + T3( L-II+1, I-II+2 )*
     $                                                      C( L, UJ+1 )
                              FR5 = FR5 + T3( L-II+1, I-II+3 )*
     $                                                        C( L, UJ )
                              FR6 = FR6 + T3( L-II+1, I-II+3 )*
     $                                                      C( L, UJ+1 )
                              FR7 = FR7 + T3( L-II+1, I-II+4 )*
     $                                                        C( L, UJ )
                              FR8 = FR8 + T3( L-II+1, I-II+4 )*
     $                                                      C( L, UJ+1 )
  320                      CONTINUE
                           FR7 = FR7 + T3( I-II+4, I-II+4 )*C( I+3, UJ )
                           FR8 = FR8 + T3( I-II+4, I-II+4 )*
     $                                                    C( I+3, UJ+1 )
                           FR7 = FR7 + T3( I-II+3, I-II+4 )*C( I+2, UJ )
                           FR8 = FR8 + T3( I-II+3, I-II+4 )*
     $                                                    C( I+2, UJ+1 )
                           FR7 = FR7 + T3( I-II+2, I-II+4 )*C( I+1, UJ )
                           FR8 = FR8 + T3( I-II+2, I-II+4 )*
     $                                                    C( I+1, UJ+1 )
                           FR7 = FR7 + T3( I-II+1, I-II+4 )*C( I, UJ )
                           FR8 = FR8 + T3( I-II+1, I-II+4 )*C( I, UJ+1 )
                           C( I+3, UJ ) = FR7
                           C( I+3, UJ+1 ) = FR8
                           FR5 = FR5 + T3( I-II+3, I-II+3 )*C( I+2, UJ )
                           FR6 = FR6 + T3( I-II+3, I-II+3 )*
     $                                                    C( I+2, UJ+1 )
                           FR5 = FR5 + T3( I-II+2, I-II+3 )*C( I+1, UJ )
                           FR6 = FR6 + T3( I-II+2, I-II+3 )*
     $                                                    C( I+1, UJ+1 )
                           FR5 = FR5 + T3( I-II+1, I-II+3 )*C( I, UJ )
                           FR6 = FR6 + T3( I-II+1, I-II+3 )*C( I, UJ+1 )
                           C( I+2, UJ ) = FR5
                           C( I+2, UJ+1 ) = FR6
                           FR3 = FR3 + T3( I-II+2, I-II+2 )*C( I+1, UJ )
                           FR4 = FR4 + T3( I-II+2, I-II+2 )*
     $                                                    C( I+1, UJ+1 )
                           FR3 = FR3 + T3( I-II+1, I-II+2 )*C( I, UJ )
                           FR4 = FR4 + T3( I-II+1, I-II+2 )*C( I, UJ+1 )
                           C( I+1, UJ ) = FR3
                           C( I+1, UJ+1 ) = FR4
                           FR1 = FR1 + T3( I-II+1, I-II+1 )*C( I, UJ )
                           FR2 = FR2 + T3( I-II+1, I-II+1 )*C( I, UJ+1 )
                           C( I, UJ ) = FR1
                           C( I, UJ+1 ) = FR2
  330                   CONTINUE
                        DO 350, RI = II+RISEC-1, II, -1
                           FR1 = ZERO
                           FR2 = ZERO
                           DO 340, L = II, RI
                              FR1 = FR1 + T3( L-II+1, RI-II+1 )*
     $                                                        C( L, UJ )
                              FR2 = FR2 + T3( L-II+1, RI-II+1 )*
     $                                                      C( L, UJ+1 )
  340                      CONTINUE
                           C( RI, UJ ) = FR1
                           C( RI, UJ+1 ) = FR2
  350                   CONTINUE
                     ELSE IF( UJSEC.EQ.1 )THEN
                        DO 370, RI = II+ISEC-1, II, -1
                           FR1 = ZERO
                           DO 360, L = II, RI
                              FR1 = FR1 + T3( L-II+1, RI-II+1 )*
     $                                                        C( L, UJ )
  360                      CONTINUE
                           C( RI, UJ ) = FR1
  370                   CONTINUE
                     END IF
  380             CONTINUE
*
*                 C := alpha*A'*C + C, general matrix multiply
*                 involving a rectangular block of A.
*
                  IF( II.GT.1 )THEN
                     CALL DGEMM ( 'N', 'N', ISEC, N, II-1, ALPHA,
     $                                  A( II, 1 ), LDA, C( 1, 1 ), LDC,
     $                                            ONE, C( II, 1 ), LDC )
                  END IF
  390          CONTINUE
            ELSE
*
*              Form  C := alpha*A'*C. Left, Lower, Transpose.
*
               DO 510, IX = MOD( M-1, RCB )+1, M, RCB
                  II = MAX( 1, IX-RCB+1 )
                  ISEC = IX-II+1
*
*                 T3 := alpha*A, copy a lower triangular diagonal block
*                 of alpha*A to T3.
*
                  IF( NOUNIT )THEN
                     DO 410, L = II, II+ISEC-1, 2
                        T3( L-II+1, L-II+1 ) = ALPHA*A( L, L )
                        RISEC = MOD( II+ISEC-L-1, 2 )+1
                        IF( RISEC.EQ.2 )THEN
                           T3( L-II+2, L-II+1 ) = ALPHA*A( L+1, L )
                           T3( L-II+2, L-II+2 ) = ALPHA*A( L+1, L+1 )
                        END IF
                        DO 400, I = L+RISEC, II+ISEC-1, 2
                           T3( I-II+1, L-II+1 ) = ALPHA*A( I, L )
                           T3( I-II+1, L-II+2 ) = ALPHA*A( I, L+1 )
                           T3( I-II+2, L-II+1 ) = ALPHA*A( I+1, L )
                           T3( I-II+2, L-II+2 ) = ALPHA*A( I+1, L+1 )
  400                   CONTINUE
  410                CONTINUE
                  ELSE
                     DO 430, L = II, II+ISEC-1, 2
                        T3( L-II+1, L-II+1 ) = ALPHA
                        RISEC = MOD( II+ISEC-L-2, 2 )+2
                        IF( RISEC.GE.2 )THEN
                           T3( L-II+2, L-II+1 ) = ALPHA*A( L+1, L )
                           T3( L-II+2, L-II+2 ) = ALPHA
                        END IF
                        IF( RISEC.EQ.3 )THEN
                           T3( L-II+3, L-II+1 ) = ALPHA*A( L+2, L )
                           T3( L-II+3, L-II+2 ) = ALPHA*A( L+2, L+1 )
                        END IF
                        DO 420, I = L+RISEC, II+ISEC-1, 2
                           T3( I-II+1, L-II+1 ) = ALPHA*A( I, L )
                           T3( I-II+1, L-II+2 ) = ALPHA*A( I, L+1 )
                           T3( I-II+2, L-II+1 ) = ALPHA*A( I+1, L )
                           T3( I-II+2, L-II+2 ) = ALPHA*A( I+1, L+1 )
  420                   CONTINUE
  430                CONTINUE
                  END IF
*
*                 C := alpha*T3'*C, update a rectangular block of C
*                 using the lower triangular diagonal block of A
*                 stored in T3'.
*
                  UISEC = ISEC-MOD( ISEC, 4 )
                  DO 500, JX = N, 1, -2
                     UJ = MAX( 1, JX-2+1 )
                     UJSEC = JX-UJ+1
*
*                    Four by two unrolling.
*
                     IF( UJSEC.EQ.2 )THEN
                        DO 450, I = II, II+UISEC-1, 4
                           FR1 = ZERO
                           FR2 = ZERO
                           FR3 = ZERO
                           FR4 = ZERO
                           FR5 = ZERO
                           FR6 = ZERO
                           FR7 = ZERO
                           FR8 = ZERO
                           DO 440, L = I+4, II+ISEC-1
                              FR1 = FR1 + T3( L-II+1, I-II+1 )*
     $                                                        C( L, UJ )
                              FR2 = FR2 + T3( L-II+1, I-II+1 )*
     $                                                      C( L, UJ+1 )
                              FR3 = FR3 + T3( L-II+1, I-II+2 )*
     $                                                        C( L, UJ )
                              FR4 = FR4 + T3( L-II+1, I-II+2 )*
     $                                                      C( L, UJ+1 )
                              FR5 = FR5 + T3( L-II+1, I-II+3 )*
     $                                                        C( L, UJ )
                              FR6 = FR6 + T3( L-II+1, I-II+3 )*
     $                                                      C( L, UJ+1 )
                              FR7 = FR7 + T3( L-II+1, I-II+4 )*
     $                                                        C( L, UJ )
                              FR8 = FR8 + T3( L-II+1, I-II+4 )*
     $                                                      C( L, UJ+1 )
  440                      CONTINUE
                           FR1 = FR1 + T3( I-II+1, I-II+1 )*C( I, UJ )
                           FR2 = FR2 + T3( I-II+1, I-II+1 )*C( I, UJ+1 )
                           FR1 = FR1 + T3( I-II+3, I-II+1 )*C( I+2, UJ )
                           FR2 = FR2 + T3( I-II+3, I-II+1 )*
     $                                                    C( I+2, UJ+1 )
                           FR1 = FR1 + T3( I-II+4, I-II+1 )*C( I+3, UJ )
                           FR2 = FR2 + T3( I-II+4, I-II+1 )*
     $                                                    C( I+3, UJ+1 )
                           FR1 = FR1 + T3( I-II+2, I-II+1 )*C( I+1, UJ )
                           FR2 = FR2 + T3( I-II+2, I-II+1 )*
     $                                                    C( I+1, UJ+1 )
                           C( I, UJ ) = FR1
                           C( I, UJ+1 ) = FR2
                           FR3 = FR3 + T3( I-II+2, I-II+2 )*C( I+1, UJ )
                           FR4 = FR4 + T3( I-II+2, I-II+2 )*
     $                                                    C( I+1, UJ+1 )
                           FR3 = FR3 + T3( I-II+4, I-II+2 )*C( I+3, UJ )
                           FR4 = FR4 + T3( I-II+4, I-II+2 )*
     $                                                    C( I+3, UJ+1 )
                           FR3 = FR3 + T3( I-II+3, I-II+2 )*C( I+2, UJ )
                           FR4 = FR4 + T3( I-II+3, I-II+2 )*
     $                                                    C( I+2, UJ+1 )
                           C( I+1, UJ ) = FR3
                           C( I+1, UJ+1 ) = FR4
                           FR5 = FR5 + T3( I-II+3, I-II+3 )*C( I+2, UJ )
                           FR6 = FR6 + T3( I-II+3, I-II+3 )*
     $                                                    C( I+2, UJ+1 )
                           FR5 = FR5 + T3( I-II+4, I-II+3 )*C( I+3, UJ )
                           FR6 = FR6 + T3( I-II+4, I-II+3 )*
     $                                                    C( I+3, UJ+1 )
                           C( I+2, UJ ) = FR5
                           C( I+2, UJ+1 ) = FR6
                           FR7 = FR7 + T3( I-II+4, I-II+4 )*C( I+3, UJ )
                           FR8 = FR8 + T3( I-II+4, I-II+4 )*
     $                                                    C( I+3, UJ+1 )
                           C( I+3, UJ ) = FR7
                           C( I+3, UJ+1 ) = FR8
  450                   CONTINUE
                        DO 470, RI = II+UISEC, II+ISEC-1
                           FR1 = ZERO
                           FR2 = ZERO
                           DO 460, L = RI, II+ISEC-1
                              FR1 = FR1 + T3( L-II+1, RI-II+1 )*
     $                                                        C( L, UJ )
                              FR2 = FR2 + T3( L-II+1, RI-II+1 )*
     $                                                      C( L, UJ+1 )
  460                      CONTINUE
                           C( RI, UJ ) = FR1
                           C( RI, UJ+1 ) = FR2
  470                   CONTINUE
                     ELSE IF( UJSEC.EQ.1 )THEN
                        DO 490, RI = II, II+ISEC-1
                           FR1 = ZERO
                           DO 480, L = RI, II+ISEC-1
                              FR1 = FR1 + T3( L-II+1, RI-II+1 )*
     $                                                        C( L, UJ )
  480                      CONTINUE
                           C( RI, UJ ) = FR1
  490                   CONTINUE
                     END IF
  500             CONTINUE
*
*                 C := alpha*A'*C + C, general matrix multiply
*                 involving the transpose of a rectangular block
*                 of A.
*
                  IF( II+ISEC.LE.M )THEN
                     CALL DGEMM ( 'T', 'N', ISEC, N, M-II-ISEC+1,
     $                                     ALPHA, A( II+ISEC, II ), LDA,
     $                                        C( II+ISEC, 1 ), LDC, ONE,
     $                                                 C( II, 1 ), LDC )
                  END IF
  510          CONTINUE
            END IF
         END IF
      ELSE
         IF( UPPER )THEN
            IF( NOTR )THEN
*
*              Form  C := alpha*C*A. Right, Upper, No transpose.
*
               DO 630, JJ = N-MOD( N-1, RB ), 1, -RB
                  JSEC = MIN( RB, N-JJ+1 )
                  DO 620, II = 1, M, CB
                     ISEC = MIN( CB, M-II+1 )
*
*                    T1 := alpha*C', copy the transpose of a rectangular
*                    block of alpha*C to T1.
*
                     UISEC = ISEC-MOD( ISEC, 2 )
                     UJSEC = JSEC-MOD( JSEC, 2 )
                     DO 530, J = JJ, JJ+UJSEC-1, 2
                        DO 520, I = II, II+UISEC-1, 2
                           T1( J-JJ+1, I-II+1 ) = ALPHA*C( I, J )
                           T1( J-JJ+2, I-II+1 ) = ALPHA*C( I, J+1 )
                           T1( J-JJ+1, I-II+2 ) = ALPHA*C( I+1, J )
                           T1( J-JJ+2, I-II+2 ) = ALPHA*C( I+1, J+1 )
  520                   CONTINUE
                        IF( UISEC.LT.ISEC )THEN
                           T1( J-JJ+1, ISEC ) = ALPHA*C( II+ISEC-1, J )
                           T1( J-JJ+2, ISEC ) =
     $                                        ALPHA*C( II+ISEC-1, J+1 )
                        END IF
  530                CONTINUE
                     IF( UJSEC.LT.JSEC )THEN
                        DO 540, I = II, II+ISEC-1
                           T1( JSEC, I-II+1 ) = ALPHA*C( I, JJ+JSEC-1 )
  540                   CONTINUE
                     END IF
*
*                    C := alpha*T1'*A, update a rectangular block of C
*                    using the transpose of a rectangular block of C
*                    stored in T1'.
*
                     UISEC = ISEC-MOD( ISEC, 4 )
                     DO 610, JX = JJ+JSEC-1, JJ, -2
                        UJ = MAX( JJ, JX-2+1 )
                        UJSEC = JX-UJ+1
*
*                       Four by two unrolling.
*
                        IF( UJSEC.EQ.2 )THEN
                           DO 560, I = II, II+UISEC-1, 4
                              FR1 = ZERO
                              FR2 = ZERO
                              FR3 = ZERO
                              FR4 = ZERO
                              FR5 = ZERO
                              FR6 = ZERO
                              FR7 = ZERO
                              FR8 = ZERO
                              DO 550, L = JJ, UJ-1
                                 FR1 = FR1 + T1( L-JJ+1, I-II+1 )*
     $                                                        A( L, UJ )
                                 FR2 = FR2 + T1( L-JJ+1, I-II+1 )*
     $                                                      A( L, UJ+1 )
                                 FR3 = FR3 + T1( L-JJ+1, I-II+2 )*
     $                                                        A( L, UJ )
                                 FR4 = FR4 + T1( L-JJ+1, I-II+2 )*
     $                                                      A( L, UJ+1 )
                                 FR5 = FR5 + T1( L-JJ+1, I-II+3 )*
     $                                                        A( L, UJ )
                                 FR6 = FR6 + T1( L-JJ+1, I-II+3 )*
     $                                                      A( L, UJ+1 )
                                 FR7 = FR7 + T1( L-JJ+1, I-II+4 )*
     $                                                        A( L, UJ )
                                 FR8 = FR8 + T1( L-JJ+1, I-II+4 )*
     $                                                      A( L, UJ+1 )
  550                         CONTINUE
                              FR2 = FR2 + T1( UJ-JJ+1, I-II+1 )*
     $                                                     A( UJ, UJ+1 )
                              FR4 = FR4 + T1( UJ-JJ+1, I-II+2 )*
     $                                                     A( UJ, UJ+1 )
                              FR6 = FR6 + T1( UJ-JJ+1, I-II+3 )*
     $                                                     A( UJ, UJ+1 )
                              FR8 = FR8 + T1( UJ-JJ+1, I-II+4 )*
     $                                                     A( UJ, UJ+1 )
                              IF( NOUNIT )THEN
                                 FR1 = FR1 + T1( UJ-JJ+1, I-II+1 )*
     $                                                       A( UJ, UJ )
                                 FR2 = FR2 + T1( UJ-JJ+2, I-II+1 )*
     $                                                   A( UJ+1, UJ+1 )
                                 C( I, UJ ) = FR1
                                 C( I, UJ+1 ) = FR2
                                 FR3 = FR3 + T1( UJ-JJ+1, I-II+2 )*
     $                                                       A( UJ, UJ )
                                 FR4 = FR4 + T1( UJ-JJ+2, I-II+2 )*
     $                                                   A( UJ+1, UJ+1 )
                                 C( I+1, UJ ) = FR3
                                 C( I+1, UJ+1 ) = FR4
                                 FR5 = FR5 + T1( UJ-JJ+1, I-II+3 )*
     $                                                       A( UJ, UJ )
                                 FR6 = FR6 + T1( UJ-JJ+2, I-II+3 )*
     $                                                   A( UJ+1, UJ+1 )
                                 C( I+2, UJ ) = FR5
                                 C( I+2, UJ+1 ) = FR6
                                 FR7 = FR7 + T1( UJ-JJ+1, I-II+4 )*
     $                                                       A( UJ, UJ )
                                 FR8 = FR8 + T1( UJ-JJ+2, I-II+4 )*
     $                                                   A( UJ+1, UJ+1 )
                                 C( I+3, UJ ) = FR7
                                 C( I+3, UJ+1 ) = FR8
                              ELSE
                                 C( I, UJ ) = FR1 +
     $                                             T1( UJ-JJ+1, I-II+1 )
                                 C( I, UJ+1 ) = FR2 +
     $                                             T1( UJ-JJ+2, I-II+1 )
                                 C( I+1, UJ ) = FR3 +
     $                                             T1( UJ-JJ+1, I-II+2 )
                                 C( I+1, UJ+1 ) = FR4 +
     $                                             T1( UJ-JJ+2, I-II+2 )
                                 C( I+2, UJ ) = FR5 +
     $                                             T1( UJ-JJ+1, I-II+3 )
                                 C( I+2, UJ+1 ) = FR6 +
     $                                             T1( UJ-JJ+2, I-II+3 )
                                 C( I+3, UJ ) = FR7 +
     $                                             T1( UJ-JJ+1, I-II+4 )
                                 C( I+3, UJ+1 ) = FR8 +
     $                                             T1( UJ-JJ+2, I-II+4 )
                              END IF
  560                      CONTINUE
                           DO 580, RI = II+UISEC, II+ISEC-1
                              FR1 = ZERO
                              FR2 = ZERO
                              DO 570, L = JJ, UJ-1
                                 FR1 = FR1 + T1( L-JJ+1, RI-II+1 )*
     $                                                        A( L, UJ )
                                 FR2 = FR2 + T1( L-JJ+1, RI-II+1 )*
     $                                                      A( L, UJ+1 )
  570                         CONTINUE
                              FR2 = FR2 + T1( UJ-JJ+1, RI-II+1 )*
     $                                                     A( UJ, UJ+1 )
                              IF( NOUNIT )THEN
                                 FR1 = FR1 + T1( UJ-JJ+1, RI-II+1 )*
     $                                                       A( UJ, UJ )
                                 FR2 = FR2 + T1( UJ-JJ+2, RI-II+1 )*
     $                                                   A( UJ+1, UJ+1 )
                              ELSE
                                 FR1 = FR1 + T1( UJ-JJ+1, RI-II+1 )
                                 FR2 = FR2 + T1( UJ-JJ+2, RI-II+1 )
                              END IF
                              C( RI, UJ ) = FR1
                              C( RI, UJ+1 ) = FR2
  580                      CONTINUE
                        ELSE IF( UJSEC.EQ.1 )THEN
                           DO 600, RI = II, II+ISEC-1
                              FR1 = ZERO
                              DO 590, L = JJ, UJ-1
                                 FR1 = FR1 + T1( L-JJ+1, RI-II+1 )*
     $                                                        A( L, UJ )
  590                         CONTINUE
                              IF( NOUNIT )THEN
                                 FR1 = FR1 + T1( UJ-JJ+1, RI-II+1 )*
     $                                                       A( UJ, UJ )
                              ELSE
                                 FR1 = FR1 + T1( UJ-JJ+1, RI-II+1 )
                              END IF
                              C( RI, UJ ) = FR1
  600                      CONTINUE
                        END IF
  610                CONTINUE
  620             CONTINUE
*
*                 C := alpha*C*A + C, general matrix multiply
*                 involving a rectangular block of A.
*
                  IF( II.GT.1 )THEN
                     CALL DGEMM ( 'N', 'N', M, JSEC, JJ-1, ALPHA,
     $                                  C( 1, 1 ), LDC, A( 1, JJ ), LDA,
     $                                            ONE, C( 1, JJ ), LDC )
                  END IF
  630          CONTINUE
            ELSE
*
*              Form  C := alpha*C*A. Right, Upper, Transpose.
*
               DO 790, JJ = 1, N, RB
                  JSEC = MIN( RB, N-JJ+1 )
*
*                 T2 := alpha*A', copy the transpose of a upper
*                 triangular diagonal block of alpha*A to T2.
*
                  IF( NOUNIT )THEN
                     DO 650, J = JJ+JSEC-2, JJ-1, -2
                        UJSEC = J-JJ+1 - MOD( J-JJ+1, 2 )
                        DO 640, L = JJ, JJ+UJSEC-1, 2
                           T2( J-JJ+1, L-JJ+1 ) = ALPHA*A( L, J )
                           T2( J-JJ+2, L-JJ+1 ) = ALPHA*A( L, J+1 )
                           T2( J-JJ+1, L-JJ+2 ) = ALPHA*A( L+1, J )
                           T2( J-JJ+2, L-JJ+2 ) = ALPHA*A( L+1, J+1 )
  640                   CONTINUE
                        IF( MOD( J-JJ+1, 2 ).EQ.1 )THEN
                           T2( J-JJ+1, J-JJ+1 ) = ALPHA*A( J, J )
                           T2( J-JJ+2, J-JJ+1 ) = ALPHA*A( J, J+1 )
                        END IF
                        T2( J-JJ+2, J-JJ+2 ) = ALPHA*A( J+1, J+1 )
  650                CONTINUE
                  ELSE
                     DO 670, J = JJ+JSEC-2, JJ-1, -2
                        UJSEC = J-JJ - MOD( J-JJ, 2 )
                        DO 660, L = JJ, JJ+UJSEC-1, 2
                           T2( J-JJ+1, L-JJ+1 ) = ALPHA*A( L, J )
                           T2( J-JJ+2, L-JJ+1 ) = ALPHA*A( L, J+1 )
                           T2( J-JJ+1, L-JJ+2 ) = ALPHA*A( L+1, J )
                           T2( J-JJ+2, L-JJ+2 ) = ALPHA*A( L+1, J+1 )
  660                   CONTINUE
                        IF( MOD( J-JJ, 2 ).EQ.1 )THEN
                           T2( J-JJ+1, J-JJ ) = ALPHA*A( J-1, J )
                           T2( J-JJ+2, J-JJ ) = ALPHA*A( J-1, J+1 )
                        END IF
                        IF( J-JJ.GE.0 )THEN
                           T2( J-JJ+1, J-JJ+1 ) = ALPHA
                           T2( J-JJ+2, J-JJ+1 ) = ALPHA*A( J, J+1 )
                        END IF
                        T2( J-JJ+2, J-JJ+2 ) = ALPHA
  670                CONTINUE
                  END IF
                  DO 780, II = 1, M, CB
                     ISEC = MIN( CB, M-II+1 )
*
*                    T1 := C', copy the transpose of a rectangular block
*                    of C to T1.
*
                     UISEC = ISEC-MOD( ISEC, 2 )
                     UJSEC = JSEC-MOD( JSEC, 2 )
                     DO 690, J = JJ, JJ+UJSEC-1, 2
                        DO 680, I = II, II+UISEC-1, 2
                           T1( J-JJ+1, I-II+1 ) = C( I, J )
                           T1( J-JJ+2, I-II+1 ) = C( I, J+1 )
                           T1( J-JJ+1, I-II+2 ) = C( I+1, J )
                           T1( J-JJ+2, I-II+2 ) = C( I+1, J+1 )
  680                   CONTINUE
                        IF( UISEC.LT.ISEC )THEN
                           T1( J-JJ+1, ISEC ) = C( II+ISEC-1, J )
                           T1( J-JJ+2, ISEC ) = C( II+ISEC-1, J+1 )
                        END IF
  690                CONTINUE
                     IF( UJSEC.LT.JSEC )THEN
                        DO 700, I = II, II+ISEC-1
                           T1( JSEC, I-II+1 ) = C( I, JJ+JSEC-1 )
  700                   CONTINUE
                     END IF
*
*                    C := alpha*T1'*T2, update a rectangular block of C
*                    using the transpose of a rectangular block of C
*                    stored in T1' and the transpose of a upper
*                    triangular diagonal block of A stored in T2.
*
                     RISEC = MOD( ISEC, 4 )
                     DO 770, JY = JJ+JSEC-1, JJ, -2
                        UJ = MAX( JJ, JY-2+1 )
                        UJSEC = JY-UJ+1
*
*                       Four by two unrolling.
*
                        IF( UJSEC.EQ.2 )THEN
                           DO 720, I = II+ISEC-4, II+RISEC, -4
                              FR1 = ZERO
                              FR2 = ZERO
                              FR3 = ZERO
                              FR4 = ZERO
                              FR5 = ZERO
                              FR6 = ZERO
                              FR7 = ZERO
                              FR8 = ZERO
                              DO 710, L = UJ+2, JJ+JSEC-1
                                 FR1 = FR1 + T1( L-JJ+1, I-II+1 )*
     $                                             T2( L-JJ+1, UJ-JJ+1 )
                                 FR2 = FR2 + T1( L-JJ+1, I-II+1 )*
     $                                             T2( L-JJ+1, UJ-JJ+2 )
                                 FR3 = FR3 + T1( L-JJ+1, I-II+2 )*
     $                                             T2( L-JJ+1, UJ-JJ+1 )
                                 FR4 = FR4 + T1( L-JJ+1, I-II+2 )*
     $                                             T2( L-JJ+1, UJ-JJ+2 )
                                 FR5 = FR5 + T1( L-JJ+1, I-II+3 )*
     $                                             T2( L-JJ+1, UJ-JJ+1 )
                                 FR6 = FR6 + T1( L-JJ+1, I-II+3 )*
     $                                             T2( L-JJ+1, UJ-JJ+2 )
                                 FR7 = FR7 + T1( L-JJ+1, I-II+4 )*
     $                                             T2( L-JJ+1, UJ-JJ+1 )
                                 FR8 = FR8 + T1( L-JJ+1, I-II+4 )*
     $                                             T2( L-JJ+1, UJ-JJ+2 )
  710                         CONTINUE
                              FR1 = FR1 + T1( UJ-JJ+1, I-II+1 )*
     $                                            T2( UJ-JJ+1, UJ-JJ+1 )
                              FR1 = FR1 + T1( UJ-JJ+2, I-II+1 )*
     $                                            T2( UJ-JJ+2, UJ-JJ+1 )
                              FR2 = FR2 + T1( UJ-JJ+2, I-II+1 )*
     $                                            T2( UJ-JJ+2, UJ-JJ+2 )
                              C( I, UJ ) = FR1
                              C( I, UJ+1 ) = FR2
                              FR3 = FR3 + T1( UJ-JJ+1, I-II+2 )*
     $                                            T2( UJ-JJ+1, UJ-JJ+1 )
                              FR3 = FR3 + T1( UJ-JJ+2, I-II+2 )*
     $                                            T2( UJ-JJ+2, UJ-JJ+1 )
                              FR4 = FR4 + T1( UJ-JJ+2, I-II+2 )*
     $                                            T2( UJ-JJ+2, UJ-JJ+2 )
                              C( I+1, UJ ) = FR3
                              C( I+1, UJ+1 ) = FR4
                              FR5 = FR5 + T1( UJ-JJ+1, I-II+3 )*
     $                                            T2( UJ-JJ+1, UJ-JJ+1 )
                              FR5 = FR5 + T1( UJ-JJ+2, I-II+3 )*
     $                                            T2( UJ-JJ+2, UJ-JJ+1 )
                              FR6 = FR6 + T1( UJ-JJ+2, I-II+3 )*
     $                                            T2( UJ-JJ+2, UJ-JJ+2 )
                              C( I+2, UJ ) = FR5
                              C( I+2, UJ+1 ) = FR6
                              FR7 = FR7 + T1( UJ-JJ+1, I-II+4 )*
     $                                            T2( UJ-JJ+1, UJ-JJ+1 )
                              FR7 = FR7 + T1( UJ-JJ+2, I-II+4 )*
     $                                            T2( UJ-JJ+2, UJ-JJ+1 )
                              FR8 = FR8 + T1( UJ-JJ+2, I-II+4 )*
     $                                            T2( UJ-JJ+2, UJ-JJ+2 )
                              C( I+3, UJ ) = FR7
                              C( I+3, UJ+1 ) = FR8
  720                      CONTINUE
                           DO 740, RI = II+RISEC-1, II, -1
                              FR1 = ZERO
                              FR2 = ZERO
                              DO 730, L = UJ+1, JJ+JSEC-1
                                 FR1 = FR1 + T1( L-JJ+1, RI-II+1 )*
     $                                             T2( L-JJ+1, UJ-JJ+1 )
                                 FR2 = FR2 + T1( L-JJ+1, RI-II+1 )*
     $                                             T2( L-JJ+1, UJ-JJ+2 )
  730                         CONTINUE
                              FR1 = FR1 + T1( UJ-JJ+1, RI-II+1 )*
     $                                            T2( UJ-JJ+1, UJ-JJ+1 )
                              C( RI, UJ ) = FR1
                              C( RI, UJ+1 ) = FR2
  740                      CONTINUE
                        ELSE IF( UJSEC.EQ.1 )THEN
                           DO 760, RI = II+ISEC-1, II, -1
                              FR1 = ZERO
                              DO 750, L = UJ, JJ+JSEC-1
                                 FR1 = FR1 + T1( L-JJ+1, RI-II+1 )*
     $                                             T2( L-JJ+1, UJ-JJ+1 )
  750                         CONTINUE
                              C( RI, UJ ) = FR1
  760                      CONTINUE
                        END IF
  770                CONTINUE
  780             CONTINUE
*
*                 C := alpha*C*A' + C, general matrix multiply
*                 involving the transpose of a rectangular block
*                 of A.
*
                  IF( JJ+JSEC.LE.N )THEN
                     CALL DGEMM ( 'N', 'T', M, JSEC, N-JJ-JSEC+1,
     $                                      ALPHA, C( 1, JJ+JSEC ), LDC,
     $                                       A( JJ, JJ+JSEC ), LDA, ONE,
     $                                                 C( 1, JJ ), LDC )
                  END IF
  790          CONTINUE
            END IF
         ELSE
            IF( NOTR )THEN
*
*              Form  C := alpha*C*A. Right, Lower, No transpose.
*
               DO 910, JX = MOD( N-1, RB )+1, N, RB
                  JJ = MAX( 1, JX-RB+1 )
                  JSEC = JX-JJ+1
                  DO 900, II = 1, M, CB
                     ISEC = MIN( CB, M-II+1 )
*
*                    T1 := alpha*C', copy the transpose of a rectangular
*                    block of alpha*C to T1.
*
                     UISEC = ISEC-MOD( ISEC, 2 )
                     UJSEC = JSEC-MOD( JSEC, 2 )
                     DO 810, J = JJ, JJ+UJSEC-1, 2
                        DO 800, I = II, II+UISEC-1, 2
                           T1( J-JJ+1, I-II+1 ) = ALPHA*C( I, J )
                           T1( J-JJ+2, I-II+1 ) = ALPHA*C( I, J+1 )
                           T1( J-JJ+1, I-II+2 ) = ALPHA*C( I+1, J )
                           T1( J-JJ+2, I-II+2 ) = ALPHA*C( I+1, J+1 )
  800                   CONTINUE
                        IF( UISEC.LT.ISEC )THEN
                           T1( J-JJ+1, ISEC ) = ALPHA*C( II+ISEC-1, J )
                           T1( J-JJ+2, ISEC ) =
     $                                        ALPHA*C( II+ISEC-1, J+1 )
                        END IF
  810                CONTINUE
                     IF( UJSEC.LT.JSEC )THEN
                        DO 820, I = II, II+ISEC-1
                           T1( JSEC, I-II+1 ) = ALPHA*C( I, JJ+JSEC-1 )
  820                   CONTINUE
                     END IF
*
*                    C := alpha*T1'*A, update a rectangular block of C
*                    using the transpose of a rectangular block of C
*                    stored in T1'.
*
                     RISEC = MOD( ISEC, 4 )
                     DO 890, JY = JJ+JSEC-1, JJ, -2
                        UJ = MAX( JJ, JY-2+1 )
                        UJSEC = JY-UJ+1
*
*                       Four by two unrolling.
*
                        IF( UJSEC.EQ.2 )THEN
                           DO 840, I = II+ISEC-4, II+RISEC, -4
                              FR1 = ZERO
                              FR2 = ZERO
                              FR3 = ZERO
                              FR4 = ZERO
                              FR5 = ZERO
                              FR6 = ZERO
                              FR7 = ZERO
                              FR8 = ZERO
                              DO 830, L = UJ+2, JJ+JSEC-1
                                 FR1 = FR1 + T1( L-JJ+1, I-II+1 )*
     $                                                        A( L, UJ )
                                 FR2 = FR2 + T1( L-JJ+1, I-II+1 )*
     $                                                      A( L, UJ+1 )
                                 FR3 = FR3 + T1( L-JJ+1, I-II+2 )*
     $                                                        A( L, UJ )
                                 FR4 = FR4 + T1( L-JJ+1, I-II+2 )*
     $                                                      A( L, UJ+1 )
                                 FR5 = FR5 + T1( L-JJ+1, I-II+3 )*
     $                                                        A( L, UJ )
                                 FR6 = FR6 + T1( L-JJ+1, I-II+3 )*
     $                                                      A( L, UJ+1 )
                                 FR7 = FR7 + T1( L-JJ+1, I-II+4 )*
     $                                                        A( L, UJ )
                                 FR8 = FR8 + T1( L-JJ+1, I-II+4 )*
     $                                                      A( L, UJ+1 )
  830                         CONTINUE
                              FR1 = FR1 + T1( UJ-JJ+2, I-II+1 )*
     $                                                     A( UJ+1, UJ )
                              FR3 = FR3 + T1( UJ-JJ+2, I-II+2 )*
     $                                                     A( UJ+1, UJ )
                              FR5 = FR5 + T1( UJ-JJ+2, I-II+3 )*
     $                                                     A( UJ+1, UJ )
                              FR7 = FR7 + T1( UJ-JJ+2, I-II+4 )*
     $                                                     A( UJ+1, UJ )
                              IF( NOUNIT )THEN
                                 FR1 = FR1 + T1( UJ-JJ+1, I-II+1 )*
     $                                                       A( UJ, UJ )
                                 FR2 = FR2 + T1( UJ-JJ+2, I-II+1 )*
     $                                                   A( UJ+1, UJ+1 )
                                 C( I, UJ ) = FR1
                                 C( I, UJ+1 ) = FR2
                                 FR3 = FR3 + T1( UJ-JJ+1, I-II+2 )*
     $                                                       A( UJ, UJ )
                                 FR4 = FR4 + T1( UJ-JJ+2, I-II+2 )*
     $                                                   A( UJ+1, UJ+1 )
                                 C( I+1, UJ ) = FR3
                                 C( I+1, UJ+1 ) = FR4
                                 FR5 = FR5 + T1( UJ-JJ+1, I-II+3 )*
     $                                                       A( UJ, UJ )
                                 FR6 = FR6 + T1( UJ-JJ+2, I-II+3 )*
     $                                                   A( UJ+1, UJ+1 )
                                 C( I+2, UJ ) = FR5
                                 C( I+2, UJ+1 ) = FR6
                                 FR7 = FR7 + T1( UJ-JJ+1, I-II+4 )*
     $                                                       A( UJ, UJ )
                                 FR8 = FR8 + T1( UJ-JJ+2, I-II+4 )*
     $                                                   A( UJ+1, UJ+1 )
                                 C( I+3, UJ ) = FR7
                                 C( I+3, UJ+1 ) = FR8
                              ELSE
                                 C( I, UJ ) = FR1 +
     $                                             T1( UJ-JJ+1, I-II+1 )
                                 C( I, UJ+1 ) = FR2 +
     $                                             T1( UJ-JJ+2, I-II+1 )
                                 C( I+1, UJ ) = FR3 +
     $                                             T1( UJ-JJ+1, I-II+2 )
                                 C( I+1, UJ+1 ) = FR4 +
     $                                             T1( UJ-JJ+2, I-II+2 )
                                 C( I+2, UJ ) = FR5 +
     $                                             T1( UJ-JJ+1, I-II+3 )
                                 C( I+2, UJ+1 ) = FR6 +
     $                                             T1( UJ-JJ+2, I-II+3 )
                                 C( I+3, UJ ) = FR7 +
     $                                             T1( UJ-JJ+1, I-II+4 )
                                 C( I+3, UJ+1 ) = FR8 +
     $                                             T1( UJ-JJ+2, I-II+4 )
                              END IF
  840                      CONTINUE
                           DO 860, RI = II+RISEC-1, II, -1
                              FR1 = ZERO
                              FR2 = ZERO
                              DO 850, L = UJ+2, JJ+JSEC-1
                                 FR1 = FR1 + T1( L-JJ+1, RI-II+1 )*
     $                                                        A( L, UJ )
                                 FR2 = FR2 + T1( L-JJ+1, RI-II+1 )*
     $                                                      A( L, UJ+1 )
  850                         CONTINUE
                              FR1 = FR1 + T1( UJ-JJ+2, RI-II+1 )*
     $                                                     A( UJ+1, UJ )
                              IF( NOUNIT )THEN
                                 FR1 = FR1 + T1( UJ-JJ+1, RI-II+1 )*
     $                                                       A( UJ, UJ )
                                 FR2 = FR2 + T1( UJ-JJ+2, RI-II+1 )*
     $                                                   A( UJ+1, UJ+1 )
                              ELSE
                                 FR1 = FR1 + T1( UJ-JJ+1, RI-II+1 )
                                 FR2 = FR2 + T1( UJ-JJ+2, RI-II+1 )
                              END IF
                              C( RI, UJ ) = FR1
                              C( RI, UJ+1 ) = FR2
  860                      CONTINUE
                        ELSE IF( UJSEC.EQ.1 )THEN
                           DO 880, RI = II+ISEC-1, II, -1
                              FR1 = ZERO
                              DO 870, L = UJ+1, JJ+JSEC-1
                                 FR1 = FR1 + T1( L-JJ+1, RI-II+1 )*
     $                                                        A( L, UJ )
  870                         CONTINUE
                              IF( NOUNIT )THEN
                                 FR1 = FR1 + T1( UJ-JJ+1, RI-II+1 )*
     $                                                       A( UJ, UJ )
                              ELSE
                                 FR1 = FR1 + T1( UJ-JJ+1, RI-II+1 )
                              END IF
                              C( RI, UJ ) = FR1
  880                      CONTINUE
                        END IF
  890                CONTINUE
  900             CONTINUE
*
*                 C := alpha*C*A + C, general matrix multiply
*                 involving a rectangular block of A.
*
                  IF( JJ+JSEC.LE.N )THEN
                     CALL DGEMM ( 'N', 'N', M, JSEC, N-JJ-JSEC+1,
     $                                      ALPHA, C( 1, JJ+JSEC ), LDC,
     $                                       A( JJ+JSEC, JJ ), LDA, ONE,
     $                                                 C( 1, JJ ), LDC )
                  END IF
  910          CONTINUE
            ELSE
*
*              Form  C := alpha*C*A'. Right, Lower, Transpose.
*
               DO 1070, JX = N, 1, -RB
                  JJ = MAX( 1, JX-RB+1 )
                  JSEC = JX-JJ+1
*
*                 T2 := alpha*A', copy the transpose of a lower
*                 triangular diagonal block of alpha*A to T2.
*
                  IF( NOUNIT )THEN
                     DO 930, J = JJ, JJ+JSEC-1, 2
                        T2( J-JJ+1, J-JJ+1 ) = ALPHA*A( J, J )
                        RJSEC = MOD( JJ+JSEC-J-1, 2 )+1
                        IF( RJSEC.EQ.2 )THEN
                           T2( J-JJ+1, J-JJ+2 ) = ALPHA*A( J+1, J )
                           T2( J-JJ+2, J-JJ+2 ) = ALPHA*A( J+1, J+1 )
                        END IF
                        DO 920, L = J+RJSEC, JJ+JSEC-1, 2
                           T2( J-JJ+1, L-JJ+1 ) = ALPHA*A( L, J )
                           T2( J-JJ+2, L-JJ+1 ) = ALPHA*A( L, J+1 )
                           T2( J-JJ+1, L-JJ+2 ) = ALPHA*A( L+1, J )
                           T2( J-JJ+2, L-JJ+2 ) = ALPHA*A( L+1, J+1 )
  920                   CONTINUE
  930                CONTINUE
                  ELSE
                     DO 950, J = JJ, JJ+JSEC-1, 2
                        T2( J-JJ+1, J-JJ+1 ) = ALPHA
                        RJSEC = MOD( JJ+JSEC-J-2, 2 )+2
                        IF( RJSEC.GE.2 )THEN
                           T2( J-JJ+1, J-JJ+2 ) = ALPHA*A( J+1, J )
                           T2( J-JJ+2, J-JJ+2 ) = ALPHA
                        END IF
                        IF( RJSEC.EQ.3 )THEN
                           T2( J-JJ+1, J-JJ+3 ) = ALPHA*A( J+2, J )
                           T2( J-JJ+2, J-JJ+3 ) = ALPHA*A( J+2, J+1 )
                        END IF
                        DO 940, L = J+RJSEC, JJ+JSEC-1, 2
                           T2( J-JJ+1, L-JJ+1 ) = ALPHA*A( L, J )
                           T2( J-JJ+2, L-JJ+1 ) = ALPHA*A( L, J+1 )
                           T2( J-JJ+1, L-JJ+2 ) = ALPHA*A( L+1, J )
                           T2( J-JJ+2, L-JJ+2 ) = ALPHA*A( L+1, J+1 )
  940                   CONTINUE
  950                CONTINUE
                  END IF
                  DO 1060, II = 1, M, CB
                     ISEC = MIN( CB, M-II+1 )
*
*                    T1 := C', copy the transpose of a rectangular block
*                    of C to T1.
*
                     UISEC = ISEC-MOD( ISEC, 2 )
                     UJSEC = JSEC-MOD( JSEC, 2 )
                     DO 970, J = JJ, JJ+UJSEC-1, 2
                        DO 960, I = II, II+UISEC-1, 2
                           T1( J-JJ+1, I-II+1 ) = C( I, J )
                           T1( J-JJ+2, I-II+1 ) = C( I, J+1 )
                           T1( J-JJ+1, I-II+2 ) = C( I+1, J )
                           T1( J-JJ+2, I-II+2 ) = C( I+1, J+1 )
  960                   CONTINUE
                        IF( UISEC.LT.ISEC )THEN
                           T1( J-JJ+1, ISEC ) = C( II+ISEC-1, J )
                           T1( J-JJ+2, ISEC ) = C( II+ISEC-1, J+1 )
                        END IF
  970                CONTINUE
                     IF( UJSEC.LT.JSEC )THEN
                        DO 980, I = II, II+ISEC-1
                           T1( JSEC, I-II+1 ) = C( I, JJ+JSEC-1 )
  980                   CONTINUE
                     END IF
*
*                    C := alpha*T1'*T2, update a rectangular block of C
*                    using the transpose of a rectangular block of C
*                    stored in T1' and the transpose of a lower
*                    triangular diagonal block of A stored in T2.
*
                     UISEC = ISEC-MOD( ISEC, 4 )
                     DO 1050, JY = JJ+JSEC-1, JJ, -2
                        UJ = MAX( JJ, JY-2+1 )
                        UJSEC = JY-UJ+1
*
*                       Four by two unrolling.
*
                        IF( UJSEC.EQ.2 )THEN
                           DO 1000, I = II, II+UISEC-1, 4
                              FR1 = ZERO
                              FR2 = ZERO
                              FR3 = ZERO
                              FR4 = ZERO
                              FR5 = ZERO
                              FR6 = ZERO
                              FR7 = ZERO
                              FR8 = ZERO
                              DO 990, L = JJ, UJ-1
                                 FR1 = FR1 + T1( L-JJ+1, I-II+1 )*
     $                                             T2( L-JJ+1, UJ-JJ+1 )
                                 FR2 = FR2 + T1( L-JJ+1, I-II+1 )*
     $                                             T2( L-JJ+1, UJ-JJ+2 )
                                 FR3 = FR3 + T1( L-JJ+1, I-II+2 )*
     $                                             T2( L-JJ+1, UJ-JJ+1 )
                                 FR4 = FR4 + T1( L-JJ+1, I-II+2 )*
     $                                             T2( L-JJ+1, UJ-JJ+2 )
                                 FR5 = FR5 + T1( L-JJ+1, I-II+3 )*
     $                                             T2( L-JJ+1, UJ-JJ+1 )
                                 FR6 = FR6 + T1( L-JJ+1, I-II+3 )*
     $                                             T2( L-JJ+1, UJ-JJ+2 )
                                 FR7 = FR7 + T1( L-JJ+1, I-II+4 )*
     $                                             T2( L-JJ+1, UJ-JJ+1 )
                                 FR8 = FR8 + T1( L-JJ+1, I-II+4 )*
     $                                             T2( L-JJ+1, UJ-JJ+2 )
  990                         CONTINUE
                              FR1 = FR1 + T1( UJ-JJ+1, I-II+1 )*
     $                                            T2( UJ-JJ+1, UJ-JJ+1 )
                              FR2 = FR2 + T1( UJ-JJ+1, I-II+1 )*
     $                                            T2( UJ-JJ+1, UJ-JJ+2 )
                              FR2 = FR2 + T1( UJ-JJ+2, I-II+1 )*
     $                                            T2( UJ-JJ+2, UJ-JJ+2 )
                              C( I, UJ ) = FR1
                              C( I, UJ+1 ) = FR2
                              FR3 = FR3 + T1( UJ-JJ+1, I-II+2 )*
     $                                            T2( UJ-JJ+1, UJ-JJ+1 )
                              FR4 = FR4 + T1( UJ-JJ+1, I-II+2 )*
     $                                            T2( UJ-JJ+1, UJ-JJ+2 )
                              FR4 = FR4 + T1( UJ-JJ+2, I-II+2 )*
     $                                            T2( UJ-JJ+2, UJ-JJ+2 )
                              C( I+1, UJ ) = FR3
                              C( I+1, UJ+1 ) = FR4
                              FR5 = FR5 + T1( UJ-JJ+1, I-II+3 )*
     $                                            T2( UJ-JJ+1, UJ-JJ+1 )
                              FR6 = FR6 + T1( UJ-JJ+1, I-II+3 )*
     $                                            T2( UJ-JJ+1, UJ-JJ+2 )
                              FR6 = FR6 + T1( UJ-JJ+2, I-II+3 )*
     $                                            T2( UJ-JJ+2, UJ-JJ+2 )
                              C( I+2, UJ ) = FR5
                              C( I+2, UJ+1 ) = FR6
                              FR7 = FR7 + T1( UJ-JJ+1, I-II+4 )*
     $                                            T2( UJ-JJ+1, UJ-JJ+1 )
                              FR8 = FR8 + T1( UJ-JJ+1, I-II+4 )*
     $                                            T2( UJ-JJ+1, UJ-JJ+2 )
                              FR8 = FR8 + T1( UJ-JJ+2, I-II+4 )*
     $                                            T2( UJ-JJ+2, UJ-JJ+2 )
                              C( I+3, UJ ) = FR7
                              C( I+3, UJ+1 ) = FR8
 1000                      CONTINUE
                           DO 1020, RI = II+UISEC, II+ISEC-1
                              FR1 = ZERO
                              FR2 = ZERO
                              DO 1010, L = JJ, UJ
                                 FR1 = FR1 + T1( L-JJ+1, RI-II+1 )*
     $                                             T2( L-JJ+1, UJ-JJ+1 )
                                 FR2 = FR2 + T1( L-JJ+1, RI-II+1 )*
     $                                             T2( L-JJ+1, UJ-JJ+2 )
 1010                         CONTINUE
                              FR2 = FR2 + T1( UJ-JJ+2, RI-II+1 )*
     $                                            T2( UJ-JJ+2, UJ-JJ+2 )
                              C( RI, UJ ) = FR1
                              C( RI, UJ+1 ) = FR2
 1020                      CONTINUE
                        ELSE IF( UJSEC.EQ.1 )THEN
                           DO 1040, RI = II, II+ISEC-1
                              FR1 = ZERO
                              DO 1030, L = JJ, UJ
                                 FR1 = FR1 + T1( L-JJ+1, RI-II+1 )*
     $                                             T2( L-JJ+1, UJ-JJ+1 )
 1030                         CONTINUE
                              C( RI, UJ ) = FR1
 1040                      CONTINUE
                        END IF
 1050                CONTINUE
 1060             CONTINUE
*
*                 C := alpha*C*A' + C, general matrix multiply
*                 involving the transpose of a rectangular block
*                 of A.
*
                  IF( JJ.GT.1 )THEN
                     CALL DGEMM ( 'N', 'T', M, JSEC, JJ-1, ALPHA,
     $                                  C( 1, 1 ), LDC, A( JJ, 1 ), LDA,
     $                                            ONE, C( 1, JJ ), LDC )
                  END IF
 1070          CONTINUE
            END IF
         END IF
      END IF
*
      RETURN
*
*     End of DTRMM.
*
      END
