      SUBROUTINE DSYR2K( UPLO, TRANS, N, K, ALPHA, A, LDA, B, LDB,
     $                   BETA, C, LDC )
*     .. Scalar Arguments ..
      CHARACTER*1        UPLO, TRANS
      INTEGER            N, K, LDA, LDB, LDC
      DOUBLE PRECISION   ALPHA, BETA
*     .. Array Arguments ..
      DOUBLE PRECISION   A( LDA, * ), B( LDB, * ), C( LDC, * )
*     ..
*
*  Purpose
*  =======
*
*  DSYR2K  performs one of the symmetric rank 2k operations
*
*     C := alpha*A*B' + alpha*B*A' + beta*C,
*
*  or
*
*     C := alpha*A'*B + alpha*B'*A + beta*C,
*
*  where  alpha and beta  are scalars, C is an  n by n  symmetric matrix
*  and  A and B  are  n by k  matrices  in the  first  case  and  k by n
*  matrices in the second case.
*
*  Parameters
*  ==========
*
*  UPLO   - CHARACTER*1.
*           On  entry,   UPLO  specifies  whether  the  upper  or  lower
*           triangular  part  of the  array  C  is to be  referenced  as
*           follows:
*
*              UPLO = 'U' or 'u'   Only the  upper triangular part of  C
*                                  is to be referenced.
*
*              UPLO = 'L' or 'l'   Only the  lower triangular part of  C
*                                  is to be referenced.
*
*           Unchanged on exit.
*
*  TRANS  - CHARACTER*1.
*           On entry,  TRANS  specifies the operation to be performed as
*           follows:
*
*              TRANS = 'N' or 'n'   C := alpha*A*B' + alpha*B*A' +
*                                        beta*C.
*
*              TRANS = 'T' or 't'   C := alpha*A'*B + alpha*B'*A +
*                                        beta*C.
*
*              TRANS = 'C' or 'c'   C := alpha*A'*B + alpha*B'*A +
*                                        beta*C.
*
*           Unchanged on exit.
*
*  N      - INTEGER.
*           On entry,  N specifies the order of the matrix C.  N must be
*           at least zero.
*           Unchanged on exit.
*
*  K      - INTEGER.
*           On entry with  TRANS = 'N' or 'n',  K  specifies  the number
*           of  columns  of the  matrices  A and B,  and on  entry  with
*           TRANS = 'T' or 't' or 'C' or 'c',  K  specifies  the  number
*           of rows of the matrices  A and B.  K must be at least  zero.
*           Unchanged on exit.
*
*  ALPHA  - DOUBLE PRECISION.
*           On entry, ALPHA specifies the scalar alpha.
*           Unchanged on exit.
*
*  A      - DOUBLE PRECISION array of DIMENSION ( LDA, ka ), where ka is
*           k  when  TRANS = 'N' or 'n',  and is  n  otherwise.
*           Before entry with  TRANS = 'N' or 'n',  the  leading  n by k
*           part of the array  A  must contain the matrix  A,  otherwise
*           the leading  k by n  part of the array  A  must contain  the
*           matrix A.
*           Unchanged on exit.
*
*  LDA    - INTEGER.
*           On entry, LDA specifies the first dimension of A as declared
*           in  the  calling  (sub)  program.   When  TRANS = 'N' or 'n'
*           then  LDA must be at least  max( 1, n ), otherwise  LDA must
*           be at least  max( 1, k ).
*           Unchanged on exit.
*
*  B      - DOUBLE PRECISION array of DIMENSION ( LDB, kb ), where kb is
*           k  when  TRANS = 'N' or 'n',  and is  n  otherwise.
*           Before entry with  TRANS = 'N' or 'n',  the  leading  n by k
*           part of the array  B  must contain the matrix  B,  otherwise
*           the leading  k by n  part of the array  B  must contain  the
*           matrix B.
*           Unchanged on exit.
*
*  LDB    - INTEGER.
*           On entry, LDB specifies the first dimension of B as declared
*           in  the  calling  (sub)  program.   When  TRANS = 'N' or 'n'
*           then  LDB must be at least  max( 1, n ), otherwise  LDB must
*           be at least  max( 1, k ).
*           Unchanged on exit.
*
*  BETA   - DOUBLE PRECISION.
*           On entry, BETA specifies the scalar beta.
*           Unchanged on exit.
*
*  C      - DOUBLE PRECISION array of DIMENSION ( LDC, n ).
*           Before entry  with  UPLO = 'U' or 'u',  the leading  n by n
*           upper triangular part of the array C must contain the upper
*           triangular part  of the  symmetric matrix  and the strictly
*           lower triangular part of C is not referenced.  On exit, the
*           upper triangular part of the array  C is overwritten by the
*           upper triangular part of the updated matrix.
*           Before entry  with  UPLO = 'L' or 'l',  the leading  n by n
*           lower triangular part of the array C must contain the lower
*           triangular part  of the  symmetric matrix  and the strictly
*           upper triangular part of C is not referenced.  On exit, the
*           lower triangular part of the array  C is overwritten by the
*           lower triangular part of the updated matrix.
*
*  LDC    - INTEGER.
*           On entry, LDC specifies the first dimension of C as declared
*           in  the  calling  (sub)  program.   LDC  must  be  at  least
*           max( 1, n ).
*           Unchanged on exit.
*
*
*  Level 3 Blas routine.
*
*  -- Written on 8-February-1989.
*     Jack Dongarra, Argonne National Laboratory.
*     Iain Duff, AERE Harwell.
*     Jeremy Du Croz, Numerical Algorithms Group Ltd.
*     Sven Hammarling, Numerical Algorithms Group Ltd.
*
*  -- Rewritten in December-1993
*     GEMM-Based Level 3 BLAS.
*     Per Ling, Institute of Information Processing,
*     University of Umea, Sweden.
*
*  -- Rewritten in Mars-1995.
*     Superscalar GEMM-Based Level 3 BLAS (Version 0.1).
*     Per Ling, Institute of Information Processing,
*     University of Umea, Sweden.
*
*  -- Modified in October-1997.
*     Superscalar GEMM-Based Level 3 BLAS (Version 0.1).
*     Per Ling, Institute of Information Processing,
*     University of Umea, Sweden.
*
*
*     .. Local Scalars ..
      INTEGER            INFO, NROWA
      INTEGER            I, II, ISEC, J
      INTEGER            UISEC, RISEC
      LOGICAL            UPPER, NOTR
*     .. Intrinsic Functions ..
      INTRINSIC          MAX, MIN, MOD
*     .. External Functions ..
      LOGICAL            LSAME
      EXTERNAL           LSAME
*     .. External Subroutines ..
      EXTERNAL           XERBLA
      EXTERNAL           DGEMM
*     .. Parameters ..
      DOUBLE PRECISION   ONE, ZERO
      PARAMETER        ( ONE = 1.0D+0, ZERO = 0.0D+0 )
*     .. User specified parameters for DSYR2K ..
      INTEGER            RCB
      PARAMETER        ( RCB = 96 )
*     .. Local Arrays ..
      DOUBLE PRECISION   T1( RCB, RCB )
*     ..
*     .. Executable Statements ..
*
*     Test the input parameters.
*
      UPPER = LSAME( UPLO, 'U' )
      NOTR = LSAME( TRANS, 'N' )
      IF( NOTR )THEN
         NROWA = N
      ELSE
         NROWA = K
      END IF
      INFO = 0
      IF( ( .NOT.UPPER ).AND.( .NOT.LSAME( UPLO, 'L' ) ) )THEN
         INFO = 1
      ELSE IF( ( .NOT.NOTR ).AND.( .NOT.LSAME( TRANS, 'T' ) ).AND.
     $                                ( .NOT.LSAME( TRANS, 'C' ) ) )THEN
         INFO = 2
      ELSE IF( N.LT.0 )THEN
         INFO = 3
      ELSE IF( K.LT.0 )THEN
         INFO = 4
      ELSE IF( LDA.LT.MAX( 1, NROWA ) )THEN
         INFO = 7
      ELSE IF( LDB.LT.MAX( 1, NROWA ) )THEN
         INFO = 9
      ELSE IF( LDC.LT.MAX( 1, N ) )THEN
         INFO = 12
      END IF
      IF( INFO.NE.0 )THEN
         CALL XERBLA( 'DSYR2K', INFO )
         RETURN
      END IF
*
*     Quick return if possible.
*
      IF( ( N.EQ.0 ).OR.
     $    ( ( ( ALPHA.EQ.ZERO ).OR.( K.EQ.0 ) ).AND.( BETA.EQ.ONE ) ) )
     $   RETURN
*
*     And when alpha.eq.zero or k.eq.0.
*
      IF( ALPHA.EQ.ZERO.OR.K.EQ.0 )THEN
         IF( UPPER )THEN
            IF( BETA.EQ.ZERO )THEN
               DO 30, J = 1, N
                  UISEC = J-MOD( J, 4 )
                  DO 10, I = 1, UISEC, 4
                     C( I, J ) = ZERO
                     C( I+1, J ) = ZERO
                     C( I+2, J ) = ZERO
                     C( I+3, J ) = ZERO
   10             CONTINUE
                  DO 20, I = UISEC+1, J
                     C( I, J ) = ZERO
   20             CONTINUE
   30          CONTINUE
            ELSE
               DO 60, J = 1, N
                  UISEC = J-MOD( J, 4 )
                  DO 40, I = 1, UISEC, 4
                     C( I, J ) = BETA*C( I, J )
                     C( I+1, J ) = BETA*C( I+1, J )
                     C( I+2, J ) = BETA*C( I+2, J )
                     C( I+3, J ) = BETA*C( I+3, J )
   40             CONTINUE
                  DO 50, I = UISEC+1, J
                     C( I, J ) = BETA*C( I, J )
   50             CONTINUE
   60          CONTINUE
            END IF
         ELSE
            IF( BETA.EQ.ZERO )THEN
               DO 100, J = 1, N
                  RISEC = MOD( N-J, 4 )+1
                  DO 80, I = J, J+RISEC-1
                     C( I, J ) = ZERO
   80             CONTINUE
                  DO 90, I = J+RISEC, N, 4
                     C( I, J ) = ZERO
                     C( I+1, J ) = ZERO
                     C( I+2, J ) = ZERO
                     C( I+3, J ) = ZERO
   90             CONTINUE
  100          CONTINUE
            ELSE
               DO 130, J = 1, N
                  RISEC = MOD( N-J, 4 )+1
                  DO 110, I = J, J+RISEC-1
                     C( I, J ) = BETA*C( I, J )
  110             CONTINUE
                  DO 120, I = J+RISEC, N, 4
                     C( I, J ) = BETA*C( I, J )
                     C( I+1, J ) = BETA*C( I+1, J )
                     C( I+2, J ) = BETA*C( I+2, J )
                     C( I+3, J ) = BETA*C( I+3, J )
  120             CONTINUE
  130          CONTINUE
            END IF
         END IF
         RETURN
      END IF
*
*     Start the operations.
*
      IF( UPPER )THEN
         IF( NOTR )THEN
*
*           Form  C := alpha*A*B' + alpha*B*A' + beta*C. Upper, Notr.
*
            DO 160, II = 1, N, RCB
               ISEC = MIN( RCB, N-II+1 )
*
*              T1 := alpha*A*B', general matrix multiplication of
*              rectangular blocks of A and B. An upper triangular
*              diagonal block of alpha*A*B' + alpha*B*A' can be
*              constructed from T1. T1 is square.
*
               CALL DGEMM ( 'N', 'T', ISEC, ISEC, K, ALPHA, A( II, 1 ),
     $                     LDA, B( II, 1 ), LDB, ZERO, T1( 1, 1 ), RCB )
*
*              C := T1 + T1' + beta*C, the upper triangular part of C
*              is updated with beta, the upper triangular part of T1,
*              and the transpose of the lower triangular part
*              of T1.
*
               DO 150, J = II+ISEC-2, II-1, -2
                  UISEC = J-II+1-MOD( J-II+1, 2 )
                  DO 140, I = II, II+UISEC-1, 2
                     C( I, J ) = BETA*C( I, J ) +
     $                       T1( I-II+1, J-II+1 ) + T1( J-II+1, I-II+1 )
                     C( I+1, J ) = BETA*C( I+1, J ) +
     $                       T1( I-II+2, J-II+1 ) + T1( J-II+1, I-II+2 )
                     C( I, J+1 ) = BETA*C( I, J+1 ) +
     $                       T1( I-II+1, J-II+2 ) + T1( J-II+2, I-II+1 )
                     C( I+1, J+1 ) = BETA*C( I+1, J+1 ) +
     $                       T1( I-II+2, J-II+2 ) + T1( J-II+2, I-II+2 )
  140             CONTINUE
                  IF( MOD( J-II+1, 2 ).EQ.1 )THEN
                     C( J, J ) = BETA*C( J, J ) +
     $                       T1( J-II+1, J-II+1 ) + T1( J-II+1, J-II+1 )
                     C( J, J+1 ) = BETA*C( J, J+1 ) +
     $                       T1( J-II+1, J-II+2 ) + T1( J-II+2, J-II+1 )
                     C( J+1, J+1 ) = BETA*C( J+1, J+1 ) +
     $                       T1( J-II+2, J-II+2 ) + T1( J-II+2, J-II+2 )
                  ELSE
                     C( J+1, J+1 ) = BETA*C( J+1, J+1 ) +
     $                       T1( J-II+2, J-II+2 ) + T1( J-II+2, J-II+2 )
                  END IF
  150          CONTINUE
*
*              C := alpha*A*B' + beta*C  and  C := alpha*B*A' + C,
*              general matrix multiplication of rectangular blocks of C
*              consisting of ISEC columns stretching from 1 to II-1.
*
               IF( II.GT.1 )THEN
                  CALL DGEMM ( 'N', 'T', II-1, ISEC, K, ALPHA,
     $                            A( 1, 1 ), LDA, B( II, 1 ), LDB, BETA,
     $                                                 C( 1, II ), LDC )
                  CALL DGEMM ( 'N', 'T', II-1, ISEC, K, ALPHA,
     $                             B( 1, 1 ), LDB, A( II, 1 ), LDA, ONE,
     $                                                 C( 1, II ), LDC )
               END IF
  160       CONTINUE
         ELSE
*
*           Form  C := alpha*A'*B + alpha*B'*A + beta*C. Upper, Trans.
*
            DO 190, II = 1, N, RCB
               ISEC = MIN( RCB, N-II+1 )
*
*              T1 := alpha*A'*B, general matrix multiplication of
*              rectangular blocks of A and B. An upper triangular
*              diagonal block of alpha*A*B' + alpha*B*A' can be
*              constructed from T1. T1 is square.
*
               CALL DGEMM ( 'T', 'N', ISEC, ISEC, K, ALPHA, A( 1, II ),
     $                     LDA, B( 1, II ), LDB, ZERO, T1( 1, 1 ), RCB )
*
*              C := T1 + T1' + beta*C, the upper triangular part of C
*              is updated with beta, the upper triangular part of T1,
*              and the transpose of the lower triangular part
*              of T1.
*
               DO 180, J = II+ISEC-2, II-1, -2
                  UISEC = J-II+1-MOD( J-II+1, 2 )
                  DO 170, I = II, II+UISEC-1, 2
                     C( I, J ) = BETA*C( I, J ) +
     $                       T1( I-II+1, J-II+1 ) + T1( J-II+1, I-II+1 )
                     C( I+1, J ) = BETA*C( I+1, J ) +
     $                       T1( I-II+2, J-II+1 ) + T1( J-II+1, I-II+2 )
                     C( I, J+1 ) = BETA*C( I, J+1 ) +
     $                       T1( I-II+1, J-II+2 ) + T1( J-II+2, I-II+1 )
                     C( I+1, J+1 ) = BETA*C( I+1, J+1 ) +
     $                       T1( I-II+2, J-II+2 ) + T1( J-II+2, I-II+2 )
  170             CONTINUE
                  IF( MOD( J-II+1, 2 ).EQ.1 )THEN
                     C( J, J ) = BETA*C( J, J ) +
     $                       T1( J-II+1, J-II+1 ) + T1( J-II+1, J-II+1 )
                     C( J, J+1 ) = BETA*C( J, J+1 ) +
     $                       T1( J-II+1, J-II+2 ) + T1( J-II+2, J-II+1 )
                     C( J+1, J+1 ) = BETA*C( J+1, J+1 ) +
     $                       T1( J-II+2, J-II+2 ) + T1( J-II+2, J-II+2 )
                  ELSE
                     C( J+1, J+1 ) = BETA*C( J+1, J+1 ) +
     $                       T1( J-II+2, J-II+2 ) + T1( J-II+2, J-II+2 )
                  END IF
  180          CONTINUE
*
*              C := alpha*A'*B + beta*C  and  C := alpha*B'*A + C,
*              general matrix multiplication of rectangular blocks of C
*              consisting of ISEC columns stretching from 1 to II-1.
*
               IF( II.GT.1 )THEN
                  CALL DGEMM ( 'T', 'N', II-1, ISEC, K, ALPHA,
     $                            A( 1, 1 ), LDA, B( 1, II ), LDB, BETA,
     $                                                 C( 1, II ), LDC )
                  CALL DGEMM ( 'T', 'N', II-1, ISEC, K, ALPHA,
     $                             B( 1, 1 ), LDB, A( 1, II ), LDA, ONE,
     $                                                 C( 1, II ), LDC )
               END IF
  190       CONTINUE
         END If
      ELSE
         IF( NOTR )THEN
*
*           Form  C := alpha*A*B' + alpha*B*A' + beta*C. Upper, Notr.
*
            DO 220, II = 1, N, RCB
               ISEC = MIN( RCB, N-II+1 )
*
*              T1 := alpha*A*B', general matrix multiplication of
*              rectangular blocks of A and B. An upper triangular
*              diagonal block of alpha*A*B' + alpha*B*A' can be
*              constructed from T1. T1 is square.
*
               CALL DGEMM ( 'N', 'T', ISEC, ISEC, K, ALPHA, A( II, 1 ),
     $                     LDA, B( II, 1 ), LDB, ZERO, T1( 1, 1 ), RCB )
*
*              C := T1 + T1' + beta*C, the lower triangular part of C
*              is updated with beta, the lower triangular part of T1,
*              and the transpose of the lower triangular part
*              of T1.
*
               DO 210, J = II, II+ISEC-1, 2
                  UISEC = II+ISEC-1-J-MOD( II+ISEC-1-J, 2 )
                  IF( MOD( ISEC-UISEC, 2 ).EQ.0 )THEN
                     C( J, J ) = BETA*C( J, J ) +
     $                       T1( J-II+1, J-II+1 ) + T1( J-II+1, J-II+1 )
                     C( J+1, J ) = BETA*C( J+1, J ) +
     $                       T1( J-II+2, J-II+1 ) + T1( J-II+1, J-II+2 )
                     C( J+1, J+1 ) = BETA*C( J+1, J+1 ) +
     $                       T1( J-II+2, J-II+2 ) + T1( J-II+2, J-II+2 )
                  ELSE
                     C( J, J ) = BETA*C( J, J ) +
     $                       T1( J-II+1, J-II+1 ) + T1( J-II+1, J-II+1 )
                  END IF
                  DO 200, I = II+ISEC-UISEC, II+ISEC-1, 2
                     C( I, J ) = BETA*C( I, J ) +
     $                       T1( I-II+1, J-II+1 ) + T1( J-II+1, I-II+1 )
                     C( I+1, J ) = BETA*C( I+1, J ) +
     $                       T1( I-II+2, J-II+1 ) + T1( J-II+1, I-II+2 )
                     C( I, J+1 ) = BETA*C( I, J+1 ) +
     $                       T1( I-II+1, J-II+2 ) + T1( J-II+2, I-II+1 )
                     C( I+1, J+1 ) = BETA*C( I+1, J+1 ) +
     $                       T1( I-II+2, J-II+2 ) + T1( J-II+2, I-II+2 )
  200             CONTINUE
  210          CONTINUE
*
*              C := alpha*A*B' + beta*C  and  C := alpha*B*A' + C,
*              general matrix multiplication of rectangular blocks of C
*              consisting of ISEC columns stretching from 1 to II-1.
*
               IF( II+ISEC-1.LT.N )THEN
                  CALL DGEMM ( 'N', 'T', N-II-ISEC+1, ISEC, K, ALPHA,
     $                      A( II+ISEC, 1 ), LDA, B( II, 1 ), LDB, BETA,
     $                                           C( II+ISEC, II ), LDC )
                  CALL DGEMM ( 'N', 'T', N-II-ISEC+1, ISEC, K, ALPHA,
     $                       B( II+ISEC, 1 ), LDB, A( II, 1 ), LDA, ONE,
     $                                           C( II+ISEC, II ), LDC )
               END IF
  220       CONTINUE
         ELSE
*
*           Form  C := alpha*A'*B + alpha*B'*A + beta*C. Upper, Trans.
*
            DO 250, II = 1, N, RCB
               ISEC = MIN( RCB, N-II+1 )
*
*              T1 := alpha*A'*B, general matrix multiplication of
*              rectangular blocks of A and B. An lower triangular
*              diagonal block of alpha*A*B' + alpha*B*A' can be
*              constructed from T1. T1 is square.
*
               CALL DGEMM ( 'T', 'N', ISEC, ISEC, K, ALPHA, A( 1, II ),
     $                     LDA, B( 1, II ), LDB, ZERO, T1( 1, 1 ), RCB )
*
*              C := T1 + T1' + beta*C, the lower triangular part of C
*              is updated with beta, the lower triangular part of T1,
*              and the transpose of the lower triangular part
*              of T1.
*
               DO 240, J = II, II+ISEC-1, 2
                  UISEC = II+ISEC-1-J-MOD( II+ISEC-1-J, 2 )
                  IF( MOD( ISEC-UISEC, 2 ).EQ.0 )THEN
                     C( J, J ) = BETA*C( J, J ) +
     $                       T1( J-II+1, J-II+1 ) + T1( J-II+1, J-II+1 )
                     C( J+1, J ) = BETA*C( J+1, J ) +
     $                       T1( J-II+2, J-II+1 ) + T1( J-II+1, J-II+2 )
                     C( J+1, J+1 ) = BETA*C( J+1, J+1 ) +
     $                       T1( J-II+2, J-II+2 ) + T1( J-II+2, J-II+2 )
                  ELSE
                     C( J, J ) = BETA*C( J, J ) +
     $                       T1( J-II+1, J-II+1 ) + T1( J-II+1, J-II+1 )
                  END IF
                  DO 230, I = II+ISEC-UISEC, II+ISEC-1, 2
                     C( I, J ) = BETA*C( I, J ) +
     $                       T1( I-II+1, J-II+1 ) + T1( J-II+1, I-II+1 )
                     C( I+1, J ) = BETA*C( I+1, J ) +
     $                       T1( I-II+2, J-II+1 ) + T1( J-II+1, I-II+2 )
                     C( I, J+1 ) = BETA*C( I, J+1 ) +
     $                       T1( I-II+1, J-II+2 ) + T1( J-II+2, I-II+1 )
                     C( I+1, J+1 ) = BETA*C( I+1, J+1 ) +
     $                       T1( I-II+2, J-II+2 ) + T1( J-II+2, I-II+2 )
  230             CONTINUE
  240          CONTINUE
*
*              C := alpha*A'*B + beta*C  and  C := alpha*B'*A + C,
*              general matrix multiplication of rectangular blocks of C
*              consisting of ISEC columns stretching from 1 to II-1.
*
               IF( II+ISEC-1.LT.N )THEN
                  CALL DGEMM ( 'T', 'N', N-II-ISEC+1, ISEC, K, ALPHA,
     $                      A( 1, II+ISEC ), LDA, B( 1, II ), LDB, BETA,
     $                                           C( II+ISEC, II ), LDC )
                  CALL DGEMM ( 'T', 'N', N-II-ISEC+1, ISEC, K, ALPHA,
     $                       B( 1, II+ISEC ), LDB, A( 1, II ), LDA, ONE,
     $                                           C( II+ISEC, II ), LDC )
               END IF
  250       CONTINUE
         END IF
      END IF
*
      RETURN
*
*     End of DSYR2K.
*
      END
