;***********************************************************
;
; TMS320C26 Weaver SSB Modulator Program
; Carlos M. Puig
; April 1993
;
;***********************************************************

#include "320C26.h"
.MSFIRST

; Conditional assembly flags

TIMERINT .equ   1               ; do a timer interrupt?
NOTUSED  .equ   0               ; for dead code

.title   "TMS320C26 Weaver SSB Modulator Program"

; NOTE: TRATE must be a multiple of 16 to avoid strange beat frequencies
;       with the S/C filter clock.

TRATE   .equ    416             ; Timer period for 24 kHz

; Processor I/O ports

ADC     .equ    0               ; Read/write to this port
DAC1    .equ    2               ; Write to this port
DAC2    .equ    3               ; Write to this port
LDAC    .equ    2               ; Read from this port

; ADC MUX addresses

ADCEXT   .equ   0               ; ADCIN connected to external input
ADC1     .equ   1               ; ADCIN grounded externally
ADCDAC1  .equ   2               ; ADCIN connected to DAC1 output
ADC1     .equ   3               ; ADCIN grounded externally
ADCDAC2  .equ   4               ; ADCIN connected to DAC2 output
ADC1     .equ   5               ; ADCIN grounded externally
ADCGND   .equ   6               ; ADCIN grounded internally
;ADCOPN  .equ   7               ; ADCIN open (DON'T USE);

;***********************************************************
;
; Data Area
;
;***********************************************************

ONE     .equ    B2D              ; = 1 (constant)
MAXPOS  .equ    B2D+1            ; = 32767 (constant)

ADCBUF  .equ    B2D+10           ; dummy location for ADC
ADCVAL  .equ    B2D+11           ; value read from aDC

DACBUF  .equ    B2D+12           ; dummy location for DAC
DACVAL1 .equ    B2D+13           ; DAC output values
DACVAL2 .equ    B2D+14
DACOFF1 .equ    B2D+15           ; DAC offsets
DACOFF2 .equ    B2D+16

DECVAL  .equ    B2D+17           ; Value at output of decimator
USEVAL  .equ    B2D+18           ; Decimator output value actually used

PHASE   .equ    B2D+20           ; Phase (=0,1) for polyphase filters
P0OUT   .equ    B2D+21           ; Result for phase 0 LP filter
P1OUT   .equ    B2D+22           ; Result for phase 1 LP filter

PAB     .equ    B0P              ; internal program area base
CAB     .equ    B1P              ; internal coefficient area base
PABD    .equ    B0D              ; program area in data space
CABD    .equ    B1D              ; coefficient area in data space
DAB     .equ    B3D              ; data area base

DDEC    .equ    DAB              ; data area for decimation filter
DINT1   .equ    DDEC+LP3KL+1     ; data area for interpolation filter 1
DINT2   .equ    DINT1+LP3KL+1    ; data area for interpolation filter 2
DLP1    .equ    DINT2+LP3KL+1    ; data area for 1.5 kHz LP filter 1
DLP2    .equ    DLP1+LP1K5L+1    ; data area for 1.5 kHz LP filter 2
SINCOS  .equ    DLP2+LP1K5L+1    ; SIN/COS table
DLAST   .equ    SINCOS+SINLEN    ; first available location in data area

DLEN    .equ    DLAST-DAB        ; total length of data area

;***********************************************************
;
; Program Area
;
;***********************************************************

        .org    000h            ; External program memory starts at 0
;
; Interrupt vectors
;
RESET   B       INIT            ; Reset vector
INT0    B       ISRV            ; Interrupt 0
INT1    B       ISRV            ; Interrupt 1
INT2    B       ISRV            ; Interrupt 2
        .block  16              ; Reserved locations (16 words)
TINT    B       TIMER+POF1      ; Timer
RINT    B       ISRV            ; Serial port receive
XINT    B       ISRV            ; Serial port transmit
USER    B       ISRV            ; Trap vector

;*****************************************************************
; DAC Offset Initialization Routine
;*****************************************************************

P0START:                        ; Start of internal memory program
POF0    .EQU    PAB-P0START     ; offset for branch addresses

;
; Get offset for DAC1
;
        LACK    ADCDAC1         ; Select DAC1 input
        CALL    ADCCONV+POF0    ; Do the A/D conversion

        ZAC
        LAC     ADCVAL          ; multiply by 5/8 to correct for FS diff
        SFR                     ; divide by 8
        SFR
        SFR
        SACL    DACOFF1,2       ; *5
        ADD     DACOFF1
        ADDK    15*16           ; patch for greater accuracy

        SACL    DACOFF1         ; save the offset correction
;
; Get offset for DAC2
;
        LACK    ADCDAC2         ; Select DAC1 input
        CALL    ADCCONV+POF0    ; Do the A/D conversion

        ZAC
        LAC     ADCVAL          ; multiply by 5/8 to correct for FS diff
        SFR                     ; divide by 8
        SFR
        SFR
        SACL    DACOFF2,2       ; *5
        ADD     DACOFF2
        SACL    DACOFF2         ; save the offset correction

        RET

;
; Subroutine: Do an ADC conversion and wait for result.
;             The ADC MUX address should be in the low ACC on entry.
;
ADCCONV:
        SACL    ADCBUF          ; Begin the next conversion
        OUT     ADCBUF,ADC
        RPTK    200-1           ; Wait 20 microseconds for the conversion
        NOP
        IN      ADCVAL,ADC      ; Read the result
        LAC     ADCVAL
        ANDK    0FFF0h          ; Mask to exactly 12 bits
        SACL    ADCVAL
        RET

LPROG0  .equ    $-P0START

;*****************************************************************
; Idle loop runs while timer interrupt is enabled
;*****************************************************************

P1START:                        ; Start of internal program for processing
POF1    .equ    PAB-P1START     ; offset for branch addresses in internal memory

EVER:
        IDLE
        B       EVER+POF1

;*****************************************************************
; Timer interrupt routine
;*****************************************************************

TIMER:
        LARP    7               ; Use ARP 7 by default

;-----------------------------------------------------------------------
; DAC and ADC cycles
;-----------------------------------------------------------------------

        ; Send the interpolator outputs to the DACs

        SOVM
        ZALH    DACVAL1         ; Add offset to value with overflow mode
        ADDH    DACVAL1         ; Double it to reverse original scaling
        ADDH    DACOFF1         ; Correct for DC offset
        SACH    DACVAL1         ; Reload so we can deal with the sign bit
        LAC     DACVAL1
        XORK    1,15            ; Flip the sign bit for DACS
        SACL    DACVAL1

        ZALH    DACVAL2         ; Add offset to value with overflow mode
        ADDH    DACVAL2         ; Double it to reverse original scaling
        ADDH    DACOFF2         ; Correct for DC offset
        SACH    DACVAL2
        LAC     DACVAL2         ; Repeat for DAC2
        XORK    1,15
        SACL    DACVAL2

        OUT     DACVAL1,DAC1    ; Output data to both DACS
        OUT     DACVAL2,DAC2
        IN      DACBUF,LDAC     ; Simultaneous DAC load

        ; Get the next A/D sample and save it

        IN      ADCVAL,ADC      ; Read the results of the last A/D conversion
        LAC     ADCVAL
        SFR                     ; Scale: Divide by 2 to avoid overflows
        ANDK    0FFF8H          ; Mask to exactly 12 data bits
        LRLK    AR7,DDEC        ; store at the start of the decimator's buffer
        SACL    *

        ; Initiate the next A/D conversion

        LACK    ADCEXT          ; Select ADC external input
        SACL    ADCBUF          ; and begin the next conversion
        OUT     ADCBUF,ADC

;-----------------------------------------------------------------------
; Input decimation filter - flat to 3kHz, down 65 dB at 6 kHz
;-----------------------------------------------------------------------

        ZAC                     ; Clear the accumulator
        MPYK    0               ; Clear the P register
        LRLK    AR7,DDEC+LP3KL-1 ; Point to the last sample

        RPTK    LP3KL-1         ; Do the convolution
        MACD    LP3K+COF,*-

        APAC                    ; add the last product
        SACH    DECVAL,1        ; save the decimator output

        ; Branch on the phase

        LAC     PHASE           ; branch on phase
        BNZ     PHASE1+POF1

;-----------------------------------------------------------------------
; Phase 0 processing
;-----------------------------------------------------------------------

        LAC     DECVAL          ; use the decimator sample when phase = 0
        SACL    USEVAL          ; this value is used for BOTH phase 0 and 1
                                ; the deimator output in phase 1 is discarded
;
; Modulate by the SIN subcarrier
;
        LARP    1               ; sw to AR1
        LT      USEVAL          ; load the data sample
        MPY     *+,7            ; multiply by the SIN value (via AR1); sw to AR7
                                ; and increment the table pointer (to get COS)

        LRLK    AR7,DLP1        ; prepare to save product in DLP1
        SPM     1               ; shift product left 1 bit during SPH
        SPH     *,1             ; save the product for 1.5 kHz filter; sw to AR1
        SPM     0               ; back to no PM shift

        ; Test for table limits

        CMPR    2               ; Is AR1 > AR0
        BBZ     LPFILT1+POF1    ; If AR1 <= AR0, skip the next instr
        LRLK    AR1,SINCOS      ; Point to base of table again

;
; Do the sharp LP filter (fp = 1.2 kHz; fs = 1.5 kHz) for the SIN subcarrier
;

LPFILT1:
        LARP    7               ; sw to AR7

        ZAC                     ; Clear the accumulator
        MPYK    0               ; Clear the P register
        LRLK    AR7,DLP1+LP1K5L-1 ; Point to the last sample

        RPTK    LP1K5L-1        ; Do the convolution
        MACD    LP1K5+COF,*-

        APAC                    ; add the last product

        LRLK    AR7,P0OUT       ; save the filter's output
        SACH    *,2             ; 2x correction for subcarrier modulation

;
; Send 0's to the interpolators for phase 0
;

        ZAC
        LRLK    AR7,DINT1       ; 0 to interpolator 1
        SACL    *
        LRLK    AR7,DINT2       ; 0 to interpolator 2
        SACL    *

        B       INTERPOLATE+POF1 ; done with phase 0 processing

;-----------------------------------------------------------------------
; Phase 1 processing
;-----------------------------------------------------------------------

PHASE1:
                                ; discard the decimator sample when phase = 1
;
; Modulate by the COS subcarrier
;

        LARP    2               ; sw to AR2
        LT      USEVAL          ; load the data sample
        MPY     *+,7            ; multiply by the COS value (via AR2); sw to AR7

        LRLK    AR7,DLP2        ; prepare to save product in DLP2
        SPM     1               ; shift product left 1 bit during SPH
        SPH     *,2             ; save the product for 1.5 kHz filter; sw to AR2
        SPM     0               ; back to no PM shift

        ; Test for table limits

        CMPR    2               ; Is AR2 > AR0
        BBZ     LPFILT2+POF1    ; If AR2 <= AR0, skip the next instr
        LRLK    AR2,SINCOS      ; Point to base of table again
;
; Do the sharp LP filter (fp = 1.2 kHz; fs = 1.5 kHz) for the SIN subcarrier
;
LPFILT2:
        LARP    7

        ZAC                     ; Clear the accumulator
        MPYK    0               ; Clear the P register
        LRLK    AR7,DLP2+LP1K5L-1 ; Point to the last sample

        RPTK    LP1K5L-1        ; Do the convolution
        MACD    LP1K5+COF,*-

        APAC                    ; add the last product

        LRLK    AR7,P1OUT       ; save the filter's output
        SACH    *,2             ; 2x correction for subcarrier modulation

;
; Send the 1.5 kHz filter outputs to the interpolators in phase 1
;

        LRLK    AR7,P0OUT       ; move P0OUT to DINT1
        LAC     *
        LRLK    AR7,DINT1
        SACL    *

        LRLK    AR7,P1OUT       ; move P1OUT to DINT2
        LAC     *
        LRLK    AR7,DINT2
        SACL    *

;-----------------------------------------------------------------------
; Interpolation (2X)
;-----------------------------------------------------------------------

INTERPOLATE:
;
; Run interpolation filter 1 - same coeff. as the input filter
;
        ZAC                     ; Clear the accumulator
        MPYK    0               ; Clear the P register
        LRLK    AR7,DINT1+LP3KL-1 ; Point to the last sample

        RPTK    LP3KL-1         ; Do the convolution
        MACD    LP3K+COF,*-

        APAC                    ; add the last product
        SACH    DACVAL1,2       ; and save the final result
                                ;   x2 for interpolator
;
; Run interpolation filter 2 - same coeff. as the input filter
;
        ZAC                     ; Clear the accumulator
        MPYK    0               ; Clear the P register
        LRLK    AR7,DINT2+LP3KL-1 ; Point to the last sample

        RPTK    LP3KL-1         ; Do the convolution
        MACD    LP3K+COF,*-

        APAC                    ; add the last product
        SACH    DACVAL2,2       ; save the final result after doubling it
                                ; Scaling is as follows:
                                ;   x2 for interpolator

;
; Flip the phase bit (alternating 0/1)
;
        LAC     PHASE
        XOR     ONE
        SACL    PHASE
;
; return from interrupt
;
        EINT
        RET

LPROG1  .equ    $-P1START

;*****************************************************************
; FIR Filter coefficients -- IN REVERSE ORDER
;*****************************************************************

COEFF:
COF     .EQU    CAB-COEFF  ; offset for coefficient addresses

;-----------------------------------------------------------------------
; 3 kHz LP filter: fp = 3 kHz; fs = 6 kHz (-60 dB); at fs = 24 kHz
;-----------------------------------------------------------------------

LP3K:
        .DW     -27     ; h[30] = -0.0008270074
	.DW	-37   	; h[29] = -0.001137216
	.DW	38    	; h[28] = 0.00115065
	.DW	145   	; h[27] = 0.004432736
	.DW	82    	; h[26] = 0.002490338
	.DW	-231  	; h[25] = -0.007057638
	.DW	-419  	; h[24] = -0.01277569
	.DW	6     	; h[23] = 0.0001743103
	.DW	800   	; h[22] = 0.02440476
	.DW	820   	; h[21] = 0.02503499
	.DW	-615  	; h[20] = -0.01876843
	.DW	-2193 	; h[19] = -0.06691592
	.DW	-1198 	; h[18] = -0.036562
	.DW	3540  	; h[17] = 0.1080174
	.DW	9530  	; h[16] = 0.2908378
	.DW	12277 	; h[15] = 0.3746673
	.DW	9530  	; h[14] = 0.2908378
	.DW	3540  	; h[13] = 0.1080174
	.DW	-1198 	; h[12] = -0.036562
	.DW	-2193 	; h[11] = -0.06691592
	.DW	-615  	; h[10] = -0.01876843
	.DW	820   	; h[ 9] = 0.02503499
	.DW	800   	; h[ 8] = 0.02440476
	.DW	6     	; h[ 7] = 0.0001743103
	.DW	-419  	; h[ 6] = -0.01277569
	.DW	-231  	; h[ 5] = -0.007057638
	.DW	82    	; h[ 4] = 0.002490338
	.DW	145   	; h[ 3] = 0.004432736
	.DW	38    	; h[ 2] = 0.00115065
	.DW	-37   	; h[ 1] = -0.001137216
	.DW	-27   	; h[ 0] = -0.0008270074
LP3KL   .EQU    $-LP3K

;-----------------------------------------------------------------------
; 1.2 kHz LP filter: fp = 1.2 kHz; fs = 1.5 kHz (-60 dB); at fs = 12 kHz
;-----------------------------------------------------------------------

#if NOTUSED
LP1K5:
        .DW     0
        .DW     32767/4
        .DW     0

LP1K5L  .EQU    $-LP1K5
#endif


LP1K5:
        .DW     12      ; h[114] = 0.0003579154
	.DW	-4    	; h[113] = -0.0001160035
	.DW	-16   	; h[112] = -0.0004779771
	.DW	-31   	; h[111] = -0.0009583998
	.DW	-44   	; h[110] = -0.001351943
	.DW	-47   	; h[109] = -0.00142799
	.DW	-34   	; h[108] = -0.001041864
	.DW	-8    	; h[107] = -0.0002372037
	.DW	24    	; h[106] = 0.000723273
	.DW	47    	; h[105] = 0.001435165
	.DW	50    	; h[104] = 0.00152474
	.DW	28    	; h[103] = 0.00084648
	.DW	-13   	; h[102] = -0.0003930746
	.DW	-55   	; h[101] = -0.001667773
	.DW	-77   	; h[100] = -0.002336739
	.DW	-64   	; h[99] = -0.00194942
	.DW	-17   	; h[98] = -0.0005109013
	.DW	47    	; h[97] = 0.001437934
	.DW	98    	; h[96] = 0.003000967
	.DW	109   	; h[95] = 0.003322037
	.DW	66    	; h[94] = 0.002028882
	.DW	-16   	; h[93] = -0.0004947418
	.DW	-104  	; h[92] = -0.003178654
	.DW	-154  	; h[91] = -0.004697695
	.DW	-134  	; h[90] = -0.00410251
	.DW	-45   	; h[89] = -0.001359473
	.DW	82    	; h[88] = 0.002487245
	.DW	186   	; h[87] = 0.00567566
	.DW	213   	; h[86] = 0.006505311
	.DW	138   	; h[85] = 0.004202633
	.DW	-18   	; h[84] = -0.0005518302
	.DW	-188  	; h[83] = -0.005741972
	.DW	-289  	; h[82] = -0.00883278
	.DW	-261  	; h[81] = -0.007958497
	.DW	-97   	; h[80] = -0.002958992
	.DW	140   	; h[79] = 0.004286409
	.DW	344   	; h[78] = 0.01048941
	.DW	406   	; h[77] = 0.01237811
	.DW	273   	; h[76] = 0.008327125
	.DW	-19   	; h[75] = -0.0005804994
	.DW	-349  	; h[74] = -0.010665
	.DW	-559  	; h[73] = -0.01705455
	.DW	-521  	; h[72] = -0.01589387
	.DW	-210  	; h[71] = -0.006411244
	.DW	268   	; h[70] = 0.008169306
	.DW	704   	; h[69] = 0.02148374
	.DW	869   	; h[68] = 0.02652662
	.DW	619   	; h[67] = 0.01889345
	.DW	-19   	; h[66] = -0.0005924444
	.DW	-824  	; h[65] = -0.02513188
	.DW	-1431 	; h[64] = -0.04367707
	.DW	-1466 	; h[63] = -0.04473386
	.DW	-685  	; h[62] = -0.02091374
	.DW	902   	; h[61] = 0.02753454
	.DW	3006  	; h[60] = 0.09172653
	.DW	5121  	; h[59] = 0.1562853
	.DW	6686  	; h[58] = 0.2040362
	.DW	7262  	; h[57] = 0.2216273
	.DW	6686  	; h[56] = 0.2040362
	.DW	5121  	; h[55] = 0.1562853
	.DW	3006  	; h[54] = 0.09172653
	.DW	902   	; h[53] = 0.02753454
	.DW	-685  	; h[52] = -0.02091374
	.DW	-1466 	; h[51] = -0.04473386
	.DW	-1431 	; h[50] = -0.04367707
	.DW	-824  	; h[49] = -0.02513188
	.DW	-19   	; h[48] = -0.0005924444
	.DW	619   	; h[47] = 0.01889345
	.DW	869   	; h[46] = 0.02652662
	.DW	704   	; h[45] = 0.02148374
	.DW	268   	; h[44] = 0.008169306
	.DW	-210  	; h[43] = -0.006411244
	.DW	-521  	; h[42] = -0.01589387
	.DW	-559  	; h[41] = -0.01705455
	.DW	-349  	; h[40] = -0.010665
	.DW	-19   	; h[39] = -0.0005804994
	.DW	273   	; h[38] = 0.008327125
	.DW	406   	; h[37] = 0.01237811
	.DW	344   	; h[36] = 0.01048941
	.DW	140   	; h[35] = 0.004286409
	.DW	-97   	; h[34] = -0.002958992
	.DW	-261  	; h[33] = -0.007958497
	.DW	-289  	; h[32] = -0.00883278
	.DW	-188  	; h[31] = -0.005741972
	.DW	-18   	; h[30] = -0.0005518302
	.DW	138   	; h[29] = 0.004202633
	.DW	213   	; h[28] = 0.006505311
	.DW	186   	; h[27] = 0.00567566
	.DW	82    	; h[26] = 0.002487245
	.DW	-45   	; h[25] = -0.001359473
	.DW	-134  	; h[24] = -0.00410251
	.DW	-154  	; h[23] = -0.004697695
	.DW	-104  	; h[22] = -0.003178654
	.DW	-16   	; h[21] = -0.0004947418
	.DW	66    	; h[20] = 0.002028882
	.DW	109   	; h[19] = 0.003322037
	.DW	98    	; h[18] = 0.003000967
	.DW	47    	; h[17] = 0.001437934
	.DW	-17   	; h[16] = -0.0005109013
	.DW	-64   	; h[15] = -0.00194942
	.DW	-77   	; h[14] = -0.002336739
	.DW	-55   	; h[13] = -0.001667773
	.DW	-13   	; h[12] = -0.0003930746
	.DW	28    	; h[11] = 0.00084648
	.DW	50    	; h[10] = 0.00152474
	.DW	47    	; h[ 9] = 0.001435165
	.DW	24    	; h[ 8] = 0.000723273
	.DW	-8    	; h[ 7] = -0.0002372037
	.DW	-34   	; h[ 6] = -0.001041864
	.DW	-47   	; h[ 5] = -0.00142799
	.DW	-44   	; h[ 4] = -0.001351943
	.DW	-31   	; h[ 3] = -0.0009583998
	.DW	-16   	; h[ 2] = -0.0004779771
	.DW	-4    	; h[ 1] = -0.0001160035
	.DW	12    	; h[ 0] = 0.0003579154
LP1K5L  .EQU    $-LP1K5


LCOEFF  .EQU    $-COEFF

;*****************************************************************
; Dummy Interrupt Service Routine (not actually used)
;*****************************************************************

ISRV    EINT
        RET

;*****************************************************************
; Processor Initialization
;*****************************************************************

INIT    ROVM                    ; Set overflow mode
        SSXM                    ; Set sign extension mode
        RXF                     ; Clear the XF bit
        LDPK    0               ; Point to data memory page 0
        LARP    7               ; Point to auxiliary register 7
        CONF    0               ; Configure all internal RAM
                                ; blocks as data memory
#IF TIMERINT
        LACK    08h             ; Load accumulator with interrupt mask
#ELSE
        LACK    00h             ; Disable all interrupts
#ENDIF
        SACL    IMR             ; Enable timer interrupt only
;
; Initialize the DACS to 0
;
        LALK    8000h           ; This value corresponds to 0V at the DACs
        SACL    DACVAL1
        OUT     DACVAL1,DAC1    ; Output data to both DACS
        OUT     DACVAL1,DAC2
        IN      DACBUF,LDAC     ; Simultaneous DAC load
;
; Initialize internal data memory
;
        ZAC                     ; Zero the accumulator
        LARK    AR7,B2D         ; Point to block B2D
        RPTK    31              ; Store zero in all 32 locations
        SACL    *+
;
        LRLK    AR7,B0D         ; Point to block B0D
        LARK    AR6,5           ; Repeat LOOP1 6 times
LOOP1   RPTK    255             ; Zeroing block B0D, B1D, and B3D
        SACL    *+              ; Zero the pages 4-15
        LARP    AR6
        BANZ    LOOP1,*-,AR7    ; Repeat 6 times
;
; Set constant values
;
        LACK    1               ; Value = 1
        SACL    ONE
        LALK    32767           ; Value = max positive number
        SACL    MAXPOS
;
; Copy the DAC initialization routine to internal memory
;
        LARP    7
        LRLK    AR7,PABD        ; this copy is done into data space !
        RPTK    LPROG0-1
        BLKP    P0START,*+
;
; Go to the initialization routine in internal memory
;
        CONF    2               ; Set blocks B0 and B1 as program memory

        LARP    7               ; Long delay to let the DACs settle down
        LARK    AR7,10          ; Repeat delay 10 times
LOOP2:  RPTK    255             ; Zeroing block B0D, B1D, and B3D
        NOP
        BANZ    LOOP2,*-

        CALL    PAB             ; Read and store the DAC DC offsets
;
; Copy the timer interrupt routine to internal program memory
;
        CONF    0
        LARP    7
        LRLK    AR7,PABD        ; this copy is done into data space !
        RPTK    LPROG1-1
        BLKP    P1START,*+
;
; Copy the FIR filter coefficients to internal program memory
;
        LRLK    AR7,CABD        ; this copy is done into data space !
        RPTK    LCOEFF-1
        BLKP    COEFF,*+
;
; Copy the SIN/COS table to internal data memory
;
        LRLK    AR7,SINCOS
        RPTK    SINLEN-1
        BLKP    SINTAB,*+
;
; Initialize the SINCOS table pointers
;
        LRLK    AR1,SINCOS          ; SIN current entry (moves)
        LRLK    AR2,SINCOS+2        ; COS current entry (moves)
        LRLK    AR0,SINCOS+SINLEN-1 ; Top of table (fixed)
;
; Initialize the timer interrupt rate
;
        LALK    TRATE-1         ; Interrupt every TRATE ticks
        SACL    PRD
;
; Go to idle loop in internal memory
;
        CONF    2               ; Set blocks B0 and B1 as program memory
#IF TIMERINT
        EINT                    ; enable the timer interrupt
#ENDIF
        B       PAB             ; branch to PSTART in B0P

;*****************************************************************
; Sine/Cosine table for 1500 Hz subcarrier generation
;*****************************************************************

SINTAB:
        .DW     0       ; sin(0)
        .DW     23170   ; sin(pi/4)
        .DW     32767   ; sin(pi/2)
        .DW     23170   ; sin(3pi/4)
        .DW     0       ; sin(pi)
        .DW     -23170  ; sin(5pi/4)
        .DW     -32767  ; sin(3pi/2)
        .DW     -23170  ; sin(7pi/4)

SINLEN  .equ    $-SINTAB

;*****************************************************************
; PUTWORD: Send a word to the host via the parallel port (for debugging)
;*****************************************************************

POF9    .EQU    0

PUTWORD:
        SXF                     ; Send the "start" bit

;
; Send a byte LSB first
;
        LARP    AR6
        LARK    AR6,16          ; Loop 17 times

BYTELOOP:
        BIOZ    BYTELOOP+POF9   ; Wait till BIO\ goes high
        NOP
        NOP

WAITLOW:                        ; Wait till BIO\ goes low
        BIOZ    BIOLOW+POF9
        B       WAITLOW+POF9
BIOLOW:

        SFR                     ; Shift LSB into carry
        BC      HIBIT+POF9
        RXF                     ; Send 0 bit
        B       NEXTBIT+POF9
HIBIT:
        SXF                     ; Send 1 bit

NEXTBIT:
        BANZ    BYTELOOP+POF9,*-

        RXF                     ; Send the "stop" bit

        RET

        .END
