/****************************************************************/
/*                                                              */
/* AVRSTEST: Hybrid UHF/microwave synthesizer test bed          */
/*           for ATmega128                                      */
/*                                                              */
/* Compile with AVR-GCC                                         */
/*                                                              */
/* ke5fx@qsl.net                                                */
/*                                                              */
/****************************************************************/

#define VERSION "1.00"

#include <iom128.h>
#include <interrupt.h>
#include <sig-avr.h>

#include <stdlib.h>
#include <string.h>

#include "typedefs.h"      
#include "avr232.cpp"      // Serial-port access for terminal emulation
#include "synth.cpp"       // Multiplatform synthesizer access class

//
// Misc. globals
//

volatile S32 wall_clock_msec = 0;

/******************************************/
//
// 1-millisecond timer for profiling
//
/******************************************/

SIGNAL(SIG_OVERFLOW0)
{
   ++wall_clock_msec;
}

//******************************************
//
// main()
//
//******************************************

int main(void)
{
   //
   // Initialize UART for 38400 bps, given 16 MHz system clock
   //

   InitUART(38400, 16000000);

   //
   // Set up timer 0 to generate 1-millisecond interrupts for profiling
   //

   outp(0x00,TCCR0);    // stop timer
   outp(0x00,TCNT0);    // set count value
   outp(0x03,TCCR0);    // start timer with x32 prescaling (div clk by 512*32)
    
   outp(OCIE0,TIMSK);   // Enable timer 0 interrupt

   //
   // Display signon banner
   // 

   printf("\n-------------------------------------------------------------------------\n");
   printf("Atmel AVR test bed for DDS/PLL hybrid synthesizer\n");
   printf("V"VERSION" of "__DATE__" by John Miles (ke5fx@qsl.net)\n");
   printf("\n");
   printf("    FRQ n sets frequency in Hz (3- or 4-digit value treated as MHz)\n");
   printf("\n");
   printf("-------------------------------------------------------------------------\n");

   //
   // Initialize synthesizer
   //

   SYNTH synth(SYNTH_PORT_A,     // Use AVR port A
               ADF411X,          // Talk to Analog Devices ADF4112 chip
               970000000,        // Minimum frequency 970 MHz for ROS-2150VW VCO
               10700000,         // DDS center frequency = 10.7 MHz
               10000000,         // DDS clock input = 10 MHz
               8,                // Minimum DDS clock multiplier = 8X
               12);              // Maximum DDS clock multiplier = 12X 

   //
   // Enter terminal mode
   //

   C8 command[32];
   S8 command_received = FALSE;
   S8 command_done     = TRUE;
   U8 i = 0;

   S32 command_start_time = 0;
   S32 command_end_time   = 0;

   strcpy(command,"FRQ 1000");
   command_received = TRUE;

   while (1)
      {
      //
      // Check for incoming keypress and add it to command string
      //

      if (command_done)
         {
         printf("\nCommand: ");
         command_done = FALSE;
         }

      if (DataInReceiveBuffer())
         {
         C8 ch = ReceiveByte();
         TransmitByte(ch);

         if (ch == '\r')
            {
            TransmitByte('\n');
            command[i] = 0;
            i = 0;
            command_received = TRUE;
            }
         else if (ch == 8)
            {
            if (i > 0)
               {
               i--;
               }
            }
         else
            {
            if (i >= sizeof(command))
               {
               printf("Command too long! Max %d characters\n",sizeof(command));
               i = 0;
               }
            else
               {
               command[i++] = ch;
               }
            }
         }

      //
      // If we have a command, process it
      //

      if (!command_received)
         {
         continue;
         }

      command_received = FALSE;

      strupr(command);

      if (!strncmp(command,"FRQ",3))
         {
         // ----------------------------------------
         // FRQ: Set frequency
         // ----------------------------------------

         U32 Hz = atol(&command[4]);

         if (Hz < 9999)
            {
            Hz *= 1000000;
            }

         printf("Setting frequency=%lu Hz ... ",Hz);
 
         command_start_time = wall_clock_msec;

         synth.set_frequency(Hz);        

         command_end_time = wall_clock_msec;

         //
         // Wait 1000 ms for lock
         //

         S8 locked = FALSE;

         while (wall_clock_msec < (command_end_time + 1000))
            {
            if (synth.locked())
               {
               S32 lock_time = wall_clock_msec - command_end_time;

               printf("Locked in %ld milliseconds\n",lock_time);
               locked = TRUE;
               break;
               }
            }

         if (!locked)
            {
            printf("ERROR: Lock failed\n");
            }  

         printf("Processed by AVR in %ld milliseconds\n",command_end_time - command_start_time);
         }
      else
         {
         // ----------------------------------------
         // Unknown or blank command
         // ----------------------------------------

         if (command[0])
            {
            printf("Unknown command: %s\n",command);
            }
         }

      command_done = TRUE;
      }
}
