//
// RS-232 support routines for ATmega128 in native mode
//

#include <stdarg.h>
#include <ctype.h>
#include <string.h>

/* UART Buffer Defines */
#define UART_RX_BUFFER_SIZE 32     /* 2,4,8,16,32,64,128 or 256 bytes */
#define UART_TX_BUFFER_SIZE 32

#define UART_RX_BUFFER_MASK ( UART_RX_BUFFER_SIZE - 1 )
#if ( UART_RX_BUFFER_SIZE & UART_RX_BUFFER_MASK )
   #error RX buffer size is not a power of 2
#endif

#define UART_TX_BUFFER_MASK ( UART_TX_BUFFER_SIZE - 1 )
#if ( UART_TX_BUFFER_SIZE & UART_TX_BUFFER_MASK )
   #error TX buffer size is not a power of 2
#endif

/* Static Variables */
static unsigned char UART_RxBuf[UART_RX_BUFFER_SIZE];
static volatile unsigned char UART_RxHead;
static volatile unsigned char UART_RxTail;
static unsigned char UART_TxBuf[UART_TX_BUFFER_SIZE];
static volatile unsigned char UART_TxHead;
static volatile unsigned char UART_TxTail;

/* Prototypes */
void InitUART( unsigned long baud, unsigned long Hz);
unsigned char ReceiveByte( void );
void TransmitByte( unsigned char data );
unsigned char DataInReceiveBuffer( void );

/* Main - a simple test program*/
#ifdef STANDALONE_AVR232

#include <iom128.h>
#include <interrupt.h>
#include <sig-avr.h>

int main( void )
{
   InitUART( 38400, 16000000 );      // 38400 baud at 16 MHz

   for( ; ; )                        /* Forever */
   {
   TransmitByte( ReceiveByte() );    /* Echo the received character */
   }

   return 0;
}
#endif

/* Initialize UART */
void InitUART( unsigned long baud, unsigned long Hz)
{
   cli();

   unsigned char x;

   UBRR0L = ((((Hz * 10) / (baud * 16)) + 5) / 10) - 1; 
   UBRR0H = 0;

   //
   // Enable UART receiver and transmitter and receive interrupt
   // Use default frame format: 8 bits, no parity, 1 stop bit
   //

   UCSR0B = ( (1<<RXCIE) | (1<<RXEN) | (1<<TXEN) );

   x = 0;              /* Flush receive buffer */

   UART_RxTail = x;
   UART_RxHead = x;
   UART_TxTail = x;
   UART_TxHead = x;

   sei();            /* Enable interrupts => enable UART interrupts */
}

/* Interrupt handlers */
SIGNAL(SIG_UART0_RECV)
{
   unsigned char data;
   unsigned char tmphead;

   data = UDR0;                /* Read the received data */
                               /* Calculate buffer index */
   tmphead = ( UART_RxHead + 1 ) & UART_RX_BUFFER_MASK;
   UART_RxHead = tmphead;      /* Store new index */

   if ( tmphead == UART_RxTail )
   {
      /* ERROR! Receive buffer overflow */
   }
   
   UART_RxBuf[tmphead] = data; /* Store received data in buffer */
}

SIGNAL(SIG_UART0_DATA)
{
   unsigned char tmptail;
                                    /* Check if all data is transmitted */
   if ( UART_TxHead != UART_TxTail )
   {
                                    /* Calculate buffer index */
      tmptail = ( UART_TxTail + 1 ) & UART_TX_BUFFER_MASK;
      UART_TxTail = tmptail;        /* Store new index */
   
      UDR0 = UART_TxBuf[tmptail];   /* Start transmition */
   }
   else
   {
      UCSR0B &= ~(1<<UDRIE);        /* Disable UDRE interrupt */
   }
}

/* Read and write functions */
unsigned char ReceiveByte( void )
{
   unsigned char tmptail;
   
   while ( UART_RxHead == UART_RxTail ); /* Wait for incoming data */

   tmptail = ( UART_RxTail + 1 ) & UART_RX_BUFFER_MASK;/* Calculate buffer index */
   
   UART_RxTail = tmptail;                /* Store new index */
   
   return UART_RxBuf[tmptail];           /* Return data */
}

void TransmitByte( unsigned char data )
{
   unsigned char tmphead;
                                         /* Calculate buffer index */
   tmphead = ( UART_TxHead + 1 ) & UART_TX_BUFFER_MASK; /* Wait for free space in buffer */

   while ( tmphead == UART_TxTail );

   UART_TxBuf[tmphead] = data;           /* Store data in buffer */
   UART_TxHead = tmphead;                /* Store new index */

   UCSR0B |= (1<<UDRIE);                 /* Enable UDRE interrupt */
}

unsigned char DataInReceiveBuffer( void )
{
   return ( UART_RxHead != UART_RxTail ); /* Return 0 (FALSE) if the receive buffer is empty */
}

// ---------------------------------------
//
// vsprintf() adapted from http://www.mikrocontroller.net/avr-gcc-faq.en.htm
//
// ---------------------------------------

#define SCRATCH 32

int vsprintf(char *buf, const char *format, va_list ap)
/* simplified sprintf */
{
  char scratch[SCRATCH];
  unsigned char format_flag;
  unsigned char long_flag = 0, sign_flag = 0;
  unsigned long u_val=0, sign, base;
  int field_len = -1;
  char *ptr;

  *buf=0;

  for (;;)
    {
    while ((format_flag = *format++) != '%')
      {      /* Until '%' or '\0' */
      if (!format_flag)
         { 
         return (0);
         }

      *buf = format_flag; 
      buf++; 
      *buf=0;
      }

   if (isdigit(*format))
      {
      field_len = *format++ - '0';
      }

HANDLE_FORMAT:
    switch (format_flag = *format++)
    {
    case 'c':
      format_flag = va_arg(ap,int);
    default:
      *buf = format_flag; buf++; *buf=0;
      continue;
    case 'l':
      long_flag = 1;
      goto HANDLE_FORMAT;
    case 'S':
    case 's':
      ptr = va_arg(ap,char *);
      strcat(buf, ptr);
      buf += strlen(ptr);
      continue;
    case 'o':
      base = 8;
      sign_flag = 0;
      *buf = '0'; buf++; *buf=0;
      goto CONVERSION_LOOP;
    case 'i':
    case 'd':
      base = 10;
      sign_flag = 1;
      goto CONVERSION_LOOP;
      /* no break -> run into next case */
    case 'u':
      base = 10;
      sign_flag = 0;
      goto CONVERSION_LOOP;
    case 'x':
    case 'X':
      base = 16;
      sign_flag = 0;

    CONVERSION_LOOP:
      
      sign = 0;

      if (long_flag)
         {
         long_flag = 0;
         u_val = va_arg(ap,unsigned long);

         if (sign_flag && (u_val > 0x7fffffff))
            {
            sign = '-';
            u_val = -u_val;
            }
         }
      else
         {
         u_val = va_arg(ap,unsigned int);

         if (sign_flag && (u_val > 0x7fff))
            {
            sign = '-';
            u_val = 65536L - u_val;
            }
         }

      ptr = scratch + SCRATCH;
      *--ptr = 0;

      do 
         {
         char ch = (u_val % base) + '0';

         if (ch > '9')
            {
            ch += 'A' - '9' - 1;
            }

         *--ptr = ch;
         u_val /= base;
         } 
      while (u_val);

      if (sign)
         {
         *--ptr = sign;
         }

      while ((scratch+SCRATCH)-ptr <= field_len)
         {
         *--ptr = '0';
         }

      field_len = 0;

      strcat(buf, ptr);
      buf += strlen(ptr);
    }
  }
}

int sprintf(char *buf, char *format, ...)
{
   va_list ap;

   va_start(ap, 
            format);

   int result = vsprintf(buf,
                         format,
                         ap);
   va_end(ap);

   return result;
}

int printf(char *format, ...)
{
   char buf[256];

   va_list ap;

   va_start(ap, 
            format);

   int result = vsprintf(buf,
                         format,
                         ap);
   va_end(ap);

   int len = strlen(buf);

   for (int i=0; i < len; i++)
      {
      if (buf[i] == 0x0a)
         {
         TransmitByte(0x0d);
         }

      TransmitByte(buf[i]);
      }

   return result;
}



