unit SerialPort;

{$undef USE_CODESITE}

interface

uses
  Windows, Messages, Classes, Graphics, Controls, Forms, Dialogs;

const
  EvtRxChar  = 0;
  EvtRxFlag  = 1;
  EvtTxEmpty = 2;
  EvtCTS     = 3;
  EvtDSR     = 4;
  EvtRLSD    = 5;
  EvtBreak   = 6;
  EvtErr     = 7;
  EvtRing    = 8;

  (*
   * From WINDOWS.PAS
   * ----------------
   * EV_RXCHAR  =   $1;  // Any Character received
   * EV_RXFLAG  =   $2;  // Received certain character
   * EV_TXEMPTY =   $4;  // Transmit Queue Empty
   * EV_CTS     =   $8;  // CTS changed state
   * EV_DSR     =  $10;  // DSR changed state
   * EV_RLSD    =  $20;  // RLSD changed state
   * EV_BREAK   =  $40;  // BREAK received
   * EV_ERR     =  $80;  // Line status error occurred
   * EV_RING    = $100;  // Ring signal detected
   *)

type
  TmeControlState = ( ctrlDisable, ctrlEnable, ctrlHandshake, ctrlToggle );
  TmeParity       = ( parNone, parOdd, parEven, parMark, parSpace );
  TmeStopBits     = ( sbOne, sbOneHalf, sbTwo );

const
  // Defaults
  DEF_ABORTONERROR                = false;
  DEF_BAUD                        = 9600;
  DEF_BINARY                      = true;  // only option in Windows
  DEF_BYTESIZE                    = 8;
  DEF_DSRSENSITIVITY              = false;
  DEF_DTRCONTROL                  = ctrlDisable;
  DEF_EOFCHAR                     = #0;
  DEF_ERRORCHAR                   = #0;
  DEF_EVTCHAR                     = #0;
  DEF_INX                         = false;
  DEF_OUTX                        = false;
  DEF_OUTXCTSFLOW                 = false;
  DEF_OUTXDTRFLOW                 = false;
  DEF_PARITY                      = parNone;
  DEF_PARITYCHECK                 = false;
  DEF_REPLACEERROR                = false;
  DEF_RTSCONTROL                  = ctrlDisable;
  DEF_STOPBITS                    = sbOne;
  DEF_STRIPNULL                   = false;
  DEF_TXCONTINUONXOFF             = false;
  DEF_XOFFCHAR                    = #$19;  // DC3
  DEF_XOFFLIM                     = 0;
  DEF_XONCHAR                     = #$17;  // DC1
  DEF_XONLIM                      = 0;
  DEF_READINTERVALTIMEOUT         = 0;
  DEF_READTOTALTIMEOUTMULTIPLIER  = 0;
  DEF_READTOTALTIMEOUTCONSTANT    = 0;
  DEF_WRITETOTALTIMEOUTMULTIPLIER = 0;
  DEF_WRITETOTALTIMEOUTCONSTANT   = 0;

type
  TmeComPort = class;

  TmeComThread = class( TThread )
  private
    fOwner        : TmeComPort;
    fEventMask    : DWORD;
    fPauseEvent   : THandle;
    fEventHandles : array[0..1] of THandle;
    fOverlapped   : TOverlapped;
    fComHandle    : THandle;
    procedure SetComHandle( aComHandle : THandle );
  protected
    procedure Execute; override;
  public
    constructor create( aOwner : TmeComPort );
    destructor destroy; override;

    property ComHandle : THandle
      read fComHandle write SetComHandle;
    procedure DoEvents;
  end;

  TmeComPort = class( TComponent )
  private
    fComHandle   : THandle;
    fEventThread : TmeComThread;
    fConnected   : Boolean;
    fPortNum     : byte;

    fWriteOverlapped   : TOverlapped;
    fReadOverlapped    : TOverlapped;
    fAbortEvent        : THandle;
    fReadEventHandles  : array[0..1] of THandle;
    fWriteEventHandles : array[0..1] of THandle;

    // DCB info
    fAbortOnError     : Boolean;
    fBaudRate         : DWORD;
    fBinary           : Boolean;
    fByteSize         : byte;
    fDSRSensitivity   : Boolean;
    fDTRControl       : TmeControlState;
    fEOFChar          : char;
    fErrorChar        : char;
    fEvtChar          : char;
    fInX              : Boolean;
    fOutX             : Boolean;
    fOutxCTSFlow      : Boolean;
    fOutxDSRFlow      : Boolean;
    fParity           : TmeParity;
    fParityCheck      : Boolean;
    fReplaceError     : Boolean;
    fRTSControl       : TmeControlState;
    fStopBits         : TmeStopBits;
    fStripNull        : Boolean;
    fTXContinueOnXOFF : Boolean;
    fXOFFChar         : char;
    fXOFFLim          : word;
    fXONChar          : char;
    fXONLim           : word;

    // timeout
    fReadIntervalTimeout         : DWORD;
    fReadTotalTimeoutMultiplier  : DWORD;
    fReadTotalTimeoutConstant    : DWORD;
    fWriteTotalTimeoutMultiplier : DWORD;

    fWriteTotalTimeoutConstant   : DWORD;

    fEvtMask : DWORD;
    fComEvent : array[EvtRxChar .. EvtRing] of TNotifyEvent;

    function GetComEvent( const aIndex : integer ) : TNotifyEvent;
    procedure SetComEvent( const aIndex : integer; aValue : TNotifyEvent );
    procedure SetupDefaults;
  protected
    procedure SetupState;
  public
    constructor create( aOwner: TComponent ); override;
    destructor destroy; override;

    procedure Open( const aPortNum : integer );
    procedure ResetPort;
    procedure Close;
    function  InQue: integer;
    function  OutQue: integer;
    function  WriteString( const aValue : string ) : Boolean;
    function  ReadBuffer( var Buffer; Count: integer ) : integer;
    procedure PurgeIn;
    procedure PurgeOut;
    property  Connected : Boolean read fConnected;
    property  PortNum : byte read fPortNum;

    procedure SetDTR( const aOn : Boolean );
    procedure SetRTS( const aOn : Boolean );
    procedure SetXON( const aOn : Boolean );
    procedure SetBreak( const aOn : Boolean );

  published
    // DCB Info
    property AbortOnError : Boolean
      read fAbortOnError write fAbortOnError
      default DEF_ABORTONERROR;
    property BaudRate : DWORD
      read fBaudRate write fBaudRate
      default DEF_BAUD;
    property Binary : Boolean
      read fBinary write fBinary
      default DEF_BINARY;
    property ByteSize : byte
      read fByteSize write fByteSize
      default DEF_BYTESIZE;
    property DSRSensitivity : Boolean
      read fDSRSensitivity write fDSRSensitivity
      default DEF_DSRSENSITIVITY;
    property DTRControl : TmeControlState
      read fDTRControl write fDTRControl
      default DEF_DTRCONTROL;
    property EOFChar : char
      read fEOFChar write fEOFChar
      default DEF_EOFCHAR;
    property ErrorChar : char
      read fErrorChar write fErrorChar
      default DEF_ERRORCHAR;
    property EvtChar : char
      read fEvtChar write fEvtChar
      default DEF_EVTCHAR;
    property InX : Boolean
      read fInX write fInX
      default DEF_INX;  // 0
    property OutX : Boolean
      read fOutX write fOutX
      default DEF_OUTX;
    property OutxCTSFlow : Boolean
      read fOutxCTSFlow write fOutxCTSFlow
      default DEF_OUTXCTSFLOW;
    property OutxDSRFlow : Boolean
      read fOutxDSRFlow write fOutxDSRFlow
      default DEF_OUTXDTRFLOW;
    property Parity : TmeParity
      read fParity write fParity
      default DEF_PARITY;
    property ParityCheck : Boolean
      read fParityCheck write fParityCheck
      default DEF_PARITYCHECK;
    property ReplaceError : Boolean
      read fReplaceError write fReplaceError
      default DEF_REPLACEERROR;
    property RTSControl : TmeControlState
      read fRTSControl write fRTSControl
      default DEF_RTSCONTROL;
    property StopBits : TmeStopBits
      read fStopBits write fStopBits
      default DEF_STOPBITS;
    property StripNull  : Boolean
      read fStripNull write fStripNull
      default DEF_STRIPNULL;
    property TXContinueOnXOFF : Boolean
      read fTXContinueOnXOFF write fTXContinueOnXOFF
      default DEF_TXCONTINUONXOFF;
    property XOFFChar : char
      read fXOFFChar write fXOFFChar
      default DEF_XOFFCHAR;
    property XoffLim : word
      read fXOFFLim write fXOFFLim
      default DEF_XOFFLIM;
    property XONChar : char
      read fXONChar write fXONChar
      default DEF_XONCHAR;
    property XonLim : word
      read fXONLim write fXONLim
      default DEF_XONLIM;

    // timeout
    property ReadIntervalTimeout : DWORD
      read fReadIntervalTimeout write fReadIntervalTimeout
      default DEF_READINTERVALTIMEOUT;
    property ReadTotalTimeoutMultiplier : DWORD
      read fReadTotalTimeoutMultiplier write fReadTotalTimeoutMultiplier
      default DEF_READTOTALTIMEOUTMULTIPLIER;
    property ReadTotalTimeoutConstant : DWORD
      read fReadTotalTimeoutConstant write fReadTotalTimeoutConstant
      default DEF_READTOTALTIMEOUTCONSTANT;

    property WriteTotalTimeoutMultiplier : DWORD
      read fWriteTotalTimeoutMultiplier write fWriteTotalTimeoutMultiplier
      default DEF_WRITETOTALTIMEOUTMULTIPLIER;
    property WriteTotalTimeoutConstant   : DWORD
      read fWriteTotalTimeoutConstant write fWriteTotalTimeoutConstant
      default DEF_WRITETOTALTIMEOUTCONSTANT;

    // serial port events
    property OnRxChar  : TNotifyEvent index EvtRxChar
      read GetComEvent write SetComEvent;
    property OnRxFlag  : TNotifyEvent index EvtRxFlag
      read GetComEvent write SetComEvent;
    property OnTxEmtpy : TNotifyEvent index EvtTxEmpty
      read GetComEvent write SetComEvent;
    property OnCTS     : TNotifyEvent index EvtCTS
      read GetComEvent write SetComEvent;
    property OnDSR     : TNotifyEvent index EvtDSR
      read GetComEvent write SetComEvent;
    property OnRLSD    : TNotifyEvent index EvtRLSD
      read GetComEvent write SetComEvent;
    property OnBreak   : TNotifyEvent index EvtBreak
      read GetComEvent write SetComEvent;
    property OnErr     : TNotifyEvent index EvtErr
      read GetComEvent write SetComEvent;
    property OnRing    : TNotifyEvent index EvtRing
      read GetComEvent write SetComEvent;
end;

procedure Register;


implementation

{ ---------------------------------------------------------------------------- }
{ - implementation ----------------------------------------------------------- }
{ ---------------------------------------------------------------------------- }

uses
  {$ifdef USE_CODESITE}
  rzCSIntf,
  {$endif}
  sysutils;

const
  cWriteBufSize = 2048;
  cReadBufSize  = 2048;

procedure CheckCloseHandle( var aHandle : THandle );
begin
  if aHandle <> INVALID_HANDLE_VALUE then
  begin
    CloseHandle(aHandle);
    aHandle := INVALID_HANDLE_VALUE
  end;
end;

procedure TmeComPort.SetupDefaults;
begin
  fAbortOnError     := DEF_ABORTONERROR;
  fBaudRate         := DEF_BAUD;
  fBinary           := DEF_BINARY;
  fByteSize         := DEF_BYTESIZE;
  fDSRSensitivity   := DEF_DSRSENSITIVITY;
  fDTRControl       := DEF_DTRCONTROL;
  fEOFChar          := DEF_EOFCHAR;
  fErrorChar        := DEF_ERRORCHAR;
  fEvtChar          := DEF_EVTCHAR;
  fInX              := DEF_INX;
  fOutX             := DEF_OUTX;
  fOutXCTSFlow      := DEF_OUTXCTSFLOW;
  fOutXDSRFlow      := DEF_OUTXDTRFLOW;
  fParity           := DEF_PARITY;
  fParityCheck      := DEF_PARITYCHECK;
  fReplaceError     := DEF_REPLACEERROR;
  fRTSControl       := DEF_RTSCONTROL;
  fStopBits         := DEF_STOPBITS;
  fStripNull        := DEF_STRIPNULL;
  fTxContinueOnXOff := DEF_TXCONTINUONXOFF;
  fXOFFChar         := DEF_XOFFCHAR;
  fXOFFLim          := DEF_XOFFLIM;
  fXONChar          := DEF_XONCHAR;
  fXONLim           := DEF_XONLIM;

  fReadIntervalTimeout         := DEF_READINTERVALTIMEOUT;
  fReadTotalTimeoutMultiplier  := DEF_READTOTALTIMEOUTMULTIPLIER;
  fReadTotalTimeoutConstant    := DEF_READTOTALTIMEOUTCONSTANT;
  fWriteTotalTimeoutMultiplier := DEF_WRITETOTALTIMEOUTMULTIPLIER;
  fWriteTotalTimeoutConstant   := DEF_WRITETOTALTIMEOUTCONSTANT;
end;

constructor TmeComPort.create(aOwner: TComponent);
begin
  inherited create(aOwner);
  SetupDefaults;

  fComHandle              := INVALID_HANDLE_VALUE;
  fAbortEvent             := INVALID_HANDLE_VALUE;
  fReadOverlapped.hEvent  := INVALID_HANDLE_VALUE;
  fWriteOverlapped.hEvent := INVALID_HANDLE_VALUE;

  fAbortEvent             := CreateEvent(nil, true, false, nil);
  fReadOverlapped.hEvent  := CreateEvent(nil, true, false, nil);
  fWriteOverlapped.hEvent := CreateEvent(nil, true, false, nil);

  fReadEventHandles[0]  := fAbortEvent;
  fReadEventHandles[1]  := fReadOverlapped.hEvent;
  fWriteEventHandles[0] := fAbortEvent;
  fWriteEventHandles[1] := fWriteOverlapped.hEvent;

end;


destructor TmeComPort.destroy;
begin
  fEventThread.Free;

  CheckCloseHandle(fComHandle);
  CheckCloseHandle(fAbortEvent);
  CheckCloseHandle(fReadOverlapped.hEvent);
  CheckCloseHandle(fWriteOverlapped.hEvent);

  inherited destroy;
end;


function TmeComPort.GetComEvent(const aIndex : integer) : TNotifyEvent;
begin
  result := fComEvent[aIndex];
end;


procedure TmeComPort.SetComEvent(const aIndex : integer; aValue : TNotifyEvent);
begin
  fComEvent[aIndex] := aValue;
  if assigned(aValue)
    then fEvtMask := fEvtMask or (1 shl aIndex)
    else fEvtMask := fEvtMask and not (1 shl aIndex);
end;


procedure TmeComPort.Open(const aPortNum : integer);
begin
  if aPortNum <> fPortNum then
  begin
    fEventThread.Free;
    fEventThread := TmeComThread.create(self);
    fPortNum := aPortNum;
    ResetPort;
  end;
end;


procedure TmeComPort.ResetPort;
var
  ComString : string;
begin
  Close;
  ResetEvent(fAbortEvent);
  ComString := 'COM' + IntToStr(fPortNum);
  fComHandle := CreateFile
    (
    pchar(ComString),
    GENERIC_READ or GENERIC_WRITE,
    0,
    nil,
    OPEN_EXISTING,
    FILE_FLAG_OVERLAPPED,
    0
    );
  if fComHandle = INVALID_HANDLE_VALUE
    then RaiseLastWin32Error;
  SetupState;
  fEventThread.ComHandle := fComHandle;
  fConnected := true;
end;


procedure TmeComPort.Close;
begin
  if fConnected then
  begin
    PulseEvent(fAbortEvent);
    CheckCloseHandle(fComHandle);
    fConnected := False;
  end;
end;


procedure TmeComPort.SetupState;
var
  DCB: TDCB;
  Timeouts: TCommTimeouts;
begin
  FillChar(DCB, SizeOf(DCB), 0);

  with DCB do
  begin
    DCBlength := SizeOf(DCB);
    BaudRate  := fBaudRate;
    Flags     := ord(fBinary) or
                 (ord(fParityCheck) shl 1) or
                 (ord(fOutxCTSFlow) shl 2) or
                 (ord(fOutxDSRFlow) shl 3) or
                 (ord(fDTRControl)  shl 4) or
                 (ord(fDSRSensitivity) shl 6) or
                 (ord(fTXContinueOnXOFF) shl 7) or
                 (ord(fOutX) shl 8) or
                 (ord(fInX) shl 9) or
                 (ord(fReplaceError) shl 10) or
                 (ord(fStripNull) shl 11) or
                 (ord(fRTSControl) shl 12) or
                 (ord(fAbortOnError) shl 14);
    XOFFLim   := fXOFFLim;
    ByteSize  := fByteSize;
    Parity    := ord(fParity);
    StopBits  := ord(fStopBits);
    XONChar   := fXONChar;
    XOFFChar  := fXOFFChar;
    ErrorChar := fErrorChar;
    EOFChar   := fEOFChar;
    EvtChar   := fEvtChar;
  end;

  Win32Check( SetCommState(fComHandle, DCB ));

  with Timeouts do
  begin
    ReadIntervalTimeout         := fReadIntervalTimeout;
    ReadTotalTimeoutMultiplier  := fReadTotalTimeoutMultiplier;
    ReadTotalTimeoutConstant    := fReadTotalTimeoutConstant;
    WriteTotalTimeoutMultiplier := fWriteTotalTimeoutMultiplier;
    WriteTotalTimeoutConstant   := fWriteTotalTimeoutConstant;
  end;

  Win32Check( SetCommTimeouts(fComHandle, Timeouts ));
  Win32Check( SetupComm(fComHandle, cReadBufSize, cWriteBufSize ));
end;


function TmeComPort.InQue: Integer;
var
  Errors: DWORD;
  ComStat: TComStat;
begin
  Win32Check( ClearCommError(fComHandle, Errors, @ComStat ));
  Result := ComStat.cbInQue;
end;


function TmeComPort.OutQue: integer;
var
  Errors: DWORD;
  ComStat: TComStat;
begin
  Win32Check( ClearCommError(fComHandle, Errors, @ComStat ));
  Result := ComStat.cbOutQue;
end;


function TmeComPort.WriteString(const aValue : string): Boolean;
var
  BytesWritten : DWORD;
  len : DWORD;
  dwSignal : DWORD;
  gle : integer;
begin
  result := false;
  begin
    if not fConnected then exit;
    len := length( aValue );
    if WriteFile( fComHandle, aValue[1], len, BytesWritten, @fWriteOverlapped )
    then
      result := True
    else
    begin
      gle := GetLastError;
      if gle = ERROR_IO_PENDING then
      begin
        dwSignal :=
          WaitForMultipleObjects(2, @fWriteEventHandles, false, INFINITE);
        if dwSignal = WAIT_OBJECT_0 + 1 then
        begin
//          ResetEvent( fWriteOverlapped.hEvent );
          Win32Check( GetOverlappedResult(fComHandle, fWriteOverlapped, BytesWritten, True ));
          result := BytesWritten = len
        end
      end
      else
      begin
        {$ifdef USE_CODESITE}
        // CodeSite.SendString( 'Write - WriteFile', SysErrorMessage( gle ));
        {$endif}
      end;
    end;
  end;
end;


function TmeComPort.ReadBuffer( var Buffer; Count: integer ) : integer;
var
  BytesRead : DWORD;
  dwSignal : DWORD;
  gle : integer;
begin
  result := 0;
  begin
    if not fConnected then exit;
    if ReadFile( fComHandle, Buffer, Count, BytesRead, @fReadOverlapped ) then
    begin
      result := BytesRead;
    end
    else
    begin
      gle := GetLastError;
      if gle = ERROR_IO_PENDING then
      begin
        dwSignal :=
          WaitForMultipleObjects( 2, @fReadEventHandles, false, INFINITE );
        if dwSignal = WAIT_OBJECT_0 + 1 then
        begin
//          ResetEvent( fReadOverlapped.hEvent );
          Win32Check( GetOverlappedResult( fComHandle, fReadOverlapped, BytesRead, True ));
          result := BytesRead
        end
      end
      else
      begin
        {$ifdef USE_CODESITE}
        // CodeSite.SendString( 'Read - ReadFile', SysErrorMessage( gle ));
        {$endif}
      end;
    end;
  end;
end;


procedure TmeComPort.PurgeIn;
begin
  Win32Check( PurgeComm( fComHandle, PURGE_RXABORT or PURGE_RXCLEAR ));
end;


procedure TmeComPort.PurgeOut;
begin
  Win32Check( PurgeComm(fComHandle, PURGE_TXABORT or PURGE_TXCLEAR ));
end;

procedure TmeComPort.SetDTR( const aOn : Boolean );
begin
  if aOn
    then EscapeCommFunction( fComHandle, windows.SETDTR )
    else EscapeCommFunction( fComHandle, windows.CLRDTR );
end;

procedure TmeComPort.SetRTS( const aOn : Boolean );
begin
  if aOn
    then EscapeCommFunction( fComHandle, windows.SETRTS )
    else EscapeCommFunction( fComHandle, windows.CLRRTS );
end;

procedure TmeComPort.SetBreak( const aOn : Boolean );
begin
  if aOn
    then EscapeCommFunction( fComHandle, windows.SETBREAK )
    else EscapeCommFunction( fComHandle, windows.CLRBREAK );
end;

procedure TmeComPort.SetXON( const aOn : Boolean );
begin
  if aOn
    then EscapeCommFunction( fComHandle, windows.SETXON )
    else EscapeCommFunction( fComHandle, windows.SETXOFF );
end;

{ ---------------------------------------------------------------------------- }
{ - TmeComThread ------------------------------------------------------------- }
{ ---------------------------------------------------------------------------- }

constructor TmeComThread.create(aOwner : TmeComPort);
begin
  inherited create(false);
  fOwner := aOwner;
  fPauseEvent := INVALID_HANDLE_VALUE;
  fOverlapped.hEvent := INVALID_HANDLE_VALUE;
  fPauseEvent := CreateEvent(nil, true, false, nil);
  fOverlapped.hEvent := CreateEvent(nil, true, false, nil);
  fEventHandles[0] := fPauseEvent;
  fEventHandles[1] := fOverlapped.hEvent;
end;

destructor TmeComThread.Destroy;
begin
  Terminate;
  PulseEvent(fPauseEvent);
  CheckCloseHandle(fOverlapped.hEvent);
  CheckCloseHandle(fPauseEvent);
  inherited destroy;
end;


procedure TmeComThread.SetComHandle(aComHandle : THandle);
begin
  fComHandle := aComHandle;
  Resume;
  Win32Check( SetCommMask(fComHandle, fOwner.fEvtMask ));
  PulseEvent(fPauseEvent);
end;


procedure TmeComThread.Execute;
var
  dwSignaled, BytesTrans : DWORD;
  gle : integer;
begin
  repeat
    if WaitCommEvent( fComHandle, fEventMask, @fOverlapped ) then
    begin
      Synchronize(DoEvents)
    end
    else
    begin
      gle := GetLastError;
      if gle = ERROR_IO_PENDING then
      begin
        dwSignaled := WaitForMultipleObjects( 2, @fEventHandles, False, INFINITE );
        if dwSignaled = WAIT_OBJECT_0 + 1 then
        begin
//          ResetEvent( fOverlapped.hEvent );
          Win32Check( GetOverlappedResult( fComHandle, fOverlapped, BytesTrans, True ));
          Synchronize( DoEvents );
        end
      end
      else
      begin
        // raise Exception.Create( SysErrorMessage( gle ));
      end;
    end;
  until Terminated;
end;


procedure TmeComThread.DoEvents;
var
  i, j : integer;
begin
  j := 1;
  for i := EvtRxChar to EvtRing do
  begin
    if ((fEventMask and j) <> 0) and assigned(fOwner.fComEvent[i]) then
    begin
      fOwner.fComEvent[i](fOwner);
    end;
    j := j shl 1;
  end;
end;


{ ---------------------------------------------------------------------------- }
{ - register ----------------------------------------------------------------- }
{ ---------------------------------------------------------------------------- }

procedure register;
begin
  RegisterComponents('Pegasus', [TmeComPort]);
end;

end.
